#include <stdio.h>
#include <malloc.h>

#include "global.h"

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263newPicStruct -- Allocate and init a new picture struct
 *
 * Author:           Niko Faerber
 *
 * Created:          07-March-97	
 *	
 * Purpose:          Allocate and init a new picture struct, that will be
 *                   filled with data by the H.263 demultiplexer. Note,
 *                   that this function does not allocate memory for the
 *                   actual data, which is done by H263allocPicStruct()
 *                   after the source format is known.
 *                   Typically, H263newPicStruct() is only called once for a
 *                   whole sequence, while H263allocPicStruct() and
 *                   H263freePicStruct() are called for each frame.
 *
 * Arguments in:     -
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    Pointer to the new picture struct on success
 *                   NULL on error (failed malloc)
 *
 * Example:          pic = H263newPicStruct();
 *
 * Side effects:     -
 *
 * Description:      Allocates a structure of type H263picStruct and
 *                   inits variables that need to have special values for
 *                   the very first frame decoded (like previousGFID=-1 to
 *                   indicate that there were no previous value for GFID).
 *
 * See also:         H263allocPicStruct(), H263freePicStruct()
 *	
 * Modified:         -
 *	
 *****************************************************************************/
H263picStruct *H263newPicStruct(void)
/***********************************************************CommentEnd********/
{
  H263picStruct *pic;

  pic = (H263picStruct *)malloc(sizeof(H263picStruct));
  if(pic==NULL) return NULL;

  pic->previousPTYPE=-1;  /* to indicate no previous PTYPE/GFID */
  pic->previousGFID=-1;   /* for GFID-Check */

  return(pic);  
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263allocPicStruct -- Allocate GOBs, MBs and BLKs of a picture struct
 *
 * Author:           Niko Faerber
 *
 * Created:          07-March-97	
 *	
 * Purpose:          H263allocPicStruct() is typically called each time
 *                   after a picture header is read. At this time, the source
 *                   format and picture type is available, and the appropriate
 *                   number of GOBs, MBs, MVs and BLKs can be allocated.
 *                   H263allocPicStruct() relies on a propper initialized
 *                   picture struct with valid values for numberOfGOBs,
 *                   numberOfMBs, numberOfMVs and numberOfBLKs. 
 *
 * Arguments in:     pic->numberOfGOBs
 *                   pic->numberOfMBs
 *                   pic->numberOfMVs 
 *                   pic->numberOfBLKs
 *
 * Arguments in/out: pic->gob[i] : pointers to allocated memory 
 *                   pic->mb[i]  :     ''    
 *
 * Arguments out:    -
 *
 * Return values:    TRUE  (1) on success
 *                   FALSE (0) on error (failed calloc)
 *
 * Example:          H263allocPicStruct(pic);
 *
 * Side effects:     -
 *
 * Description:      Allocates the GOB, MB and BLK structs from a picture
 *                   struct using the subroutines H263allocMbStruct() and
 *                   H263allocBlkStruct().
 *
 * See also:         H263allocMbStruct(), H263allocBlkStruct()
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int H263allocPicStruct(H263picStruct *pic)
/***********************************************************CommentEnd********/
{
  int i;
  int numberOfLevels=NUMBER_OF_LEVELS;

  /* allocate gob headers
   */
  pic->gob = (H263gobStruct **)calloc(pic->numberOfGOBs,
				      sizeof(H263gobStruct *));
  if(pic->gob==NULL) return 0;
  for(i=0;i<pic->numberOfGOBs;i++) {
    pic->gob[i] = (H263gobStruct *)calloc(1,sizeof(H263gobStruct));
    if(pic->gob[i]==NULL) return 0;
  }
  /* allocate mb data
   */
  pic->mb = (H263mbStruct **)calloc(pic->numberOfMBs*pic->numberOfGOBs,
				    sizeof(H263mbStruct *));
  if(pic->mb==NULL) return 0;
  for(i=0;i<pic->numberOfMBs*pic->numberOfGOBs;i++) {
    pic->mb[i] = H263allocMbStruct(pic->numberOfMVs, pic->numberOfBLKs, 
				   numberOfLevels);
    if(pic->mb[i]==NULL) return 0;
  }

  return 1;
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263freePicStruct -- Free GOBs, MBs and BLKs in a picture struct
 *
 * Author:           Niko Faerber
 *
 * Created:          08-March-97	
 *	
 * Purpose:          Free the data of a picture struct (not the struct itself).
 *
 * Arguments in:     -
 *
 * Arguments in/out: pic : picture struct
 *
 * Arguments out:    -
 *
 * Return values:    TRUE (1) on success
 *
 * Example:          H263freePicStruct(pic);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263AllocPicStruct()
 *	
 * Modified:         -
 *	
 *****************************************************************************/
int H263freePicStruct(H263picStruct *pic)
/***********************************************************CommentEnd********/
{
  int g, m, b, v;

  /* free GOB headers
   */
  for(g=0;g<pic->numberOfGOBs;g++) {
    free(pic->gob[g]);
  }
  free(pic->gob);

  /* free MB data
   */
  for(m=0;m<pic->numberOfMBs*pic->numberOfGOBs;m++) {
    /* free LEVELs
     */
    for(b=0;b<pic->numberOfBLKs;b++) {
      free(pic->mb[m]->blk[b]->level);
    }
    /* free BLKs
     */
    for(b=0;b<pic->numberOfBLKs;b++) {
      free(pic->mb[m]->blk[b]);
    }
    free(pic->mb[m]->blk);
    /* free MVs
     */
    for(v=0;v<pic->numberOfMVs;v++) {
      free(pic->mb[m]->mv[v]);
    }
    free(pic->mb[m]->mv);
    /* free PVs
     */
    for(v=0;v<pic->numberOfMVs;v++) {
      free(pic->mb[m]->pv[v]);
    }
    free(pic->mb[m]->pv);
    /* free CBP
     */
    free(pic->mb[m]->cbp);
    /* free MB
     */
    free(pic->mb[m]);
  }
  free(pic->mb);

  return 1;
}

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263copyPicStruct -- copy a picture struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          ???
 *	
 * Purpose:          copy the content of picture struct 1 to picture struct 2
 *
 * Arguments in:     pic1 
 *
 * Arguments in/out: -
 *
 * Arguments out:    pic2
 *
 * Return values:    -
 *
 * Example:          H263copyPicStruct(pic1, pic2);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263copyGobStruct(), H263copyMbStruct(),
 *                   H263copyBlkStruct()
 *	
 * Modified:         08-March-97, Niko Faerber, added Comment
 *	
 *****************************************************************************/
void H263copyPicStruct(H263picStruct *pic1, H263picStruct *pic2) 
/***********************************************************CommentEnd********/
{
  int i;

  pic2->currentQuant = pic1->currentQuant;
  pic2->tr = pic1->tr;
  pic2->ptype = pic1->ptype;
  pic2->pquant = pic1->pquant;
  pic2->cpm = pic1->cpm;
  pic2->psbi = pic1->psbi;
  pic2->trb = pic1->trb;
  pic2->dbquant = pic1->dbquant;
  pic2->previousPTYPE = pic1->previousPTYPE;
  pic2->previousGFID = pic1->previousGFID;
  pic2->splitScreenIndicator = pic1->splitScreenIndicator; 
  pic2->documentCameraIndicator = pic1->documentCameraIndicator;
  pic2->freezePictureRelease = pic1->freezePictureRelease; 
  pic2->sourceFormat = pic1->sourceFormat;            
  pic2->pictureCodingType = pic1->pictureCodingType;  
  pic2->umv = pic1->umv;                     
  pic2->sac = pic1->sac;                     
  pic2->ap = pic1->ap;                      
  pic2->pb = pic1->pb;                      
  pic2->widthFrame = pic1->widthFrame;
  pic2->heightFrame = pic1->heightFrame;
  pic2->numberOfGOBs = pic1->numberOfGOBs;
  pic2->numberOfMBs = pic1->numberOfMBs;    
  pic2->numberOfBLKs = pic1->numberOfBLKs;
  pic2->numberOfMVs = pic1->numberOfMVs;
  
  H263allocPicStruct(pic2);
  
  for(i=0;i<pic1->numberOfGOBs;i++) {
    H263copyGobStruct(pic1->gob[i], pic2->gob[i]);
  }
  for(i=0;i<pic1->numberOfMBs*pic1->numberOfGOBs;i++) {    
    H263copyMbStruct(pic1->numberOfBLKs, pic1->numberOfMVs, 
		     pic1->mb[i], pic2->mb[i]);
  }  
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263copyGobStruct -- copy a GOB struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          ???
 *	
 * Purpose:          copy the content of GOB struct 1 to GOB struct 2
 *
 * Arguments in:     gob1 
 *
 * Arguments in/out: -
 *
 * Arguments out:    gob2
 *
 * Return values:    -
 *
 * Example:          H263copyGobStruct(gob1, gob2);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263copyPicStruct(), H263copyMbStruct(), H263copyBlkStruct()
 *	
 * Modified:         08-March-97, Niko Faerber, added Comment
 *	
 *****************************************************************************/
void H263copyGobStruct(H263gobStruct *gob1, H263gobStruct *gob2) 
/***********************************************************CommentEnd********/
{
  gob2->gbscPresent = gob1->gbscPresent;
  gob2->gn = gob1->gn;
  gob2->gquant = gob1->gquant;
  gob2->gsbi = gob1->gsbi;
  gob2->gfid = gob1->gfid;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263allocMbStruct -- Allocate a macroblock struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          07-March-97	
 *	
 * Purpose:          Allocate a macroblockblock struct
 *
 * Arguments in:     numberOfMVs : number of motion vectors (usually 0,1,4 or 5)
 *                   numberOfBLKs : number of blocks (usually 6 or 12)
 *                   numberOfLevels : number of levels (usually 64)
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    Pointer to the new macrblock struct on success
 *                   NULL on error (failed calloc)
 *
 * Example:          mb = H263allocMbStruct(1,6,64);
 *
 * Side effects:     -
 *
 * Description:      Allocates a structure of type H263mbStruct with
 *                   the specified number of motion vectors, blocks and levels.
 *
 * See also:         H263allocBlkStruct(), H263allocPicStruct()
 *	
 * Modified:         -
 *	
 *****************************************************************************/
H263mbStruct *H263allocMbStruct(int numberOfMVs, int numberOfBLKs, 
				int numberOfLevels)
/***********************************************************CommentEnd********/
{
  int i;
  H263mbStruct *mb = (H263mbStruct *)calloc(1,sizeof(H263mbStruct));
  if(mb==NULL) return 0;

  /* allocate CBP */
  mb->cbp = (int *)calloc(numberOfBLKs,sizeof(int));
  if(mb->cbp==NULL) return 0;

  /* allocate MVs */
  mb->mv = (MVector **)malloc(numberOfMVs*sizeof(MVector *));    
  if(mb->mv==NULL) return 0;
  mb->pv = (MVector **)malloc(numberOfMVs*sizeof(MVector *));    
  if(mb->pv==NULL) return 0;
  for(i=0;i<numberOfMVs;i++) {
    mb->mv[i] = AllocMVector();
    ResetMVector(mb->mv[i]);
    mb->mv[i]->sF = 2;
    mb->pv[i] = AllocMVector();
    ResetMVector(mb->pv[i]);
    mb->pv[i]->sF = 2;
  }  

  /* allocate BLKs */
  mb->blk = (H263blkStruct **)calloc(numberOfBLKs,sizeof(H263blkStruct *));
  if(mb->blk==NULL) return 0;
  for(i=0;i<numberOfBLKs;i++) {
    mb->blk[i] = H263allocBlkStruct(numberOfLevels);
    if(mb->blk[i]==NULL) return 0;
  }  

  return(mb);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263copyMbStruct -- copy a MB struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          ???
 *	
 * Purpose:          copy the content of MB struct 1 to MB struct 2
 *
 * Arguments in:     numberOfBLKs
 *                   numberOfMVs
 *                   mb1 
 *
 * Arguments in/out: -
 *
 * Arguments out:    mb2
 *
 * Return values:    -
 *
 * Example:          H263copyMbStruct(mb1, mb2);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263copyPicStruct(), H263copyGOBStruct(), H263copyBlkStruct()
 *	
 * Modified:         08-March-97, Niko Faerber, added Comment
 *	
 *****************************************************************************/
void H263copyMbStruct(int numberOfBLKs, int numberOfMVs,
		      H263mbStruct *mb1, H263mbStruct *mb2) 
/***********************************************************CommentEnd********/
{
  int i;

  mb2->type = mb1->type;
  mb2->quant = mb1->quant;
  for(i=0;i<numberOfBLKs;++i) {
    mb2->cbp[i] = mb1->cbp[i];
    H263copyBlkStruct(mb1->blk[i],mb2->blk[i]); 
  }
  for(i=0;i<numberOfMVs;++i) {
    CopyMVector(mb1->mv[i],mb2->mv[i]); 
    CopyMVector(mb1->pv[i],mb2->pv[i]); 
  }

  mb2->rate = mb1->rate;
  mb2->dist = mb1->dist;
}

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263allocBlkStruct -- Allocate a block struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          07-March-97	
 *	
 * Purpose:          Allocate a block struct
 *
 * Arguments in:     numberOfLevels : number of levels (usually 64)
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    Pointer to the new block struct on success
 *                   NULL on error (failed calloc)
 *
 * Example:          blk = H263allocBlkStruct(64);
 *
 * Side effects:     -
 *
 * Description:      Allocates a structure of type H263blkStruct with
 *                   the specified number of levels (usually 64).
 *
 * See also:         H263allocMbStruct(), H263allocPicStruct()
 *	
 * Modified:         -
 *	
 *****************************************************************************/
H263blkStruct *H263allocBlkStruct(int numberOfLevels)
/***********************************************************CommentEnd********/
{
  H263blkStruct *blk = (H263blkStruct *)calloc(1,sizeof(H263blkStruct));
  
  /* allocate LEVELs */
  blk->level = (int *)calloc(numberOfLevels,sizeof(int));
  if(blk->level==NULL) return 0;      

  return(blk);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263copyBlkStruct -- copy a block struct
 *
 * Author:           Thomas Wiegand
 *
 * Created:          ???
 *	
 * Purpose:          copy the content of block struct 1 to block struct 2
 *
 * Arguments in:     blk1 
 *
 * Arguments in/out: -
 *
 * Arguments out:    blk2
 *
 * Return values:    -
 *
 * Example:          H263copyBlkStruct(pic1, pic2);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263copyPicStruct(), H263copyGobStruct(), H263copyMbStruct()
 *	
 * Modified:         08-March-97, Niko Faerber, added Comment
 *	
 *****************************************************************************/
void H263copyBlkStruct(H263blkStruct *blk1, H263blkStruct *blk2) 
/***********************************************************CommentEnd********/
{
  int i;

  blk2->intradc = blk1->intradc;

  for(i=0;i<NUMBER_OF_LEVELS;++i) {
    blk2->level[i] = blk1->level[i]; 
  }
}

