/***********************************************************HeaderBegin*******
 *                                                                         
 * File: idct.c
 *
 * Author: Telenor, TMN
 * Created: 
 *                                                                         
 * Description: inverse discrete cosine transform
 *
 * Notes:  
 *
 * Modified: T.W., adopted from MoMuSys VM 96Nov07
 *
 ***********************************************************HeaderEnd*********/


/************************    INCLUDE FILES    ********************************/
#define ZIGZAG
#include "global.h"
#include <math.h>


#define W1 2841 /* 2048*sqrt(2)*cos(1*pi/16) */
#define W2 2676 /* 2048*sqrt(2)*cos(2*pi/16) */
#define W3 2408 /* 2048*sqrt(2)*cos(3*pi/16) */
#define W5 1609 /* 2048*sqrt(2)*cos(5*pi/16) */
#define W6 1108 /* 2048*sqrt(2)*cos(6*pi/16) */
#define W7 565  /* 2048*sqrt(2)*cos(7*pi/16) */


/***********************************************************CommentBegin******
 ****************************************************************************
 *
 * -- IdctFastRow -- fast inverse DCT-transform of a row
 *
 * Author :            Robert Danielsen
 *
 * Purpose :           Fast inverse DCT-transform of a block row with
 *                     fixed point arithmetic.
 *                     Two dimensional inverse discrete cosine transform.
 *
 * Arguments in :      -
 *                     
 * Arguments in/out :  Short *block : pointer to a block
 *
 * Arguments out :     -
 * 
 * Return values :     none
 *
 * Side effects :      none
 *
 * Description :       row (horizontal) IDCT
 *
 *                                 7                       pi         1
 *                       dst[k] = sum c[l] * src[l] * cos( -- * ( k + - ) * l )
 *                                l=0                      8          2
 *
 *                     where: c[0]    = 128
 *                            c[1..7] = 128*sqrt(2)
 *
 * See also :          IdctRef, IdctFastFloat, IdctFast, IdctFastClipCol
 *
 * Modified :          T.W., 11-Nov-1996, Adopted from Momusys VM
 *                     K.S., 23-Jun-1997, Adopted towards CIDS
 *
 ****************************************************************************/
Void IdctFastRow(Short *blk)
/***********************************************************CommentEnd********/
{
  Int x0, x1, x2, x3, x4, x5, x6, x7, x8;


  /* shortcut */
  if (!((x1 = blk[4]<<11) | (x2 = blk[6]) | (x3 = blk[2]) |
        (x4 = blk[1]) | (x5 = blk[7]) | (x6 = blk[5]) | (x7 = blk[3])))
  {
    blk[0]=blk[1]=blk[2]=blk[3]=blk[4]=blk[5]=blk[6]=blk[7]=blk[0]<<3;
    return;
  }

  x0 = (blk[0]<<11) + 128; /* for proper rounding in the fourth stage */

  /* first stage */
  x8 = W7*(x4+x5);
  x4 = x8 + (W1-W7)*x4;
  x5 = x8 - (W1+W7)*x5;
  x8 = W3*(x6+x7);
  x6 = x8 - (W3-W5)*x6;
  x7 = x8 - (W3+W5)*x7;
  
  /* second stage */
  x8 = x0 + x1;
  x0 -= x1;
  x1 = W6*(x3+x2);
  x2 = x1 - (W2+W6)*x2;
  x3 = x1 + (W2-W6)*x3;
  x1 = x4 + x6;
  x4 -= x6;
  x6 = x5 + x7;
  x5 -= x7;
  
  /* third stage */
  x7 = x8 + x3;
  x8 -= x3;
  x3 = x0 + x2;
  x0 -= x2;
  x2 = (181*(x4+x5)+128)>>8;
  x4 = (181*(x4-x5)+128)>>8;
  
  /* fourth stage */
  blk[0] = (x7+x1)>>8;
  blk[1] = (x3+x2)>>8;
  blk[2] = (x0+x4)>>8;
  blk[3] = (x8+x6)>>8;
  blk[4] = (x8-x6)>>8;
  blk[5] = (x0-x4)>>8;
  blk[6] = (x3-x2)>>8;
  blk[7] = (x7-x1)>>8;
}


/***********************************************************CommentBegin******
 ****************************************************************************
 *
 * -- IdctFastColClip -- fast inverse DCT-transform of a column and clipping
 *
 * Author :            Robert Danielsen
 *
 * Purpose :           Fast inverse DCT-transform of a block column with
 *                     fixed point arithmetic.
 *                     One dimensional vertical inverse discrete cosine
 *                     transform.
 *                     The result is clipped to the range -256...255.
 *
 * Arguments in :      -
 *                     
 * Arguments in/out :  Short *block : pointer to a block
 *
 * Arguments out :     -
 * 
 * Return values :     none
 *
 * Side effects :      none
 *
 * Description :       column (vertical) IDCT
 *
 *                              7                         pi         1
 *                  dst[8*k] = sum c[l] * src[8*l] * cos( -- * ( k + - ) * l )
 *                             l=0                        8          2
 *
 *                     where: c[0]    = 1/1024
 *                            c[1..7] = (1/1024)*sqrt(2)
 *
 * See also :          IdctRef, IdctFastFloat, IdctFast, IdctFastRow
 *
 * Modified :          T.W., 11-Nov-1996, Adopted from Momusys VM
 *                     K.S., 23-Jun-1997, Adopted towards CIDS
 *
 ****************************************************************************/
Void IdctFastColClip(Short *blk)
/***********************************************************CommentEnd********/
{
  Int          x0, x1, x2, x3, x4, x5, x6, x7, x8;
  Int          i;
  static Int   virgin = 1;
  static Short iclip[1024]; /* clipping table */
  static Short *iclp;


  if (virgin) {
    iclp = iclip + 512;
    for (i= -512; i <512; i++)
      iclp[i] = (i<-256) ? -256 : ((i>255) ? 255 : i);
    virgin = 0;
  }

  /* shortcut */
  if (!((x1 = (blk[8*4]<<8)) | (x2 = blk[8*6]) | (x3 = blk[8*2]) |
        (x4 = blk[8*1]) | (x5 = blk[8*7]) | (x6 = blk[8*5]) | (x7 = blk[8*3])))
  {
    blk[8*0]=blk[8*1]=blk[8*2]=blk[8*3]=blk[8*4]=blk[8*5]=blk[8*6]=blk[8*7]=
      iclp[(blk[8*0]+32)>>6];
    return;
  }

  x0 = (blk[8*0]<<8) + 8192;

  /* first stage */
  x8 = W7*(x4+x5) + 4;
  x4 = (x8+(W1-W7)*x4)>>3;
  x5 = (x8-(W1+W7)*x5)>>3;
  x8 = W3*(x6+x7) + 4;
  x6 = (x8-(W3-W5)*x6)>>3;
  x7 = (x8-(W3+W5)*x7)>>3;
  
  /* second stage */
  x8 = x0 + x1;
  x0 -= x1;
  x1 = W6*(x3+x2) + 4;
  x2 = (x1-(W2+W6)*x2)>>3;
  x3 = (x1+(W2-W6)*x3)>>3;
  x1 = x4 + x6;
  x4 -= x6;
  x6 = x5 + x7;
  x5 -= x7;
  
  /* third stage */
  x7 = x8 + x3;
  x8 -= x3;
  x3 = x0 + x2;
  x0 -= x2;
  x2 = (181*(x4+x5)+128)>>8;
  x4 = (181*(x4-x5)+128)>>8;
  
  /* fourth stage */
  blk[8*0] = iclp[(x7+x1)>>14];
  blk[8*1] = iclp[(x3+x2)>>14];
  blk[8*2] = iclp[(x0+x4)>>14];
  blk[8*3] = iclp[(x8+x6)>>14];
  blk[8*4] = iclp[(x8-x6)>>14];
  blk[8*5] = iclp[(x0-x4)>>14];
  blk[8*6] = iclp[(x3-x2)>>14];
  blk[8*7] = iclp[(x7-x1)>>14];
}


/***********************************************************CommentBegin******
 ****************************************************************************
 *
 * -- IdctFast -- fast inverse DCT-transform with fixed point arith.
 *
 * Author :            Robert Danielsen
 *
 * Purpose :           Fast inverse DCT-transform with fixed point
 *                     arithmetic.
 *                     Two dimensional inverse discrete cosine transform.
 *
 * Arguments in :      Int *coeff : pointer to coefficients in
 *                                  zigzag-scan order.
 *                     
 * Arguments in/out :  none
 *
 * Arguments out :     Short *block : pointer to a block.
 * 
 * Return values :     none
 *
 * Side effects :      none
 *
 * Description :       Descans zigzag-scanned coefficients and does
 *                     inverse DCT-transform on 64 coefficients.
 *                     
 *
 * See also :          IdctRef, IdctFastFloat, IdctFastRow, IdctFastColClip
 *
 * Modified :          T.W., 11-Nov-1996, Adopted from Momusys VM
 *
 ****************************************************************************/
Void IdctFast(Int *coeff, Short *block)
/***********************************************************CommentEnd********/
{
  Int          i, j;


  /* Inverse zigzag-scanning */
  for (j = 0; j < 8; ++j) {
    for (i = 0; i < 8; ++i) {
      block[j * 8 + i] = (Short)(coeff[zigzag[j][i]]);
    }
  }

  /* Do DCTs */
  for (i = 0; i < 8; i++)
    IdctFastRow(block + 8 * i);
  for (i = 0; i < 8; i++)
    IdctFastColClip(block + i);
}


/***********************************************************CommentBegin******
 ****************************************************************************
 *
 * -- IdctFastFloat -- fast inverse DCT-transform with floating point arith.
 *
 * Author :            Robert Danielsen
 *
 * Purpose :           Fast inverse DCT-transform with floating point
 *                     arithmetic.
 * 
 * Arguments in :      Int *coeff : pointer to coefficients in
 *                                  zigzag-scan order.
 *                     
 * Arguments in/out :  none
 *
 * Arguments out :     Int *block : pointer to a block.
 * 
 * Return values :     none
 *
 * Side effects :      none
 *
 * Description :       Descans zigzag-scanned coefficients and does
 *                     inverse DCT-transform on 64 coefficients.
 *                     
 *
 * See also :          IdctRef, IdctFast
 *
 * Modified :          T.W., 11-Nov-1996, Adopted from Momusys VM
 *
 ****************************************************************************/
Void IdctFastFloat(Int *coeff, Int *block) 
/***********************************************************CommentEnd********/
{
  Int                j1, i, j;
  Double b[8], b1[8], dd[8][8];
  Double f0=.7071068;
  Double f1=.4903926;
  Double f2=.4619398;
  Double f3=.4157348;
  Double f4=.3535534;
  Double f5=.2777851;
  Double f6=.1913417;
  Double f7=.0975452;
  Double e, f, g, h;

  /* Horizontal */

  /* Descan coefficients first */

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      b[j] = coeff[zigzag[i][j]];
    }
    e = b[1] * f7 - b[7] * f1;
    h = b[7] * f7 + b[1] * f1;
    f = b[5] * f3 - b[3] * f5;
    g = b[3] * f3 + b[5] * f5;

    b1[0] = (b[0] + b[4]) * f4;
    b1[1] = (b[0] - b[4]) * f4;
    b1[2] = b[2] * f6 - b[6] * f2;
    b1[3] = b[6] * f6 + b[2] * f2;
    b[4] = e + f;
    b1[5] = e - f;
    b1[6] = h - g;
    b[7] = h + g;
    
    b[5] = (b1[6] - b1[5]) * f0;
    b[6] = (b1[6] + b1[5]) * f0;
    b[0] = b1[0] + b1[3];
    b[1] = b1[1] + b1[2];
    b[2] = b1[1] - b1[2];
    b[3] = b1[0] - b1[3];

    for (j = 0; j < 4; j++) {
      j1 = 7 - j;
      dd[i][j] = b[j] + b[j1];
      dd[i][j1] = b[j] - b[j1];
    }
  }

  /* Vertical */
  
  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      b[j] = dd[j][i];
    }
    e = b[1] * f7 - b[7] * f1;
    h = b[7] * f7 + b[1] * f1;
    f = b[5] * f3 - b[3] * f5;
    g = b[3] * f3 + b[5] * f5;

    b1[0] = (b[0] + b[4]) * f4;
    b1[1] = (b[0] - b[4]) * f4;
    b1[2] = b[2] * f6 - b[6] * f2;
    b1[3] = b[6] * f6 + b[2] * f2;
    b[4] = e + f;
    b1[5] = e - f;
    b1[6] = h - g;
    b[7] = h + g;

    b[5] = (b1[6] - b1[5]) * f0;
    b[6] = (b1[6] + b1[5]) * f0;
    b[0] = b1[0] + b1[3];
    b[1] = b1[1] + b1[2];
    b[2] = b1[1] - b1[2];
    b[3] = b1[0] - b1[3];

    for (j = 0; j < 4; j++) {
      j1 = 7 - j;
      dd[j][i] = b[j] + b[j1];
      dd[j1][i] = b[j] - b[j1];
    }
  }

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      block[i*8+j] = MNINT(dd[i][j]);
    }
  }
}


/***********************************************************CommentBegin******
 ****************************************************************************
 *
 * -- IdctRef -- inverse DCT-transform with Double precision
 *
 * Author :            unknown
 *
 * Purpose :           Inverse DCT-transform
 * 
 * Arguments in :      Double *coeff : pointer to coefficients in
 *                                     zigzag-scan order.
 *                     
 * Arguments in/out :  none
 * 
 * Arguments out :     Int *block : pointer to a block.
 * 
 * Return values :     none
 *
 * Side effects :      none
 *
 * Description :       Perform IDCT matrix multiply for 8x8 coefficient block.
 *                     Perform IEEE 1180 reference (64-bit floating point, 
 *                     separable 8x1 direct matrix multiply) Inverse Discrete 
 *                     Cosine Transform.
 *                     Here we use math.h to generate constants.  
 *                     Compiler results may vary a little.
 *
 * See also :
 *
 * Modified :          T.W., 11-Nov-1996, Adopted from Momusys VM
 *
 ****************************************************************************/
Void IdctRef(Double *coeff, Int *block) 
/***********************************************************CommentEnd********/
{
  Int i, j, k, v;
  Double partial_product;
  Double tmp[64];
  static Double c[8][8];
  static int virgin = TRUE;

  if(virgin) {
    Int freq, time;
    Double scale;

    virgin = FALSE;
    for (freq=0; freq < 8; freq++) {
      scale = (freq == 0) ? sqrt(0.125) : 0.5;
      
      for (time=0; time<8; time++) {
	c[freq][time] = scale*cos((PI/8.0)*freq*(time + 0.5));
      }
    }
  }

  for (i=0; i<8; i++) {
    for (j=0; j<8; j++) {     
      partial_product = 0.0;
      for (k=0; k<8; k++) {
	partial_product+= c[k][j] * coeff[zigzag[i][j]];
      }
      tmp[8*i+j] = partial_product;
    }	
  }
  
  /* Transpose operation is integrated into address mapping by switching 
     loop order of i and j */

  for (j=0; j<8; j++) {    
    for (i=0; i<8; i++) {
      partial_product = 0.0;
      for (k=0; k<8; k++) {
	partial_product+= c[k][i]*tmp[8*k+j];
      }
      v = (Int) floor (partial_product + 0.5);
      block[8*i+j] = (v<-256) ? -256 : ((v>255) ? 255 : v);
    }
  }
}
