/***********************************************************HeaderBegin*******
 *                                                                         
 * File:	motion.c
 *
 * Author:	K.S.
 * Created:	6-Aug-96
 *                                                                         
 * Description: Operations for Motion Structures.
 *
 * Notes: 	-
 *
 * Modified:	-
 *
 ***********************************************************HeaderEnd*********/

#include "global.h"

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AllocMVector -- Allocate memory for a MVector structure
 *
 * Author:             T.W., X.Z.
 *
 * Created:            8-Oct-96
 *	
 *
 * Purpose:            Allocate all memory for a motion structure.
 * 
 * Arguments in:       -
 *
 * Arguments in/out:   -	
 *	
 * Arguments out:      -
 *
 * Return values:      A pointer to the allocated MVector structure.
 *
 * Example:            mv = AllocMVector();
 *
 * Side effects:       -
 *
 * Description:        see above
 *
 * See also:           FreeMVector
 *	
 * Modified:	        -
 *
 *****************************************************************************/
MVector *AllocMVector(void)
/***********************************************************CommentEnd********/
{
  return(NEW(MVector));
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- FreeMVector -- Free memory of MVector structure
 *
 * Author:            T.W., X.Z.
 *	
 * Created:           8-Oct-96
 *
 * Purpose:	      Frees up all the memory associated with a MVector struct
 * 
 * Arguments in:      -
 *
 * Arguments in/out:  MVector **mv_p   pointer to the MVector struct to be 
 *                                    freed. The pointer is set to NULL.
 *
 * Arguments out:     -
 *
 * Return values:     -
 *
 * Example:           FreeMVector(&mv);
 *
 * Side effects:      -
 *
 * Description:       -
 *
 * See also:          AllocMVector
 *
 * Modified:          -
 *
 *****************************************************************************/
void FreeMVector(MVector **mv_p)
/***********************************************************CommentEnd********/
{
  free(*mv_p);
  *mv_p = NULL;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ResetMVector -- Reset an MVector
 *
 * Author:             K.S.
 *
 * Created:            7-Aug-96
 *	
 * Purpose:            Reset an MVector
 * 
 * Arguments in:       -
 *
 * Arguments in/out:   MVector *mv          MVector to be reset.
 *
 * Arguments out:      -
 *
 * Return values:      -
 *
 * Example:            ResetMVector(mv);
 *
 * Side effects:       -
 *	
 * Description:        Fullpel and subpel components of the vector are set
 *                     to 0.
 *                     The subpel-fraction 'sF' is not changed!
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
void ResetMVector(MVector *mv)
/***********************************************************CommentEnd********/
{
  mv->x = 0;
  mv->y = 0;
  mv->sx = 0;
  mv->sy = 0;
  mv->t = 0;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ZeroMVector -- Verify if a motion vector is zero
 *
 * Author:             K.S.
 *
 * Created:            19-Sep-96
 *	
 * Purpose:            Verifies if a motion vector ('MVector') is zero.
 * 
 * Arguments in:       MVector *mv          MVector to be verified.
 *
 * Arguments in/out:   -
 *
 * Arguments out:      -
 *
 * Return values:      0  if any component of 'mv' is not 0.
 *                     1  if all components of 'mv' are 0.
 *
 * Example:            zero = ZeroMVector(mv);
 *
 * Side effects:       -
 *	
 * Description:        The MVectors m is checked.
 *
 * See also:           ZeroMVector2D
 *
 * Modified:           23-May-97 K.S.: 't' is also checked now.
 *
 *****************************************************************************/
int ZeroMVector(MVector *mv)
/***********************************************************CommentEnd********/
{
  return(mv->x == 0 && mv->sx == 0 && mv->y == 0 && mv->sy == 0 && mv->t == 0);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ZeroMVector2D -- Verify if a motion vector is zero in 2 dimensions (x,y)
 *
 * Author:             K.S.
 *
 * Created:            23-May-97
 *	
 * Purpose:            Verifies if a motion vector ('MVector') is zero in
 *                     2 dimensions, i.e. if x, y, sx and sy are zero.
 *                     Note: the time displacement is not checked.
 * 
 * Arguments in:       MVector *mv          MVector to be verified.
 *
 * Arguments in/out:   -
 *
 * Arguments out:      -
 *
 * Return values:      0  if any component of 'mv' is not 0.
 *                     1  if all components of 'mv' are 0.
 *
 * Example:            zero = ZeroMVector2D(mv);
 *
 * Side effects:       -
 *	
 * Description:        The MVectors m is checked.
 *
 * See also:           ZeroMVector
 *
 * Modified:           -
 *
 *****************************************************************************/
int ZeroMVector2D(MVector *mv)
/***********************************************************CommentEnd********/
{
  return(mv->x == 0 && mv->sx == 0 && mv->y == 0 && mv->sy == 0);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AdjustSubpelMVector -- Adjust the subpel component of a motion vector
 *
 * Author:             K.S.
 *
 * Created:            22-May-97
 *	
 * Purpose:            Adjusts the subpel component of a motion vector
 *                     ('MVector') to the range 0,...,sF-1.
 * 
 * Arguments in:       MVector *mv          MVector to be adjusted.
 *
 * Arguments in/out:   -
 *
 * Arguments out:      -
 *
 * Return values:      1 if success
 *                     0 if error (e.g. sF <= 0)
 *
 * Example:            AdjustSubpelMVector(mv);
 *
 * Side effects:       -
 *	
 * Description:        -
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
int AdjustSubpelMVector(MVector *mv)
/***********************************************************CommentEnd********/
{
  if (mv->sF <= 0)
    return (0);

  while (mv->sx < 0) {
    --(mv->x);
    mv->sx += mv->sF;
  }
  while (mv->sy < 0) {
    --(mv->y);
    mv->sy += mv->sF;
  }
  while (mv->sx >= mv->sF) {
    ++(mv->x);
    mv->sx -= mv->sF;
  }
  while (mv->sy >= mv->sF) {
    ++(mv->y);
    mv->sy -= mv->sF;
  }

  return(1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ClipMVector -- Clip an MVector
 *
 * Author:             K.S.
 *
 * Created:            1-Sep-97
 *	
 * Purpose:            Clips an MVector to the specified range.
 * 
 * Arguments in:       int     minX         minimal x value.
 *                     int     maxX         maximal x value.
 *                     int     minY         minimal y value.
 *                     int     maxY         maximal y value.
 *
 * Arguments in/out:   MVector *mv          MVector to be clipped.
 *                                          Fullpel and subpel component are
 *                                          adjusted.
 *
 * Arguments out:      -
 *
 * Return values:      -
 *
 * Example:            ClipMVector(-16, 15, -16, 15, mv);
 *
 * Side effects:       -
 *	
 * Description:        The fullpel components are clipped to be within the
 *                     specified range.
 *                     The subpel components are set to zero in case of
 *                     clipping.
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
void ClipMVector(int minX, int maxX, int minY, int maxY, MVector *mv)
/***********************************************************CommentEnd********/
{
  /* X Component */
  if (mv->x < minX) {
    mv->x = 0;
    mv->sx = 0;
  }
  if (mv->x == maxX)
    mv->sx = 0;
  if (mv->x > maxX) {
    mv->x = maxX;
    mv->sx = 0;
  }
  /* Y Component */
  if (mv->y < minY) {
    mv->y = 0;
    mv->sy = 0;
  }
  if (mv->y == maxY)
    mv->sy = 0;
  if (mv->y > maxY) {
    mv->y = maxY;
    mv->sy = 0;
  }
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- InvertMVector -- Invert an MVector
 *
 * Author:             K.S.
 *
 * Created:            7-Aug-96
 *	
 * Purpose:            Invert an MVector
 * 
 * Arguments in:       -
 *
 * Arguments in/out:   MVector *mv          MVector to be inverted.
 *                                          The resulting motion vector
 *                                          points in the opposite direction.
 *
 * Arguments out:      -
 *
 * Return values:      0  if error
 *                     1  if ok.
 *
 * Example:            InvertMVector(mv);
 *
 * Side effects:       -
 *	
 * Description:        The MVector mv is inverted (including t).
 *                     The subpel components are kept in the range
 *                           0 <= sx, sy < sF
 *                     It works only, if the subpel components of the
 *                     MVector are already in this range.
 *                     No computation overflow is checked.
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
int InvertMVector(MVector *mv)
/***********************************************************CommentEnd********/
{
  mv->x = - mv->x;
  mv->y = - mv->y;
  mv->t = - mv->t;

  if (mv->sx > 0) {
    mv->x -= 1;
    mv->sx = mv->sF - mv->sx;
  }
  if (mv->sy > 0) {
    mv->y -= 1;
    mv->sy = mv->sF - mv->sy;
  }

  return(1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CopyMVector -- Copy an MVector
 *
 * Author:             K.S.
 *
 * Created:            7-Aug-96
 *
 * Purpose:            Copy an MVector to another MVector struct
 * 
 * Arguments in:       MVector *mvIn        Input MVector that is to be copied.
 *
 * Arguments in/out:   MVector *mvOut       MVector struct that is filled
 *                                          with the values of mvIn.
 *
 * Arguments out:      -
 *
 * Return values:      0  if error
 *                     1  if ok.
 *
 * Example:            CopyMVector(mvIn, mvOut);
 *
 * Side effects:       -
 *
 * Description:        The MVector mvIn is copied to the MVector mvOut.
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
int CopyMVector(MVector *mvIn, MVector *mvOut)
/***********************************************************CommentEnd********/
{
  mvOut->x = mvIn->x;
  mvOut->sx = mvIn->sx;
  mvOut->y = mvIn->y;
  mvOut->sy = mvIn->sy;
  mvOut->sF = mvIn->sF;
  mvOut->t = mvIn->t;

  return(1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CompareMVectors -- Compare MVectors
 *
 * Author:             K.S.
 *
 * Created:            5-Nov-96
 *
 * Purpose:            Compares all components of two MVectors
 * 
 * Arguments in:       MVector *mv1         Input MVector 1
 *                     MVector *mv2         Input MVector 2
 *
 * Arguments in/out:   -
 *
 * Arguments out:      -
 *
 * Return values:      0  if 'mv1' and 'mv2' are different
 *                     1  if 'mv1' and 'mv2' are identical
 *
 * Example:            a = CompareMVectors(mv1, mv2);
 *
 * Side effects:       -
 *
 * Description:        The MVectors mv1 and mv2 are compared. If they are
 *                     identical, 1 is returned, else 0.
 *
 * See also:           CompareMVectors2D()
 *
 * Modified:           -
 *
 *****************************************************************************/
int CompareMVectors(MVector *mv1, MVector *mv2)
/***********************************************************CommentEnd********/
{
  if (mv1->x != mv2->x || mv1->sx != mv2->sx ||
      mv1->y != mv2->y || mv1->sy != mv2->sy ||
      mv1->sF != mv2->sF || mv1->t != mv2->t) {
    return(0);
  } else {
    return(1);
  }
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CompareMVectors2D -- Compare 2D components of MVectors
 *
 * Author:             K.S.
 *
 * Created:            5-Nov-96
 *
 * Purpose:            Compares the 2D components (x, sx, y, sy, sF) of
 *                     two MVectors
 * 
 * Arguments in:       MVector *mv1         Input MVector 1
 *                     MVector *mv2         Input MVector 2
 *
 * Arguments in/out:   -
 *
 * Arguments out:      -
 *
 * Return values:      0  if 'mv1' and 'mv2' are different
 *                     1  if 'mv1' and 'mv2' are identical
 *
 * Example:            a = CompareMVectors2D(mv1, mv2);
 *
 * Side effects:       -
 *
 * Description:        The 2D components (x, y, sx, sy, sF) of the MVectors
 *                     mv1 and mv2 are compared. If they are
 *                     identical, 1 is returned, else 0.
 *                     The component 't' is not regarded.
 *
 * See also:           CompareMVectors()
 *
 * Modified:           -
 *
 *****************************************************************************/
int CompareMVectors2D(MVector *mv1, MVector *mv2)
/***********************************************************CommentEnd********/
{
  if (mv1->x != mv2->x || mv1->sx != mv2->sx ||
      mv1->y != mv2->y || mv1->sy != mv2->sy ||
      mv1->sF != mv2->sF)
    return(0);
  else
    return(1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AddMVectors -- Add two MVectors
 *
 * Author:             K.S.
 *
 * Created:            7-Aug-96
 *
 * Purpose:            Add two MVectors. Note: the time displacement 
 *                     is not added.
 * 
 * Arguments in:       MVector *mvIn1        First input MVector
 *	               MVector *mvIn2        Second input MVector
 *
 * Arguments in/out:   MVector *mvOut        Resulting MVector struct
 *                                           containing  mvIn1 + mvIn2.
 *
 * Arguments out:      -
 *
 * Return values:      0  if error
 *                     1  if ok.
 *
 * Example:            AddMVectors(mvIn1, mvIn2, mvOut);
 *
 * Side effects:       -
 *
 * Description:        The MVectors mvIn1 and mvIn2 are added.
 *                     The subpel components are kept in the range
 *                           0 <= sx, sy < sF
 *                     It works only, if the subpel components of the
 *                     input MVector are already in this range.
 *                     No computation overflow is checked.
 *                     The subpel-fraction 'sF' of all motion vectors must
 *                     be the same. This is not checked. The 'sF' of the
 *                     resulting vector 'mvOut' is used.
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
int AddMVectors(MVector *mvIn1, MVector *mvIn2, MVector *mvOut)
/***********************************************************CommentEnd********/
{
  Short tmpS;

  mvOut->x = mvIn1->x + mvIn2->x;
  mvOut->y = mvIn1->y + mvIn2->y;

  tmpS = mvIn1->sx + mvIn2->sx;
  switch (tmpS < mvOut->sF) {
    case 0:
      mvOut->x++;
      mvOut->sx = tmpS % mvOut->sF;
      break;
    case 1:
      mvOut->sx = tmpS;
      break;
  }

  tmpS = mvIn1->sy + mvIn2->sy;
  switch (tmpS < mvOut->sF) {
    case 0:
      mvOut->y++;
      mvOut->sy = tmpS % mvOut->sF;
      break;
    case 1:
      mvOut->sy = tmpS;
      break;
  }

  return(1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- SubtractMVectors -- Subtract two MVectors
 *
 * Author:             K.S.
 *
 * Created:            7-Aug-96
 *
 * Purpose:            Subtract two MVectors. Note: the time displacement 
 *                     is not subtracted.
 * 
 * Arguments in:       MVector *mvInPos      First input MVector
 *	               MVector *mvInNeg      Second input MVector
 *
 * Arguments in/out:   MVector *mvOut        Resulting MVector struct
 *                                           containing  mvInPos - mvInNeg.
 *
 * Arguments out:      -
 *
 * Return values:      0  if error
 *                     1  if ok.
 *
 * Example:            SubtractMVectors(mvInPos, mvInNeg, mvOut);
 *
 * Side effects:       -
 *
 * Description:        The MVectors mvIn1 and mvIn2 are subtracted.
 *                     The subpel components are kept in the range
 *                           0 <= sx, sy < sF
 *                     It works only, if the subpel components of the
 *                     input MVector are already in this range.
 *                     No computation overflow is checked.
 *                     The subpel-fraction 'sF' of all motion vectors must
 *                     be the same. This is not checked. The 'sF' of the
 *                     resulting vector 'mvOut' is used.
 *
 * See also:           -
 *
 * Modified:           -
 *
 *****************************************************************************/
int SubtractMVectors(MVector *mvInPos, MVector *mvInNeg, MVector *mvOut)
/***********************************************************CommentEnd********/
{
  Short tmpS;

  mvOut->x = mvInPos->x - mvInNeg->x;
  mvOut->y = mvInPos->y - mvInNeg->y;

  tmpS = mvInPos->sx - mvInNeg->sx;
  switch (tmpS < 0) {
  case 0:
    mvOut->sx = tmpS;
    break;
  case 1:
    mvOut->x--;
    mvOut->sx = mvOut->sF + tmpS;
    break;
  }

  tmpS = mvInPos->sy - mvInNeg->sy;
  switch (tmpS < 0) {
  case 0:
    mvOut->sy = tmpS;
    break;
  case 1:
    mvOut->y--;
    mvOut->sy = mvOut->sF + tmpS;
    break;
  }

  return(1);
}

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- H263divideBy2HalfPelMVector -- Divide a MVector as needed in H.263
 *
 * Author:           T.W.
 *
 * Created:          15-May-97
 *
 * Purpose:          A MVector is divided for the chrominance as needed
 *                   in H.263.
 * 
 * Arguments in:     MVector    *mvIn         input MVector
 *
 * Arguments in/out: MVector    *mv           resulting MVector
 *
 * Arguments out:    -
 *
 * Return values:    -
 *
 * Example:          H263divideBy2HalfPelMVector(mvIn, mvOut);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         H263divideFourMVectors
 *      
 * Modified:         -
 *      
 *****************************************************************************/
void DivideBy2HalfPelMVector(MVector *mv1, MVector *mv2)
/***********************************************************CommentEnd********/
{
  int x,y;
 
  x = 2*mv1->x+mv1->sx;
  if (x%4==0) {
    mv2->x = x/4;
    mv2->sx = 0;
  } else {
    mv2->x = MFLOOR((float)(x-1)/4);
    mv2->sx = 1;
  }
 
  y = 2*mv1->y+mv1->sy;
  if (y%4==0) {
    mv2->y = y/4;
    mv2->sy = 0;
  } else {
    mv2->y = MFLOOR((float)(y-1)/4);
    mv2->sy = 1;
  }
 
  mv2->t = mv1->t;
  mv2->sF = mv1->sF;
}

