/*
 * net.h -- declarations for network access and feedback for jittr atoms.
 *
 * note: this requires <sys/time.h>, <sys/socket.h> (and thus <sys/types.h>)
 *
 * 29.04.95 jw.
 */

/*
 * This switches the input gather loop into greedy mode.
 * You will never get double prompts per command line then, but
 * commands will not execute unless input seizes temporarily. 
 * This makes `cd <atom>; get .' different from `get <atom>'.
 * 
 * Fixme: The select() call will block unexpectedly in nongreedy mode.  
 *        Although we have cached input in at least one of our buffers.
 *        We need a workaround there.
 */
#define GREEDY_INTERPRETER

extern char *con_type_name[];

/* keep this in sync with char *con_type_name[] in net.c */
enum con_type
{
  CTYPE_INVALID,
  CTYPE_TCP,
  CTYPE_TCP_OR_TELNET,
  CTYPE_TELNET,
  CTYPE_UDP,
  CTYPE_WFD,		/* a write-only filedescriptor */
  CTYPE_FD,		/* a read-write filedescriptor */
  CTYPE_BUF
};

#define IS_CTYPE_TCP(a) ((a) == CTYPE_TCP || (a) == CTYPE_TCP_OR_TELNET || (a) == CTYPE_TELNET)

struct conn
{
  int fd;		/* filedescriptor if already connected */
  enum con_type type;	/* CTYPE_... type of communicaton line */
  /*
   * If the client connection is unidirectional, only one of the
   * following two buffers is used. But if we have only one unidirectional
   * connection, we may try to make it bidirectional, and see, if he responds.
   * Vice versa we must accept that a client is talking back to us
   * through an outbound connection. In this case we discard his 'other'
   * inbound connection and make this one a full duplex connection.
   * A connection is full duplex, if both conn pointers o and i point
   * to the same conn structure. In that case we sure need both buffers.
   */
  struct dstring *ibuf;	/* an inbound buffer for the connection */
  struct dstring *obuf;	/* an outbound buffer for the connection */
  int read_only;	/* if nonzero: this conn cannot change system state */
  struct sockaddr sa;	/* the peer, if remote */
  /*
   * as not all struct sockaddr_.. variants (e.g. AF_UNIX) fit into 
   * struct sockaddr, it is the last enty in struct conn.
   * For shared memory buffers this structure may even be totally inadequate.
   * A struct conn may be malloced to be large enough.
   */
};

enum prompt_type
{
  PT_DISABLED,		/* This is a dumb mechanical client. Do not prompt */
  PT_COMMENT,		/* Prompts welcome, if they look like #...\n comments */
  PT_PLAIN		/* Client is human. Understands everything. */
};

struct prompt
{
  enum prompt_type type;	/* how to prompt */
  struct dstring *prompt_ok, *prompt_bad, *prompt_more;
};


/*
 * If a client lives on another host, his clock may be different than ours.
 * If we are both NTP synchronised the difference that we measure is mostly
 * due to asymetric network latencies. 
 * If we are not synchronized, we should expect a few seconds or even minutes
 * of difference. If one of the hosts has a wrong concept about summertime
 * the clocks may differ by one hour. If one of the clients does not want to
 * talk in UTC (e.g. if he is a human administrator coming via telnet), we
 * may have several hours deviation here.
 *
 * If a packet with an absolute timestamp is sent, the sender uses his clock
 * to create the timestamp and the receiver interprets the timestamp according
 * to his clock. If a packet is sent without or with a relative timestamp,
 * only the receivers clock is involved for creation of an internal absolute
 * timestamp.
 *
 * Possible effects of differenig clocks are:
 *  1) The receiver discards the command, because its effect has already been
 *     superceeded by a younger command.
 *  2) The receiver delays the execution of the command, because the clock 
 *     difference transported it into the future.
 *
 * These effects do not disturb any internal mechanisms of the session,
 * but they degrade the usability of the overall system, if the clocks differ
 * by an amount that a human user could notice.
 *
 * Thus a mechanism for measuring the clock offset (during the opening dialogue
 * of a session) and another mechansim for adjusting timestamps (when receiving
 * a packet) has been established.
 * 
 * The jittr library has a lifelock prevention feature that is based on 
 * timestamps. Modifying timestamps can ruin this feature, unless done 
 * carefully.
 *
 * Whenever a packet can bounce back to its originator, the timestamp should be
 * set exactly as it was when the packet left the originator. 
 * 
 * If the sum of corrections is not exactly NULL after a roundtrip, the 
 * originators livelock prevention is spoiled whenver the sum of corrections is
 * positive: The packet is assumed to contain an update of the previous command
 * and it is redistributed, entering a livelock as soon as the redistributed
 * packet returns again (with increased timestamp).
 *
 * This can be prevented on any full duplex link by copying the offset value
 * from one party (that measured the value) to the other party (with inverted
 * sign).  Thus a direct bounce can be detected savely.
 *
 * Jittr networks usually have star-topology, where the only way a packet can
 * return is a direct bounce.
 *
 * In the unusual case where a ring topology has been (unintentionally)
 * established, a different route may be used from client A to B, than from B
 * back to A. In this case time differences are measured by multiple instances
 * and add up to the total offset for each route. Due to lacenies and rounding
 * errors we cannot expect that all measurements on the route from A to B back
 * to A sum up to exactly 0.
 *
 * To help the livelock prevention in a ring topology, we try to make the sum
 * of corrections per roundtrip either excatly zero, or noticably different
 * from zero. The approach chosen here is to round all the correction to a
 * certain granularity. Thus any sum of corrections is either exactly 0 or
 * an exact integer multiple of the granularity.
 *
 * The knowledge about exact integer multiples of the granularity can be used
 * in the livelock prevention feature, if necessary. Increasing the granularity
 * can also be extremly helpful if livelocks occur in a ring topology.
 * If the granularity is too high, one of the effects 1) or 2) may occur.
 *
 * Caution: The granularity must be identical on all clients, thus it is only
 * adjustable at compile time!
 */
#define CLOCK_OFF_GRAN	50000	

struct tv_offset
{
  struct timeval tv;	/* this clients clock is off by sec/usec */
  int sign;		/* 0: undef, >0 he lags behind, <0 he is fast */
};

#define IOT_DONT_READ	0x01	/* Our input_notify_fn wants to read itself */
#define IOT_DONT_WRITE	0x02	/* Client is an alien, cannot talk to him */

struct client
{
  struct client *next;	/* chain of clients */
  int id;		/* unique id of the client, given by server */
  struct cwp cwp;	/* where we are. Consider this part of the "context" */
  struct dstring *name;	/* name of client, given by client */
  struct prompt prompt;	/* how to prompt for input */
  char *crypted_pw;	/* a password for write access, if any */
  struct conn *i, *o;	/* inbound / outbound connection pointer */
  struct conn _i, _o;	/* builtin inbound / outbound connection data area */
  /*
   * i and o may be NULL, may both point to _i, may point to _i and _o
   * respectively, or may point to allocated memory containing 
   * a conn structure.
   * Clients may be unidirectionally connected, then they have i or o Zeroed.
   * Clients may have one bidirectional connection, then i and o are identical.
   * Or clients may have different inbound and outbound connections.
   * Some connection structures (e.g. AF_UNIX)  may have larger size than 
   * what would fit into _i and _o, these require allocated structures.
   */

  int iotype;			/* IOT_DONT_READ | IOT_DONT_WRITE */
  int (* input_notify_fn) __P((void *cl, char *buf, int len));
  /*
   * If the above method is NULL, DoIo will use the default notify handler
   * XXX: Maybe this should be a stack, and we should
   *      have a jittr command to push and pop handlers.
   */
  void *input_notify_data;	/* each input handler may have private data */

  struct tv_offset off;		/* if he tells us timestamps, they may be off */

  struct client *close_when_empty;	/* a parent used by jittr_nuke_client */
  dstring *init_port, *init_opts;	/* how jittr_init_tty ran */
};

extern struct client *clients;	/* list of all known clients */

#define INVALID_CLIENT ((struct client *)-1)	/* used by server ports */

struct server
{
  int tcp_port;
  int tcp_port_fd;
  int udp_port;
  int udp_port_fd;
  int connect_timeout;
  struct dstring *name;
  struct dstring *netmask;		/* connect & read permission */
  struct dstring *netmask_write;	/* modify permission */
  int (* access_fn) __P((struct server *s, struct client *cl, char *buf, int len));
  /*
   * access_fn() should return 0 to allow the client in. Note that access_fn
   * must not memorize client pointers.
   */
  void *access_data;			/* private data for the allow_fn */
  char *crypted_pw;			/* a pointer into "pass" atom. If any */
};

#ifndef SYS_ERRLIST_DECLARED
extern char *sys_errlist[];
#endif
#ifdef sun
# ifndef __SVR4			/* how Rudi detects Solaris */
#  define strerror(e)   (((e) > 0) ? sys_errlist[(e)] : "Dammned error!")
# endif
#endif

#ifndef __P
# ifdef __STDC__
#  define __P(a) a
# else
#  define __P(a) ()
# endif
#endif

#if defined(sun) || defined(SOLARIS) || defined(linux)
# define SIGHASARG
#endif

#if defined(sun) || defined(linux) || defined(__sgi)
# define SIGVOID
#endif

#ifdef SIGVOID
# define SIGRETURN
# define sigret_t void
#else
# define SIGRETURN return 0;
# define sigret_t int
#endif

#ifdef SIGHASARG
# define SIGDEFARGHACK()
# define SIGPROTOARG   (int)
# define SIGDEFARG     sigsig)int sigsig;SIGDEFARGHACK(
# define SIGARG        0
#else
# define SIGPROTOARG   (void)
# define SIGDEFARG
# define SIGARG
#endif

#define SOCKBUFSIZE (48 * 1024)
#define ESTIMATED_PACKET_SIZE 1024

/* return value masks for client input notify functions: */
#define CIN_SCHED_MASK	0x01	/* command(s) complete, scheduled */
#define CIN_MORE_MASK	0x02	/* last command incomplete, need more input */
#define CIN_VOID_MASK	0x04	/* last command corrupt, discarded */
#define CIN_CLOSE_MASK	0x08	/* parser layer rejects connection */

/* declarations */
char *Hostname __P((void));
struct sockaddr_in *Str2Si __P((char *host, int port, struct sockaddr_in *sip));
int ListenPort __P((int sock_dgram_or_stream, int port, int *fd_ret, dstring **errp));
int ListenPort5 __P((int dgr_or_str, int port, int *port_ret, int *fd_ret, dstring **errp));
int PortShutdown __P((int clientfd, struct client **fd_client, dstring **errp));
int TcpPortAccept __P((dstring **errp));
int TcpPortAccept3 __P((int server_fd, dstring *netmask, dstring **errp));
int CheckNetmask __P((dstring *masks, struct sockaddr_in *peer));
int NetmaskDefault __P((dstring **mask_ret, unsigned long def_mask, char *hostname, dstring **errp));
int TcpPortConnect __P((char *hostname, int port, int timeout, dstring **errp));
int UdpSetTTL __P((int serverfd, int ttl, dstring **errp));
int UdpSend __P((struct sockaddr_in *peer, char *buf, int len, dstring **errp));
int UdpRecv __P((char *buf, int len, struct sockaddr_in *peer, dstring **errp));
int UdpRecv5 __P((int fd, char *buf, int len, struct sockaddr_in *peer, dstring **errp));
struct client **ClientByName __P((struct client **root, char *name));
struct client **ClientByPeer __P((struct client **root, struct sockaddr *peer));
struct client **ClientByFd __P((struct client **root, int fd));
struct client **ClientById __P((struct client **root, int id));
int DoIo __P((struct timeval *, int (*timo_fn)(struct timeval *now), int (*cin_fn)(void *cl, char *buf, int len)));
int jittr_do_prompt __P((struct prompt *pr, struct conn *conn, int));
int jittr_register_fd __P((int fd, struct client *cl, struct conn *conn));
int conn2dstr __P((struct dstring **dp, int offset, struct conn *conn));
int ParsePeer __P((char *peerstring, int default_port, char **end_ret));
int jittr_is_selectable __P((int fd));
int jittr_mk_selectable __P((int fd, int *pid_ret));
