/*
 * schedule.h -- definitions and declarations for performing timed actions. 
 * You may call it a "cooperative scheduler", or "not a real scheduler at all",
 * depending on your point of view. The jittr world is not ment to be CPU 
 * intensive, thus it is sufficient.
 *
 * The motto here is: "An action combined with a timing is an event"
 *
 * 29.7.94 (jw)
 *
 * Limit for exponential interval change added.
 * 14.7.96
 */

#ifndef __P
# ifdef __STDC__
#  define __P(a) a
# else
#  define __P(a) ()
# endif
#endif

struct sched_timing
{
  struct timeval when;	/* when this event is scheduled next (absolute time) */
  struct timeval iv;	/* interval between next and second next (rel time) */
  struct timeval li;	/* exponenet is applied to iv until this limit hit */
  double exponent;	/* scale factor for iv, after event occured. */
  long count;		/* repetition number of next occurence (1 is first) */
  long max_rep;		/* max number of repetitions, (0 is infinite) */
};

/*
 * Initialize a timing from a timeval. A single event, without repeats.
 */
#define TV2TIMING(v,i)					\
   do { (i)->when.tv_sec = (v) ? (v)->tv_sec : 0;	\
	(i)->when.tv_usec = (v) ? (v)->tv_usec : 0;	\
	(i)->iv.tv_sec = (i)->iv.tv_usec = 0;		\
	(i)->li.tv_sec = (i)->li.tv_usec = 0;		\
	(i)->exponent = 1.0;				\
	(i)->count = 0;					\
	(i)->max_rep = 1; } while (0)

#define SCHED_ARGC	8

struct sched_action
{
  void *argv[SCHED_ARGC];	/* object data */
  int (*fn) __P((void **argv));	/* object method */
  int flag;			/* bitfield, free argv[], where bit is set. */
  struct sched_action *next;	/* cycle forward, if nonzero */
};

struct sched_event
{
  struct sched_action action;	/* action to process, when time reached */
  struct sched_timing time;	/* event timing */
  long id;			/* constant uniq event identifier */
  struct sched_event *next;	/* event chain */
};

/*
 * Parse a string specifying a relative or absolute time stamp.
 * If now is NULL, the timing syntax must indicate an absolute time interval.
 * If now is nonzero, it should point to a timeval returned by 
 * gettimeofday(2), and will be used to resolve relative timings
 * into absolute timestamps. It is ignored, when the syntax already
 * indicates absolute time.  Absolute specifications include a date.
 * The string pointer strp is advanced by the amount of sucessfully parsed
 * characters.
 * Returns negative and does not advance on error.
 */
int sched_parse_tv __P((char **strp, struct timeval *tv, struct timeval *now));

/* 
 * Parse a timing and repetition specification given in an odd syntax.
 * A nonzero now parameter (as provided by gettimeofday) is required to help 
 * esolve relative start timings.
 * The string pointer strp is advanced by the amount of sucessfully parsed
 * characters.
 */
int sched_parse_timing __P((char **strp, struct sched_timing *t, struct timeval *now));

/*
 * prints the timeval pointed to by tv into the character string 
 * starting at *str in a syntax understood by sched_parse_tv().
 * Tv is expected to be an absolute timestamp. An absolute timestamp spec is
 * printed when now is NULL. A relative spec is calcualted and printed, when
 * now is nonzero.
 * relative timstamps print like this:
 *	+1h15m2.502s
 * or if in the past:
 *	+-1h15m2.502s
 *
 * An absolute timestamp means that a complete description (including month and
 * year) is constructed.  The character string is delimited by a null byte.
 * The return value is as for sched_format_timing().
 */
int sched_format_tv __P((char *str, int len, struct timeval *tv, struct timeval *now));

/*
 * Prints a complete timing specification as found in *t into the character
 * string pointed to by *str, in the syntax understood by sched_parse_timing()
 * using at most len bytes. The character string is delimited by a 
 * null byte. sched_format_tv() is used. If now is nonzero, the timing is 
 * returned relative.
 *
 * If more than len bytes would be used, an even over-estimate of the
 * number of bytes is returned as a negative number. If successful,
 * the (positive) number of written bytes is returned. 
 * Returns -1 on error.
 */
int sched_format_timing __P((char *str, int len, struct sched_timing *t, struct timeval *now));

/*
 * Allocates one sched_action structure, chains must be constructed by hand.
 * The address of the allocated structure is written to *ap. Flag bits 1, 2, 
 * 4, 8 ... indicate that the corresponding element of argv refers 
 * to malloced data and should be freed when the action is destroyed.
 * A maximum number of SCHED_ARGC elements is recorded from argv. Argv should 
 * be NULL terminated if shorter.
 *
 * Returns negative on error.
 *
 * simple example usage (ouch, printf doesnt like an argv vector):
 *
 *   struct sched_action *new_action;
 *   static char argv[SCHED_ARGC] = { "%s %s\n", "Hello", "world", };
 *   sched_make_action(new_action, printf, 0, argv);
 */
int sched_make_action __P((struct sched_action **ap, int (*fn)(), int flag, void **argv));

/*
 * Combine a timing spec and an action spec into a scheduled event and
 * returns its (positive) id. Returns negative on error.
 * Both structures may reside in malloced memory. Flag should be set to 1, 2 or
 * 3 accordingly. Chained action structures must reside in alloced memory.
 * sched_timeout() should be called after entering new events, to learn
 * about the updated timings.
 */
int sched_enter_event __P((struct sched_timing *, struct sched_action *, int flag));

/* 
 * Used to post an event. *e is malloced memory and should be initialized by
 * a function similar to sched_alloc()
 * The id of the posted event e is returned for convenience.
 */
int sched_enter __P((struct sched_event *e));

/*
 * Create a sched_event structure, that can (after additional patching) be 
 * passed to sched_enter.
 * If stardate_fn is NULL, the invocation time is fetched from tv_sec, tv_usec
 * as an absolute timestamp. Otherwise stardate_fn(struct timeval *) is called
 * to initialize the invocation time stamp -- if tv_sec or tv_usec are also 
 * nonzero, infinite repetitions at each tv_sec.tc_usec interval are performed.
 * fn(void **argv) will be called with argv[0] == data1, argv[1] == data2;
 * If data1 or data2 points to malloced memory, the flag bits can be patched
 * in the returned structure, to avoid memory loss (after the last invocation
 * is done).
 */
struct sched_event *sched_alloc __P((int (*stardate_fn) __P((struct timeval *)), int tv_sec, int tv_usec, int (*fn) __P((void **)), void *data1, void *data2));

/* 
 * remove a timed event from the queue. Returns the event structure
 * but does not free any data. Returns NULL, if the event was not queued.
 */
struct sched_event *sched_unlink_event __P((int id));

/* 
 * Free a single or chain of actions. Private data pointers are passed to 
 * free, if indicated by the flags.
 * May also be used to free sched_event structures.
 */
void sched_free_action __P((struct sched_action *a));

/* 
 * Sets *tv to indicate when the next event is due. If now is nonzero, *tv
 * holds the number of seconds and microseconds until the next event occurs,
 * measured from now on. If now is zero, the absolute timestamp of the next
 * event queued is returned in *tv.
 * The return value is the positive event id of the next event. If the event was
 * due in the past, its id is returned negative. If there are no events,
 * 0 is returned and *tv is not altered.
 */
int sched_timeout __P((struct timeval *tv, struct timeval *now));

/* 
 * Processes one scheduled event that is due at the specified timeval now.
 * The timing and action data is recalculated after the registered
 * function is called. The function has only one parmeter: void **argv.
 * The structure *(struct sched_event *)argv reflects the internal data, while
 * the function executes. If time.count equals time.max_rep, no (further)
 * repetitions will be scheduled.
 * If the processing function returns non-negative, the event is rescheduled
 * if its repetition counter still permits. The return value of the processing
 * function is stored at *ret.  The (positive) id of the executed event is
 * returned. When no event was due now, 0 is returned. 
 *
 * As execution may take time, sched_timeout() should be called after 
 * sched_process() returns, to find out if more events are due then. 
 * If the actions function takes long, it should call sched_timeout() to 
 * see how much time it has left and may consider to return prematurly.
 *
 * The repetition mechanism is implemented in sched_process().
 * It attempts but does not guarantee invocation once per interval e->iv.  It
 * only guarantees that a repetitive event with an unlimited number of
 * invocations is invoked at most twice in any time-interval of length e->iv.
 */ 
int sched_process __P((int *ret, struct timeval *now));

/*
 * When called with both arguments zero, the peek pointer is reset to 
 * the head of the queue and the id of the event at head of queue is returned.
 * When called with one or both arguments nonzero, pointers to the events
 * action and/or timing structure are returned in *tp / *ap.
 * the event id is returned and the peek pointer is advanced to the next event.
 * If there are no (more) events or one of sched_process, sched_enter_event, 
 * sched_unlink_event have been called since the last reset, 0 is returned.
 */
int sched_peek_event __P((struct sched_timing **tp, struct sched_action **ap));

/* 
 * sched_timerdiff() returns the difference timeval in result, between future
 * and past.  If the diffeence is negative, result remains untouched and -1 is
 * returned.  Returns 0 on success.
 */
extern int sched_timerdiff __P((struct timeval *result, struct timeval *future, struct timeval *past));
