#include <string.h>

#include "Util.h"
#include "Codebook.h"

#include "global.h"
#include "extern.h"

#include "BlockDecoder.h"
#include "Input.h"
#include "DecodeSymbol.h"

/*
#define S1SPY 0.705901
#define S2SPY 0.840002
#define S1SPC 0.619113
#define S2SPC 0.817945
#define S1TPY 0.600735
#define S2TPY 0.811468
#define S1TPC 0.619113
#define S2TPC 0.817945
*/
/*
#define S1SPY 1
#define S2SPY 1
#define S1SPC 1
#define S2SPC 1
#define S1TPY 1
#define S2TPY 1
#define S1TPC 1
#define S2TPC 1
*/

/* #define CSCALE ROUND(0.5*scale) */
#define CSCALE (0.85*scale)
/* ============================================================================== */
/* To make things faster ... */
/* ============================================================================== */

static int TableIndex;

/*
static int Shell1SPYScaled[120][8];
static int Shell1SPCScaled[120][8];
static int Shell1TPYScaled[120][8];
static int Shell1TPCScaled[120][8];
*/

int Yrowbytes;
int Crowbytes;

static int YRowOffset[8];
static int CRowOffset[8];

static int YBlockStart8[4];
static int YBlockStart4[4];
static int CBlockStart4[4];

void InitBlockDecoder(int w) {
  int i, j;

  Yrowbytes = w;
  Crowbytes = w/2;

  for (j=0; j<2; j++)
    for (i=0; i<4; i++)
      YRowOffset[(j*4)+i] = j+(i*w);

  for (j=0; j<2; j++)
    for (i=0; i<4; i++)
      CRowOffset[(j*4)+i] = j+(i*(w/2));

  YBlockStart8[0] = 0;
  YBlockStart8[1] = 8;
  YBlockStart8[2] = 8*w;
  YBlockStart8[3] = 8*w+8;

  YBlockStart4[0] = 0;
  YBlockStart4[1] = 4;
  YBlockStart4[2] = 4*w;
  YBlockStart4[3] = 4*w+4;

  CBlockStart4[0] = 0;
  CBlockStart4[1] = 4;
  CBlockStart4[2] = 4*(w/2);
  CBlockStart4[3] = 4*(w/2)+4;

}

/* ============================================================================== */
/* Decode Intra mean */
/* ============================================================================== */

static int MeanYPredictor, MeanUPredictor, MeanVPredictor;

void ResetMeanPredictors() {
  MeanYPredictor = MeanUPredictor = MeanVPredictor = 128;
}

int DecodeIntraMean (Global *this, int *MeanPred)
{
  int MeanSize, base, diff;
  
  MeanSize = GetMeanSizeHuffCode(this);

  if (MeanSize) {
    base = (1<<MeanSize);

    diff = GetBits(this, MeanSize);

    if (diff<(base>>1))
      diff = diff - base + 1;

  } else diff = 0;

  *MeanPred = *MeanPred + diff;
  return(*MeanPred);
}

static void ReconstructVector(Global *this, Byte *dec, int *offset, double scale, double *v) {
  dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[0])));
  dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[1])));
  dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[2])));
  dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[3])));
  dec++;
  dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[4])));
  dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[5])));
  dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[6])));
  dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[7])));
}

/* 
static int SymIndices[3][8] = {
  { 4, 5, 6, 7, 0, 1, 2, 3 },
  { 3, 2, 1, 0, 7, 6, 5, 4 },
  { 7, 6, 5, 4, 3, 2, 1, 0 }
};
*/

#if 0
static void ReconstructAndInterVector(Global *this, Byte *dec, int *offset, int scale, double *v) {

  double x[2];

  x[0] = v[0]*scale; x[1] = v[4]*scale;

  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);

  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  x[0] = v[1]*scale; x[1] = v[5]*scale;

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  x[0] = v[2]*scale; x[1] = v[6]*scale;

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  x[0] = v[3]*scale; x[1] = v[7]*scale;

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);

  dec += offset[1]; /* next line */
  dec[0] = ROUNDCLIP2(this, dec[0] + x[0]); dec[1] = ROUNDCLIP2(this, dec[1] + x[0]); dec[2] = ROUNDCLIP2(this, dec[2] + x[0]); dec[3] = ROUNDCLIP2(this, dec[3] + x[0]);
  dec[4] = ROUNDCLIP2(this, dec[4] + x[1]); dec[5] = ROUNDCLIP2(this, dec[5] + x[1]); dec[6] = ROUNDCLIP2(this, dec[6] + x[1]); dec[7] = ROUNDCLIP2(this, dec[7] + x[1]);
}

static void ReconstructAndInterUVVector(Global *this, Byte *udec, Byte *vdec, int *offset, int scale, double *v) {

  double x[4];

  x[0] = v[0]*scale; x[1]=v[1]*scale;
  x[2] = v[4]*scale; x[3]=v[5]*scale;

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  x[0] = v[2]*scale; x[1]=v[3]*scale;
  x[2] = v[6]*scale; x[3]=v[7]*scale;

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

  udec += offset[1]; vdec += offset[1];

  udec[0] = ROUNDCLIP2(this, udec[0] + x[0]);   udec[1] = ROUNDCLIP2(this, udec[1] + x[0]); udec[2] = ROUNDCLIP2(this, udec[2] + x[0]); udec[3] = ROUNDCLIP2(this, udec[3] + x[0]);
  vdec[0] = ROUNDCLIP2(this, vdec[0] + x[2]);   vdec[1] = ROUNDCLIP2(this, vdec[1] + x[2]); vdec[2] = ROUNDCLIP2(this, vdec[2] + x[2]); vdec[3] = ROUNDCLIP2(this, vdec[3] + x[2]);

  udec[4] = ROUNDCLIP2(this, udec[4] + x[1]);   udec[5] = ROUNDCLIP2(this, udec[5] + x[1]); udec[6] = ROUNDCLIP2(this, udec[6] + x[1]); udec[7] = ROUNDCLIP2(this, udec[7] + x[1]);
  vdec[4] = ROUNDCLIP2(this, vdec[4] + x[3]);   vdec[5] = ROUNDCLIP2(this, vdec[5] + x[3]); vdec[6] = ROUNDCLIP2(this, vdec[6] + x[3]); vdec[7] = ROUNDCLIP2(this, vdec[7] + x[3]);

}
#endif


static void ReconstructTransformedVector(Global *this, Byte *dec, int *offset, double scale, double *v, int sy) {

  switch (sy) {
  case 0:
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[4])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[5])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[6])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[7])));
    dec++;
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[0])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[1])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[2])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[3])));
    break;
  case 1:
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[3])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[2])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[1])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[0])));
    dec++;
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[7])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[6])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[5])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[4])));
    break;
  case 2:
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[7])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[6])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[5])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[4])));
    dec++;
    dec[0] = ROUNDCLIP2(this, dec[0] + (scale*(v[3])));
    dec[offset[1]] = ROUNDCLIP2(this, dec[offset[1]] + (scale*(v[2])));
    dec[offset[2]] = ROUNDCLIP2(this, dec[offset[2]] + (scale*(v[1])));
    dec[offset[3]] = ROUNDCLIP2(this, dec[offset[3]] + (scale*(v[0])));
    break;
  }
}

/* ============================================================================== */
/* Shell 1: */
/* ============================================================================== */
static void DecodeShell1TPCVector(Global *this, Byte *dec, int *offset, int scale) {
  int Index;

  Index = this->GetShell1TPCHuffCode(this);
  if (GetOneBit(this) == 1) scale = -scale;
#ifdef S1TPC
  ReconstructVector(this, dec, offset, S1TPC*scale, Shell1TPCCodebook+8*Index);
#else
  switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell1TPCCodebook0+8*Index); break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell1TPCCodebook1+8*Index); break;
  }
#endif
}

static void DecodeShell1TPYVector(Global *this, Byte *dec, int *offset, int scale) {
  int Index;

  Index = this->GetShell1TPYHuffCode(this);
  if (GetOneBit(this) == 1) scale = -scale;
#ifdef S1TPY
  ReconstructVector(this, dec, offset, S1TPY*scale, Shell1TPYCodebook+8*Index);
#else
  switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell1TPYCodebook0+8*Index); break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell1TPYCodebook1+8*Index); break;
  }
#endif
}

static void DecodeShell1SPYVector(Global *this, Byte *dec, int *offset, int scale) {
  int Index;

  Index = this->GetShell1SPYHuffCode(this);
  if (GetOneBit(this) == 1) scale = -scale;
#ifdef S1SPY
  ReconstructVector(this, dec, offset, S1SPY*scale, Shell1SPYCodebook+8*Index);
#else
  switch(TableIndex) {
  case 0:   ReconstructVector(this, dec, offset, scale, Shell1SPYCodebook0+8*Index); break;
  case 1:   ReconstructVector(this, dec, offset, scale, Shell1SPYCodebook1+8*Index); break;
  }
#endif
}

static void DecodeShell1SPCVector(Global *this, Byte *dec, int *offset, int scale) {
  int Index;

  Index = this->GetShell1SPCHuffCode(this);
  if (GetOneBit(this) == 1) scale = -scale;
#ifdef S1SPC
  ReconstructVector(this, dec, offset, S1SPC*scale, Shell1SPCCodebook+8*Index);
#else
  switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell1SPCCodebook0+8*Index); break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell1SPCCodebook1+8*Index); break;
  }
#endif
}

/* ============================================================================== 
 * Shell 2: 
 * Interpretation of bits: b3 b2 b1 b0
 * b3: sign, negative if zero, positive otherwise
 * b2 b1 b0: 3 bit offset -or-
 * b0: flip, b2 b1: 2 bit offset
 * ============================================================================== */

static void DecodeShell2TPYVector(Global *this, Byte *dec, int *offset, int scale) {
  int sign, trans, sy, Index;

  Index = this->GetShell2TPYHuffCode(this);
  sign = GetOneBit(this);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(this);
  else trans = 0;

  if (sign==1) scale = -scale;

  if (trans==1) {
#ifdef S2TPY
    ReconstructTransformedVector(this, dec, offset, S2TPY*scale, Shell2TPYCodebook+8*Index,sy-1);
#else
    switch(TableIndex) {
    case 0: ReconstructTransformedVector(this, dec, offset, scale, Shell2TPYCodebook0+8*Index,sy-1); break;
    case 1: ReconstructTransformedVector(this, dec, offset, scale, Shell2TPYCodebook1+8*Index,sy-1); break;
    }
#endif
  } else {
#ifdef S2TPY
    ReconstructVector(this, dec, offset, S2TPY*scale, Shell2TPYCodebook+8*Index);    
#else
    switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell2TPYCodebook0+8*Index);    break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell2TPYCodebook1+8*Index);    break;
    }
#endif
  }
}

static void DecodeShell2TPCVector(Global *this, Byte *dec, int *offset, int scale) {
  int sign, trans, sy, Index;

  Index = this->GetShell2TPCHuffCode(this);
  sign = GetOneBit(this);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(this);
  else trans = 0;

  if (sign==1) scale = -scale;

  if (trans==1) {
#ifdef S2TPC
    ReconstructTransformedVector(this, dec, offset, S2TPC*scale, Shell2TPCCodebook+8*Index,sy-1);
#else
    switch(TableIndex) {
    case 0: ReconstructTransformedVector(this, dec, offset, scale, Shell2TPCCodebook0+8*Index,sy-1); break;
    case 1: ReconstructTransformedVector(this, dec, offset, scale, Shell2TPCCodebook1+8*Index,sy-1); break;
    }
#endif
  } else {
#ifdef S2TPC
    ReconstructVector(this, dec, offset, S2TPC*scale, Shell2TPCCodebook+8*Index);    
#else
    switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell2TPCCodebook0+8*Index);    break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell2TPCCodebook1+8*Index);    break;
    }
#endif
  }
}




/* ============================================================================== */


static void DecodeShell2SPYVector(Global *this, Byte *dec, int *offset, int scale) {
  int sign, trans, sy, Index;

  Index = this->GetShell2SPYHuffCode(this);
  sign = GetOneBit(this);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(this);
  else trans = 0;

  if (sign==1) scale = -scale;

  if (trans==1) {
#ifdef S2SPY
    ReconstructTransformedVector(this, dec, offset, S2SPY*scale, Shell2SPYCodebook+8*Index,sy-1);
#else
    switch(TableIndex) {
    case 0: ReconstructTransformedVector(this, dec, offset, scale, Shell2SPYCodebook0+8*Index,sy-1); break;
    case 1: ReconstructTransformedVector(this, dec, offset, scale, Shell2SPYCodebook1+8*Index,sy-1); break;
    }
#endif
  } else {
#ifdef S2SPY
    ReconstructVector(this, dec, offset, S2SPY*scale, Shell2SPYCodebook+8*Index);    
#else
    switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell2SPYCodebook0+8*Index);    break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell2SPYCodebook1+8*Index);    break;
    }
#endif
  }
}

static void DecodeShell2SPCVector(Global *this, Byte *dec, int *offset, int scale) {
  int sign, trans, sy, Index;

  Index = this->GetShell2SPCHuffCode(this);
  sign = GetOneBit(this);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(this);
  else trans = 0;

  if (sign==1) scale = -scale;

  if (trans==1) {
#ifdef S2SPC
    ReconstructTransformedVector(this, dec, offset, S2SPC*scale, Shell2SPCCodebook+8*Index,sy-1);
#else
    switch(TableIndex) {
    case 0: ReconstructTransformedVector(this, dec, offset, scale, Shell2SPCCodebook0+8*Index,sy-1); break;
    case 1: ReconstructTransformedVector(this, dec, offset, scale, Shell2SPCCodebook1+8*Index,sy-1); break;
    }
#endif
  } else {
#ifdef S2SPC
    ReconstructVector(this, dec, offset, S2SPC*scale, Shell2SPCCodebook+8*Index);    
#else
    switch(TableIndex) {
    case 0: ReconstructVector(this, dec, offset, scale, Shell2SPCCodebook0+8*Index);    break;
    case 1: ReconstructVector(this, dec, offset, scale, Shell2SPCCodebook1+8*Index);    break;
    }
#endif
  }
}

/* ============================================================================== */
/* Decoding of 2 SP vectors */
/* ============================================================================== */

static void Decode2SPYVectors (Global *this, Byte *z, int scale) {

  double v[8];
  int mode;

  mode = this->GetIntraShellCombYHuffCode(this) + 1;

  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1SPYVector(this, z, YRowOffset, scale);
    break;
  case 2: 
    DecodeShell2SPYVector(this, z, YRowOffset, scale);
    break;
  case 3:
    DecodeExteriorVector(this, v, 0, 1);
    ReconstructVector(this, z, YRowOffset, scale, v);
    break;
  }
  switch (mode & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1SPYVector(this, z+2, YRowOffset, scale);
    break;
  case 2: 
    DecodeShell2SPYVector(this, z+2, YRowOffset, scale);
    break;
  case 3:
    DecodeExteriorVector(this, v, 0, 1);
    ReconstructVector(this, z+2, YRowOffset, scale, v);
    break;
  }
}

static void Decode2SPCVectors (Global *this, Byte *z, int scale) {

  double v[8];
  int mode;

  mode = this->GetIntraShellCombCHuffCode(this) + 1;

  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1SPCVector(this, z, CRowOffset, scale);
    break;
  case 2: 
    DecodeShell2SPCVector(this, z, CRowOffset, scale);
    break;
  case 3:
    DecodeExteriorVector(this, v, 1, 1);
    ReconstructVector(this, z, CRowOffset, scale, v);
    break;
  }
  switch (mode & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1SPCVector(this, z+2, CRowOffset, scale);
    break;
  case 2: 
    DecodeShell2SPCVector(this, z+2, CRowOffset, scale);
    break;
  case 3:
    DecodeExteriorVector(this, v, 1, 1);
    ReconstructVector(this, z+2, CRowOffset, scale, v);
    break;
  }
}

/* ============================================================================== */
/* Decoding of 2 TP vectors */
/* ============================================================================== */

static void Decode2TPYVectors (Global *this, Byte *dec, int scale) {

  double v[8];
  int mode;

  mode = this->GetInterShellCombYHuffCode(this) + 1;

  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1TPYVector(this, dec, YRowOffset, scale);
    break;
  case 2: 
    DecodeShell2TPYVector(this, dec, YRowOffset, scale);
    break;
  case 3: 
    DecodeExteriorVector(this, v, 0, 0);
    ReconstructVector(this, dec, YRowOffset, scale, v);
    break;
  }

  switch (mode & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1TPYVector(this, dec+2, YRowOffset, scale);
    break;
  case 2: 
    DecodeShell2TPYVector(this, dec+2, YRowOffset, scale);
    break;
  case 3: 
    DecodeExteriorVector(this, v, 0, 0);
    ReconstructVector(this, dec+2, YRowOffset, scale, v);
    break;
  }
}

static void Decode2TPCVectors (Global *this, Byte *dec, int scale) {

  double v[8];
  int mode;

  mode = this->GetInterShellCombCHuffCode(this) + 1;

  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1TPCVector(this, dec, CRowOffset, scale);
    break;
  case 2: 
    DecodeShell2TPCVector(this, dec, CRowOffset, scale);
    break;
  case 3: 
    DecodeExteriorVector(this, v, 1, 0);
    ReconstructVector(this, dec, CRowOffset, scale, v);
    break;
  }

  switch (mode & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1TPCVector(this, dec+2, CRowOffset, scale);
    break;
  case 2: 
    DecodeShell2TPCVector(this, dec+2, CRowOffset, scale);
    break;
  case 3: 
    DecodeExteriorVector(this, v, 1, 0);
    ReconstructVector(this, dec+2, CRowOffset, scale, v);
    break;
  }
}

/* ============================================================================== */
/* Decoding of different Block types */
/* ============================================================================== */


static void Set8x8Mean(Byte *dec, int w, int mean) {
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
  dec += w;
  dec[0] = dec[1] = dec[2] = dec[3] = dec[4] = dec[5] = dec[6] = dec[7] = mean;
} 

void DecodeIntraMeans(Global *this, Byte *ydec, Byte *udec, Byte *vdec) {
  int i, j;

  for (j=0; j<16; j+=8) {
    for (i=0; i<16; i+=8) {
      DecodeIntraMean(this, &MeanYPredictor);
      Set8x8Mean(ydec+(j*Yrowbytes)+i, Yrowbytes, MeanYPredictor);
    }
  }

  DecodeIntraMean(this, &MeanUPredictor);
  Set8x8Mean(udec, Crowbytes, MeanUPredictor);

  DecodeIntraMean(this, &MeanVPredictor);
  Set8x8Mean(vdec, Crowbytes, MeanVPredictor);

}

/* ============================================================================== */
/* Decoding of different Macroblock types */
/* ============================================================================== */

void DecodeSPMB(Global *this, Byte *ydec, Byte *udec, Byte *vdec, int scale, int bs) {

  int cbpy1, cbpy2, cbpc;

  TableIndex = GetOneBit(this);

  switch(TableIndex) {
  case 0: 
    this->GetSPYClassHuffCode = GetSPYClass0HuffCode;
    this->GetIntraShellCombYHuffCode = GetIntraShellCombY0HuffCode;
    this->GetIntraCBPYHuffCode = GetIntraCBPY0HuffCode;
    this->GetIntraCBPCHuffCode = GetIntraCBPC0HuffCode;
    this->GetIntraShellCombCHuffCode = GetIntraShellCombC0HuffCode;
    this->GetShell1SPYHuffCode = GetShell1SPY0HuffCode;
    this->GetShell2SPYHuffCode = GetShell2SPY0HuffCode;
    this->GetShell1SPCHuffCode = GetShell1SPC0HuffCode;
    this->GetShell2SPCHuffCode = GetShell2SPC0HuffCode;
    this->GetSPCClassHuffCode = GetSPCClass0HuffCode;
    break;
  case 1: 
    this->GetSPYClassHuffCode = GetSPYClass1HuffCode;
    this->GetIntraShellCombYHuffCode = GetIntraShellCombY1HuffCode;
    this->GetIntraCBPYHuffCode = GetIntraCBPY1HuffCode;
    this->GetIntraCBPCHuffCode = GetIntraCBPC1HuffCode;
    this->GetIntraShellCombCHuffCode = GetIntraShellCombC1HuffCode;
    this->GetShell1SPYHuffCode = GetShell1SPY1HuffCode;
    this->GetShell2SPYHuffCode = GetShell2SPY1HuffCode;
    this->GetShell1SPCHuffCode = GetShell1SPC1HuffCode;
    this->GetShell2SPCHuffCode = GetShell2SPC1HuffCode;
    this->GetSPCClassHuffCode = GetSPCClass1HuffCode;
    break;
  }

  cbpy1 = this->GetIntraCBPYHuffCode(this);

  if ( cbpy1 & 128 )
    Decode2SPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[0], scale);

  if ( cbpy1 & 64 )
    Decode2SPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[1], scale);

  if ( cbpy1 & 32 )
    Decode2SPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[2], scale);

  if ( cbpy1 & 16 )
    Decode2SPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[3], scale);

  if ( cbpy1 & 8 )
    Decode2SPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[0], scale);

  if ( cbpy1 & 4 )
    Decode2SPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[1], scale);

  if ( cbpy1 & 2 )
    Decode2SPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[2], scale);

  if ( cbpy1 & 1 )
    Decode2SPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[3], scale);

  cbpy2 = this->GetIntraCBPYHuffCode(this);

  if ( cbpy2 & 128 )
    Decode2SPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[0], scale);

  if ( cbpy2 & 64 )
    Decode2SPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[1], scale);

  if ( cbpy2 & 32 )
    Decode2SPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[2], scale);

  if ( cbpy2 & 16 )
    Decode2SPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[3], scale);

  if ( cbpy2 & 8 )
    Decode2SPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[0], scale);

  if ( cbpy2 & 4 )
    Decode2SPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[1], scale);

  if ( cbpy2 & 2 )
    Decode2SPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[2], scale);

  if ( cbpy2 & 1 )
    Decode2SPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[3], scale);

  cbpc = this->GetIntraCBPCHuffCode(this);

  if (cbpc & 128) 
    Decode2SPCVectors(this, udec+CBlockStart4[0], CSCALE);
  
  if (cbpc & 64) 
    Decode2SPCVectors(this, udec+CBlockStart4[1], CSCALE);
  
  if (cbpc & 32) 
    Decode2SPCVectors(this, udec+CBlockStart4[2], CSCALE);
  
  if (cbpc & 16) 
    Decode2SPCVectors(this, udec+CBlockStart4[3], CSCALE);

  if (cbpc & 8) 
    Decode2SPCVectors(this, vdec+CBlockStart4[0], CSCALE);
  
  if (cbpc & 4) 
    Decode2SPCVectors(this, vdec+CBlockStart4[1], CSCALE);
  
  if (cbpc & 2) 
    Decode2SPCVectors(this, vdec+CBlockStart4[2], CSCALE);
  
  if (cbpc & 1) 
    Decode2SPCVectors(this, vdec+CBlockStart4[3], CSCALE);
}

void CompensateTPLP_MB(Global *this, Byte *ydec, Byte *udec, Byte *vdec, 
		       Byte *yref, Byte *uref, Byte *vref,
		       int fx,    int fy) {

  int sx, sy, xh, yh;

  sx = (fx>>1); sy = (fy>>1);
  xh = fx&1; yh = fy&1;

  yref += (sy*Yrowbytes) + sx;

  CompensateHalfPelMotion16(this, ydec, yref, xh, yh, Yrowbytes);

  /*
  fx = ( fx % 4 == 0 ? fx >> 1 : (fx>>1)|1 );
  fy = ( fy % 4 == 0 ? fy >> 1 : (fy>>1)|1 );
  */

  fx >>= 1; fy >>= 1;

  sx = fx>>1; sy = fy>>1;
  xh = fx&1; yh = fy&1;

  uref += (sy*Crowbytes) + sx;
  CompensateHalfPelMotion8(this, udec, uref, xh, yh, Crowbytes);

  vref += (sy*Crowbytes) + sx;
  CompensateHalfPelMotion8(this, vdec, vref, xh, yh, Crowbytes);
}

void DecodeTPMB(Global *this, Byte *ydec, Byte *udec, Byte *vdec, int scale, int bs)
{
  int cbpy1, cbpy2, cbpc;

  TableIndex = GetOneBit(this);

  switch(TableIndex) {
  case 0:
    this->GetInterCBPYHuffCode = GetInterCBPY0HuffCode;
    this->GetInterCBPCHuffCode = GetInterCBPC0HuffCode;
    this->GetInterShellCombYHuffCode = GetInterShellCombY0HuffCode;
    this->GetInterShellCombCHuffCode = GetInterShellCombC0HuffCode;
    this->GetShell1TPYHuffCode = GetShell1TPY0HuffCode;
    this->GetShell1TPCHuffCode = GetShell1TPC0HuffCode;
    this->GetShell2TPYHuffCode = GetShell2TPY0HuffCode;
    this->GetShell2TPCHuffCode = GetShell2TPC0HuffCode;
    this->GetTPYClassHuffCode = GetTPYClass0HuffCode;
    this->GetTPCClassHuffCode = GetTPCClass0HuffCode;
    break;
  case 1:
    this->GetInterCBPYHuffCode = GetInterCBPY1HuffCode;
    this->GetInterCBPCHuffCode = GetInterCBPC1HuffCode;
    this->GetInterShellCombYHuffCode = GetInterShellCombY1HuffCode;
    this->GetInterShellCombCHuffCode = GetInterShellCombC1HuffCode;
    this->GetShell1TPYHuffCode = GetShell1TPY1HuffCode;
    this->GetShell1TPCHuffCode = GetShell1TPC1HuffCode;
    this->GetShell2TPYHuffCode = GetShell2TPY1HuffCode;
    this->GetShell2TPCHuffCode = GetShell2TPC1HuffCode;
    this->GetTPYClassHuffCode = GetTPYClass1HuffCode;
    this->GetTPCClassHuffCode = GetTPCClass1HuffCode;
    break;
  }

  cbpy1 = this->GetInterCBPYHuffCode(this);

  if ( cbpy1 & 128 )
    Decode2TPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[0], scale);

  if ( cbpy1 & 64 )
    Decode2TPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[1], scale);

  if ( cbpy1 & 32 )
    Decode2TPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[2], scale);

  if ( cbpy1 & 16 )
    Decode2TPYVectors(this, ydec+YBlockStart8[0]+YBlockStart4[3], scale);

  if ( cbpy1 & 8 )
    Decode2TPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[0], scale);

  if ( cbpy1 & 4 )
    Decode2TPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[1], scale);

  if ( cbpy1 & 2 )
    Decode2TPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[2], scale);

  if ( cbpy1 & 1 )
    Decode2TPYVectors(this, ydec+YBlockStart8[1]+YBlockStart4[3], scale);

  cbpy2 = this->GetInterCBPYHuffCode(this);

  if ( cbpy2 & 128 )
    Decode2TPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[0], scale);

  if ( cbpy2 & 64 )
    Decode2TPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[1], scale);

  if ( cbpy2 & 32 )
    Decode2TPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[2], scale);

  if ( cbpy2 & 16 )
    Decode2TPYVectors(this, ydec+YBlockStart8[2]+YBlockStart4[3], scale);

  if ( cbpy2 & 8 )
    Decode2TPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[0], scale);

  if ( cbpy2 & 4 )
    Decode2TPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[1], scale);

  if ( cbpy2 & 2 )
    Decode2TPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[2], scale);

  if ( cbpy2 & 1 )
    Decode2TPYVectors(this, ydec+YBlockStart8[3]+YBlockStart4[3], scale);

  cbpc = this->GetInterCBPCHuffCode(this);

  if (cbpc & 128) 
    Decode2TPCVectors(this, udec+CBlockStart4[0], CSCALE);
  
  if (cbpc & 64) 
    Decode2TPCVectors(this, udec+CBlockStart4[1], CSCALE);
  
  if (cbpc & 32) 
    Decode2TPCVectors(this, udec+CBlockStart4[2], CSCALE);
  
  if (cbpc & 16) 
    Decode2TPCVectors(this, udec+CBlockStart4[3], CSCALE);

  if (cbpc & 8) 
    Decode2TPCVectors(this, vdec+CBlockStart4[0], CSCALE);
  
  if (cbpc & 4) 
    Decode2TPCVectors(this, vdec+CBlockStart4[1], CSCALE);
  
  if (cbpc & 2) 
    Decode2TPCVectors(this, vdec+CBlockStart4[2], CSCALE);
  
  if (cbpc & 1) 
    Decode2TPCVectors(this, vdec+CBlockStart4[3], CSCALE);

}

void Print16x16Block(Byte *b, int w) {
  int i, j;

  for (j=0; j<16; j++) {
    for (i=0; i<16; i++) {
      fprintf(stderr,"%3d ", b[(j*w)+i]);
    }
    fprintf(stderr,"\n");
  }
}

/* ============================================================================== */
/* Motion vector decoding */
/* ============================================================================== */

int DecodeMVComponent(Global *this, int pred)
{
  int diff;

  diff = GetMVHuffCode(this) - 32;

  diff += pred;
  
  if (diff > 31) diff -= 64;
  else if (diff < -32) diff += 64;

  return (diff);
}

int DecodeRefineMVComponent(Global *this)
{
  int diff;

  diff = GetMVHuffCode(this) - 32;

  return (diff);
}

