/* 
 * pmod.c -- a synchronized sender for video & audio data. 
 * based on Uwe's pmod.c 
 *
 * usage: 
 *	pmod [-fps 16] [[-abpf 187.5 ] audiofile.e571 < file.pyra > file.tcp
 *
 * 21.4.97, jw
 */
#include <stdio.h>
#include "Syntax.h"

/* Gets the next bytes
 */
unsigned char GetByte() {
  return((unsigned char) getchar());
}

/* Checks for a left justified sync in its argument
 * Returns -1 if none, otherwise the sync type is returned and
 * info contains the side info 
 */

int CheckSync(unsigned long x, int *info) {
  if ((x&0xffffff00) != 0x000000100) return(-1); 
  *info = x&0x0f; 
  switch(x&SYNC_MASK) {
  case SEQUENCE_START_CODE: return(SEQUENCE_START_CODE); break;
  case PICTURE_START_CODE: return(PICTURE_START_CODE); break;
  case LAYER_START_CODE: return(LAYER_START_CODE); break;
  case SLICE_START_CODE: return(SLICE_START_CODE); break;
  case SEQUENCE_END_CODE : return(SEQUENCE_END_CODE); break;
  default:
    fprintf(stderr,"No valid Sync type after Sync Header\n");
    return(-1);
    break;
  }
}

/* If set to 0 output is switched off
 */

static int OutputOn;

/* Outputs nb bytes from buff, starting with the left most one
 */
void FlushBytes(unsigned long buff, int nb) {
  int i;
  int shift;

  if (OutputOn) {
    for (i=0; i<nb; i++) {
      shift = 24-(8*i);
      putchar(buff>>shift);
    }
  }
}

main(int argc, char *argv[]) {

  unsigned long BitBuff = 0;
  int SyncType, info;

  OutputOn = 1; /* Output on by default */

  /* Load BitBuff */

  BitBuff = (GetByte()<<24)|(GetByte()<<16)|(GetByte()<<8)|(GetByte());

  /* Exit if stream does not begin with START_OF_SEQUENCE  */
  if (CheckSync(BitBuff, &info)!=SEQUENCE_START_CODE) {
    fprintf(stderr,"Not a valid bitstream\n");
    exit(1);
  }

  /* Otherwise output first byte */
  FlushBytes(BitBuff, 1);


  while (1) { /* Run over the whole bitstream, exit when SEQUENCE_END is reached */

    BitBuff = (BitBuff<<8) | GetByte(); /* Get next byte */
    FlushBytes(BitBuff, 1); /* Output one buffered byte */

    while ( (SyncType=CheckSync(BitBuff,&info)) == -1) { /* No sync, keep on ... */
      BitBuff = (BitBuff<<8) | GetByte();
      FlushBytes(BitBuff, 1); /* Output buffered byte */
    }

    /* fprintf(stderr,"%d %d\n", SyncType, info); */

    if (SyncType==SEQUENCE_END_CODE) { /* Exit loop when SEQUENCE_END is reached */
      /* fprintf(stderr,"End of Sequence reached\n"); */
      break;
    }

    if (feof(stdin)) { /* Exit loop when end-of-file is reached */
      fprintf(stderr,"End of stream reached\n");
      break;
    }

    if ( ((SyncType==LAYER_START_CODE)&&((info&3)==0))  /* Layer 0 data */ 
       ||((SyncType==PICTURE_START_CODE)&&((info&3)==0))) { /* or a picture containing only layer 0 data */
      OutputOn=0; /* Switch off output */
      /* fprintf(stderr,"Out off\n"); */
    }

    if ((SyncType==PICTURE_START_CODE) /* Picture start and */
	&&((info&3)>0)) {               /* picture contains at least layer 1 data */
      OutputOn=1;                      /* Switch output on */
      /* fprintf(stderr,"Out on\n"); */
    }

  }
  OutputOn=1; 
  FlushBytes(BitBuff, 4);
}

