/* 
 * pmux.c -- a synchronized sender for video & audio data. 
 * based on Uwe's pmod.c 
 *
 *	pmux [-qcif] [-fps 16] [-abpf 187.5 ] [-audio] file_base > file.tcp
 *
 * Opens file_base.pyra for video in, file_base.g723 for audio in and writes
 * (as fast as possible) to stdout.
 *
 * 21.4.97, jw
 */
#include <stdio.h>
#include <stdlib.h>		/* atof() */
#include <string.h>		/* strncmp() */
#include <fcntl.h>		/* open() */
#include <assert.h>		/* assert() */
#include "Syntax.h"

#if 0	/* raw 16bit Audio */
#define ROUND_CHUNK(x) ((((int)(x)) + 3) & ~3)
#else
#define ROUND_CHUNK(x) (((((int)(x))+3)/3)*3)
#endif

struct sync
{
  int qcif_only;	/* Pyramid layer to transport */
  int chunk;		/* granularity for audio packets in bytes */
  double fps;		/* video frames per second */
  double abps;		/* audio bytes per second */
  double abpf;		/* audio bytes per video frame */
  double mark;		/* desired audio bytes water mark */
  double water;		/* current audio bytes water level */
  FILE *vfp;
  FILE *afp;
} AudioSync = { 0, 0, 0.0, 3000.0, 0.0, 0.0, 0.0, NULL, NULL };

/* Gets the next bytes
 */
unsigned char GetByte(fp) 
FILE *fp;
{
  return((unsigned char) getc(fp));
}

/* Checks for a left justified sync in its argument
 * Returns -1 if none, otherwise the sync type is returned and
 * info contains the side info 
 */

int CheckSync(unsigned long x, int *info) {
  if ((x&0xffffff00) != 0x000000100) return(-1); 
  *info = x&0x0f; 
  switch(x&SYNC_MASK) {
  case SEQUENCE_START_CODE: return(SEQUENCE_START_CODE); break;
  case PICTURE_START_CODE: return(PICTURE_START_CODE); break;
  case LAYER_START_CODE: return(LAYER_START_CODE); break;
  case SLICE_START_CODE: return(SLICE_START_CODE); break;
  case SEQUENCE_END_CODE : return(SEQUENCE_END_CODE); break;
  default:
    fprintf(stderr,"No valid Sync type after Sync Header\n");
    return(-1);
    break;
  }
}

/* If set to 0 output is switched off
 */

static int OutputOn;

/* Outputs nb bytes from buff, starting with the left most one
 */
void FlushBytes(unsigned long buff, int nb) {
  int i;
  int shift;

  if (OutputOn) {
    for (i=0; i<nb; i++) {
      shift = 24-(8*i);
      putchar(buff>>shift);
    }
  }
}

int AudioBytesNow(s)
struct sync *s;
{
  int d = 0;

  if (!s->afp)
    return 0;
  
  assert(s->water >= 0.0);	/* were too slow (audio data missing) */

  while (s->water < s->mark)
    {
      d += s->chunk;
      s->water += (double)s->chunk;
    }
  return d;
}


int EmitAudio(s, n)
struct sync *s;
int n;
{
  static unsigned char *buf = NULL;
  static int bsize = 0;
  int i;

  if (bsize < n)
    {
      if (buf) free((char *)buf);
      bsize = (n+255) & ~255;
      buf = (unsigned char *)malloc(bsize);
    }

  for (i = 0; i < n; i++)
    {
      if (feof(s->afp))
        break;
      buf[i] = fgetc(s->afp);
    }

#ifdef DEBUG
  fprintf(stderr, "EmitAudio: %d bytes audio\n", i);
#endif

  putchar(i>>8);  
  putchar(i&255);  

  for (n = 0; n < i; n++)
    putchar(buf[n]);
  
  return 0;
}

int exit_with_usage(av0)
char *av0;
{
  fprintf(stderr, "Usage: %s [opts] filebasename > file.tcp\n", av0);
  fprintf(stderr, "Valid options are:\n");
  fprintf(stderr, "       -qcif          pass only qcif layer. Default: All layers (cif).\n");
  fprintf(stderr, "       -audio         Read audio data from filebasename.g723.\n");
  fprintf(stderr, "                      Default: Readonly video from filebasename.pyra\n");
#if 0
  fprintf(stderr, "       -fps ff.f      calculate ff.f frames per second. Default 30(15 if qcif).\n");
#endif
  fprintf(stderr, "       -abps nnnn.n   calculate nnn.n audio-bytes per second. Default %g.\n", AudioSync.abps);
#if 0
  fprintf(stderr, "       -aoff s.s      keep audio s.s seconds ahead of video. Default 2/ff.f.\n");
#endif
  exit(0);
}


int main(int argc, char *argv[]) 
{
  unsigned long BitBuff = 0;
  int SyncType, info;
  struct sync *s = &AudioSync;
  char *av0 = argv[0];
  int off = 0, au = 0;
  char *filebasename = NULL;
  char buf[1024];

  argv++;
  argc--;

  while(*argv)
    {
      if (**argv == '-')
        {
	       if (!strncmp(*argv, "-q",   2)) s->qcif_only = 1;
	  else if (!strncmp(*argv, "-f",   2)) s->fps    = atof(*(++argv));
	  else if (!strncmp(*argv, "-ab",  3)) s->abps   = atof(*(++argv));
	  else if (!strncmp(*argv, "-au",  3)) au = 1;
	  else if (!strncmp(*argv, "-ao",  3)) off       = atoi(*(++argv));
	  else exit_with_usage(av0);
	}
      else if (!filebasename)
        filebasename = *argv;
      else 
        exit_with_usage(av0);
      argv++;
    }

  if (!filebasename)
    exit_with_usage(av0);

  if (strlen(filebasename) > sizeof(buf) - 10)
    return 0;

  sprintf(buf, "%s.pyra", filebasename);
  if (!(s->vfp = fopen(buf, "r")))
    {
      fprintf(stderr, "%s: cannot open video file %s.\n", av0, buf);
      exit(0);
    }

  if (au)
    {
      sprintf(buf, "%s.g723", filebasename);
      s->afp = fopen(buf, "r");
    }

  if (!s->fps)
    s->fps = s->qcif_only ? 15.0 : 30.0;
  
  s->abpf = s->abps/s->fps;
  s->mark = 4.0 * s->abpf;
  s->chunk = ROUND_CHUNK(s->abpf);

  OutputOn = 1; /* Output on by default */

  /* Load BitBuff */

  BitBuff = (GetByte(s->vfp)<<24)|(GetByte(s->vfp)<<16)|
            (GetByte(s->vfp)<<8)|(GetByte(s->vfp));

  /* Exit if stream does not begin with START_OF_SEQUENCE  */
  if (CheckSync(BitBuff, &info)!=SEQUENCE_START_CODE) {
    fprintf(stderr,"Not a valid bitstream\n");
    exit(1);
  }

  /* Otherwise output first byte */
  FlushBytes(BitBuff, 1);

  if (off && s->afp)
    {
      int i;

      fprintf(stderr, "pmux skipping %d audio bytes\n", off);
      for (i = 0; i < off; i++)
	getc(s->afp);
    }

  while (1) { /* Run over the whole bitstream, exit when SEQUENCE_END is reached */

    BitBuff = (BitBuff<<8) | GetByte(s->vfp); /* Get next byte */
    FlushBytes(BitBuff, 1); /* Output one buffered byte */

    while ( (SyncType=CheckSync(BitBuff,&info)) == -1) { /* No sync, keep on ... */
      BitBuff = (BitBuff<<8) | GetByte(s->vfp);
      FlushBytes(BitBuff, 1); /* Output buffered byte */
    }

    /* fprintf(stderr,"%d %d\n", SyncType, info); */

    if (SyncType==SEQUENCE_END_CODE) { /* Exit loop when SEQUENCE_END is reached */
      /* fprintf(stderr,"End of Sequence reached\n"); */
      break;
    }

    if (feof(stdin)) { /* Exit loop when end-of-file is reached */
      fprintf(stderr,"End of stream reached\n");
      break;
    }

    if ( ((SyncType==LAYER_START_CODE)&&((info&3)==0))  /* Layer 0 data */ 
       ||((SyncType==PICTURE_START_CODE)&&((info&3)==0))) { /* or a picture containing only layer 0 data */
      if (s->qcif_only)
        OutputOn=0; /* Switch off output */
      /* fprintf(stderr,"Out off\n"); */
    }

    if ((SyncType==PICTURE_START_CODE) /* Picture start and */
	&&((info&3)>0)) {               /* picture contains at least layer 1 data */
      OutputOn=1;                      /* Switch output on */
      /* fprintf(stderr,"Out on\n"); */
    }

    if (SyncType == PICTURE_START_CODE && OutputOn)
      {
        int n;
	static int p = 0;

#ifdef DEBUG
	fprintf(stderr, "%04d: ", p++);
#endif
        if ((n = AudioBytesNow(s)))
	  {
	    BitBuff |= 1<<3;
	    BitBuff <<= 8;
	    FlushBytes(BitBuff, 3);
	    EmitAudio(s, n);
	    BitBuff = (GetByte(s->vfp)<<16)|(GetByte(s->vfp)<<8)|(GetByte(s->vfp));
	  }
	s->water -= s->abpf;
      }
  }
  OutputOn=1; 
  FlushBytes(BitBuff, 4);
  return 0;
}

