#include "global.h"

static int VectorFieldAllocated = 0;
static int XScaled[4], YScaled[4];

MultiScaleFieldPtr NewVectorField (Global *this) {

  int                l;
  int                ow,oh,bw,bh;
  MVFieldPtr         myField;
  MultiScaleFieldPtr tmpField;

  tmpField = NEW(MultiScaleFieldStruct);

  for (l = 0; l < this->SeqHdr.Layers; l++) {

    ow = this->SeqHdr.HSize / this->SeqHdr.XSubsamp[l];
    bw = this->SeqHdr.BlockWidth[l];
    oh = this->SeqHdr.VSize / this->SeqHdr.YSubsamp[l];
    bh = this->SeqHdr.BlockHeight[l];

    tmpField->Layer[l] = myField = NEW(MVFieldStruct);

    myField->layer = l;
    myField->width = ow/bw;
    myField->height = oh/bh;
    myField->size = myField->width*myField->height;
    myField->bw = bw;
    myField->bh = bh;

    myField->XComp = NEWELEMS(short, myField->size);
    myField->YComp = NEWELEMS(short, myField->size);

  }
  VectorFieldAllocated = 1;
  return(tmpField);
}

void FreeVectorField (Global *this, MultiScaleFieldPtr field1, MultiScaleFieldPtr field2) {

  int                l;

  if (VectorFieldAllocated) {

    for (l=0; l<this->SeqHdr.Layers; l++) {
      free(field1->Layer[l]->XComp);
      free(field1->Layer[l]->YComp);
      free(field1->Layer[l]);
    }
    free(field1);

    for (l=0; l<this->SeqHdr.Layers; l++) {
      free(field2->Layer[l]->XComp);
      free(field2->Layer[l]->YComp);
      free(field2->Layer[l]);
    }
    free(field2);
  }
}


void ComputeMVScaling (Global *this) {

  int l;
  int w,h,bw,bh;
  int w0, h0, bw0, bh0;

  for (l=0; l<this->SeqHdr.Layers-1; l++) {

    bw = this->SeqHdr.BlockWidth[l];
    bh = this->SeqHdr.BlockHeight[l];
    w = this->SeqHdr.HSize/this->SeqHdr.XSubsamp[l];
    h = this->SeqHdr.VSize/this->SeqHdr.YSubsamp[l];

    bw0 = this->SeqHdr.BlockWidth[l+1];
    bh0 = this->SeqHdr.BlockHeight[l+1];
    w0 = this->SeqHdr.HSize/this->SeqHdr.XSubsamp[l+1];
    h0 = this->SeqHdr.VSize/this->SeqHdr.YSubsamp[l+1];

    if ( (w/bw)==(w0/bw0) ) 
      XScaled[l] = 1;
    else if ( (w/bw) == 2*(w0/bw0) ) {
      XScaled[l] = 0;
    }
    else {
      fprintf(stderr,"Don't know how to scale initial x component\n");
      exit(1);
    }

    if ( (h/bh)==(h0/bh0) ) 
      YScaled[l] = 1;
    else if ( (h/bh) == 2*(h0/bh0) ) {
      YScaled[l] = 0;
    }
    else {
      fprintf(stderr,"Don't know how to scale initial y component\n");
      exit(1);
    }
  }
  XScaled[this->SeqHdr.Layers] = YScaled[this->SeqHdr.Layers] = -1;
}

void GetMotionVector (MVFieldPtr mv, int i, int j, int *x, int *y)
{
  int mvAddress;

  if (mv==NULL) 
    *x = *y = 0;

  else {

    mvAddress = (j/mv->bh)*mv->width+(i/mv->bw);

    *x = mv->XComp[mvAddress];
    *y = mv->YComp[mvAddress];
  }
}

void InterpolateMVField2 (MVFieldPtr in, MVFieldPtr out, int inRefTempRef, int inActTempRef, int outRefTempRef, int outActTempRef, int fullpel) {

  int i, j;
  int refdist, actdist, spatialdist;
  short x, y;
  short *xto, *yto;

  /* As far this only works if blocksizes in the two different layers are the same */


  /*
     Compute temporal distance for appropriate scaling
     refdist: distance between the reference and the frame this vector field estimates
     actdist: distance between the reference and the actual frame for which a vector field
     should be predicted
     */

  out->ActTempRef = outActTempRef;
  out->RefTempRef = outRefTempRef;

  in->ActTempRef = inActTempRef;
  in->RefTempRef = inRefTempRef;

  refdist =  (in->ActTempRef - in->RefTempRef);
  actdist =   (outActTempRef - outRefTempRef);

  spatialdist = (in->layer - out->layer);

  /*
     Now change either refdist or actdist to account for different spatial scales
     */

  if (spatialdist < 0) 
    refdist *= (1<<(-spatialdist));
  else if (spatialdist > 0)
    actdist *= (1<<spatialdist);

  /* Half-pel accuracy is assumed for each vector.
     If we need a full-pel accuracy prediction we have to scale down MVs by a factor of 2
     */
  
  if (fullpel) refdist *= 2; 
  
  if ((in->width==out->width)&&(in->height==out->height)) {
    for (i=0; i<in->size; i++) {
      out->XComp[i] = (in->XComp[i] * actdist) / refdist;
      out->YComp[i] = (in->YComp[i] * actdist) / refdist;
    }
  } else if ((2*in->width==out->width)&&(2*in->height==out->height)) {
    for (j=0; j<in->height; j++) {
      xto = out->XComp + (2*j*out->width);
      yto = out->YComp + (2*j*out->width);
      for (i=0; i<in->width; i++) {
	x = in->XComp[j*in->width+i];
	y = in->YComp[j*in->width+i];
/*
	if (fullpel) {
	  x >>= 1;
	  y >>= 1;
	}  */

	/* Simple upsampling */
	*xto = *(xto+1) = *(xto+out->width) = *(xto+out->width+1) = (x * actdist)/refdist;
	*yto = *(yto+1) = *(yto+out->width) = *(yto+out->width+1) = (y * actdist)/refdist;

	xto += 2; yto += 2;
      }
    }
  } else {
    fprintf(stderr,"Don't know how to interpolate MV Field\n");
    exit(1);
  }
}


void SetMotionVector (MVFieldPtr mv, int i, int j, int x, int y)
{
  int row, off;

  if (mv != NULL) {
    row = (j/mv->bh)*mv->width;
    off = (i/mv->bw);
  
    (mv->XComp + row)[off] = x;
    (mv->YComp + row)[off] = y;
  }
}

