/*
 * Play a G723/24kbps encoded audio file
 *
 * compile with:  cc -fast -xO4 play_723.c # -laio
 *
 * $Id: audio.c,v 1.3 1998/06/15 01:00:19 jnweiger Exp $
 */

#ifndef WITHOUT_AUDIO
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef __sun
# ifdef FILE
#  include <sun/audioio.h>
# else
#  include <sys/audioio.h>
# endif
#endif
#include <sys/ioctl.h>
#include <sys/conf.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/asynch.h>
#include <errno.h>
#include "debug.h"

#define INLEN    8001
#define DEC_BITS 3

static  int fdaudio;
static  struct g72x_state state;
static  struct audio_info ai;
static  unsigned char outbuf[INLEN*8/3];
static  int StartSample;
static  int SamplesWritten = 0;

/*---------------------------------------------------------------------------*
 *  Decode a G723/24 bitstream to a G711/ulaw bytestream
 */

#include "g711.c"
#include "g72x.c"
#include "g723_24.c"

void ResetAudioSampleTimer() {
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  StartSample = ai.play.samples;
}

int AudioSamplesPlayed(int num) {
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  debug2("diff=%d, num=%d\n", ai.play.samples - StartSample, num);
  /* ueberlauf???? */
  if ((ai.play.samples - StartSample)<num)
    return(0);
  else
    return(1);
}

void
sleep1000(msec)
int msec;
{
  struct timeval t;

  t.tv_sec = (long) (msec / 1000);
  t.tv_usec = (long) ((msec % 1000) * 1000);
  select(0, (fd_set *)0, (fd_set *)0, (fd_set *)0, &t);
}

void WaitForCompletion(int fps, int tempref) {
  static double nsamples = -999999.0;
  double spf;

  if (!fps) 
    return;

  spf = 8000.0/fps;

  if (nsamples < -999998.0)
    {
      AUDIO_INITINFO(&ai);
      ai.play.pause = 0;
      ioctl(fdaudio, AUDIO_SETINFO, &ai);
      nsamples = (double)ai.play.samples;
    }
  debug1("%04d: WaitForCompletion:\n", tempref);

  nsamples += spf;
  while (!ioctl(fdaudio, AUDIO_GETINFO, &ai))
    {
      debug2("\t spf=%g diff=%d\n", spf, SamplesWritten - ai.play.samples);
      debug3("\t SamplesWritten=%d, nsamples=%g, ai.play.samples=%d\n",
             SamplesWritten, nsamples, ai.play.samples);
      if ((ai.play.samples >= (int)nsamples) ||
          (SamplesWritten - ai.play.samples <= 200))
        break;

      sleep1000(1);
    }
}

int decode_buf(int num, unsigned char *inbuf, unsigned char *outbuf)
{
  unsigned char *in, *out;
  int i, bits;
  unsigned long bitbuf;
  unsigned int code, sample;


  out = outbuf;
  in = inbuf;
  bits = 0;
  bitbuf = 0L;

  /*
   *  Read 3 bytes from input, containing 8 samples; decode
   */
  for(i=num/8; i; i--) {
    bitbuf = *in | (*(in+1)<<8) | (*(in+2)<<16);
    in += 3;

    *out = g723_24_decoder(bitbuf & 7, AUDIO_ENCODING_ULAW, &state);
    out[1] = g723_24_decoder((bitbuf>>3) & 7, AUDIO_ENCODING_ULAW, &state);
    out[2] = g723_24_decoder((bitbuf>>6) & 7, AUDIO_ENCODING_ULAW, &state);
    out[3] = g723_24_decoder((bitbuf>>9) & 7, AUDIO_ENCODING_ULAW, &state);
    out[4] = g723_24_decoder((bitbuf>>12) & 7, AUDIO_ENCODING_ULAW, &state);
    out[5] = g723_24_decoder((bitbuf>>15) & 7, AUDIO_ENCODING_ULAW, &state);
    out[6] = g723_24_decoder((bitbuf>>18) & 7, AUDIO_ENCODING_ULAW, &state);
    out[7] = g723_24_decoder((bitbuf>>21) & 7, AUDIO_ENCODING_ULAW, &state);
    out +=8;
  }

  /*
   *  Deal with rest in input buffer
   */
  for(i=num%8; i; i--) {
    if(bits < 3) {
      bitbuf |= *in++ << bits;
      bits += 8;
    }
    code = bitbuf & 0x7;
    bitbuf >>= 3;
    bits -= 3;

    sample = g723_24_decoder(code, AUDIO_ENCODING_ULAW, &state);
    *out++ = (unsigned char) sample;
    ASSERT(0);		/* shall not be used */
  }
  return out - outbuf;
}

/*---------------------------------------------------------------------------*
 *  Write an ulaw sample sequence asynchronously to the audio device
 */

void play_buf(int num, unsigned char *buf)
{
#ifdef DEBUG
  static int tfd = -1;

  if (tfd < 0)
    tfd = open("/tmp/a.g723", O_CREAT|O_WRONLY, 0644);
#endif

  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  if (ai.play.error) {
    debug("underflow\n");
    ai.play.error = 0;
    ioctl(fdaudio, AUDIO_SETINFO, &ai);
  }
  write(fdaudio, buf, num);
#ifdef DEBUG
  write(tfd, buf, num);
#endif
  SamplesWritten += num;
}

/*---------------------------------------------------------------------------*/

int PlayAudioPacket(int compressed_samples, unsigned char *inbuf) {
  /*
   *  Read to inbuffer; decode to outbuffer; play
   */

  if (compressed_samples > INLEN) {
    debug("Audio buffer overflow\n");
    compressed_samples = INLEN;
  }

  ASSERT(compressed_samples%3 == 0);

  decode_buf(compressed_samples*8/DEC_BITS, inbuf, outbuf);
  play_buf(compressed_samples*8/DEC_BITS, outbuf);
  return compressed_samples;
}

int CloseAudio() {
  return close(fdaudio);  /* wait for audio buffer to empty */
}

int InitAudio() {
  

  if((fdaudio = open("/dev/audio", O_WRONLY|O_NDELAY, 0666)) < 0) 
    {
      if (errno == EBUSY || errno == EAGAIN)
	{
	  perror("/dev/audio is busy, discarding sound");
	  fdaudio = open("/dev/null", O_WRONLY, 0666);
	  return 0;
	}
      perror("/dev/audio");
      exit(3);
    }
  debug1("InitAudio: fdaudio=%d\n", fdaudio);
  if(ioctl(fdaudio, AUDIO_GETINFO, &ai) < 0) {
    perror("ioctl AUDIO_GETINFO");
    exit(1);
  }
  debug("InitAudio: AUDIO_GETINFO\n");
  ai.play.sample_rate = 8000;
  ai.play.channels = 1;
  ai.play.precision = 8;
  ai.play.encoding = AUDIO_ENCODING_ULAW;
  ai.play.pause = 1;
  ai.play.error = 0;
  ai.play.samples = 0;

  if(ioctl(fdaudio, AUDIO_SETINFO, &ai) < 0) {
    perror("ioctl AUDIO_SETINFO");
    exit(1);
  }
  debug("InitAudio: AUDIO_SETINFO\n");

  return 0;
}

#else /* WITHOUT_AUDIO */

void WaitForCompletion(int fps, int tempref) {}
int InitAudio() { return 0; }
int CloseAudio() { return 0; }
int PlayAudioPacket(int s, unsigned char *inbuf) { return s; }

#endif /* WITHOUT_AUDIO */
