/* display.c, X11 interface                                                 */

 /* the Xlib interface is closely modeled after
  * mpeg_play 2.0 by the Berkeley Plateau Research Group
  */


#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "global.h"
#include "error.h"
#include "display.h"

#ifdef QUIET
static int quiet = 1;
#else
static int quiet = 0;
#endif

unsigned char *rgb_image;
int coded_picture_width, coded_picture_height, chrom_width;

/* X11 related variables */
static int preload = 0;
static Display *display = NULL;
static Window window;
static Colormap cmap;
static GC gc;
static XImage *ximage;
static Visual visual;
static int screen;
unsigned char pixel[256];
int visdepth;


#ifndef WITHOUT_SHM
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>

#include "extern.h"

static int HandleXError (Display *dpy, XErrorEvent *event);
static void InstallXErrorHandler (void);
static void DeInstallXErrorHandler (void);

static XShmSegmentInfo shminfo1, shminfo2;
static int gXErrorFlag;
static int CompletionType = -1;
#endif /* !WITHOUT_SHM */

#ifndef WITHOUT_DGA
#include <dga/dga.h>
#include <dga/dga_externaldefs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fbio.h>

#include <fcntl.h>
#include <sys/mman.h>
#include <sys/cg14io.h>

Dga_drawable dga_id;
unsigned long *dga_map;
int dga_fb_width;
int dga_fb_height;
#endif /* !WITHOUT_DGA */


/* for colormap generation on pseudocolor visual */
static int matrix_coefficients = 5;
static int convmat[8][4] =
{
  {117504, 138453, 13954, 34903}, /* no sequence_display_extension */
  {117504, 138453, 13954, 34903}, /* ITU-R Rec. 709 (1990) */
  {104597, 132201, 25675, 53279}, /* unspecified */
  {104597, 132201, 25675, 53279}, /* reserved */
  {104448, 132798, 24759, 53109}, /* FCC */
  {104597, 132201, 25675, 53279}, /* ITU-R Rec. 624-4 System B, G */
  {104597, 132201, 25675, 53279}, /* SMPTE 170M */
  {117579, 136230, 16907, 35559}  /* SMPTE 240M (1987) */
};

/* ---------------------------------------------------------------------------
 *  X error handling
 */

static int HandleXError(dpy, event)
Display *dpy;
XErrorEvent *event;
{
#ifdef USE_SHM
  /* if (this->Options.use_shm)	 XXX: Can't access this from an error handler */
    gXErrorFlag = 1;
#endif
  return 0;
}

static void InstallXErrorHandler()
{
  XSetErrorHandler(HandleXError);
  XFlush(display);
}

static void DeInstallXErrorHandler()
{
  XSetErrorHandler(NULL);
  XFlush(display);
}


/* ---------------------------------------------------------------------------
 *  Build colormap fuer PseudoColor visuals
 */

static void AllocColors(Global *this)
{
  int crv, cbu, cgu, cgv;
  int y, u, v, r, g, b;
  int private;
  int i;
  unsigned long tmp_pixel;
  XColor xcolor;
  XWindowAttributes xwa;

  /* matrix coefficients */
  crv = convmat[matrix_coefficients][0];
  cbu = convmat[matrix_coefficients][1];
  cgu = convmat[matrix_coefficients][2];
  cgv = convmat[matrix_coefficients][3];

  /* allocate colors */

  cmap = DefaultColormap(display, screen);
  private = 0;

  /* color allocation:
   * i is the (internal) 8 bit color number, it consists of separate
   * bit fields for Y, U and V: i = (yyyyuuvv), we don't use yyyy=0000
   * yyyy=0001 and yyyy=1111, this leaves 48 colors for other applications
   *
   * the allocated colors correspond to the following Y, U and V values:
   * Y:   40, 56, 72, 88, 104, 120, 136, 152, 168, 184, 200, 216, 232
   * U,V: -48, -16, 16, 48
   *
   * U and V values span only about half the color space; this gives
   * usually much better quality, although highly saturated colors can
   * not be displayed properly
   *
   * translation to R,G,B is implicitly done by the color look-up table
   */
  for (i=32; i<240; i++)
  {
    /* color space conversion */
    y = 16*((i>>4)&15) + 8;
    u = 32*((i>>2)&3)  - 48;
    v = 32*(i&3)       - 48;

    y = 76309 * (y - 16); /* (255/219)*65536 */

    r = CLIP2(this, ((y + crv*v + 32768)>>16) );
    g = CLIP2(this, ((y - cgu*u -cgv*v + 32768)>>16) );
    b = CLIP2(this, ((y + cbu*u + 32786)>>16) );

    /* X11 colors are 16 bit */
    xcolor.red   = r << 8;
    xcolor.green = g << 8;
    xcolor.blue  = b << 8;

    if (XAllocColor(display, cmap, &xcolor) != 0)
      pixel[i] = xcolor.pixel;
    else
    {
      /* allocation failed, have to use a private colormap */

      if (private)
        FATAL( "Couldn't allocate private colormap");

      private = 1;

      if (!quiet)
        fprintf(stderr, "Using private colormap (%d colors were available).\n",
          i-32);

      /* Free colors. */
      while (--i >= 32)
      {
        tmp_pixel = pixel[i]; /* because XFreeColors expects unsigned long */
        XFreeColors(display, cmap, &tmp_pixel, 1, 0);
      }

      /* i is now 31, this restarts the outer loop */

      /* create private colormap */

      XGetWindowAttributes(display, window, &xwa);
      cmap = XCreateColormap(display, window, xwa.visual, AllocNone);
      XSetWindowColormap(display, window, cmap);
      XInstallColormap(display, cmap);
    }
  }
}


/* ---------------------------------------------------------------------------
 *  Create XImage to store RGB image in
 */

static void NewXImage(Global *this, int ww, int wh) {
  char dummy;

#ifndef WITHOUT_SHM
  /*  In GrayScale mode we use the Y part of the YUV frame in the XImage
   *  directly, sparing to copy it around. Of course we could put the frames
   *  into shmem, but that'd be much work (and probably exceed constraints)
   */
  if (this->Options.use_shm)
    {
      if ((this->Options.visgray && !this->Options.visexpand) ||
          !XShmQueryExtension(display))
	{
	  this->Options.use_shm = 0;
	  if (!quiet)
	    fprintf(stderr,
	      "Shared memory not supported, reverting to normal Xlib\n");
	}
    }

  if (this->Options.use_shm)
    CompletionType = XShmGetEventBase(display) + ShmCompletion;

  InstallXErrorHandler();

  if (this->Options.use_shm) {
    ximage = XShmCreateImage(display, &visual, visdepth, ZPixmap, NULL,
			     &shminfo1, ww, wh);

    /* If no go, then revert to normal Xlib calls. */

    if (ximage==NULL) {
      if (ximage!=NULL) XDestroyImage(ximage);
      if (!quiet)
        fprintf(stderr, "Shared memory error, disabling (Ximage error)\n");
      goto shmemerror;
    }

    /* Success here, continue. */

    shminfo1.shmid =
      shmget(IPC_PRIVATE, ximage->bytes_per_line * ximage->height,
             IPC_CREAT | 0700);

    if (shminfo1.shmid<0) {
      XDestroyImage(ximage);
      if (!quiet) {
        fprintf(stderr, "Shared memory error, disabling (seg id error)\n");
	perror("shmget");
      }
      goto shmemerror;
    }

    shminfo1.shmaddr = (char *) shmat(shminfo1.shmid, 0, 0);
    shminfo2.shmaddr = (char *) shmat(shminfo2.shmid, 0, 0);

    if (shminfo1.shmaddr==((char *) -1)) {
      XDestroyImage(ximage);
      if (shminfo1.shmaddr!=((char *) -1))
        shmdt(shminfo1.shmaddr);
      if (!quiet) {
        fprintf(stderr, "Shared memory error, disabling (address error)\n");
      }
      goto shmemerror;
    }

    ximage->data = shminfo1.shmaddr;
    rgb_image = (unsigned char *)ximage->data;
    shminfo1.readOnly = False;
    XShmAttach(display, &shminfo1);

    XSync(display, False);

    if (gXErrorFlag) {
      /* Ultimate failure here. */
      XDestroyImage(ximage);
      shmdt(shminfo1.shmaddr);
      if (!quiet)
        fprintf(stderr, "Shared memory error, disabling.\n");
      gXErrorFlag = 0;
      goto shmemerror;
    } else {
      shmctl(shminfo1.shmid, IPC_RMID, 0);
    }

    if (!quiet) {
      fprintf(stderr, "Sharing memory.\n");
    }
  }
  else {
shmemerror:
    this->Options.use_shm = 0;
#endif /* !WITHOUT_SHM */

    ximage = XCreateImage(display,None,visdepth,ZPixmap,0,&dummy, ww,wh,8,0);

    if (ximage==NULL) {
      fprintf(stderr, "XCreateImage failed\n");
      exit(1);
    }

    /* For Grayscale we take Y part of YUV image, no further modif. req. */
    if(!this->Options.visgray || this->Options.visexpand) {
      if (!(rgb_image = (unsigned char *)malloc(ximage->bytes_per_line * wh)))
	FATAL("malloc failed");
    }
#ifndef WITHOUT_SHM
  }
  DeInstallXErrorHandler();
#endif /* !WITHOUT_SHM */
  /* set image parameters */
  ximage->byte_order = MSBFirst;
  ximage->bitmap_bit_order = MSBFirst;
  _XInitImageFuncPtrs(ximage);
}


/* ---------------------------------------------------------------------------
 *  Purge XImage
 */

void FreeXImage(Global *this) {
#ifndef WITHOUT_SHM
  if (this->Options.use_shm) {
    /* This dumps core !?
    XDestroyImage(ximage);
    */
    XShmDetach(display, &shminfo1);
    shmdt(shminfo1.shmaddr);
  }
#else
  XDestroyImage(ximage);
#endif
}

void preload_dpy(Global *this, Display *d, int s, Window w, GC g, Visual *v,
	int depth)
{
  /* XXX: fixme: another bunch of static variables */
  if (display)
    abort();	/* must not be already initialized! */
  display = d;
  screen = s;
  window = w;
  gc = g;
  visual = *v;
  visdepth = depth;
  preload = 1;
}

unsigned long
PixelColor(dpy, cmap, r, g, b)
Display *dpy;
Colormap cmap;
int r, g, b;
{
  XColor c;
 
  c.red = r << 8;
  c.green = g << 8;
  c.blue = b << 8;
 
  XAllocColor(dpy, cmap, &c);
  return c.pixel;
}

/* ---------------------------------------------------------------------------
 *  Open X Display
 */

void init_display (Global *this, int ww, int wh)
{
  int vnum;
  unsigned int fg, bg;
  double gamma;
  XSizeHints hint;
  XEvent xev;
  XSetWindowAttributes xswa;
  XVisualInfo vtempl, *vlist;
  Colormap colormap;
  unsigned int xswamask;
  XGCValues xgcv;

  /* save parameters for later */
  coded_picture_height = wh;
  coded_picture_width = ww;
  chrom_width = ww/2;

  quiet = this->Options.quiet;

  if (!preload)
    {
      display = XOpenDisplay(NULL);

      if (display == NULL)
	FATAL("Can not open display\n");

      screen = DefaultScreen(display);

      /* Get some colors */

      bg = WhitePixel (display, screen);
      fg = BlackPixel (display, screen);

      /* First search a TrueColor visual */
      vlist = NULL;
      if(!this->Options.visgray && !this->Options.vis8) {
	vtempl.screen = screen;
	vtempl.depth  = visdepth = 24;
	vlist = XGetVisualInfo (display,
				VisualScreenMask | VisualDepthMask,
				&vtempl, &vnum);
        if (!vnum)
	  {
	    vtempl.depth = visdepth = 32;
	    vlist = XGetVisualInfo (display,
	    			VisualScreenMask | VisualDepthMask,
				&vtempl, &vnum);
	  }
      }
      else vnum = 0;

      /* Else take a PseudoColor (or GrayScale in the first place) */
      if(vnum == 0) {
	long vinfomask = VisualScreenMask | VisualDepthMask;
	if(vlist) XFree(vlist);
	vtempl.screen = screen;
	vtempl.depth  = visdepth = 8;
	if(this->Options.visgray) {
	  vtempl.class = StaticGray;
	  vinfomask |= VisualClassMask;
	}
	vlist = XGetVisualInfo (display, vinfomask, &vtempl, &vnum);
	if(vnum == 0) {
	  fprintf(stderr, "No supported visuals found\n");
	  exit(1);
	}
      }
      visual = *vlist[0].visual;
      if(!quiet) printf("Using visual 0x%x\n", (unsigned int)visual.visualid);

#if defined(__sun) && !defined(FILE)
      if(!quiet) {
	XSolarisGetVisualGamma(display, screen, visual, &gamma);
	printf("Gamma of display is: %lf\n", gamma);
      }
#endif

      colormap = XCreateColormap(display,
				 RootWindow(display, screen),
				 vlist[0].visual, AllocNone);

      if (visdepth >= 24)
        {
	  /* WhitePixel() and BlackPixel() do not work with TrueColor */
	  bg = PixelColor(display, colormap, 0, 0, 64);
	  fg = PixelColor(display, colormap, 255, 255, 255);
	}

      if (this->Options.cover_window)
        {
	  hint.x = 0;
	  hint.y = 0;
	}
      else
        {
	  hint.x = 200;
	  hint.y = 200;
	}
      
      if (this->Options.visexpand && (visdepth>=24 || this->Options.visgray)) {
	hint.width = 2*ww;  /* width of the display window */
	hint.height = 2*wh;  /* height of the display window */
      }
      else {
	if(this->Options.visexpand && !this->Options.visgray) {
	  this->Options.visexpand = 0;
	  fprintf(stderr, "Big display not supported for PseudoColor visual, use -gray.\n");
	}
	hint.width = ww;
	hint.height = wh;
      }
      hint.flags = PPosition | PSize;

      /* Make the window */
      xswa.background_pixel = fg;
      xswa.border_pixel     = bg;
      xswa.colormap = colormap;
      xswamask = CWBackPixel | CWBorderPixel;
      if(visdepth >= 24 || this->Options.visgray) xswamask |= CWColormap;

      window = XCreateWindow(display,
      			     this->Options.cover_window ? 
			       (Window)this->Options.cover_window :
			       RootWindow(display, screen),
			     hint.x, hint.y, hint.width, hint.height, 0,
			     vlist[0].depth, InputOutput, vlist[0].visual,
			     xswamask, &xswa);
      if (!window) {
	  fprintf(stderr, "Unable to create window\n");
	  exit(1);
      }
      XFree(vlist);

      XSelectInput(display, window, ExposureMask | KeyPressMask |
		     ButtonPressMask | StructureNotifyMask);

      /* Tell other applications about this window */

      XSetStandardProperties (display, window, "ntcodec","ntcodec",
			      None, NULL, 0, &hint);

      /* Map window. */
      XMapRaised(display, window);

      /* Wait for map. */
      do  {
	XNextEvent(display, &xev);
      }
      while (xev.type != MapNotify || xev.xmap.event != window); 

      if(visdepth >= 24 || this->Options.visgray) {
	gc = XCreateGC(display, window, 0L, &xgcv);
	init_yuv2rgb(this);
      }
      else {
	gc = DefaultGC(display, screen);
	AllocColors(this);
	init_dither();
      }
    }
  else	/* display was preloaded */
    {
      if (visdepth >= 24 || this->Options.visgray)
        init_yuv2rgb(this);
      else
        {
	  AllocColors(this);
	  init_dither();
	}
    }

#ifndef WITHOUT_DGA
  if (this->Options.use_dga)
    {
      /*
       *  Set up DGA
       *  - supported only on 24 bit displays (would need seperate routines for 8)
       *  - query which fb is being used
       *  - map fb device memory into client address space
       *  Doesn't work with Creator (ffb), since fbio(7I) ioctls are unsupported
       */
      dga_id = 0;
      if(visdepth >= 24) {
	DGA_INIT();
	if((dga_id = XDgaGrabDrawable(display, window)) == NULL) {
	  if(!quiet) fprintf(stderr, "DGA not available\n");
	}
      }
      if(dga_id) {            /* device specific initialization */
	struct fbgattr fba;

	DGA_DRAW_LOCK(dga_id, (short)-1);

	if (ioctl(dga_draw_devfd(dga_id), FBIOGATTR, &fba) == -1) {
	    if(!quiet) perror("ioctl(FBIOGATTR) error");
	    DGA_DRAW_UNLOCK(dga_id);
	    dga_id = 0;
	}
	else {
	  long mapoff, mapsize;
	  /*
	  printf("fb type: %d,%d,%d,%d,%d,%d\n", fba.real_type,
		 fba.fbtype.fb_height, fba.fbtype.fb_width, fba.fbtype.fb_depth,
		 fba.fbtype.fb_cmsize, fba.fbtype.fb_size);
	  */

	  switch(fba.real_type) {
	    case FBTYPE_MDICOLOR:
	      mapoff = MDI_CHUNKY_BGR_MAP;
	      mapsize = fba.fbtype.fb_size;
	      break;
	    case 21: /* SUNW,tcx */
	      mapoff = 0x01000000;   /* TCX_VADDR_DFB24 */
	      mapsize = fba.fbtype.fb_width * (fba.fbtype.fb_depth / 8) *
			fba.fbtype.fb_height;
	      break;
	    default:
	      if(!quiet)
		fprintf(stderr, "Unsupported framebuffer for DGA: %d\n",
				fba.real_type);
	      DGA_DRAW_UNLOCK(dga_id);
	      XDgaUnGrabDrawable(dga_id);
	      dga_id = 0;
	  }
	  if(dga_id) {
	    dga_fb_width = fba.fbtype.fb_width;
	    dga_fb_height = fba.fbtype.fb_height;

	    /* get the DGA mappings for the frame buffer */
	    dga_map = (unsigned long *)mmap(0, mapsize,
			   PROT_READ|PROT_WRITE, MAP_SHARED,
			   dga_draw_devfd(dga_id), mapoff);
	    if(dga_map == MAP_FAILED) {
	      perror("mmap");
	      DGA_DRAW_UNLOCK(dga_id);
	      XDgaUnGrabDrawable(dga_id);
	      dga_id = 0;
	    }
	  }
	}
	if(dga_id) DGA_DRAW_UNLOCK(dga_id);
      }
      if(dga_id == 0) {
	if(this->Options.visexpand) NewXImage(this, 2*ww,2*wh);
	else NewXImage(this, ww,wh);
      }
      else if(!quiet) fprintf(stderr, "Will use DGA\n");
    }
  else
#endif /* !WITHOUT_DGA */
    {
      if(this->Options.visexpand) NewXImage(this, 2*ww,2*wh);
      else NewXImage(this, ww,wh);
    }
}


/* ---------------------------------------------------------------------------
 *  Resize
 */

void ResizeDisplayWindow (Global *this, int ww, int wh)
{
  int i;
  
  XResizeWindow (display, window, ww, wh);

  for (i=0; i<100000; i++) ;

  FreeXImage(this);

  for (i=0; i<100000; i++) ;
  NewXImage(this, ww, wh);
}


/* ---------------------------------------------------------------------------
 *  unused
 */

void SetDisplayWindowTitle (char *title) {
  if (display != NULL)
    XStoreName(display, window, title);
}


/* ---------------------------------------------------------------------------
 *  Close X Display
 */

void exit_display(Global *this)
{
  if (display != NULL) {
    if (!preload)
      XDestroyWindow(display,window);
    FreeXImage(this);
  }
}

/* ---------------------------------------------------------------------------
 *  Display complete RGB image in window
 */
  
/* display dithered image */
void display_image(Global *this) {
#ifndef WITHOUT_SHM
  XEvent xev;
#endif

#ifndef WITHOUT_DGA
  if(this->Options.use_dga && dga_id) {
    fprintf(stderr,"FATAL: display_image called for DGA window\n");
    exit(-1);
  }
  else
#endif

#ifndef WITHOUT_SHM
  if (this->Options.use_shm)
  {
    XShmPutImage(display, window, gc, ximage, 
       	         0, 0, 0, 0, ximage->width, ximage->height, True);
    XFlush(display);
      
    while (1)
    {
      XNextEvent(display, &xev);
      if (xev.type == CompletionType)
        break;
    }
  }
  else 
#endif /* !WITHOUT_SHM */
  {
    ximage->data = (char *) rgb_image;
    XPutImage(display, window, gc, ximage, 0, 0, 0, 0, ximage->width, ximage->height);
  }
}

