#include <math.h>
#include <string.h>
#include <stdio.h>

#include "Parameter.h"
#include "Motion.h"
#include "Crop.h"
#include "Codebook.h"
#include "Lattice.h"
#include "Output.h"
#include "VLC.h"
#include "Statistics.h"
#include "dpcm.h"
/* #include "ExtVect.h" */
#include "ExtVectNew.h"

#include "BlockCoder.h"
#include "CodeSymbol.h"

#include "Scaling.h"
#include "Training.h"

#include "config.h"
#include "compand.h"

/* #define COMPAND
#define COMPSCALE 200.0
*/

#define CATT 0.85

#define SPY1CONST 0.0
#define SPY2CONST 0.0

#define SPYCONST 0.0
#define TPYCONST 0.0

extern double gyVar, guVar, gvVar;

/* Students */
/*
static double AttY[4] = { 0.85, 0.9, 0.95, 0.95 };
static double AttC    = 0.75;
static double AttT    = 0.75;
*/
/* Flower */
/*
static double AttY[4] = { 0.85, 0.9, 0.95, 0.95 };
static double AttC    = 0.75;
static double AttT    = 0.75;
*/
/* Table */
/*
static double AttY[4] = { 0.75,0.9,0.95,1.0 };
static double AttC    = 0.75;
static double AttT    = 0.75;
*/
/* Football */
/*
static double AttY[4] = { 0.75,0.9,0.95,0.95 };
static double AttC    = 0.75;
static double AttT    = 0.75;
*/
/* Mobile */
/*
static double AttY[4] = { 0.9,0.9,0.95,0.95 };
static double AttC    = 0.75;
static double AttT    = 0.75;
*/

#define MEAN_SCALE 6

/* ****************************************************************************** */

typedef struct {
  
  int coded;

  VECTOR vmem[48];
  VECTOR inpt[48];
     int norm[48];
     int abssum[48];
     int Polarity[48];
     int GroupIndex[48];
     int Flip[48];
     int var;
     int class;
     int bits;
  double rescale[48];
  double Variance;
} QBlockStruct;

typedef QBlockStruct *QBlockPtr;

typedef struct {
  int Index;
  int FrameType;
  int Layer;
  int IsLowestLayer;
  int IsHighestLayer;
  int Scale;
  int lScale;
  int BlockSize;
  int CodingMode;
  int TableIndex;
  int UTableIndex;
  int VTableIndex;
  int T[3];
  int coded;
  int hasMV;
  int ymean[4];
  int cmean[2];
  int cvpy[4];
  int cvpc[2];
  int cbpy1;
  int cbpy2;
  int cbpc;
  int mvfx, mvfy;
  int mvbx, mvby;
} SideInfoStruct;

static SideInfoStruct SideInfo;
static QBlockStruct QBlock;

void SetLayerBlockThresholds(int *t) {
  SideInfo.T[0] = t[0];
  SideInfo.T[1] = t[1];
  SideInfo.T[2] = t[2];
}

static FILE *prot;

void OpenProtFile(char *name) {

  prot = fopen(name,"w");
}

void CloseProtFile() {
  fclose(prot);
}

/*
========================================================================================================================
*/

static int ymean_pred, umean_pred, vmean_pred;

void ResetIntraPredictors() {
  ymean_pred = umean_pred = vmean_pred = 128;
}

static int SizeTab[256] = { 
  0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5,
  5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
  8, 8, 8 };

void CodeIntraMean (int mean, int *MeanPred)
{
  int diff, size;
  int UsedBits = 0;

  diff = mean - *MeanPred;
  size = SizeTab[ABS(diff)];

  UsedBits += PutMeanSizeHuffCode(size);

  if (size) 
    UsedBits += (diff>0) ? EmitBits(diff,size,"Mean") : EmitBits((1<<size)+diff-1,size,"Mean");

  SMIncrementResidBitCount(UsedBits);
  *MeanPred = mean; 
}

/* ======================================================================================================================== */
static void TrainAndReconShell1Vector(int Index, int Sign, VECTOR y, VECTOR x, int mode, int comp) {
  VECTOR z;

  ApplySymmetry(x, z, 0, Sign);

  if (comp==0) {
    if (mode==SP_BLOCK)
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell1SPYCellEntry(Index, z, Shell1SPYCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1SPYCodebook0+8*Index, y, 0, Sign); 
	break;
      case 1:
	AddShell1SPYCellEntry(Index, z, Shell1SPYCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1SPYCodebook1+8*Index, y, 0, Sign); 
	break;
      }
    else
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell1TPYCellEntry(Index, z, Shell1TPYCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1TPYCodebook0+8*Index, y, 0, Sign); 
	break;
      case 1:
	AddShell1TPYCellEntry(Index, z, Shell1TPYCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1TPYCodebook1+8*Index, y, 0, Sign); 
	break;
      }
  } else {
    if (mode==SP_BLOCK)
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell1SPCCellEntry(Index, z, Shell1SPCCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1SPCCodebook0+8*Index, y, 0, Sign); 
	break;
      case 1:
	AddShell1SPCCellEntry(Index, z, Shell1SPCCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1SPCCodebook1+8*Index, y, 0, Sign); 
	break;
      }
    else
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell1TPCCellEntry(Index, z, Shell1TPCCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1TPCCodebook0+8*Index, y, 0, Sign); 
	break;
      case 1:
	AddShell1TPCCellEntry(Index, z, Shell1TPCCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell1TPCCodebook1+8*Index, y, 0, Sign); 
	break;
      }
  }
}
/* ======================================================================================================================== */

static void TrainAndReconShell2Vector(int Index, int Sign, int sy, VECTOR y, VECTOR x, int mode, int comp) {
  VECTOR z;

  ApplySymmetry(x, z, sy, Sign);

  if (comp==0) {
    if (mode==SP_BLOCK)
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell2SPYCellEntry(Index, z, Shell2SPYCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2SPYCodebook0+8*Index, y, sy, Sign); 
	break;
      case 1:
	AddShell2SPYCellEntry(Index, z, Shell2SPYCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2SPYCodebook1+8*Index, y, sy, Sign); 
	break;
      }
    else
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell2TPYCellEntry(Index, z, Shell2TPYCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2TPYCodebook0+8*Index, y, sy, Sign); 
	break;
      case 1:
	AddShell2TPYCellEntry(Index, z, Shell2TPYCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2TPYCodebook1+8*Index, y, sy, Sign); 
	break;
      }
  } else {
    if (mode==SP_BLOCK)
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell2SPCCellEntry(Index, z, Shell2SPCCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2SPCCodebook0+8*Index, y, sy, Sign); 
	break;
      case 1:
	AddShell2SPCCellEntry(Index, z, Shell2SPCCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2SPCCodebook1+8*Index, y, sy, Sign); 
	break;
      }
    else
      switch(SideInfo.TableIndex) {
      case 0:
	AddShell2TPCCellEntry(Index, z, Shell2TPCCodebook0+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2TPCCodebook0+8*Index, y, sy, Sign); 
	break;
      case 1:
	AddShell2TPCCellEntry(Index, z, Shell2TPCCodebook1+8*Index, SideInfo.TableIndex);
	ApplySymmetry(Shell2TPCCodebook1+8*Index, y, sy, Sign); 
	break;
      }
  }
}


/* ======================================================================================================================== */
/* From this table the sign of a shell 1 vector can be computed */

int Shell1SignTab[120] = {
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,-1,1,1,1,1,1,-1,1,-1,-1,-1,-1,1,1,1,1,1,1,
  -1,1,-1,-1,-1,-1,1,1,-1,-1,-1,-1,1,-1,-1,-1,1,-1,-1,1,-1,1,1,1,1,-1,1,1,
  -1,1,-1,1,-1,1,-1,1,-1,1,-1,-1,1,-1,-1,1,-1,1,-1,1,-1,-1,1,-1,-1,-1,1,-1,
  -1,-1,1,1,-1,1,-1,1,-1,-1,1,-1,-1,-1,1,-1,-1,-1,1,-1,1,-1,-1,-1,1,-1,1,1,1,-1,-1
};

static int SignOfFirstNonZero(double x[8]) {
  int i;

  i = 0;
  while (x[i]==0.0) 
    i++;

  return((x[i] < 0.0) ? -1 : 1);
}

static void Shell1Info(double x[8], int *index, int *sign) {
  int gn;
  gn = LatticePointGN(x);        /* a group number between 1 and 256 */
  *index = GN2GroupIndex[gn-1];  /* maps gn to indices suitable for Shell1 codebook */
  *sign = (SignOfFirstNonZero(x)==Shell1SignTab[*index]) ? 0 : 1;
}
/* ======================================================================================================================== */


static void CodeShell1Vector(QBlockPtr qblock, int n, int mode, int comp) { 

  int index, sign;
  int UsedBits = 0;

  Shell1Info(qblock->vmem[n], &index, &sign);

  if (comp==0) {
    if (mode==SP_BLOCK)
      UsedBits += PutShell1SPYHuffCode(index, SideInfo.TableIndex);
    else
      UsedBits += PutShell1TPYHuffCode(index, SideInfo.TableIndex);
  } else {
    if (mode==SP_BLOCK)
      UsedBits += PutShell1SPCHuffCode(index, SideInfo.TableIndex);
    else
      UsedBits += PutShell1TPCHuffCode(index, SideInfo.TableIndex);
  }

  UsedBits += EmitBits(sign,1,"Shell1 sign");

  TrainAndReconShell1Vector(index, (sign==0) ? 1 : -1, qblock->vmem[n], qblock->inpt[n], mode, comp);
  SMIncrementResidBitCount(UsedBits);
}

/* ======================================================================================================================== */
static void Shell2Info(double x[8], int *index, int *sign, int *trans, int *sy) {
  int gn, gi, ci, si, symtype;

  gn = LatticePointGN(x);             /* a group number between 1 and 256 */
  ci = GN2GroupIndex[gn-1];           /* Class index between 0 and 134 */
  si = Shell2Groups[ci].VMemAddress;  /* maps ci to the start index of ci */

  /* ABS(gi) is the row index WITHIN the matrix of 8 orthogonal vectors spanning one Shell2 group */
  gi = (int) ScalarMul(x, Shell2Groups[ci].AddressVector);
  *sign = (gi < 0) ? 1 : 0;
  gi = ABS(gi);

  symtype = Shell2Groups[ci].Symmetry; /* type of symmetry */

  if (symtype == 0) {
    gi = gi - 1; *trans = -1;  *sy = 0; /* trans=-1: no t bit need to be sent, no symmetry appliable */
  } else {
    if ((gi & 1) == 0) { gi = (gi >> 1) - 1; *trans = 1; *sy = symtype; } else { gi = (gi >> 1); *trans = 0; *sy = 0; }
  }

  *index = si + gi;
}
/* ======================================================================================================================== */

static int VEQ(double *x1, double *x2) {
  int i;

  for (i=0; i<8; i++) {
    if (x1[i] != x2[i]) return(0);
  }
  return(1);
}

static void CodeShell2Vector(QBlockPtr qblock, int n, int mode, int comp) {
  int index, sign, trans, sy;
  int UsedBits = 0;

  Shell2Info(qblock->vmem[n], &index, &sign, &trans, &sy);

  if (comp==0) {
    if (mode==SP_BLOCK)
      UsedBits += PutShell2SPYHuffCode(index, SideInfo.TableIndex);
    else
      UsedBits += PutShell2TPYHuffCode(index, SideInfo.TableIndex);
  } else {
    if (mode==SP_BLOCK)
      UsedBits += PutShell2SPCHuffCode(index, SideInfo.TableIndex);
    else
      UsedBits += PutShell2TPCHuffCode(index, SideInfo.TableIndex);
  }

  UsedBits += EmitBits(sign,1,"Shell2 sign");
  if (trans != -1)
    UsedBits += EmitBits(trans,1,"Shell2 trans");

  TrainAndReconShell2Vector(index, (sign==0) ? 1 : -1, sy, qblock->vmem[n], qblock->inpt[n], mode, comp); 

  SMIncrementResidBitCount(UsedBits);
}

static void GetShell2Centro(double v[8], double c[8], int comp) {
  int Index, sign, trans, sy, polarity;

  Shell2Info(v, &Index, &sign, &trans, &sy);
  polarity = (sign==0) ? 1 : -1;

  if (SideInfo.CodingMode == SP_BLOCK) {
    if (comp==0) {
      switch (SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell2SPYCodebook0+8*Index, c, sy, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell2SPYCodebook1+8*Index, c, sy, polarity); 
	break;
      }
    } else {
      switch (SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell2SPCCodebook0+8*Index, c, sy, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell2SPCCodebook1+8*Index, c, sy, polarity); 
	break;
      }
    }
  } else {
    if (comp==0) {
      switch (SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell2TPYCodebook0+8*Index, c, sy, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell2TPYCodebook1+8*Index, c, sy, polarity); 
	break;
      }
    } else {
      switch (SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell2TPCCodebook0+8*Index, c, sy, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell2TPCCodebook1+8*Index, c, sy, polarity); 
	break;
      }
    }
  }
}

static void GetShell1Centro(double v[8], double c[8], int comp) {

  int Index, sign, polarity;

  Shell1Info(v, &Index, &sign);
  polarity = (sign==0) ? 1 : -1;

  if (SideInfo.CodingMode == SP_BLOCK) {
    if (comp==0) {
      switch(SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell1SPYCodebook0+8*Index, c, 0, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell1SPYCodebook1+8*Index, c, 0, polarity); 
	break;
      }
    } else {
      switch(SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell1SPCCodebook0+8*Index, c, 0, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell1SPCCodebook1+8*Index, c, 0, polarity); 
	break;
      }
    }
  } else {
    if (comp==0) {
      switch(SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell1TPYCodebook0+8*Index, c, 0, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell1TPYCodebook1+8*Index, c, 0, polarity); 
	break;
      }
    } else {
      switch(SideInfo.TableIndex) {
      case 0:
	ApplySymmetry(Shell1TPCCodebook0+8*Index, c, 0, polarity); 
	break;
      case 1:
	ApplySymmetry(Shell1TPCCodebook1+8*Index, c, 0, polarity); 
	break;
      }
    }
  }
}

static void GetCentroid(int shell, int comp, double v[8], double c[8]) {
  int n;

  switch (shell) {
  case 0: for (n=0; n<8; n++) c[n] = 0.0; break;
  case 2: GetShell1Centro(v, c, comp); break;
  case 4: GetShell2Centro(v, c, comp); break;
  default:
    for (n=0; n<8; n++) c[n] = v[n]; break;
  }
}


/* ======================================================================================================================== */

void AnalyzeVect(VECTOR v) {
  int i;
  int norm, type;

  norm = l2_norm(v);

  if ((norm==6)||(norm==8)) {
    fprintf(stderr,"Norm=%d\t", norm);
    switch (norm) {
    case 6:
      if ((v[0]==1.5)||(v[0]==0.5)) type=2;
      else if (l1_norm(v)==6) type=0;
      else type = 1;
      break;
    case 8:
      switch(l1_norm(v)) {
      case 4: type = 3; break;
      case 6: if ((ABS(v[0])==2.5)||(ABS(v[0])==0.5)) type=7; else type = 4; break;
      case 7: type = 6; break;
      case 8: type = 5; break;
      default:
	for (i=0; i<8; i++) fprintf(stderr,"%4.1f ", v[i]);
	
      }
      break;
    }
    fprintf(stderr,"Type=%d\n", type);
  }
}

void EncodeOtherVectors(double v[8], int mode, int ti) {
  int i;
  int v1, v2, s1, s2, sc,vc;
  int translate;
  int UsedBits = 0;

  if (v[0]!=(int)v[0])
    translate = 1;
  else 
    translate = 0;

  UsedBits += EmitBits(translate,1,"Other translate");

  for (i=0; i<4; i++) {

    s1 = ZSGN(v[i]);
    if (translate) {
      if (s1==1) v1 = (int) (ABS(v[i])-0.5);
      else if (s1==-1) v1 = (int) (ABS(v[i])-0.5);
    } else v1 = (int) ABS(v[i]);

    if (v1>7) { v1 = 7; }

    s2 = ZSGN(v[i+4]);
    if (translate) {
      if (s2==1) v2 = (int) (ABS(v[i+4])-0.5);
      else if (s2==-1) v2 = (int) (ABS(v[i+4])-0.5);
    } else v2 = (int) ABS(v[i+4]);

    if (v2>7) { v2 = 7; }

    sc = 3*(s1+1) + (s2+1);
    UsedBits += PutCombSignHuffCode(sc);

    if ((s1 != 0)&&(s2 !=0)) {
      vc = ((v1)<<3)|(v2);
      UsedBits += PutVCombHuffCode(vc);
    } else {
      if (s1 != 0) UsedBits += PutVMagHuffCode(v1);
      if (s2 != 0) UsedBits += PutVMagHuffCode(v2);
    }

    if (translate) {
      if (s1==1) v[i] = v1 + 0.5;
      else if (s1==-1) v[i] = -v1 - 0.5;
    } else v[i] = s1*v1;

    if (translate) {
      if (s2==1) v[i+4] = v2 + 0.5;
      else if (s2==-1) v[i+4] = -v2 - 0.5;
    } else v[i+4] = s2*v2;
  }
  SMIncrementResidBitCount(UsedBits);
}

static void CodeExteriorVector(int comp, int n) {
  int index;
  int sign, nsign;
  int UsedBits = 0;

  index = FindCBIndex(QBlock.vmem[n], &sign, &nsign);

  if (comp==0) {
    if (SideInfo.CodingMode==SP_BLOCK)
      UsedBits += PutSPYClassHuffCode(index, SideInfo.TableIndex); 
    else
      UsedBits += PutTPYClassHuffCode(index, SideInfo.TableIndex); 
  } else {
    if (SideInfo.CodingMode==SP_BLOCK)
      UsedBits += PutSPCClassHuffCode(index, SideInfo.TableIndex); 
    else
      UsedBits += PutTPCClassHuffCode(index, SideInfo.TableIndex); 
  }

  if (index==64) {
    EncodeOtherVectors(QBlock.vmem[n], SideInfo.CodingMode, SideInfo.TableIndex);
  }
  else
    UsedBits += EncodeExtVector(QBlock.vmem[n], index, sign, nsign);

    SMIncrementResidBitCount(UsedBits);
}

/**********************************************************************
 *
 *      Name:        QuantizeInterBlock   
 *      Description: Quantizes a predicted block   
 *
 *      Input:       Number of block within macroblock   
 *                   Prediction signal
 *                   Original signal
 *                   vectors per block
 *      Returns:     zero if all vectors were quantized to zero
 *                   non-zero otherwise
 *                      
 *      Side effects: Writes quantization information to QBlock
 *
 *      Author: Uwe Horn, Telecommunications Institute, Univ. Erlangen
 *              uhorn@nt.e-technik.uni-erlangen.de
 ***********************************************************************/

static void QuantizeVect(int comp, int *orig, int *pred, double scale, double *y, double *x) {

  int i;
  int diff;
  int l2;
  int class;
  double in_mse;
  double mse, uncoded_mse;
  double r[8];

  mse = in_mse = 0;
  for (i=0; i<8; i++) {
    diff = orig[i] - pred[i];
    in_mse += (diff*diff);
    x[i] = (double) diff / scale;
    mse += (diff*diff);
  }

  if (SideInfo.CodingMode==SP_BLOCK) {
    if (comp==0)
      in_mse = gyVar/(double)(scale*scale);
    else if (comp==1) in_mse = guVar/(double)(scale*scale);
    else in_mse = gvVar/(double)(scale*scale);
  } else in_mse = 0.075;

  if (in_mse<0.1) class = 0;
  else if (in_mse<1.0) class = 1;
  else class = 2;

  uncoded_mse = mse;

  NearestE8LatticePoint (x, y);
  l2 = l2_norm(y);

  GetCentroid(l2, comp, y, r);

  mse = 0;
  for (i=0; i<8; i++) {
    diff = orig[i] - ROUNDCLIP(pred[i] + r[i]*scale);
    mse += (diff*diff);
  }

  if ((uncoded_mse)<=(mse)) for (i=0; i<8; i++) y[i] = 0;
  /*
  if (l2>4) {
    vvar = 0; for (i=0; i<8; i++) vvar += (x[i]*x[i]); 
    fprintf(prot, "%d %f ", class, in_mse);
    for (i=0; i<8; i++) fprintf(prot, "%6.3f ", x[i]);
    fprintf(prot, "\t");
    for (i=0; i<8; i++) fprintf(prot, "%4.1f ", y[i]);
    fprintf(prot, "\n");
  }
  */
}

static int QuantizeInterBlock(int n, int comp, int *pred, int *orig) {
  int coded;
  int scale;

  if (comp==0)
    scale = SideInfo.Scale;
  else
    scale = SideInfo.Scale*CATT;

  coded = 0;

  QuantizeVect(comp, orig, pred, scale, QBlock.vmem[n], QBlock.inpt[n]);

  QBlock.norm[n] = l2_norm(QBlock.vmem[n]);
  QBlock.abssum[n] = l1_norm(QBlock.vmem[n]);

  if (QBlock.norm[n] != 0) coded = 1;

  QuantizeVect(comp, orig+8, pred+8, scale, QBlock.vmem[n+1], QBlock.inpt[n+1]);

  QBlock.norm[n+1] = l2_norm(QBlock.vmem[n+1]);
  QBlock.abssum[n+1] = l1_norm(QBlock.vmem[n+1]);

  if (QBlock.norm[n+1] != 0) coded = 1;

  if (n<16) { SideInfo.cbpy1 <<= 1; SideInfo.cbpy1 |= coded; }
  else if (n<32) { SideInfo.cbpy2 <<= 1; SideInfo.cbpy2 |= coded; }
  else { SideInfo.cbpc <<= 1; SideInfo.cbpc |= coded; }

  return(coded);
}

/* ======================================================================================================================== */

/**********************************************************************
 *
 *      Name:        CodeBlock    
 *      Description: Encodes a block by jointly coding pairs of two vectors   
 *
 *      Input:       Number of block within macroblock   
 *                   Pointer to decoded block   
 *                   Pointer to prediction   
 *                      
 *      Returns:        
 *                      
 *      Side effects:   
 *      Author: Uwe Horn, Telecommunications Institute, Univ. Erlangen
 *              uhorn@nt.e-technik.uni-erlangen.de
 ***********************************************************************/

static int meanhist[256] = { 0 } ;

void PrintMeanHist(){

  int i;

  for (i=0; i<256; i++)
    fprintf(stderr,"%d ", meanhist[i]);
}

static void ReconstructVector(int comp, int n, int *dec, int *pre) {
  int i;
  int cscale;
  double f;
  /* Students */

  /*
  if ((SideInfo.Layer==1) && (SideInfo.CodingMode==SP_BLOCK)) {
    if (comp==0) f = 0.94; else f = 0.85;
  } else if ((SideInfo.Layer==0) && (SideInfo.CodingMode==SP_BLOCK)) {
    if (comp==0) f = 0.90; else f = 0.70;

  } else   if ((SideInfo.Layer==1) && (SideInfo.CodingMode==FWD_TPLP_BLOCK)) {
    if (comp==0) f = 0.70; else f = 0.7;
  } else if ((SideInfo.Layer==0) && (SideInfo.CodingMode==FWD_TPLP_BLOCK)) {
    if (comp==0) f = 0.70; else f = 1.0;
  } else f = 1.0;
  */

  f = 1;

  if (comp==0)
    for (i=0; i<8; i++) {
      dec[i] = ROUNDCLIP(pre[i] + (QBlock.vmem[n][i]*SideInfo.Scale)); 
    }
  else {
    cscale = (SideInfo.Scale*CATT);
    for (i=0; i<8; i++) {
      dec[i] = ROUNDCLIP(pre[i] + (QBlock.vmem[n][i]*cscale)); 
    }
  }
}

static void CodeAndReconstructExterior(int comp, int n, int *dec, int *pre) {
  CodeExteriorVector(comp, n);
  ReconstructVector(comp, n, dec, pre);
}

static void CodeBlock(int n, int comp, int *dec, int *pre) {

  int norm1, norm2;
  int mode1, mode2, mode;
  int i;
  int UsedBits = 0;
  
  norm1 = QBlock.norm[n];
  norm2 = QBlock.norm[n+1];

  if ( (norm1 == 0) && (norm2 == 0) ) {

    for (i=0; i<16; i++) dec[i] = pre[i];

  } else {

    switch(norm1) {
    case 0: mode1 = 0; break;
    case 2: mode1 = 1; break;
    case 4: mode1 = 2; break;
    default: mode1 = 3; break;
    }

    switch(norm2) {
    case 0: mode2 = 0; break;
    case 2: mode2 = 1; break;
    case 4: mode2 = 2; break;
    default: mode2 = 3; break;
    }

    mode = ((mode1<<2)+mode2);

    if (comp==0) {
      if (SideInfo.CodingMode==SP_BLOCK) {
	UsedBits += PutIntraShellCombYHuffCode(mode-1, SideInfo.TableIndex); 
      } else {
	UsedBits += PutInterShellCombYHuffCode(mode-1, SideInfo.TableIndex); 
      }
    } else {
      if (SideInfo.CodingMode==SP_BLOCK) {
	UsedBits += PutIntraShellCombCHuffCode(mode-1, SideInfo.TableIndex);  
      } else {
	UsedBits += PutInterShellCombCHuffCode(mode-1, SideInfo.TableIndex); 
      }
    }

    switch (mode1) {
    case 0:
      for (i=0; i<8; i++) dec[i] = pre[i];
      break;
    case 1:
      CodeShell1Vector(&QBlock, n, SideInfo.CodingMode, comp); 
      ReconstructVector(comp, n, dec, pre);
      break;
    case 2:
      CodeShell2Vector(&QBlock, n, SideInfo.CodingMode, comp); 
      ReconstructVector(comp, n, dec, pre);
      break;
    case 3:
      CodeAndReconstructExterior(comp, n, dec, pre);
      break;
    }

    n++; dec += 8; pre += 8;

    switch (mode2) {
    case 0:
      for (i=0; i<8; i++) dec[i] = pre[i];
      break;
    case 1:
      CodeShell1Vector(&QBlock, n, SideInfo.CodingMode, comp);
      ReconstructVector(comp, n, dec, pre);
      break;
    case 2:
      CodeShell2Vector(&QBlock, n, SideInfo.CodingMode, comp);
      ReconstructVector(comp, n, dec, pre);
      break;
    case 3:
      CodeAndReconstructExterior(comp, n, dec, pre);
      break;
    }
  }
  SMIncrementResidBitCount(UsedBits);
}

/* ======================================================================================================================== */

void SetFrameCodingInfo (int frametype, int tempref) {
  SideInfo.Index         = tempref;
  SideInfo.FrameType     = frametype;
}

void SetLayerCodingInfo (int IsLowestLayer, int IsHighestLayer, int layer) {
  SideInfo.Layer = layer;
  SideInfo.IsLowestLayer = IsLowestLayer;
  SideInfo.IsHighestLayer = IsHighestLayer;
}

static int pmvfx, pmvfy;

void ResetMVPredictor() {
  pmvfx = pmvfy = 0;
}

void SetMBCodingInfo (int bsize, int scale, int mode, int mvfx, int mvfy, int mvbx, int mvby) {

  SideInfo.Scale          = scale;
  SideInfo.lScale          = scale;
  SideInfo.BlockSize      = bsize;
  SideInfo.CodingMode     = mode;

  SideInfo.mvfx           = mvfx;
  SideInfo.mvfy           = mvfy;

  SideInfo.mvbx           = mvbx;
  SideInfo.mvby           = mvby;
}

/* ======================================================================================================================== */

void QuantizeInterMB(int *yact, int *uact, int *vact, int *ypre, int *upre, int *vpre)
{
  int n;
  int cvp;
  int i;
  int err;
  double scaled_var, mse,scale;

  mse = 0;
  for (i=0; i<256; i++) {
    err = yact[i] - ypre[i];
    mse += (err*err);
  }
  mse /= 256.0;

  scale = SideInfo.Scale;

  if (SideInfo.FrameType == I_FRAME) {
    if (mse/(scale*scale)<0.25) {
      fprintf(stderr,"L"); 
      SideInfo.TableIndex = 0;
    } else {
      fprintf(stderr,"H"); 
      SideInfo.TableIndex = 1;
    }
  } else if (SideInfo.FrameType == P_FRAME) {
    if (mse/(scale*scale)<0.1) {
      fprintf(stderr,"L"); 
      SideInfo.TableIndex = 0;
    } else {
      fprintf(stderr,"H"); 
      SideInfo.TableIndex = 1;
    }
  }

  /* SideInfo.TableIndex = SideInfo.Layer; */

  SideInfo.cbpy1 = SideInfo.cbpy2 = SideInfo.cbpc = 0;

  for (n=0; n<16; n++) {
    QuantizeInterBlock(2*n, 0, ypre+(n*16), yact+(n*16));
  }

  for (n=0; n<4; n++) 
    QuantizeInterBlock(32+(2*n), 1, upre+(n*16), uact+(n*16)) ;

  for (n=0; n<4; n++) 
    QuantizeInterBlock(40+(2*n), 2, vpre+(n*16), vact+(n*16)) ; 

  cvp = SideInfo.cbpy1 | SideInfo.cbpy2 | SideInfo.cbpc; 

  if (SideInfo.FrameType == P_FRAME) {
    SideInfo.hasMV =  (SideInfo.mvfx!=0) || (SideInfo.mvfy!=0);
    SideInfo.coded = ((cvp | SideInfo.hasMV)!=0);
  } else if (SideInfo.FrameType == B_FRAME) {
    SideInfo.hasMV =  (SideInfo.mvfx!=0) || (SideInfo.mvfy!=0) || (SideInfo.mvbx!=0) || (SideInfo.mvby!=0);
    SideInfo.coded = ((cvp | SideInfo.hasMV)!=0);
  } else SideInfo.coded = (cvp != 0);
}

/* ======================================================================================================================== */

static int BlockMSE (int *b1, int *b2, int size) {
  int i;
  int d, sum = 0;

  for (i=0; i<size; i++) {
    d = b1[i] - b2[i];
    sum += (d*d);
  }
  return(ROUND(sum/(double)size));
}

void SendMB(int *ydec, int *udec, int *vdec, int *ypre, int *upre, int *vpre,  int *ymean, int *umean, int *vmean) {

  int i, n;
  int pmode;
  int UsedBits = 0;
  int MBBits;
  int yact[256];

  SMStartMBCoding();
  SMStartBlock(3);

  memcpy(yact, ydec, 256*sizeof(int));

  if (SideInfo.IsLowestLayer) {
    if (SideInfo.FrameType == P_FRAME) {
      if (SideInfo.CodingMode == SP_BLOCK) {
	/* P-Frame, lowest layer, intra coded */
	ResetMVPredictor();
	if (SideInfo.coded) pmode = 2; else pmode = 3;
	UsedBits += PutPLowMode(pmode);
	/*
	for (i=0; i<16; i++) CodeIntraMean(ymean[i], &ymean_pred);
	for (i=0; i<4; i++) CodeIntraMean(umean[i], &umean_pred);
	for (i=0; i<4; i++) CodeIntraMean(vmean[i], &vmean_pred);
	*/

	for (i=0; i<4; i++) CodeIntraMean(ymean[i], &ymean_pred);
	CodeIntraMean(umean[0], &umean_pred);
	CodeIntraMean(vmean[0], &vmean_pred);

      } else if (SideInfo.CodingMode==FWD_TPLP_BLOCK) {
	/* P-Frame, lowest layer, temporally predicted */
	if (SideInfo.coded) pmode = 1; else pmode = 0;
	UsedBits += PutPLowMode(pmode);

	if (SideInfo.coded) {
	  CodeFXComponent(SideInfo.mvfx-pmvfx);
	  CodeFYComponent(SideInfo.mvfy-pmvfy);
	  pmvfx = SideInfo.mvfx; pmvfy = SideInfo.mvfy;
	} else pmvfx = pmvfy = 0;

      }
    } else if (SideInfo.FrameType == I_FRAME) {
      if (SideInfo.coded) {
	UsedBits += EmitBits(0,1,"MB coded");
      } else {
	UsedBits += EmitBits(1,1,"MB not coded");
      }
    }
  } else if (SideInfo.FrameType == P_FRAME) {
    if (SideInfo.CodingMode == SP_BLOCK) {
      /* P-Frame, not lowest layer, spatially predicted */
      ResetMVPredictor();
      if (SideInfo.coded) pmode = 0+1; else pmode = 0+0;
      UsedBits += PutPMode(pmode);
    } else if (SideInfo.CodingMode==FWD_TPLP_BLOCK) {
      /* P-Frame, not lowest layer, temporally predicted */
      if (SideInfo.coded) pmode = 2+1; else pmode = 2+0;
      UsedBits += PutPMode(pmode);

      if (SideInfo.coded) {
	CodeFXRefineComponent(SideInfo.mvfx);
	CodeFYRefineComponent(SideInfo.mvfy);
      } else pmvfx = pmvfy = 0;

    } else if (SideInfo.CodingMode==FWD_TPBP_BLOCK) {
      /* P-Frame, not lowest layer, temporally predicted */
      if (SideInfo.coded) pmode = 4+1; else pmode = 4+0;
      UsedBits += PutPMode(pmode);

      if (SideInfo.coded) {
	CodeFXRefineComponent(SideInfo.mvfx);
	CodeFYRefineComponent(SideInfo.mvfy);
      } else pmvfx = pmvfy = 0;
    }
  } else if (SideInfo.FrameType == I_FRAME) {
    if (SideInfo.coded) {
      UsedBits += EmitBits(0,1,"MB coded");
    } else {
      UsedBits += EmitBits(1,1,"MB not coded");
    }
  } 
  SMIncrementResidBitCount(UsedBits); UsedBits = 0;

  if (SideInfo.coded) {

    fprintf(stderr,"*");

    SMStartBlock(0);

    UsedBits += EmitBits(SideInfo.TableIndex,1,"H/L Block");

    if (SideInfo.CodingMode==SP_BLOCK) {
      ActivityBits += PutIntraCBPYHuffCode(SideInfo.cbpy1,SideInfo.TableIndex);
    } else {
      ActivityBits += PutInterCBPYHuffCode(SideInfo.cbpy1,SideInfo.TableIndex); 
    }


    MBBits = GlobalBitCount;
    for (n=0; n<8; n++) {
      if (SideInfo.cbpy1 & (1<<(7-n))) {
	CodeBlock(2*n,0,ydec+(n*16), ypre+(n*16));
      } else {
	for (i=0; i<16; i++) ydec[(n*16)+i] = ypre[(n*16)+i];
      }
    }
    QBlock.var = BlockMSE(yact, ydec, 256);
    MBBits = GlobalBitCount - MBBits;

    if (SideInfo.CodingMode==SP_BLOCK) {
      ActivityBits += PutIntraCBPYHuffCode(SideInfo.cbpy2,SideInfo.TableIndex);
    } else {
      ActivityBits += PutInterCBPYHuffCode(SideInfo.cbpy2,SideInfo.TableIndex); 
    }

    for (n=0; n<8; n++) {
      if (SideInfo.cbpy2 & (1<<(7-n))) {
	CodeBlock(16+(2*n),0,ydec+(128+(n*16)), ypre+(128+(n*16)));
      } else {
	for (i=0; i<16; i++) ydec[128+(n*16)+i] = ypre[128+(n*16)+i];
      }
    }

    SMIncrementResidBitCount(UsedBits); UsedBits = 0;
    SMStartBlock(1);

    if (SideInfo.CodingMode==SP_BLOCK)
      ActivityBits += PutIntraCBPCHuffCode(SideInfo.cbpc,SideInfo.TableIndex);
    else
      ActivityBits += PutInterCBPCHuffCode(SideInfo.cbpc,SideInfo.TableIndex); 

    for (n=0; n<4; n++) {
      if (SideInfo.cbpc & (1<<(7-n))) {
	CodeBlock(32+(2*n),1,udec+(n*16), upre+(n*16));
      } else {
	for (i=0; i<16; i++) udec[(n*16)+i] = upre[(n*16)+i];
      }
    }

    SMIncrementResidBitCount(UsedBits); UsedBits = 0;
    SMStartBlock(2);

    for (n=4; n<8; n++) {
      if (SideInfo.cbpc & (1<<(7-n))) {
	CodeBlock(32+(2*n),2,vdec+((n-4)*16), vpre+((n-4)*16));
      } else {
	for (i=0; i<16; i++) vdec[((n-4)*16)+i] = vpre[((n-4)*16)+i];
      }
    }

    SMIncrementResidBitCount(UsedBits);   

  } else {
    MBBits = 0;
    fprintf(stderr," ");

    for (i=0; i<256; i++) ydec[i] = ypre[i];

    for (i=0; i<64; i++) {
      udec[i] = upre[i];
      vdec[i] = vpre[i];
    }
    ResetMVPredictor();
  }
}
 
void QuantizeIntraImage(ImagePtr in, ImagePtr decoded) {
  int i, j;
  int v[8];

  for (j=0; j<in->h; j+=4) {
    for (i=0; i<in->w; i+=2) {
      v[0] = in->data[(j*in->w)+i] - 128; v[4] = in->data[(j*in->w)+i+1] - 128;
      v[1] = in->data[(j*in->w)+i] - 128; v[5] = in->data[(j*in->w)+i+1] - 128;
      v[2] = in->data[(j*in->w)+i] - 128; v[6] = in->data[(j*in->w)+i+1] - 128;
      v[3] = in->data[(j*in->w)+i] - 128; v[7] = in->data[(j*in->w)+i+1] - 128;
    }
  }
}

