#include <stdio.h>
#include <string.h>
#include <math.h>
#include <values.h>
#include <stdlib.h>

#include "Crop.h"
#include "Filter.h"

/* ****************************************************************************** */

static unsigned short ITab[256], I3Tab[256], I98Tab[256];

void InitInterpolator()
{
  int i;

  for (i=0; i<256; i++) {
    I3Tab[i] = i * 3;
    I98Tab[i] = i * 9 + 8;   /* +8 for rouding */
    ITab[i] = 3*i+2;
  }
}
/* ****************************************************************************** */

void InterpolateHV_onepass (Byte *ii, Byte *oi, int iw, int ih)
{
  Byte *outData, *lastData, *lii;
  int i,j;
  int ow = iw * 2;
  int ns1, ns2;
  /*
   *  variable naming: i for "input", m for "minus", p for "plus"
   *    w for width as in variable iw. Hence: ipwm1 = *(ii+iw-1).
   *    appended 3 means variable contains 3 times the value.
   */
  int imwm1, imw, imw_3, im1, iv, imwp1, ip1, ipwm1, ipw, ipw_3, ipwp1;
  int lim1, limwm1, lipwm1;
  int imwp2, ip2, ipwp2, out11,out12,out13,out14,out21,out22,out23,out24;

  outData = oi;
  lii = ii + (ih-1)*iw;
  lastData = oi + 2*(ih-1)*ow;

  /**
   **  Deal with first and last scanline
   **/
  im1 = *ii;
  ipwm1 = *(ii+iw);
  lim1 = *lii;
  limwm1 = *(lii-iw);
  for (i=iw-1; i; i--) {
    iv = *ii;
    ip1 = *(ii+1); 

    /* scanline above image is assumed to be identical to top line */
    imwm1 = im1;

    imw_3 = I3Tab[iv];
    imwp1 = ip1;

    ipw_3 = I3Tab[*(ii+iw)];
    ipwp1 = *(ii+iw+1);

    ns1 = I3Tab[im1] + I98Tab[iv];
    ns2 = I98Tab[iv] + I3Tab[ip1];

    *outData = (imwm1 + imw_3 + ns1)>>4;
    *(outData+1) = (imw_3 + imwp1 + ns2)>>4;
    *(outData+ow) = (ns1 + ipwm1 + ipw_3)>>4;
    *(outData+ow+1) = (ns2 + ipw_3 + ipwp1)>>4;
    outData += 2;
    ipwm1 = *(ii+iw);
    im1 = iv;
    ii++;

    /* scanline below image is assumed to be identical with last line */
    imw_3 = I3Tab[*(lii-iw)];
    imwp1 = *(lii-iw+1);

    iv = *lii;
    ip1 = *(lii+1);
    lipwm1 = lim1;
    ipw_3 = I3Tab[iv];
    ipwp1 = ip1;

    ns1 = I3Tab[lim1] + I98Tab[iv];
    ns2 = I98Tab[iv] + I3Tab[ip1];

    *lastData = (limwm1 + imw_3 + ns1)>>4;
    *(lastData+1) = (imw_3 + imwp1 + ns2)>>4;
    *(lastData+ow) = (ns1 + lipwm1 + ipw_3)>>4;
    *(lastData+ow+1) = (ns2 + ipw_3 + ipwp1)>>4;
    lastData += 2;
    limwm1 = *(lii-iw);
    lim1 = iv;
    lii++;
  }
  /* deal with last pixel of first & last scanline */
  iv = *ii;
  ip1 = iv;
  imwm1 = im1;
  imw_3 = I3Tab[iv];
  imwp1 = ip1;
  ipwm1 = *(ii+iw-1);
  ipw_3 = I3Tab[*(ii+iw)];
  ipwp1 = *(ii+iw);
  ns1 = I3Tab[im1] + I98Tab[iv];
  ns2 = I98Tab[iv] + I3Tab[ip1];
  *outData = (imwm1 + imw_3 + ns1)>>4;
  *(outData+1) = (imw_3 + imwp1 + ns2)>>4;
  *(outData+ow) = (ns1 + ipwm1 + ipw_3)>>4;
  *(outData+ow+1) = (ns2 + ipw_3 + ipwp1)>>4;
  outData += 2;
  ii++;

  imwm1 = *(lii-iw-1);
  imw_3 = I3Tab[*(lii-iw)];
  imwp1 = *(lii-iw);
  iv = *lii;
  ip1 = iv;
  ipwm1 = lim1;
  ipw_3 = I3Tab[iv];
  ipwp1 = iv;
  ns1 = I3Tab[lim1] + I98Tab[iv];
  ns2 = I98Tab[iv] + I3Tab[ip1];
  *lastData = (imwm1 + imw_3 + ns1)>>4;
  *(lastData+1) = (imw_3 + imwp1 + ns2)>>4;
  *(lastData+ow) = (ns1 + ipwm1 + ipw_3)>>4;
  *(lastData+ow+1) = (ns2 + ipw_3 + ipwp1)>>4;
  outData += ow;

  /**
   **  Blow each input pixel up to 4 interpolated pixels in output image
   **  unrolled one time
   **  - to spare a few register moves
   **  - to do 32-bit writes
   **/
  for (j=ih-2; j; j--) {
    iv = *ii;
    ipw = *(ii+iw);
    im1 = iv;          /* -1 omitted on leftmost pixel in line */
    imw = *(ii-iw);
    imwm1 = imw;
    ipwm1 = ipw;
    for (i=iw/2-1; i; i--) {
      imwp1 = *(ii-iw+1);
      ip1 = *(ii+1);
      ipwp1 = *(ii+iw+1);

      imw_3 = I3Tab[imw];
      ipw_3 = I3Tab[ipw];

      ns1 = I3Tab[im1] + I98Tab[iv];
      ns2 = I98Tab[iv] + I3Tab[ip1];

      out11 = imwm1 + imw_3 + ns1;
      out12 = imw_3 + imwp1 + ns2;
      out21 = ns1 + ipwm1 + ipw_3;
      out22 = ns2 + ipw_3 + ipwp1;

      /* unrolled */

      imwp2 = *(ii-iw+2);
      ip2 = *(ii+2);
      ipwp2 = *(ii+iw+2);

      imw_3 = I3Tab[imwp1];
      ipw_3 = I3Tab[ipwp1];

      ns1 = I3Tab[iv] + I98Tab[ip1];
      ns2 = I98Tab[ip1] + I3Tab[ip2];

      out13 = imw + imw_3 + ns1;
      out14 = imw_3 + imwp2 + ns2;
      out23 = ns1 + ipw + ipw_3;
      out24 = ns2 + ipw_3 + ipwp2;

      *((long *)outData) =
                 ((out11 & 0xff0)<<20) | ((out12 & 0xff0)<<12) |
                 ((out13 & 0xff0)<<4) | (out14>>4);
      *((long *)(outData + ow)) =
                 ((out21 & 0xff0)<<20) | ((out22 & 0xff0)<<12) |
                 ((out23 & 0xff0)<<4) | (out24>>4);
      outData += 4;
      ii += 2;
      imwm1 = imwp1;
      im1 = ip1;
      ipwm1 = ipwp1;
      imw = imwp2;
      iv = ip2;
      ipw = ipwp2;
    }
    /*
     *  Deal with last pixel(s) in scanline seperately
     *  - the 2nd last would need no special handling, but since
     *    we want to do 32-bit stroes (aka 2 pels at a time) above...
     *  - for interpolation pel right to image is same as rightmost
     *    (same as with all other borders)
     */
    imwp1 = *(ii-iw+1);
    ip1 = *(ii+1);
    ipwp1 = *(ii+iw+1);
    imw_3 = I3Tab[imw];
    ipw_3 = I3Tab[ipw];
    ns1 = I3Tab[im1] + I98Tab[iv];
    ns2 = I98Tab[iv] + I3Tab[ip1];
    out11 = imwm1 + imw_3 + ns1;
    out12 = imw_3 + imwp1 + ns2;
    out21 = ns1 + ipwm1 + ipw_3;
    out22 = ns2 + ipw_3 + ipwp1;

    /* now the very last one */
    imw_3 = I3Tab[imwp1];
    ipw_3 = I3Tab[ipwp1];
    ns1 = I3Tab[iv] + I98Tab[ip1];
    ns2 = I98Tab[ip1] + I3Tab[ip1];
    out13 = imw + imw_3 + ns1;
    out14 = imw_3 + imwp1 + ns2;
    out23 = ns1 + ipw + ipw_3;
    out24 = ns2 + ipw_3 + ipwp1;

    *((long *)outData) =
	       ((out11 & 0xff0)<<20) | ((out12 & 0xff0)<<12) |
	       ((out13 & 0xff0)<<4) | (out14>>4);
    *((long *)(outData + ow)) =
	       ((out21 & 0xff0)<<20) | ((out22 & 0xff0)<<12) |
	       ((out23 & 0xff0)<<4) | (out24>>4);
    ii += 2;
    outData += ow + 4;
  }
}

static void InterpolateImage (Byte *ii, Byte *oi, int iw, int ih)
{

  register Byte *inData, *outData;
  register Byte x1,x2,x3;
  register unsigned short t;

  int i,j;
  int ow = iw<<1;
  int ow2 = ow<<1;
  int ow4 = ow2<<1;
  int oh = ih<<1;

  /* Horizontal upsampling */

  inData = ii;
 
  for (j=0; j<ih; j++, inData+=iw) {
    x1 = inData[0]; x2 = x1; x3 = inData[1];
    outData = oi+(j*ow2);
    for (i=0; i<(iw-1); i++, outData+=2) {
      t=ITab[x2];
      outData[0] = (Byte) ((t+x1)>>2); outData[1] = (Byte) ((t+x3)>>2);
      x1 = x2; x2 = x3; x3 = inData[i+2];
    }
    t=ITab[x2];
    /* Last pixel in row (x2) is replicated */
    outData[0] = (Byte) ((t+x1)>>2); outData[1] = (Byte) ((t+x2)>>2);
  }

  /* Vertical upsampling */

  for (i=0; i<ow; i++) {
    outData = oi + i;
    x1 = outData[0]; x2 = x1; x3 = outData[ow2];
    for (j=0; j<oh-4; j+=2, outData+=(ow2)) {
      t=ITab[x2];
      outData[0] = (t+x1)>>2; outData[ow] = (t+x3)>>2;
      x1 = x2; x2 = x3; x3 = outData[ow4];
    }
    /* Replicate border pixels */
    t=ITab[x2]; outData[0] = (t+x1)>>2; outData[ow] = (t+x3)>>2;
    t=ITab[x3]; outData[ow2] = (t+x2)>>2; outData[ow2+ow] = (t+x3)>>2;
  }
}
/* ****************************************************************************** */

void UpsampleDataBlock(Byte *in, Byte *out, int iw, int ow, int ni) {
  int i, j;

  Byte data, *outdata;

  for (j=0; j<ni; j++) {
    for (i=0; i<ni; i++) {
      data = in[(j*iw)+i];
      outdata = out+(2*j*ow)+(2*i);

      outdata[0] = data;
      outdata[1] = data;
      outdata[ow] = data;
      outdata[ow+1] = data;
    }
  }
}

      
/* ****************************************************************************** */

void DownsampleImage (ImagePtr InImage, ImagePtr OutImage)
{
  int i, j;
  int p1, p2, p3, p4;
  int iw, ih, ow;

  iw = InImage->w;
  ih = InImage->h;
  ow = OutImage->w;

    for (j=0; j<ih; j+=2)

      for (i=0; i<iw; i+=2) {
      
	p1= ((Byte *) InImage->data)[iw*j+i]; p2=((Byte *) InImage->data)[iw*j+i+1];
	p3=((Byte *) InImage->data)[iw*(j+1)+i]; p4=((Byte *) InImage->data)[iw*(j+1)+i+1];

	((Byte *) OutImage->data)[(j>>1)*ow+(i>>1)] = ROUND((p1+p2+p3+p4)/4.0);
      }
}

static void BlockDownsampleImage (ImagePtr InImage, ImagePtr OutImage, int bs)
{
  int i, j, ii, jj;
  int mean;
  int iw, ih, ow;

  iw = InImage->w;
  ih = InImage->h;
  ow = OutImage->w;

  for (j=0; j<ih; j+=bs)
    for (i=0; i<iw; i+=bs) {

      mean = 0;
      for (jj=0; jj<bs; jj++) 
	for (ii=0; ii<bs; ii++) 
	  mean += (InImage->data)[iw*(j+jj)+i+ii];

      ((Byte *) OutImage->data)[(j/bs)*ow+(i/bs)] = ROUND(mean/((double)(bs*bs)));
    }
}


static void BlockDownsampleImageVectors (ImagePtr InImage, ImagePtr OutImage)
{
  int i, j, ii, jj;
  int mean;
  int iw, ih, ow;

  iw = InImage->w;
  ih = InImage->h;
  ow = OutImage->w;

  for (j=0; j<ih; j+=4)
    for (i=0; i<iw; i+=2) {

      mean = 0;
      for (jj=0; jj<4; jj++) 
	for (ii=0; ii<2; ii++) 
	  mean += (InImage->data)[iw*(j+jj)+i+ii];

      ((Byte *) OutImage->data)[(j/4)*ow+(i/2)] = ROUND(mean/8.0);
    }
}

/*
static void ReduceImage(ImagePtr inp, ImagePtr out) {
  int i, j;
  Byte *corner;

  for (j=0; j<out->h; j++)
    for (i=0; i<out->w; i++) {
      corner = inp->data[(2*j*inp->w)+(2*i)];
      out->data[(j*out->w)+i] = ROUND((corner[0] + corner[1] + corner[inp->w] + corner[inp->w+1])/4.0);
    }
}

static void ExpandImage(ImagePtr inp, ImagePtr out) {
  int i, j;
  Byte *corner;
  for (j=0; j<out->h; j++)
    for (i=0; i<out->w; i++) {
      corner = inp->data[(2*j*inp->w)+(2*i)];
      out->data[(j*out->w)+i] = ROUND((corner[0] + corner[1] + corner[inp->w] + corner[inp->w+1])/4.0);
    }
}
*/
/* ****************************************************************************** */

void DownsampleFrame(FramePtr inpFrame, FramePtr outFrame) {
  DownsampleImage(inpFrame->Y, outFrame->Y);
  DownsampleImage(inpFrame->U, outFrame->U);
  DownsampleImage(inpFrame->V, outFrame->V);
}

void BlockDownsampleFrame(FramePtr inpFrame, FramePtr outFrame, int bs) {
  BlockDownsampleImage(inpFrame->Y, outFrame->Y, bs);
  BlockDownsampleImage(inpFrame->U, outFrame->U, bs);
  BlockDownsampleImage(inpFrame->V, outFrame->V, bs);
}

void BlockDownsampleFrameVectors(FramePtr inpFrame, FramePtr outFrame) {
  BlockDownsampleImageVectors(inpFrame->Y, outFrame->Y);
  BlockDownsampleImageVectors(inpFrame->U, outFrame->U);
  BlockDownsampleImageVectors(inpFrame->V, outFrame->V);
}

/* ****************************************************************************** */

void InterpolateFrameOnepass (FramePtr in, FramePtr out) {
  InterpolateHV_onepass(in->Y->data, out->Y->data, in->Y->w, in->Y->h);
  InterpolateHV_onepass(in->U->data, out->U->data, in->U->w, in->U->h);
  InterpolateHV_onepass(in->V->data, out->V->data, in->V->w, in->V->h);
}

void InterpolateSignal (ImagePtr in, ImagePtr out) {
  InterpolateHV_onepass(in->data, out->data, in->w, in->h);
}


void InterpolateFrame (FramePtr in, FramePtr out) {
  InterpolateImage(in->Y->data, out->Y->data, in->Y->w, in->Y->h);
  InterpolateImage(in->U->data, out->U->data, in->U->w, in->U->h);
  InterpolateImage(in->V->data, out->V->data, in->V->w, in->V->h);
}

/* ****************************************************************************** */

static int f2[13] = { 2, 0, -4, -3, 5, 19, 26, 19, 5, -3, -4, 0, 2 };

static void CCIR2CIFImage(Byte *in, Byte *out, int iw, int ih, int ow, int oh, int colflag) {

  int i, j, ii, n;
  int sum;
  int xoff, yoff;

  xoff = ow - (iw/2);
  if (xoff != 0) xoff /= 2;

  yoff = oh - (ih/2);
  if (yoff != 0) yoff /= 2;

  if (colflag)
    memset(out, 128, ow*oh);
  else
    memset(out, 0, ow*oh);

  for (j=1; j<ih; j+=2) {

    for (i=0; i<iw; i+=2) {

      sum = 0;

      for (n=-6; n<=6; n++) {

	ii = i+n;
	if (ii<0) ii = -ii; else if (ii>=iw) ii = 2*iw - ii - 2;

	sum += f2[n+6] * in[(j*iw)+ii];
      }
      out[((yoff+(j>>1))*(iw>>1))+(i>>1)+xoff] = ROUNDCLIP(sum/64.0);
    }
  }
}

static void CCIR2CIFDownsampleHori(Byte *in, Byte *out, int iw, int ih, int ow, int oh) {

  int i, j, ii, n;
  int sum;

  for (j=0; j<ih; j++) {

    for (i=0; i<iw; i+=2) {

      sum = 0;

      for (n=-6; n<=6; n++) {

	ii = i+n;
	if (ii<0) ii = -ii; else if (ii>=iw) ii = 2*iw - ii - 2;

	sum += f2[n+6] * in[(j*iw)+ii];
      }
      out[j*ow+(i>>1)] = ROUNDCLIP(sum/64.0);
    }
  }
}
/* ****************************************************************************** */
void CCIR2CIFFrame (FramePtr in, FramePtr out) {
  CCIR2CIFImage(in->Y->data, out->Y->data, in->Y->w, in->Y->h, out->Y->w, out->Y->h, 0);
  CCIR2CIFImage(in->U->data, out->U->data, in->U->w, in->U->h, out->U->w, out->U->h, 1);
  CCIR2CIFImage(in->V->data, out->V->data, in->V->w, in->V->h, out->V->w, out->V->h, 1);
}
/* ****************************************************************************** */

static ntsc1[5] = { -16, 22, 116, 22, -16 };
static ntsc2[4] = { -23, 40, 110, 1 };
static ntsc3[4] = { -24, 63, 100, -11 };
static ntsc4[4] = { -20, 84, 84, -20 };
static ntsc5[4] = { -11, 100, 63, -24 };
static ntsc6[4] = { 1, 110, 40, -23 };

static void CCIRNTSC2CIFImage(Byte *in, Byte *out, int iw, int ih, int ow, int oh) {

  int i, j, jj, oj, n;
  int sum;

  for (i=0; i<iw; i++) {

    oj = 0;
    for (j=0; j<ih; j+=10) {
      
      sum = 0;

      for (n=-2; n<=2; n++) {

	jj = j+2*n+1; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc1[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);

      sum = 0;

      for (n=-2; n<=1; n++) {

	jj = j+2*n+3; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc2[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);

      sum = 0;

      for (n=-2; n<=1; n++) {

	jj = j+2*n+5; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc3[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);

      sum = 0;

      for (n=-2; n<=1; n++) {

	jj = j+2*n+7; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc4[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);

      sum = 0;

      for (n=-2; n<=1; n++) {

	jj = j+2*n+9; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc5[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);

      sum = 0;

      for (n=-2; n<=1; n++) {

	jj = j+2*n+11; /* use even field */
	if (jj<0) jj = -jj; else if (jj>=ih) jj = 2*ih - jj - 2;

	sum += ntsc6[n+2] * in[(jj*iw)+i];
      }
      out[(oj++ *ow)+i] = ROUNDCLIP(sum/128.0);
    }
  }
}
/* ****************************************************************************** */
void CCIRNTSC2CIFFrame (FramePtr in, FramePtr out) {
  FramePtr filtered;

  filtered = NewFrame(in->Y->w, out->Y->h);
  CCIRNTSC2CIFImage(in->Y->data, filtered->Y->data, in->Y->w, in->Y->h, filtered->Y->w, filtered->Y->h);
  CCIR2CIFDownsampleHori(filtered->Y->data, out->Y->data, filtered->Y->w, filtered->Y->h, out->Y->w, out->Y->h);

  CCIRNTSC2CIFImage(in->U->data, filtered->U->data, in->U->w, in->U->h, filtered->U->w, filtered->U->h);
  CCIR2CIFDownsampleHori(filtered->U->data, out->U->data, filtered->U->w, filtered->U->h, out->U->w, out->U->h);

  CCIRNTSC2CIFImage(in->V->data, filtered->V->data, in->V->w, in->V->h, filtered->V->w, filtered->V->h);
  CCIR2CIFDownsampleHori(filtered->V->data, out->V->data, filtered->V->w, filtered->V->h, out->V->w, out->V->h);

  DeleteFrame(filtered);
}
/* ****************************************************************************** */

static void AddWhiteImageNoise(ImagePtr in, ImagePtr out, int var) {
  int i;
  short d;
  double delta2, delta;
  int norm, r;
 
  norm = (1<<14);
  delta2 = sqrt((double)var*12)/2.0;
 
  for (i=0; i < (in->w*in->h); i++) {
    d = in->data[i];
 
    r = rand() - norm;
    delta = ((double) r * delta2) / norm;
    out->data[i] = ROUNDCLIP(d + delta);
  }
}

void AddWhiteNoise(FramePtr in, FramePtr out, int var) {
 AddWhiteImageNoise(in->Y, out->Y, var);
 /* AddWhiteImageNoise(in->U, out->U, var);
 AddWhiteImageNoise(in->V, out->V, var); */
}
