#include <string.h>

#include "Image.h"

/* ========================================================================================== */

ImagePtr NewImage(int w, int h) {
  ImagePtr tmp;

  tmp = NEW(ImageStruct);
  tmp->w = w;
  tmp->h = h;
  tmp->ow = w;
  tmp->oh = h;
  tmp->size = tmp->w*tmp->h;

  tmp->data = NEWELEMS(Byte, tmp->w*tmp->h);
  memset(tmp->data,0, tmp->size);
  return(tmp);
}

static void DeleteImage(ImagePtr image) {
  free(image->data);
  free(image);
}

/* ========================================================================================== */

static int InterpolatePixel (Byte *data, int w, int xh, int yh)
{
  int pixel;

  switch ((yh<<1)+xh) {
  case 0:
    return ((int) *data);
    break;
  case 1:
    pixel = *data + *(data+1);
    return((int) ((pixel+1)>>1));
    break;
  case 2:
    pixel = *data + *(data+w);
    return((int) ((pixel+1)>>1));
    break;
  case 3:
    pixel = *data + *(data+1) + *(data+w) + *(data+w+1);
    return((int) ((pixel+2)>>2));
    break;
  default:
    EXIT_WITH_ERR("Unkown Interpolation mode");
    break;
  }
  return(-1);
}

static void Get4x4TranslatedBlock(ImagePtr image, int x, int y, int xh, int yh, int *vector) {

  Byte *col;

  col = image->data+y*image->w+x;

  vector[0] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[1] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[2] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[3] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;

  col = image->data+y*image->w+x+1;

  vector[4] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[5] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[6] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[7] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;

  col = image->data+y*image->w+x+2;

  vector[8] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[9] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[10] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[11] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;

  col = image->data+y*image->w+x+3;

  vector[12] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[13] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[14] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;
  vector[15] = InterpolatePixel (col, image->w, xh, yh); col+=image->w;

}

static void Get8x8TranslatedBlock(ImagePtr image, int x, int y, int xh, int yh, int *vector) {
  Get4x4TranslatedBlock(image, x,   y,   xh, yh, vector); vector+=16;
  Get4x4TranslatedBlock(image, x+4, y,   xh, yh, vector); vector+=16;
  Get4x4TranslatedBlock(image, x,   y+4, xh, yh, vector); vector+=16;
  Get4x4TranslatedBlock(image, x+4, y+4, xh, yh, vector); vector+=16;
}

/* ========================================================================================== */

static void Get4x4Block(ImagePtr image, int x, int y, int *vector) {

  Byte *col;

  col = image->data+y*image->w+x;

  vector[0] = *col; col+=image->w;
  vector[1] = *col; col+=image->w;
  vector[2] = *col; col+=image->w;
  vector[3] = *col; col+=image->w;

  col = image->data+y*image->w+x+1;

  vector[4] = *col; col+=image->w;
  vector[5] = *col; col+=image->w;
  vector[6] = *col; col+=image->w;
  vector[7] = *col; col+=image->w;

  col = image->data+y*image->w+x+2;

  vector[8] = *col; col+=image->w;
  vector[9] = *col; col+=image->w;
  vector[10] = *col; col+=image->w;
  vector[11] = *col; col+=image->w;

  col = image->data+y*image->w+x+3;

  vector[12] = *col; col+=image->w;
  vector[13] = *col; col+=image->w;
  vector[14] = *col; col+=image->w;
  vector[15] = *col; col+=image->w;

}

static void Get8x8Block(ImagePtr image, int x, int y, int *vector) {
  Get4x4Block(image, x,   y,   vector); vector+=16;
  Get4x4Block(image, x+4, y,   vector); vector+=16;
  Get4x4Block(image, x,   y+4, vector); vector+=16;
  Get4x4Block(image, x+4, y+4, vector); vector+=16;
}

/* ========================================================================================== */

static void Put4x4Block(ImagePtr image, int x, int y, int *vector) {

  Byte *col;

  col = image->data+y*image->w+x;

  *col = vector[0] ; col+=image->w;
  *col = vector[1] ; col+=image->w;
  *col = vector[2] ; col+=image->w;
  *col = vector[3] ; col+=image->w;

  col = image->data+y*image->w+x+1;

  *col = vector[4] ; col+=image->w;
  *col = vector[5] ; col+=image->w;
  *col = vector[6] ; col+=image->w;
  *col = vector[7] ; col+=image->w;

  col = image->data+y*image->w+x+2;

  *col = vector[8] ; col+=image->w;
  *col = vector[9] ; col+=image->w;
  *col = vector[10] ; col+=image->w;
  *col = vector[11] ; col+=image->w;

  col = image->data+y*image->w+x+3;

  *col = vector[12] ; col+=image->w;
  *col = vector[13] ; col+=image->w;
  *col = vector[14] ; col+=image->w;
  *col = vector[15] ; col+=image->w;

}

static void Put8x8Block(ImagePtr image, int x, int y, int *vector) {
  Put4x4Block(image, x,   y,   vector); vector+=16;
  Put4x4Block(image, x+4, y,   vector); vector+=16;
  Put4x4Block(image, x,   y+4, vector); vector+=16;
  Put4x4Block(image, x+4, y+4, vector); vector+=16;
}

/* ========================================================================================== */

FramePtr NewFrame(int w, int h) {
  FramePtr tmp;

  tmp = NEW(FrameStruct);

  tmp->Y = NewImage(w, h);
  tmp->U = NewImage((w+1)/2, (h+1)/2);
  tmp->V = NewImage((w+1)/2, (h+1)/2);

  return(tmp);
}

/* ========================================================================================== */

static void Get8x8MB(FramePtr frame, int x, int y, int *ydata, int *udata, int *vdata) {
  Get4x4Block(frame->Y, x,   y,   ydata);
  Get4x4Block(frame->Y, x+4, y,   ydata+16);
  Get4x4Block(frame->Y, x,   y+4, ydata+32);
  Get4x4Block(frame->Y, x+4, y+4, ydata+48);

  Get4x4Block(frame->U, x/2, y/2, udata);
  Get4x4Block(frame->V, x/2, y/2, vdata);
}

static void Get16x16MB(FramePtr frame, int x, int y, int *ydata, int *udata, int *vdata) {
  Get8x8Block(frame->Y, x,   y,   ydata);
  Get8x8Block(frame->Y, x+8, y,   ydata+64);
  Get8x8Block(frame->Y, x,   y+8, ydata+128);
  Get8x8Block(frame->Y, x+8, y+8, ydata+192);

  Get8x8Block(frame->U, x/2, y/2, udata);
  Get8x8Block(frame->V, x/2, y/2, vdata);
}

void GetMB(FramePtr frame, int x, int y, int size, int *ydata, int *udata, int *vdata) {
  switch (size) {
  case 8:
    Get8x8MB(frame, x, y, ydata, udata, vdata);
    break;
  case 16:
    Get16x16MB(frame, x, y, ydata, udata, vdata);
    break;
  }
}

/* ========================================================================================== */

static void Get8x8TranslatedMB(FramePtr frame, int x, int y, int fx, int fy, int *ydata, int *udata, int *vdata) {
  int sx, sy, xh, yh;

  sx = (fx>>1); sy = (fy>>1);
  xh = fx&1; yh = fy&1;

  sx += x; sy += y;

  Get4x4TranslatedBlock(frame->Y, sx,   sy,   xh, yh, ydata);
  Get4x4TranslatedBlock(frame->Y, sx+4, sy,   xh, yh, ydata+16);
  Get4x4TranslatedBlock(frame->Y, sx,   sy+4, xh, yh, ydata+32);
  Get4x4TranslatedBlock(frame->Y, sx+4, sy+4, xh, yh, ydata+48);

  /*
  fx = ( fx % 4 == 0 ? fx >> 1 : (fx>>1)|1 );
  fy = ( fy % 4 == 0 ? fy >> 1 : (fy>>1)|1 );
  */

  fx >>= 1; fy >>= 1;

  sx = fx>>1; sy = fy>>1;
  xh = fx&1; yh = fy&1;

  Get4x4TranslatedBlock(frame->U, sx, sy, xh, yh, udata);
  Get4x4TranslatedBlock(frame->V, sx, sy, xh, yh, vdata);
}

static void Get16x16TranslatedMB(FramePtr frame, int x, int y, int fx, int fy, int *ydata, int *udata, int *vdata) {
  int sx, sy, xh, yh;

  sx = (fx>>1); sy = (fy>>1);
  xh = fx&1; yh = fy&1;

  sx = x + sx; sy = y + sy;

  Get8x8TranslatedBlock(frame->Y, sx,   sy,   xh, yh, ydata);
  Get8x8TranslatedBlock(frame->Y, sx+8, sy,   xh, yh, ydata+64);
  Get8x8TranslatedBlock(frame->Y, sx,   sy+8, xh, yh, ydata+128);
  Get8x8TranslatedBlock(frame->Y, sx+8, sy+8, xh, yh, ydata+192);

  /*
  fx = ( ((fx % 4) == 0) ? fx >> 1 : (fx>>1)|1 );
  fy = ( ((fy % 4) == 0) ? fy >> 1 : (fy>>1)|1 );
  */
  fx >>= 1; fy >>= 1;

  sx = fx>>1; sy = fy>>1;
  xh = fx&1; yh = fy&1;

  sx = (x>>1) + sx; sy = (y>>1) + sy;

  Get8x8TranslatedBlock(frame->U, sx, sy, xh, yh, udata);
  Get8x8TranslatedBlock(frame->V, sx, sy, xh, yh, vdata);
}

void GetTranslatedMB (FramePtr frame, int x, int y, int fx, int fy, int size, int *ydata, int *udata, int *vdata) {

  switch (size) {
  case 8:
    Get8x8TranslatedMB(frame, x, y, fx, fy, ydata, udata, vdata);
    break;
  case 16:
    Get16x16TranslatedMB(frame, x, y, fx, fy, ydata, udata, vdata);
    break;
  }
}

/* ========================================================================================== */

static void Put8x8MB(FramePtr frame, int x, int y, int *ydata, int *udata, int *vdata) {
  Put4x4Block(frame->Y, x, y, ydata);
  Put4x4Block(frame->Y, x+4, y, ydata+16);
  Put4x4Block(frame->Y, x, y+4, ydata+32);
  Put4x4Block(frame->Y, x+4, y+4, ydata+48);

  Put4x4Block(frame->U, x/2, y/2, udata);
  Put4x4Block(frame->V, x/2, y/2, vdata);
}

static void Put16x16MB(FramePtr frame, int x, int y, int *ydata, int *udata, int *vdata) {
  Put8x8Block(frame->Y, x, y, ydata);
  Put8x8Block(frame->Y, x+8, y, ydata+64);
  Put8x8Block(frame->Y, x, y+8, ydata+128);
  Put8x8Block(frame->Y, x+8, y+8, ydata+192);

  Put8x8Block(frame->U, x/2, y/2, udata);
  Put8x8Block(frame->V, x/2, y/2, vdata);
}

void PutMB(FramePtr frame, int x, int y, int size, int *ydata, int *udata, int *vdata) {
  switch (size) {
  case 8:
    Put8x8MB(frame, x, y, ydata, udata, vdata);
    break;
  case 16:
    Put16x16MB(frame, x, y, ydata, udata, vdata);
    break;
  }
}

void DeleteFrame(FramePtr frame) {
  DeleteImage(frame->Y);
  DeleteImage(frame->U);
  DeleteImage(frame->V);
  free(frame);
}

/* ========================================================================================== */

PyramidPtr NewPyramid(int w, int h, int size) {
  int i;
  PyramidPtr tmp;

  tmp = NEW(PyramidStruct);
  tmp->Layer = (FramePtr *) malloc (size*sizeof(void *));
  tmp->size = size;

  for (i=0; i<size; i++) {
    tmp->Layer[i] = NewFrame((w + ((1<<i)>>1)) / (1<<i), (h + ((1<<i)>>1)) / (1<<i));
  }
  return(tmp);
}

void DeletePyramid(PyramidPtr pyra) {

  int i;

  for (i=0; i<pyra->size; i++) {
    DeleteFrame(pyra->Layer[i]);
  }

  free(pyra);
}

/* ========================================================================================== */

void PrintIntBlock(int *data, char *comm, int n, int size) {
  int i;

  if (gprint_on) {
    fprintf(stderr,"\n%s %d\n", comm, n);

    for (i=0; i<size; i++)
      fprintf(stderr,"%3d ", data[i]);
  }
}

void PrintFloatBlock(double *data, char *comm, int n, int size) {
  int i;

  if (gprint_on) {
    fprintf(stderr,"\n%s %d\n", comm, n);

    for (i=0; i<size; i++)
      fprintf(stderr,"%4.1f ", data[i]);
  }
}

