#include <stdio.h>
#include <string.h>
#include <math.h>
#include "Output.h"
#include "Parameter.h"
#include "Statistics.h"

#define LAYERS 5
#define CLASSES 3

static int CodingMode[8];

static int currLayer;
static int lowestCodedLayer;
static int currFrame;
static int currComponent;
static double MSEWeight[LAYERS];

static unsigned long ResidBits[4]; /* bitcount for each component */
static unsigned long frameResidBits[4], frameMEstiBits, frameBModeBits, frameActBits;
static unsigned long layerResidBits[LAYERS][4], layerMEstiBits[LAYERS], layerActBits[LAYERS], layerBModeBits[LAYERS];
static unsigned long blkResidBits[4];

static int ClassHistory[LAYERS][CLASSES];
static int ClassMSE[LAYERS][CLASSES];

static double layerMSE[LAYERS][3], layerSNR[LAYERS][3];

static int pixhist[512];

static void ResetPixhist() {
  memset(pixhist,0,512*sizeof(int));
}

void AddDiff(int val) {
  pixhist[256+val]++;
}

void PrintPixhist() {
  int i;
  for (i=0; i<512; i++)
    fprintf(stderr,"%d ", pixhist[i]);
}


void SMInit() {
  int i,l;
  int ow, oh, OriginalPixels, pixels;
  

  ResidBits[0] = ResidBits[1] = ResidBits[2] = ResidBits[3] = 0;
  MEstiBits = BModeBits = 0;
  for (i=0; i<8; i++) CodingMode[i] = 0;

  ow = SeqHdr.HSize;
  oh = SeqHdr.VSize;
  
  OriginalPixels = ow*oh;
  MSEWeight[0] = 1;

  for (l=1; l<SeqHdr.Layers; l++) {
    pixels = (ow/SeqHdr.XSubsamp[l]) * (oh/SeqHdr.YSubsamp[l]);
    MSEWeight[l] = pixels / (double) OriginalPixels;
  }

}

void SMStartFrameCoding(int Number)
{
  currFrame = Number;

  frameResidBits[0] = ResidBits[0];
  frameResidBits[1] = ResidBits[1];
  frameResidBits[2] = ResidBits[2];
  frameResidBits[3] = ResidBits[3];

  frameActBits = ActivityBits;
  frameMEstiBits = MEstiBits;
  frameBModeBits = BModeBits;

  lowestCodedLayer = 0;
}

void SMEndFrameCoding() 
{
  int l;
  double YMSE, UMSE, VMSE;

  fprintf(stderr,"####################################################\n");
  fprintf(stderr,"Frame Statistics for frame %d (GOP Index: %d); Coding mode: %s \n", 
	  currFrame, PictHdr.TempRef,
	  (PictHdr.CodingType==1) ? "I" : (PictHdr.CodingType==2) ? "P" : "B");
  fprintf(stderr,"####################################################\n");

  frameResidBits[0] = ResidBits[0] - frameResidBits[0];
  frameResidBits[1] = ResidBits[1] - frameResidBits[1];
  frameResidBits[2] = ResidBits[2] - frameResidBits[2];
  frameResidBits[3] = ResidBits[3] - frameResidBits[3];

  frameActBits = ActivityBits - frameActBits;
  frameMEstiBits = MEstiBits - frameMEstiBits;
  frameBModeBits = BModeBits - frameBModeBits;

  fprintf(stderr,"\n");
  fprintf(stderr,"MSE\tPSNR\n");

  for (l=0; l<=lowestCodedLayer; l++) {
    fprintf(stderr,"A%d\t%6.2f\t%5.2f\n", l, layerMSE[l][0], layerSNR[l][0]);
  }

  fprintf(stderr,"\n");

  fprintf(stderr,"\tY\tU\tV\tAct\tMode\tMotion\tSBR[Y]\tSNR[U]\tSNR[V]\n");

  YMSE = UMSE = VMSE = 0;

  for (l=0; l<=lowestCodedLayer; l++) {
    fprintf(stderr,"L%d\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%6.2f\t%6.2f\t%6.2f\n",
	    l, layerResidBits[l][0], layerResidBits[l][1], layerResidBits[l][2], layerActBits[l], layerResidBits[l][3], layerMEstiBits[l],
	    layerSNR[l][0], layerSNR[l][1], layerSNR[l][2]);
  YMSE += MSEWeight[l]*layerMSE[l][0];
  UMSE += MSEWeight[l]*layerMSE[l][1];
  VMSE += MSEWeight[l]*layerMSE[l][2];
  }
  fprintf(stderr,"-------------------------------------------------------------------------------\n");

  fprintf(stderr,"Sum\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%6.2f\t%6.2f\t%6.2f\n",
	  frameResidBits[0], frameResidBits[1], frameResidBits[2], frameActBits, frameResidBits[3], frameMEstiBits,
	  YMSE, UMSE, VMSE);
  fprintf(stderr,"-------------------------------------------------------------------------------\n");
  fprintf(stderr,"\tBits\tMSE[Y][0]\tSNR[Y][0]\tMSE[U][0]\tSNR[U][0]\tMSE[V][0]\tSNR[V][0]\n");
  fprintf(stderr,"All\t%ld\t%6.2f\t\t%6.2f\t\t%6.2f\t\t%6.2f\t\t%6.2f\t\t%6.2f\n", frameActBits+frameResidBits[0]+frameResidBits[1]+frameResidBits[2]+frameResidBits[3]+frameMEstiBits,
	  layerMSE[0][0], layerSNR[0][0], layerMSE[0][1], layerSNR[0][1], layerMSE[0][2], layerSNR[0][2]);
  fprintf(stderr,"\n");
}

void SMStartLayerCoding(int layer)
{
  currLayer = layer;

  if (layer>lowestCodedLayer)
    lowestCodedLayer = layer;

  layerResidBits[layer][0] = ResidBits[0];
  layerResidBits[layer][1] = ResidBits[1];
  layerResidBits[layer][2] = ResidBits[2];
  layerResidBits[layer][3] = ResidBits[3];

  layerActBits[layer] = ActivityBits;
  layerMEstiBits[layer] = MEstiBits;
  layerBModeBits[layer] = BModeBits;

  ResetPixhist();
}

void SMEndLayerCoding()
{
  int i;

  layerResidBits[currLayer][0] = ResidBits[0] - layerResidBits[currLayer][0];
  layerResidBits[currLayer][1] = ResidBits[1] - layerResidBits[currLayer][1];
  layerResidBits[currLayer][2] = ResidBits[2] - layerResidBits[currLayer][2];
  layerResidBits[currLayer][3] = ResidBits[3] - layerResidBits[currLayer][3];

  layerMEstiBits[currLayer] = MEstiBits - layerMEstiBits[currLayer];
  layerBModeBits[currLayer] = BModeBits - layerBModeBits[currLayer];
  layerActBits[currLayer] = ActivityBits - layerActBits[currLayer];

  for (i=0; i<CLASSES; i++) {
    if (ClassHistory[currLayer][i])
      ClassMSE[currLayer][i] /= ClassHistory[currLayer][i];
    else
      ClassMSE[currLayer][i] = 0;
  }

/*  PrintPixhist(); */
}

void SMStartMBCoding() 
{
  blkResidBits[0] = ResidBits[0];
  blkResidBits[1] = ResidBits[1];
  blkResidBits[2] = ResidBits[2];
  blkResidBits[3] = ResidBits[3];

}

void SMEndBlockCoding()
{

/*  fprintf(stderr,"%ld %ld %ld\n", ResidBits[0]-blkResidBits[0], ResidBits[1]-blkResidBits[1], ResidBits[2]-blkResidBits[2]); */
}

void SMIncrementResidBitCount(int UsedBits) {
  ResidBits[currComponent] += UsedBits;
}

void SMStartBlock(int compType) {
  currComponent = compType;
}

double SignalMSE(ImagePtr actual, ImagePtr decoded) 
{
  int i;
  int size, se;
  int ap, dp, d;

  size = actual->w*actual->h;
  se = 0;

  for (i=0; i<size; i++) {
    ap = ((Byte *) actual->data)[i];
    dp = ((Byte *) decoded->data)[i];
    d = (ap-dp);
    se += (d*d);
  }
  return (se / (double) size);
}

double SignalVAR(ImagePtr actual, ImagePtr decoded) 
{
  int i;
  int size;
  int ap, dp;
  double mean, d, var;

  size = actual->w*actual->h;

  mean = 0;

  for (i=0; i<size; i++) {
    ap = ((Byte *) actual->data)[i];
    dp = ((Byte *) decoded->data)[i];
    mean += (ap-dp);
  }

  mean /= (double) size;

  var = 0;
  for (i=0; i<size; i++) {
    ap = ((Byte *) actual->data)[i];
    dp = ((Byte *) decoded->data)[i];
    d = (double) (ap-dp) - mean;
    var += (d*d);
  }
  return (var / (double) size);
}

void SMLayerQuantizationError(FramePtr Actual, FramePtr Decoded)
{

  layerMSE[currLayer][0] = SignalMSE(Actual->Y, Decoded->Y);
  layerSNR[currLayer][0] = 10*log10(65025/layerMSE[currLayer][0]);

  layerMSE[currLayer][1] = SignalMSE(Actual->U, Decoded->U);
  layerSNR[currLayer][1] = 10*log10(65025/layerMSE[currLayer][1]);

  layerMSE[currLayer][2] = SignalMSE(Actual->V, Decoded->V);
  layerSNR[currLayer][2] = 10*log10(65025/layerMSE[currLayer][2]);

}



