#include "Motion.h"
#include "Parameter.h"

static int VectorFieldAllocated = 0;

MultiScaleFieldPtr NewVectorField () {

  int                l;
  int                ow,oh,bw,bh;
  MVFieldPtr         myField;
  MultiScaleFieldPtr tmpField;

  tmpField = NEW(MultiScaleFieldStruct);

  for (l=0; l<SeqHdr.Layers; l++) {

    ow = SeqHdr.HSize / SeqHdr.XSubsamp[l];
    bw = SeqHdr.BlockWidth[l];
    oh = SeqHdr.VSize / SeqHdr.YSubsamp[l];
    bh = SeqHdr.BlockHeight[l];

    tmpField->Layer[l] = myField = NEW(MVFieldStruct);

    myField->layer = l;
    myField->width = ow/bw;
    myField->height = oh/bh;
    myField->size = myField->width*myField->height;
    myField->bw = bw;
    myField->bh = bh;

    myField->Mode  = NEWELEMS(short, myField->size);
    myField->SXComp = NEWELEMS(short, myField->size);
    myField->SYComp = NEWELEMS(short, myField->size);

    myField->XComp = NEWELEMS(short, myField->size);
    myField->YComp = NEWELEMS(short, myField->size);

  }
  VectorFieldAllocated = 1;
  return(tmpField);
}

void FreeVectorField (MultiScaleFieldPtr field1, MultiScaleFieldPtr field2) {

  int                l;

  if (VectorFieldAllocated) {

    for (l=0; l<SeqHdr.Layers; l++) {
      free(field1->Layer[l]->Mode);
      free(field1->Layer[l]->XComp);
      free(field1->Layer[l]->YComp);
      free(field1->Layer[l]->SXComp);
      free(field1->Layer[l]->SYComp);
      free(field1->Layer[l]);
    }
    free(field1);

    for (l=0; l<SeqHdr.Layers; l++) {
      free(field2->Layer[l]->Mode);
      free(field2->Layer[l]->XComp);
      free(field2->Layer[l]->YComp);
      free(field2->Layer[l]->SXComp);
      free(field2->Layer[l]->SYComp);
      free(field2->Layer[l]);
    }
    free(field2);
  }
}


void SetMotionVector (MVFieldPtr mv, int i, int j, int x, int y, int mode)
{
  int row, off;

  if (mv != NULL) {
    row = (j/mv->bh)*mv->width;
    off = (i/mv->bw);
  
    (mv->Mode  + row)[off] = mode;
    (mv->XComp + row)[off] = x;
    (mv->YComp + row)[off] = y;
  }
}

void GetMotionVector (MVFieldPtr mv, int i, int j, int *sx, int *sy, int *x, int *y, int *mode)
{
  int mvAddress;

  if (mv==NULL) 
    *x = *y = *sx = *sy = 0;

  else {

    mvAddress = (j/mv->bh)*mv->width+(i/mv->bw);

    *mode = mv->Mode[mvAddress];
       *x = mv->XComp[mvAddress];
       *y = mv->YComp[mvAddress];
       *sx = mv->SXComp[mvAddress];
       *sy = mv->SYComp[mvAddress];
  }
}

void InterpolateMVField2 (MVFieldPtr in, MVFieldPtr out, int inRefTempRef, int inActTempRef, int outRefTempRef, int outActTempRef, int fullpel) {

  int i, j;
  int refdist, actdist, spatialdist;
  short x, y;
  short *xto, *yto, *sxto, *syto;

  /* So far this only works if blocksizes in the two different layers are the same */


  /*
     Compute temporal distance for appropriate scaling
     refdist: distance between the reference and the frame this vector field estimates
     actdist: distance between the reference and the actual frame for which a vector field
     should be predicted
     */

  out->ActTempRef = outActTempRef;
  out->RefTempRef = outRefTempRef;

  in->ActTempRef = inActTempRef;
  in->RefTempRef = inRefTempRef;

  refdist =  (in->ActTempRef - in->RefTempRef);
  actdist =   (outActTempRef - outRefTempRef);

  spatialdist = (in->layer - out->layer);

  /*
     Now change either refdist or actdist to account for different spatial scales
     */

  if (spatialdist < 0) 
    refdist *= (1<<(-spatialdist));
  else if (spatialdist > 0)
    actdist *= (1<<spatialdist);

  /* Half-pel accuracy is assumed for each vector.
     If we need a full-pel accuracy prediction we have to scale down MVs by a factor of 2
     */
  
  if (fullpel) refdist *= 2; 
  
  if ((in->width==out->width)&&(in->height==out->height)) {
    for (i=0; i<in->size; i++) {
      out->XComp[i] = (in->XComp[i] * actdist) / refdist;
      out->YComp[i] = (in->YComp[i] * actdist) / refdist;
      out->SXComp[i] = out->XComp[i];
      out->SYComp[i] = out->YComp[i];
    }
  } else if ((2*in->width==out->width)&&(2*in->height==out->height)) {
    for (j=0; j<in->height; j++) {
      xto = out->XComp + (2*j*out->width);
      yto = out->YComp + (2*j*out->width);
      sxto = out->SXComp + (2*j*out->width);
      syto = out->SYComp + (2*j*out->width);
      for (i=0; i<in->width; i++) {
	x = in->XComp[j*in->width+i];
	y = in->YComp[j*in->width+i];
	/* Simple upsampling */
	*xto = *(xto+1) = *(xto+out->width) = *(xto+out->width+1) = (x * actdist)/refdist;
	*yto = *(yto+1) = *(yto+out->width) = *(yto+out->width+1) = (y * actdist)/refdist;

	*sxto = *(sxto+1) = *(sxto+out->width) = *(sxto+out->width+1) = (x * actdist)/refdist;
	*syto = *(syto+1) = *(syto+out->width) = *(syto+out->width+1) = (y * actdist)/refdist;

	xto += 2; yto += 2;
	sxto += 2; syto += 2;
      }
    }
  } else {
    fprintf(stderr,"Don't know how to interpolate MV Field\n");
    exit(1);
  }
}

void PrintMV(MVFieldPtr mv) {
  int i, j;

  for (j=0; j<mv->height; j++) {
    fprintf(stderr,"\n");
    for (i=0; i<mv->width; i++) {
      fprintf(stderr,"%3d%3d ", mv->XComp[j*mv->width+i], mv->YComp[j*mv->width+i]);
    }
  }
  fprintf(stderr,"\n");
}
 
void Read263MVField(MVFieldPtr mv, char *name, int pos) {
  int i,j,n;
  FILE *in;

  in = fopen(name,"r");

  for (n=1; n<pos; n++) {
    for (j=0; j<mv->height; j++) {
      for (i=0; i<mv->width; i++) {
	fscanf(in,"%3hd%3hd ", (short *) &mv->XComp[j*mv->width+i], (short *) &mv->YComp[j*mv->width+i]);
      }
      fscanf(in,"\n");
    }
  }
  for (j=0; j<mv->height; j++) {
    for (i=0; i<mv->width; i++) {
      fscanf(in,"%3hd%3hd ", (short *) &mv->XComp[j*mv->width+i], (short *) &mv->YComp[j*mv->width+i]);
    }
    fscanf(in,"\n");
  }
  fclose(in);
}

void DumpFwdX()
{
  int w0, h0, bw, bh, w, h;
  int l,i,j;
  short *mvx;

  w0 = SeqHdr.HSize;
  h0 = SeqHdr.VSize;

  for (l=0; l<SeqHdr.Layers; l++) {

    mvx = ((short *) ActFwdMV->Layer[l]->XComp);

    bw = SeqHdr.BlockWidth[l];
    bh = SeqHdr.BlockHeight[l];

    w = w0/SeqHdr.XSubsamp[l];
    h = h0/SeqHdr.YSubsamp[l];

    fprintf(stderr,"\n");

    for (j=0; j<(h/bh); j++) {
      fprintf(stderr,"\n");
      for (i=0; i<(w/bw); i++) {
	fprintf(stderr,"%2d ", mvx[j*(w/bw)+i]);
      }
    }
  }
}
    

void DumpFwdY()
{
  int w0, h0, bw, bh, w, h;
  int l,i,j;
  short *mvy;

  w0 = SeqHdr.HSize;
  h0 = SeqHdr.VSize;

  for (l=0; l<SeqHdr.Layers; l++) {

    bw = SeqHdr.BlockWidth[l];
    bh = SeqHdr.BlockHeight[l];

    w = w0/SeqHdr.XSubsamp[l];
    h = h0/SeqHdr.YSubsamp[l];

    mvy = ((short *) ActFwdMV->Layer[l]->YComp);

    fprintf(stderr,"\n");

    for (j=0; j<(h/bh); j++) {
      fprintf(stderr,"\n");
      for (i=0; i<(w/bw); i++) {
	fprintf(stderr,"%2d ", mvy[j*(w/bw)+i]);
      }

    }
  }
}


