/*
 * Play a G723/24kbps encoded audio file
 *
 * compile with:  cc -fast -xO4 play_723.c # -laio
 *
 * $Id: audio.c,v 1.2 1997/12/16 23:30:13 jnweiger Exp $
 */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/audioio.h>
#include <sys/ioctl.h>
#include <sys/conf.h>
#include <fcntl.h>
#include <sys/asynch.h>
#include <assert.h>		/* assert() */
#include <unistd.h>
#include <signal.h>
#include "sig.h"

#if defined(__sun) && defined(__GNUC__)
extern int write();
extern int close();
#endif


#define INLEN    8001
#define DEC_BITS 3

static  int fdaudio;
static  struct g72x_state state;
static  struct audio_info ai;
static  unsigned char outbuf[INLEN*8/3];
static  int StartSample;

/*---------------------------------------------------------------------------*
 *  Decode a G723/24 bitstream to a G711/ulaw bytestream
 */

#include "g711.c"
#include "g72x.c"
#include "g723_24.c"

void ResetAudioSampleTimer() {
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
   StartSample = ai.play.samples;
}

int AudioSamplesPlayed(int num) 
{
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  if ((ai.play.samples - StartSample)<num)
    return(0);
  else
    return(1);
}

void WaitForCompletion(int time) 
{
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  fprintf(stderr,"Before: StartSample=%d play.samples=%d\n", StartSample, ai.play.samples);
  
  while (! AudioSamplesPlayed(time)) ;
  ResetAudioSampleTimer();

  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  fprintf(stderr,"After: %d\n", ai.play.samples);

}

int decode_buf(int num, unsigned char *inbuf, unsigned char *outbuf)
{
  unsigned char *in, *out;
  int i, bits;
  unsigned long bitbuf;
  unsigned int code, sample;


  out = outbuf;
  in = inbuf;
  bits = 0;
  bitbuf = 0L;

  /*
   *  Read 3 bytes from input, containing 8 samples; decode
   */
  for(i=num/8; i; i--) {
    bitbuf = *in | (*(in+1)<<8) | (*(in+2)<<16);
    in += 3;

    *out = g723_24_decoder(bitbuf & 7, AUDIO_ENCODING_ULAW, &state);
    out[1] = g723_24_decoder((bitbuf>>3) & 7, AUDIO_ENCODING_ULAW, &state);
    out[2] = g723_24_decoder((bitbuf>>6) & 7, AUDIO_ENCODING_ULAW, &state);
    out[3] = g723_24_decoder((bitbuf>>9) & 7, AUDIO_ENCODING_ULAW, &state);
    out[4] = g723_24_decoder((bitbuf>>12) & 7, AUDIO_ENCODING_ULAW, &state);
    out[5] = g723_24_decoder((bitbuf>>15) & 7, AUDIO_ENCODING_ULAW, &state);
    out[6] = g723_24_decoder((bitbuf>>18) & 7, AUDIO_ENCODING_ULAW, &state);
    out[7] = g723_24_decoder((bitbuf>>21) & 7, AUDIO_ENCODING_ULAW, &state);
    out +=8;
  }

  /*
   *  Deal with rest in input buffer
   */
  for(i=num%8; i; i--) {
    if(bits < 3) {
      bitbuf |= *in++ << bits;
      bits += 8;
    }
    code = bitbuf & 0x7;
    bitbuf >>= 3;
    bits -= 3;

    sample = g723_24_decoder(code, AUDIO_ENCODING_ULAW, &state);
    *out++ = (unsigned char) sample;
    assert(0);		/* shall not be used */
  }
  return out - outbuf;
}

/*---------------------------------------------------------------------------*
 *  Write an ulaw sample sequence asynchronously to the audio device
 */

void play_buf(int num, unsigned char *buf)
{
  ioctl(fdaudio, AUDIO_GETINFO, &ai);
  if (ai.play.error) {
    fprintf(stderr,"underflow\n");
    ai.play.error = 0;
    ioctl(fdaudio, AUDIO_SETINFO, &ai);
  }
  write(fdaudio, buf, num);
}

/*---------------------------------------------------------------------------*/

int PlayAudioPacket(int compressed_samples, unsigned char *inbuf) {
  /*
   *  Read to inbuffer; decode to outbuffer; play
   */

  if (compressed_samples > INLEN) {
    fprintf(stderr,"Audio buffer overflow\n");
    compressed_samples = INLEN;
  }

  assert(compressed_samples%3 == 0);

  decode_buf(compressed_samples*8/DEC_BITS, inbuf, outbuf);
  play_buf(compressed_samples*8/DEC_BITS, outbuf);
  return compressed_samples;
}

int CloseAudio() {
  return close(fdaudio);  /* wait for audio buffer to empty */
}
static int timeo = 0;
 
static sigret_t 
timeo_func(SIGDEFARG) 
{ 
  timeo++; 
  SIGRETURN
}

int InitAudio() {
  sigret_t (*sig) __P(SIGPROTOARG);
  sig = (sigret_t (*)())signal(SIGALRM, timeo_func);

  timeo = 0;
  alarm(3);
  if((fdaudio = open("/dev/audio", O_WRONLY, 0)) < 0) {
    perror("/dev/audio");
    exit(1);
  }
  alarm(0);
  signal(SIGALRM, sig);

  if(ioctl(fdaudio, AUDIO_GETINFO, &ai) < 0) {
    perror("ioctl AUDIO_GETINFO");
    exit(1);
  }
  ai.play.sample_rate = 8000;
  ai.play.channels = 1;
  ai.play.precision = 8;
  ai.play.encoding = AUDIO_ENCODING_ULAW;
  ai.play.pause = 0;
  ai.play.error = 0;
  ai.play.samples = 0;

  if(ioctl(fdaudio, AUDIO_SETINFO, &ai) < 0) {
    perror("ioctl AUDIO_SETINFO");
    exit(1);
  }

  return 0;
}

