/*
 * Play a G723/24kbps encoded audio file
 *
 * compile with:  cc -fast -xO4 play_723.c # -laio
 *
 * $Id: play_723.c,v 1.1 1997/12/16 18:50:14 jnweiger Exp $
 */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/audioio.h>
#include <sys/ioctl.h>
#include <sys/conf.h>
#include <fcntl.h>
#include <sys/asynch.h>

#include "g72x.h"

#define DEC_BITS 3
#define INLEN    600

int fdaudio;

/*---------------------------------------------------------------------------*
 *  Decode a G723/24 bitstream to a G711/ulaw bytestream
 */

#include "g711.c"
#include "g72x.c"
#include "g723_24.c"

int
decode_buf(int num, unsigned char *inbuf, unsigned char *outbuf)
{
  unsigned char *in, *out;
  int i, bits;
  unsigned long bitbuf;
  unsigned int code, sample;
  struct g72x_state state;

  out = outbuf;
  in = inbuf;
  bits = 0;
  bitbuf = 0L;

  /*
   *  Read 3 bytes from input, containing 8 samples; decode
   */
  for(i=num/8; i; i--) {
    bitbuf = *in | (*(in+1)<<8) | (*(in+2)<<16);
    in += 3;

    *out = g723_24_decoder(bitbuf & 7, AUDIO_ENCODING_ULAW, &state);
    out[1] = g723_24_decoder((bitbuf>>3) & 7, AUDIO_ENCODING_ULAW, &state);
    out[2] = g723_24_decoder((bitbuf>>6) & 7, AUDIO_ENCODING_ULAW, &state);
    out[3] = g723_24_decoder((bitbuf>>9) & 7, AUDIO_ENCODING_ULAW, &state);
    out[4] = g723_24_decoder((bitbuf>>12) & 7, AUDIO_ENCODING_ULAW, &state);
    out[5] = g723_24_decoder((bitbuf>>15) & 7, AUDIO_ENCODING_ULAW, &state);
    out[6] = g723_24_decoder((bitbuf>>18) & 7, AUDIO_ENCODING_ULAW, &state);
    out[7] = g723_24_decoder((bitbuf>>21) & 7, AUDIO_ENCODING_ULAW, &state);
    out +=8;
  }

  /*
   *  Deal with rest in input buffer
   */
  for(i=num%8; i; i--) {
    if(bits < 3) {
      bitbuf |= *in++ << bits;
      bits += 8;
    }
    code = bitbuf & 0x7;
    bitbuf >>= 3;
    bits -= 3;

    sample = g723_24_decoder(code, AUDIO_ENCODING_ULAW, &state);
    *out++ = (unsigned char) sample;
  }
  return out - outbuf;
}

/*---------------------------------------------------------------------------*
 *  Write an ulaw sample sequence asynchronously to the audio device
 */

void
play_buf(int num, unsigned char *buf)
{
  write(fdaudio, buf, num);
  /*
  printf("audio written\n"); fflush(stdout);
  */
  /*
  static aio_result_t result[40];
  static int idx = 0;

  result[idx].aio_return = AIO_INPROGRESS;
  if(aiowrite(fdaudio, (char *)buf, num, 0, 0, &result[idx]) < 0) {
    perror("aiowrite");
  }
  idx++;
  printf("%d written\n", idx); fflush(stdout);
  */
}

/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  struct audio_info ai;
  unsigned char inbuf[INLEN], outbuf[INLEN*8/3];
  int fdin, rstat;

  if(argc != 2) {
    fprintf(stderr, "Usage:  %s file\n", argv[0]);
    exit(1);
  }

  if(!strcmp(argv[1], "-")) fdin = 0;
  else fdin = open(argv[1], O_RDONLY, 0);
  if(fdin < 0) {
    perror("open");
    exit(1);
  }

  /*
   *  Check for file header
   */

  read(fdin, inbuf, 8);
  if(!strncmp(inbuf, ".snd")) {
    lseek(fdin, *(int *)(inbuf+4), SEEK_SET);
  }
  else {
    lseek(fdin, 0, SEEK_SET);
  }

  /*
   *  Initialize audio device
   */

  if((fdaudio = open("/dev/audio", O_WRONLY, 0)) < 0) {
    perror("/dev/audio");
    exit(1);
  }
  if(ioctl(fdaudio, AUDIO_GETINFO, &ai) < 0) {
    perror("ioctl AUDIO_GETINFO");
    exit(1);
  }
  ai.play.sample_rate = 8000;
  ai.play.channels = 1;
  ai.play.precision = 8;
  ai.play.encoding = AUDIO_ENCODING_ULAW;
  ai.play.pause = 0;
  ai.play.error = 0;

  if(ioctl(fdaudio, AUDIO_SETINFO, &ai) < 0) {
    perror("ioctl AUDIO_SETINFO");
    exit(1);
  }

  /*
   *  Read to inbuffer; decode to outbuffer; play
   */

  while(1) {
    rstat = read(fdin, inbuf, INLEN);
    if(rstat < 0) {
      perror("read");
      exit(1);
    }
    if(rstat == 0) break;

    decode_buf(rstat*8/DEC_BITS, inbuf, outbuf);
    play_buf(rstat*8/DEC_BITS, outbuf);
  }
  /*
  printf("compute done\n"); fflush(stdout);
  */
  if(fdin) close(fdin);
  close(fdaudio);  /* wait for audio buffer to empty */
}
