 

#include <limits.h>
#include <string.h>

#include "defs.h"
#include "Util.h"
#include "bitOut.h"

#include "chacodec.h"

#include "codec.h"
#include "galois.h"



/***************************************************************************** 
 *
 * function: Channel Encoder
 *
 *    Stefan Leusch, Klaus Stuhlmueller, Michael Link
 *     Mar.1998
 *
 * Return: Number of packed packets. 
 *
 *****************************************************************************/
int chaenco(int numLayer, int **gobSize, int *nGOBs, int *maxkIl, int nIl,
	    int pckMode, int varBytesPerPck_f, int maxBytesPerPck,
	    int *bytesPerPck_p, int *bytesPerLayerPck,
	    int *forceLayer, Bitstr **bitstr, Byte *packStream, int end_f)
{
  int pck2Write;
  int varBytesPerLayerPck_f = 1;


  switch (pckMode) {
  case 0:   /* Pack layers in separate packet blocks */
    pck2Write = PackIlSeparate(numLayer, maxkIl, nIl, varBytesPerPck_f,
			       maxBytesPerPck, bytesPerPck_p, bitstr,
			       packStream);
    break;
  case 1:
  case 2:
    pck2Write = PackIlIsoBytes(numLayer, maxkIl, nIl, varBytesPerPck_f,
			       maxBytesPerPck, bytesPerPck_p, forceLayer,
			       bitstr, packStream, (end_f || (pckMode == 2)));
    break;
  case 3:
  case 4:
    pck2Write = PackIlIsoGOBs(numLayer, gobSize, nGOBs, maxkIl, nIl,
			      varBytesPerPck_f, varBytesPerLayerPck_f,
			      maxBytesPerPck, bytesPerPck_p, bytesPerLayerPck,
			      bitstr, packStream, (end_f || (pckMode == 4)));
    break;
  default:
    fprintf(stderr,"pckMode %d not supported yet!\n", pckMode);
    pck2Write = 0;
  }
  return(pck2Write);
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PackIlSeparate -- Pack layers separately into interleavers
 *
 * Author:              K.S.
 *
 * Created:             23-Jul-98
 *
 * Purpose:             Packs all layers separately into interleavers,
 *                      beginning with the baselayer.
 * 
 * Arguments in:        int    numLayer        total number of layers
 *                      int    *maxkIl         max. k of each layer
 *                      int    nIl             max. number of packets in il.
 *                      int    varBytesPerPck_f flag for variable packet size.
 *                      int    maxBytesPerPck  packetsize (if var.: max pckSi.)
 *
 * Arguments in/out:    int    *bytesPerPck_p  actual packet size.
 *                      Bitstr **bitstr        bitstreams of the layers
 *
 * Arguments out:       Byte   *packStream     packetized stream
 *
 * Return values:       int    nPackets        number of packets in packStream.
 *
 * Example              nPck =  PackIlSeparate(numLayer, maxkIl, nIl,
 *                                             varBytesPerPck_f, bytesPerPck,
 *                                             bitstr, packStream);
 *
 * Side effects:        -
 *
 * Description:         The lowest layer (highest number) in which are bytes
 *                      to pack is filled into packStream. If the interleaver
 *                      is not full, nIl is reduced proportionally. I.e.
 *                          maxkIl[l] / maxnIl <= kIl[l] / nIl
 *                      * Instead of a InterleaverHeader the first three bytes
 *                        in each packet denote the 'layernumber', 'k' and
 *                        'end_f'.
 *                      * If the 'varBytesPerPck_f' is set, 'bytesPerPck' is
 *                        the max. packetsize. The packets are smaller, if
 *                        not enough info bytes are in the bitstream.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
int PackIlSeparate(int numLayer, int *maxkIl, int nIl, int varBytesPerPck_f,
		   int maxBytesPerPck, int *bytesPerPck_p, Bitstr **bitstr,
		   Byte *packStream)
 /**********************************************************CommentEnd********/
{
  int         l, i;
  int         end_f;
  const int   bytesPerPckIlInfo = HEADER_BYTES_PCK_MODE_0;
  int         remBytes = 0;
  int         infoBytes;
  int         kIl;
  int         bytesPerLayerPck;
  int                         length = 0;
  static int                  virgin = 1, nIlmax, packetSize;
  static SYMBOL               *samplep_p;
  static unsigned long        *bytesPPvec;
  Byte                        *il0_p, *il_p, *bs_p;


  if (virgin) {
    virgin = 0;
    bytesPPvec = (unsigned long *)malloc(nIl * sizeof(unsigned long));
    samplep_p = (SYMBOL *)malloc(nIl * sizeof(SYMBOL));
    for (i = 0; i < nIl; i++){
      samplep_p[i] = i;
      bytesPPvec[i] = i * maxBytesPerPck;
    }
    nIlmax = nIl;
    packetSize = maxBytesPerPck;
  } else if (nIlmax < nIl) {
    free(bytesPPvec);
    free(samplep_p);
    bytesPPvec = (unsigned long *)malloc(nIl * sizeof(unsigned long));
    samplep_p = (SYMBOL *)malloc(nIl * sizeof(SYMBOL));
    for (i = 0; i < nIl; i++){
      samplep_p[i] = i;
      bytesPPvec[i] = i * maxBytesPerPck;
    }
    nIlmax = nIl;
    packetSize = maxBytesPerPck;
  }

 

  /*---------------------------------------------------------------------------
   * Initialize and check parameters
   *-------------------------------------------------------------------------*/

  /* Regard interleaver info */
  bytesPerLayerPck = maxBytesPerPck - bytesPerPckIlInfo;
  *bytesPerPck_p = maxBytesPerPck;

  /* Check which bitstream is not empty */
  for (l = numLayer - 1; l >= 0; l--)
    if (bitstr[l]->ind > 0)
      break;
  if (l < 0)
    return 0;    /* All bitstreams are empty */

  kIl = maxkIl[l];

  infoBytes = (bitstr[l]->ind >> 3);
  if (infoBytes > maxkIl[l] * bytesPerLayerPck) {
    end_f = 0;
    infoBytes = maxkIl[l] * bytesPerLayerPck;
    remBytes = (bitstr[l]->ind >> 3) - infoBytes;
  } else
    end_f = 1;

  /**************************************************************/
  /* Adjust vertical or horizontal dimension of the packetblock */
  /**************************************************************/
  if (end_f)
    if (varBytesPerPck_f) {
      bytesPerLayerPck = (infoBytes + kIl - 1) / kIl;
      *bytesPerPck_p = bytesPerLayerPck + bytesPerPckIlInfo;
      kIl = (infoBytes + bytesPerLayerPck - 1) / bytesPerLayerPck;
    } else {
      kIl = (infoBytes + bytesPerLayerPck - 1) / bytesPerLayerPck;
      if (kIl < maxkIl[l])
	/* Reduce number of packets if possible */
	nIl = (nIl * kIl + maxkIl[l] - 1) / maxkIl[l];
    }

  /************************/
  /* Pack one packetblock */
  /************************/
  /* code Interleaver Info */
  /* In this case three (HEADER_BYTES_PCK_MODE_0) bytes in each packet:
     l, kIl, end_f */
  il0_p = packStream;
  for (i = 0; i < nIl; i++) {
    il0_p[0] = (Byte)l;
    il0_p[1] = (Byte)kIl;
    il0_p[2] = (Byte)end_f;
    il0_p += *bytesPerPck_p;
  }
  /* codiere Layer */
  il0_p = packStream + bytesPerPckIlInfo;
  /* copy infobytes */
  bs_p = bitstr[l]->b;
  il_p = il0_p;
  for (i = infoBytes; i > 0;
       i -= bytesPerLayerPck, bs_p += bytesPerLayerPck,
	 il_p += *bytesPerPck_p) {
    length = MIN(bytesPerLayerPck, i);
    memcpy(il_p, bs_p, length * sizeof(Byte));
  }
  if (i!=0){
    il_p -= *bytesPerPck_p;
    il_p += length;
    length = bytesPerLayerPck - length;
    memset(il_p, 0, length * sizeof(Byte));
  }

  /* bytesPPvec setzen */
  if (*bytesPerPck_p != packetSize) {
    packetSize = *bytesPerPck_p;
    for (i = 0; i < nIlmax; i++)
      bytesPPvec[i] = i * packetSize;
  }

  /*  redundanz berechnen */
  for (i = 0 ; i < bytesPerLayerPck; i++) {
    /* Systematic RS(n,k)-encoder */     
    FastNewtonInterpolation(il0_p + i, bytesPPvec, samplep_p, kIl,
			    &samplep_p[kIl], nIl-kIl);
  }

  /* Treat the layer bitstreams, i.e. save the unsent bytes */
  if (remBytes > 0)
    memmove(bitstr[l]->b, bitstr[l]->b + infoBytes, remBytes * sizeof(Byte));
  bitstr[l]->ind = (remBytes << 3);

  return nIl;
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PackIlIsoBytes -- Pack layers proportionally to their number of bytes
 *
 * Author:              K.S.
 *
 * Created:             23-Jul-98
 *
 * Purpose:             Multiplexes all layers into interleaver,
 *                      The proportion of the layers in the packets is the
 *                      same as the bytes in the bitstreams of the layer.
 * 
 * Arguments in:        int    numLayer        total number of layers
 *                      int    *maxkIl         max. k of each layer
 *                      int    nIl             max. number of packets in il.
 *                      int    varBytesPerPck_f flag for variable packet size.
 *                      int    maxBytesPerPck  packetsize (if var.: max pckSi.)
 *                      int    *forceLayer     force a layer into the il.
 *                                             i.e. such a layer is prefered.
 *                      int    packAll_f       if 1, the interleaver is sent
 *                                             even if it not full.
 *                                             In this case nIl is reduced.
 *
 * Arguments in/out:    int    *bytesPerPck_p  actual packet size.
 *                      Bitstr **bitstr        bitstreams of the layers
 *
 * Arguments out:       Byte   *packStream     packetized stream
 *
 * Return values:       int    nPackets        number of packets in packStream.
 *
 * Example              nPck =  PackIlIsoBytes(numLayer, maxkIl, nIl,
 *                                             varBytesPerPck_f,maxBytesPerPck,
 *                                             bytesPerPck, forceLayer,
 *                                             bitstr, packStream, packAll_f);
 *
 * Side effects:        -
 *
 * Description:         The layer streams are filled proportionally into
 *                      the interleaver.
 *                      The proportion of the layers in the packets is the
 *                      same as the bytes in the bitstreams of the layer.
 *                      If the 'packAll_f' flag is set, the interleaver is sent
 *                      even if it not full. In this case nIl is reduced.
 *                      * With the flag 'forceLayer[.]' layers can be prefered
 *                        in packing.
 *                      * If the 'varBytesPerPck_f' is set, 'bytesPerPck' is
 *                        the max. packetsize. The packets are smaller, if
 *                        not enough info bytes are in the bitstream.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
int PackIlIsoBytes(int numLayer, int *maxkIl, int nIl, int varBytesPerPck_f,
		   int maxBytesPerPck, int *bytesPerPck_p, int *forceLayer,
		   Bitstr **bitstr, Byte *packStream, int packAll_f)
 /**********************************************************CommentEnd********/
{
  int         l;
  int         end_f;
  int         bytesPerPckIlInfo;
  int         nettoBytesPerPck, tmp_bytesPerPck, changeBytes,neededBytesPerPck;
  int         remBytes[NUM_LAYERS];
  int         processedBytes[NUM_LAYERS];
  int         infoBytes[NUM_LAYERS];
  int         kIl[NUM_LAYERS];
  int         bytesPerLayerPck[NUM_LAYERS];


 

  /*---------------------------------------------------------------------------
   * Initialize and check parameters
   *-------------------------------------------------------------------------*/

  /* Regard interleaver info */
  bytesPerPckIlInfo = BYTES_PER_PACKET_IL_INFO(numLayer, K_IL_INFO);
  nettoBytesPerPck = maxBytesPerPck - bytesPerPckIlInfo;
  *bytesPerPck_p = maxBytesPerPck;

  /* Check how many bytes per packet we would need to transmit all bytes */
  neededBytesPerPck = 0;
  for (l = 0; l < numLayer; l++) {
    processedBytes[l] = 0;
    infoBytes[l] = (bitstr[l]->ind >> 3);
    bytesPerLayerPck[l] = (infoBytes[l] + maxkIl[l] - 1) / maxkIl[l];
    neededBytesPerPck += bytesPerLayerPck[l];
  }

  if ((neededBytesPerPck < nettoBytesPerPck && !packAll_f) ||
      (neededBytesPerPck == 0))
    /* Interleaver is not full or no data to pack */
    return (0);

  /************************************************************************/
  /* Adjust the bytes per layer packet to meet the given bytes per packet */
  /************************************************************************/
  tmp_bytesPerPck = 0;
  changeBytes = nettoBytesPerPck - neededBytesPerPck;
  if (!varBytesPerPck_f || changeBytes < 0) {
    for (l = numLayer - 1; l >= 0; l--) {
      if (!forceLayer[l]) { /* We can force a layer to be packed */
	bytesPerLayerPck[l] = (bytesPerLayerPck[l] * nettoBytesPerPck)
	  / neededBytesPerPck;
	bytesPerLayerPck[l] = MAX(bytesPerLayerPck[l], 0);
      }
      tmp_bytesPerPck += bytesPerLayerPck[l];
    }
    neededBytesPerPck = tmp_bytesPerPck;

    changeBytes = nettoBytesPerPck - neededBytesPerPck;
    if (!varBytesPerPck_f && changeBytes > 0) {
      /* Spend the spare bytes for the baselayer */
      for (l = numLayer - 1; l >= 0; l--)
	if (infoBytes[l] > 0) {
	  bytesPerLayerPck[l] += changeBytes;
	  break;
	}
      neededBytesPerPck = nettoBytesPerPck;
    }
    if (changeBytes < 0) {
      /* Cut of some bytes from the toplayer */
      for (l = 0; l < numLayer; l++)
	if (bytesPerLayerPck[l] >= - changeBytes) {
	  bytesPerLayerPck[l] += changeBytes;
	  changeBytes = 0;
	  break;
	} else {
	  changeBytes += bytesPerLayerPck[l];
	  bytesPerLayerPck[l] = 0;
	}
      neededBytesPerPck = nettoBytesPerPck;
    }
  }

  /**************************************************************/
  /* Adjust horizontal or vertical dimension of the packetblock */
  /**************************************************************/
  memcpy(kIl, maxkIl, NUM_LAYERS * sizeof(int));
  if (neededBytesPerPck <= nettoBytesPerPck) {
    if (varBytesPerPck_f) {
      *bytesPerPck_p -= changeBytes;
      nettoBytesPerPck = *bytesPerPck_p - bytesPerPckIlInfo;
      GetKs(numLayer, infoBytes, bytesPerLayerPck, kIl);
    } else {
      /* The baselayer gets the rest */
      bytesPerLayerPck[numLayer - 1] += changeBytes;
      /* Adjust k's and stretch last layer to meet 'nettoBytesPerPck' */
      AdjustLayerAreas(numLayer, nettoBytesPerPck, infoBytes, maxkIl, kIl,
		       bytesPerLayerPck);

      /* Reduce number of packets if possible */
      nIl = AdjustNumberOfILPcks(numLayer, kIl, maxkIl, nIl);
    }
  }

  /* How many bytes are not packed in each layer? */
  end_f = 1;
  for (l = numLayer - 1; l >= 0; l--) {
    infoBytes[l] = MIN(infoBytes[l], (bytesPerLayerPck[l] * kIl[l]));
    remBytes[l] = (bitstr[l]->ind >> 3) - infoBytes[l];
    if (remBytes[l] > 0)
      end_f = 0;
  }

  /************************/
  /* Pack one packetblock */
  /************************/
  packIl(numLayer, processedBytes, infoBytes, bytesPerPckIlInfo,
	 bytesPerLayerPck, kIl, end_f, bitstr, nIl, packStream,
	 *bytesPerPck_p);

  /* Treat the layer bitstreams, i.e. save the unsent bytes */
  for (l = numLayer - 1; l >= 0; l--) {
    memmove(bitstr[l]->b, bitstr[l]->b + infoBytes[l],
	    remBytes[l] * sizeof(Byte));
    bitstr[l]->ind = (remBytes[l] << 3);
  }

  return nIl;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PackIlIsoGOBs -- Pack layers proportionally to the number of GOBs
 *
 * Author:              K.S.
 *
 * Created:             23-Jul-98
 *
 * Purpose:             Multiplexes all layers into interleaver,
 *                      The proportion of the layers in the packets is the
 *                      same as the given number of GOBs of the layer.
 * 
 * Arguments in:        int    numLayer        total number of layers
 *                      int    **gobSize       all GOB sizes (in bytes).
 *                      int    *nGOBs          number of GOBs to pack.
 *                      int    *maxkIl         max. k of each layer
 *                      int    nIl             max. number of packets in il.
 *                      int    varBytesPerPck_f flag for variable packet size.
 *                      int    varBytesPerLayerPck_f  if 1 the bytes per layer
 *                                             in the il is variable.
 *                      int    maxBytesPerPck  packetsize of complete packet.
 *                                             Only regarded if
 *                                               varBytesPerLayerPck_f == 1.
 *                      int    *bytesPerPck_p  The packet size may differ from
 *                                             'maxBytesPerPck' if
 *                                             'varBytesPerLayerPck_f' == 0.
 *                                             In this case the packet size is
 *                                             the sum of '*bytesPerLayerPck'
 *                                             and is returned in this var.
 *                      int    *bytesPerLayerPck  bytes that are filled with
 *                                             the layer in each packet.
 *                                             Only regarded if
 *                                               varBytesPerLayerPck_f == 0.
 *                      int    packAll_f       if 1, the interleaver is sent
 *                                             even if it not full.
 *                                             In this case nIl is reduced.
 *
 * Arguments in/out:    Bitstr **bitstr        bitstreams of the layers
 *
 * Arguments out:       Byte   *packStream     packetized stream
 *
 * Return values:       int    nPackets        number of packets in packStream.
 *
 * Example              nPck =  PackIlIsoGOBs(numLayer, gobSize, nGOBs,
 *                                            maxkIl, nIl, varBytesPerLayerPck,
 *                                            maxBytesPerPck, *bytesPerPck,
 *                                            bitstr, packStream, packAll_f);
 *
 * Side effects:        -
 *
 * Description:         The layer streams are filled proportionally into
 *                      the interleaver.
 *                      The proportion of the layers in the packets is the
 *                      same as the bytes in the bitstreams of the layer.
 *                      If the 'packAll_f' flag is set, the interleaver is sent
 *                      even if it not full. In this case nIl is reduced.
 *                      * With the flag 'forceLayer[.]' layers can be prefered
 *                        in packing.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
int PackIlIsoGOBs(int numLayer, int **gobSize, int *nGOBs, int *maxkIl,
		  int nIl, int varBytesPerPck_f, int varBytesPerLayerPck_f,
		  int maxBytesPerPck, int *bytesPerPck_p,
		  int *bytesPerLayerPck, Bitstr **bitstr, Byte *packStream,
		  int packAll_f)
 /**********************************************************CommentEnd********/
{
  static int  virgin = 1;
  static int  ngobStart = 0;  /* Keep in mind if we have filled one Il */
  static int  processedBytes[NUM_LAYERS];
                              /* Keep in mind if we have filled one Il */
  static int  infoBytes[NUM_LAYERS];
                              /* Keep in mind if the Il was not filled compl.*/
  static int  storedGOBs_f = 0; /* Did we store GOBs last time? */

  int         l, j;
  int         ngob, minNGOBs = INT_MAX, codeGOBs = -1;
  int         maxInfoBytes[NUM_LAYERS];
  int         tmp_bytesPerPck = 0;
  int         tmp_bytesPerLayerPck[NUM_LAYERS];
  int         tmp_infoBytes[NUM_LAYERS];
  int         tmp_kIl[NUM_LAYERS], kIl[NUM_LAYERS];
  int         bytesPerPckIlInfo;

 

  fprintf(stderr, "Warning: PackIlIsoGOBs is a little bit buggy!\n");

  if (virgin) {
    for (l = 0; l < NUM_LAYERS; l++) {
      processedBytes[l] = 0;
      infoBytes[l] = 0;
    }
    virgin = 0;  
  }

  /*---------------------------------------------------------------------------
   * Initialize and check parameters
   *-------------------------------------------------------------------------*/

  if (ngobStart < 0) {
    ngobStart = 0;
    return 0;
  }

  for (l = 0; l < numLayer; l++) {
    minNGOBs = MIN(minNGOBs, nGOBs[l]);
    maxInfoBytes[l] = maxkIl[l] * bytesPerLayerPck[l];
    tmp_infoBytes[l] = infoBytes[l];
  }

  bytesPerPckIlInfo = BYTES_PER_PACKET_IL_INFO(numLayer, K_IL_INFO);
  maxBytesPerPck -= bytesPerPckIlInfo;

  /*---------------------------------------------------------------------------
   * Encode data 
   *-------------------------------------------------------------------------*/

  /* Loop over GOBs for reference layer */
  for (ngob = ngobStart; ngob < minNGOBs && codeGOBs == -1; ngob++) {
    /* Loop over layer */
    for (l = numLayer - 1; l >= 0; l--) {
      /* Loop over GOBs in not reference layers */
      for (j = (ngob * nGOBs[l]) / minNGOBs;
	   j < ((ngob + 1) * nGOBs[l]) / minNGOBs; j++)
	tmp_infoBytes[l] += gobSize[l][j];
      if (varBytesPerLayerPck_f == 0) {
	if (tmp_infoBytes[l] > maxInfoBytes[l]) {
	  codeGOBs = ngob - ngobStart;
	  break;
	}
      }
    }
    if (varBytesPerLayerPck_f == 1) {
      tmp_bytesPerPck = 0;
      for (l = 0; l < numLayer; l++) {
	tmp_bytesPerLayerPck[l] = (tmp_infoBytes[l] + maxkIl[l] - 1) / maxkIl[l];
	tmp_bytesPerPck += tmp_bytesPerLayerPck[l];
      }
      if (tmp_bytesPerPck > maxBytesPerPck) {
	codeGOBs = ngob - ngobStart;
	break;
      } else {
	memcpy(bytesPerLayerPck, tmp_bytesPerLayerPck, numLayer*sizeof(int));
      }
    }
    if (codeGOBs == -1)
      memcpy(infoBytes, tmp_infoBytes, numLayer*sizeof(int));
  }
  /* After this loop we have the number of GOBs to code
     (in the layer with the least number of GOBs) in: codeGOBs.
     If 'varBytesPerLayerPck_f == 1' we have adjusted 'bytesPerLayerPck'. */

#if 0
  if (codeGOBs < 0)
    /* Not enough data */;
  if (codeGOBs == 0 && storedGOBs_f)
    /* Send one interleaver */;
  if (codeGOBs == 0 && !storedGOBs_f)
    /* Interleaver zu klein! */;
  if (codeGOBs > 0)
    /* Send one interleaver */;
#endif

  if (codeGOBs < 0 && !packAll_f) {     /* ngobStart > 0 */
    /* The interleaver is not filled completely */
    ngobStart = 0;
    storedGOBs_f = 0;
    for (l = 0; l < numLayer; l++) {
      if (infoBytes[l] > 0)
	storedGOBs_f = 1;
      memmove(bitstr[l]->b, bitstr[l]->b + processedBytes[l],
	      infoBytes[l] * sizeof(Byte));
      bitstr[l]->ind = infoBytes[l] << 3;
      processedBytes[l] = 0;
    }
    return 0;
  } else {
    if (codeGOBs == 0 && !storedGOBs_f) {
      /* We have not enough data to fill one Interleaver or the first GOB
	 does not fit into the interleaver */
      fprintf(stderr, "Interleaver too small!\n");
      fprintf(stderr, "%d bytes per packet would be needed; "
	      "%d bytes are available.\n", tmp_bytesPerPck, maxBytesPerPck);
      exit(1);
    } else {
      /* We have enough data to fill one interleaver */
      if (varBytesPerPck_f) {
	GetKs(numLayer, infoBytes, bytesPerLayerPck, kIl);
      } else {
	AdjustLayerAreas(numLayer, maxBytesPerPck, infoBytes, maxkIl, kIl,
			 bytesPerLayerPck);
	if (codeGOBs < 0) {    /* Got no interleaver complete */
	  nIl = AdjustNumberOfILPcks(numLayer, kIl, maxkIl, nIl);
	}
      }

      *bytesPerPck_p = bytesPerPckIlInfo;
      /* Reserve 'bytesPerPckIlInfo' Bytes for interleaver info */
      for (l = 0; l < numLayer; l++)
	*bytesPerPck_p += bytesPerLayerPck[l];	  

      /* Pack one interleaver */
      packIl(numLayer, processedBytes, infoBytes, bytesPerPckIlInfo,
	     bytesPerLayerPck, kIl, 1, bitstr, nIl, packStream,
	     *bytesPerPck_p);
      /* We have enough data to fill one Interleaver */
      memcpy(kIl, tmp_kIl, numLayer*sizeof(int));

      if (codeGOBs < 0) {
	ngobStart = -1;  /* To escape the outer loop */
	storedGOBs_f = 0;
	for (l = 0; l < numLayer; l++) {
	  bitstr[l]->ind = 0;
	  infoBytes[l] = 0;
	  processedBytes[l] = 0;
	}
      } else {
	ngobStart += codeGOBs;
	for (l = 0; l < numLayer; l++) {
	  processedBytes[l] += infoBytes[l];
	  infoBytes[l] = 0;
	}
      }
      return nIl;
    }
  }
}



void GetKs(int numLayer, int *infoBytes, int *bytesPerLayerPck, int *kIl)
{
  int l;


  for (l = 0; l < numLayer; l++) {
    if (bytesPerLayerPck[l]) {
      kIl[l] = (infoBytes[l] + bytesPerLayerPck[l] - 1) / bytesPerLayerPck[l];
      if (kIl[l] < 1)
	kIl[l] = 1;
    } else
      kIl[l] = 1;
  }
}


void AdjustLayerAreas(int numLayer, int maxBytesPerPck, int *infoBytes,
		      int *maxkIl, int *kIl, int *bytesPerLayerPck)
{
  int l, j, break_f;
  int tmp_bytesPerLayerPck, bytesPerPck;


  memcpy(kIl, maxkIl, numLayer*sizeof(int));
  do {
    break_f = 1;
    for (l = 0; l < numLayer; l++) {
      if (kIl[l] > 1) {
	kIl[l]--;
	tmp_bytesPerLayerPck = bytesPerLayerPck[l];
	bytesPerLayerPck[l] = (infoBytes[l] + kIl[l] - 1) / kIl[l];
	bytesPerPck = 0;
	for (j = 0; j < numLayer; j++)
	  bytesPerPck += bytesPerLayerPck[j];
	if (bytesPerPck > maxBytesPerPck) {
	  kIl[l]++;
	  bytesPerLayerPck[l] = tmp_bytesPerLayerPck;
	} else {
	  break_f = 0;
	}
      }
    }
  } while (break_f == 0);

  /* stretch last layer */
  bytesPerPck = 0;
  for (l = 0; l < numLayer; l++)
    bytesPerPck += bytesPerLayerPck[l];
  if (bytesPerPck < maxBytesPerPck)
    bytesPerLayerPck[0] += maxBytesPerPck - bytesPerPck;
}

int AdjustNumberOfILPcks(int numLayer, int *kIl, int *maxkIl, int nIl)
{
  int l;
  int a;
  int maxIl = K_IL_INFO;


  /* Diminish 'nIl' for given 'kIl[l]' */
  /* Keep the relations 'nIl/maxkIl[l]' */
  for (l = 0; l < numLayer; l++) {
    a = (nIl * kIl[l] + maxkIl[l] - 1) / maxkIl[l];
    if (a > maxIl)
      maxIl = a;
  }

  return maxIl;
}


void packIl(int numLayer, int *processedBytes, int *infoBytes,
	    int bytesPerPckIlInfo, int *bytesPerLayerPck, int *kIl, int end_f,
	    Bitstr **bitstr, int nIl, Byte *packStream, int bytesPerPck)
{
  int                         l, i;
  int                         column;
  int                         length = 0;
  static int                  virgin = 1, nIlmax, packetSize;
  static SYMBOL               *samplep_p;
  static unsigned long        *bytesPPvec;
  Byte                        *il0_p, *il_p, *bs_p;


  if (virgin) {
    virgin = 0;
    bytesPPvec = (unsigned long *)malloc(nIl * sizeof(unsigned long));
    samplep_p = (SYMBOL *)malloc(nIl * sizeof(SYMBOL));
    for (i=0;i<nIl;i++){   
      samplep_p[i]=i;
      bytesPPvec[i]=i*bytesPerPck;
    }
    packetSize = bytesPerPck;
    nIlmax = nIl;
  } else if (nIlmax < nIl) {
    free(bytesPPvec);
    free(samplep_p);
    bytesPPvec = (unsigned long *)malloc(nIl * sizeof(unsigned long));
    samplep_p = (SYMBOL *)malloc(nIl * sizeof(SYMBOL));
    for (i=0;i<nIl;i++){   
      samplep_p[i]=i;
      bytesPPvec[i]=i*bytesPerPck;
    }
    packetSize = bytesPerPck;
    nIlmax = nIl;
  } else {
    if (bytesPerPck != packetSize) {
      for (i = 0; i < nIl; i++) {   
	bytesPPvec[i] = i * bytesPerPck;
      }
      packetSize = bytesPerPck;
    }
  }


  /* code Interleaver Info */
  CodeIlInfo(numLayer, end_f, kIl, bytesPerLayerPck, bytesPerPck, packStream);

  /* fuer interleaver info redundanz berechnen*/
  for ( column=0 ; column< bytesPerPckIlInfo ; column++ ) {
    /* Systematic RS(n,k)-encoder: k = codeword[N]  n = samplepoints[N]   */   
    FastNewtonInterpolation(packStream + column, bytesPPvec, samplep_p,
			    K_IL_INFO, &samplep_p[K_IL_INFO],/*+1*/
			    nIl - K_IL_INFO);
  }  

  /* codiere Layer */
  il0_p = packStream + bytesPerPckIlInfo;
  for(l = numLayer - 1; l >= 0; l--){

    /* copy infobytes */
    bs_p = bitstr[l]->b + processedBytes[l];
    il_p = il0_p;
    for (i = infoBytes[l]; i > 0;
	 i -= bytesPerLayerPck[l], bs_p += bytesPerLayerPck[l], il_p += bytesPerPck) {
      length = MIN(bytesPerLayerPck[l], i);
      memcpy(il_p, bs_p, length * sizeof(Byte));
    }
    if (i!=0){
      il_p -= bytesPerPck;
      il_p += length;
      length = bytesPerLayerPck[l] - length;
      memset(il_p, 0, length * sizeof(Byte));
    }


    /*  redundanz berechnen */
    for ( column=0 ; column<bytesPerLayerPck[l] ; column++ ) {   
      /* Systematic RS(n,k)-encoder */     
      FastNewtonInterpolation(il0_p + column, bytesPPvec, samplep_p, kIl[l],
			      &samplep_p[kIl[l]], nIl-kIl[l]);
    }

    il0_p += bytesPerLayerPck[l];
  }
}


void CodeIlInfo(int numLayer, int end_f, int *kIl, int *bytesPerLayerPck,
		int bytesPerPck, Byte *packStream)
{
  int  k, l, i;
  Byte *il_p = packStream;


  /* schreibe interleaverinfo */
  il_p[0] = (Byte)numLayer;
  il_p[bytesPerPck] = (Byte)end_f;
  k = 2;
  for (l = numLayer - 1; l >= 0 ; l--) {
    if (k >= K_IL_INFO) {
      k = 0;
      il_p++;
    }
    il_p[k * bytesPerPck] = (Byte)kIl[l];
    k++;
  }
  for (l = numLayer - 1; l >= 0 ; l--) {
    for (i = 8; i >= 0; i -= 8) {
      if (k >= K_IL_INFO) {
	k = 0;
	il_p++;
      }
      il_p[k * bytesPerPck] = (Byte)(bytesPerLayerPck[l] >> i);
      k++;
    }
  }
}


void mbInd2gobSize(int *mbInd, int indEnd,
		   int nMBsPerGOB, int nGOBsPerFrame, int *gobSize)
{
  int i;
  int ind, ind0;


  ind0 = (mbInd[0] >> 3);
  for (i = 0; i < nGOBsPerFrame - 1; i++) {
    ind = (mbInd[(i+1)*nMBsPerGOB] >> 3);
    gobSize[i] = ind - ind0;
    ind0 = ind;
  }
  gobSize[nGOBsPerFrame - 1] = (indEnd >> 3) - ind0;
}




