/***************************************************************************** 
 * Chacodec protokol:
 *                  Extract all packetization parameters from a PRS-Streamfile
 *
 * Klaus Stuhlmueller, LNT, Uni Erlangen, 1998
 *
 * Last changes: -
 *
 * compile with: gcc -I../../include -L../../lib/sun4u.sos5/ -o ChaProt chaprot.c -lchacodec 
 *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <limits.h>

#include "Util.h"
#include "bitOut.h"
#include "chacodec.h"


#define MAX_PACKETSIZE   1024 /*MS*/

void usage (char *name)
{
	fprintf(stderr, "usage: %s  [-i InputFilename]  [-o OutputFilename] \n\n", name);
	fprintf(stderr, "K. Stuhlmueller, LNT, 07/24/1998\n\n");

	exit(1);
}



/***************************************************************************** 
 *
 * main function: Channel
 *
 *****************************************************************************/
int main(int argc, char *argv[])
{
  char                      codeddataname[80];  /* filenames and pointers */
  char                      protocolname[80];
  FILE                      *inputfilepointer = stdin;
  FILE                      *protocol = stdout;
  
  int                       iln = -1;    /* Interleaver number */
  int                       totalNumIl = 0; /* total number of interleavers */
  int                       totalLayerNumIl[NUM_LAYERS];
                                  /* number of il in which layer was packed */

  int                       n = 0;
  int                       l;
  int                       i;
  
  unsigned char             *buffer;
  unsigned short            bytesinbuffer;
  
  int                       packetSize;
  int                       nIlmax, kIlInfo, pckMode, maxLayers;
  int                       method[NUM_LAYERS];

  RSPacketHeader            rsPckHeader;

  int                       numLayer, bytesPerPckILInfo;
  int                       end_f;
  int                       packetSizeMin = INT_MAX, packetSizeMax = 0,
                            packetSizeAvg = 0;
  int                       nIlMin = INT_MAX, nIlMax = 0, nIlAvg = 0;
  int                       kIl[NUM_LAYERS], kIlMin[NUM_LAYERS],
                            kIlMax[NUM_LAYERS], kIlAvg[NUM_LAYERS];
  int                       bytesPerLayerPck[NUM_LAYERS],
                            bytesPerLayerPckMin[NUM_LAYERS],
                            bytesPerLayerPckMax[NUM_LAYERS],
                            bytesPerLayerPckAvg[NUM_LAYERS];
  int                       layerStartOffset[NUM_LAYERS];

  /*---------------------------------------------------------------------------
   * Check command line arguments
   *-------------------------------------------------------------------------*/

  for (i=1; i<argc; i++) {
    if (argv[i][0] == '-') {
      switch (argv[i][1]) {

      case 'i' :

	/* Open input file: search *, *.rs, *.263.rs */
	sprintf(codeddataname, "%s", argv[++i]);
	if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	  sprintf(codeddataname, "%s.rs", argv[i]);
	  if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	    sprintf(codeddataname, "%s.263.rs", argv[i]);
	    if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	      fprintf(stderr,"Error in %s: can't open %s for reading.\n", argv[0], argv[i]);
	      exit(3);
	    }
	  }
	}			  

	break;

      case 'o':

	/* Open output file for writing */
	sprintf(protocolname, "pro_%s", argv[++i]);
	if ( (protocol = fopen(protocolname, "w")) == NULL ) {
	  fprintf(stderr,"Error in %s: can't open %s for writing.\n",
		  argv[0], protocolname);
	  exit(3);
	}

	break;

      default  :

	usage(argv[0]);

	break;
      }
    } else
      usage(argv[0]);
  }


  /********/
  /* Init */
  /********/
  for (i = 0; i < NUM_LAYERS; i++) {
    totalLayerNumIl[i] = 0;
    kIlMin[i] = INT_MAX;
    kIlMax[i] = 0;
    kIlAvg[i] = 0;
    bytesPerLayerPckMin[i] = INT_MAX;
    bytesPerLayerPckMax[i] = 0;
    bytesPerLayerPckAvg[i] = 0;
  }

  /*---------------------------------------------------------------------------
   * Read parameters from input file
   *-------------------------------------------------------------------------*/
  ReadPrsFileHeader(inputfilepointer, &packetSize, &nIlmax, &kIlInfo, &pckMode,
		    &maxLayers, method);

  assert(maxLayers <= NUM_LAYERS);

  /* Initialize packet buffer */
  if ((buffer = (unsigned char *)malloc(packetSize * (kIlInfo + 1) *
					sizeof(unsigned char))) == NULL) {
    fprintf(stderr,"Error in %s: malloc returns NULL.\n", argv[0]);
    exit(2);
  }

  /* Write parameters to output file header */ 
  fprintf(protocol, "%%RS-packed stream\n");
  fprintf(protocol, "nIlmax     %d\n", nIlmax);
  fprintf(protocol, "kIlInfo    %d\n", kIlInfo);
  fprintf(protocol, "packetMode %d\n", pckMode);

  fprintf(protocol, "\n");


  /* Read packet header from input file */
  while (!feof(inputfilepointer)) {

    if (fread(&rsPckHeader, sizeof(RSPacketHeader), 1, inputfilepointer) ==0)
      break;

    n++;
    if (iln != (int)(rsPckHeader.ilNumber)) { /* First packet of next il. */
      if (iln >= 0) {
	/* Extract parameter from last interleaver */
	fprintf(protocol, "\n\npckSize: %d\n", packetSize);
	fprintf(protocol, "il:    %d\n", iln);
	fprintf(protocol, "N :    %d\n", n);

	if (pckMode == 0) { /* Separate layer packetization */
	  l = (int)(buffer[0]);
	  fprintf(protocol, "l :   %d\n", l);
	  kIl[l] = (int)(buffer[1]);
	  fprintf(protocol, "k :   %d\n", kIl[l]);
	  end_f = (int)(buffer[2]);
	  fprintf(protocol, "end:  %d\n", end_f);
	  /* Statistics */
	  kIlMin[l] = MIN(kIl[l], kIlMin[l]);
	  kIlMax[l] = MAX(kIl[l], kIlMax[l]);
	  kIlAvg[l] += kIl[l];
	  bytesPerLayerPckMin[l] = MIN((packetSize - HEADER_BYTES_PCK_MODE_0),
				       bytesPerLayerPckMin[l]);
	  bytesPerLayerPckMax[l] = MAX((packetSize - HEADER_BYTES_PCK_MODE_0),
				       bytesPerLayerPckMax[l]);
	  bytesPerLayerPckAvg[l] += (packetSize - HEADER_BYTES_PCK_MODE_0);
	  totalLayerNumIl[l]++;
	} else {
	  /* Extract info (k's, P's, etc.) */
	  numLayer = (int)(buffer[0]);
	  bytesPerPckILInfo =
	    BYTES_PER_PACKET_IL_INFO(numLayer, kIlInfo);
	  DecodeIlInfo(buffer, packetSize, bytesPerPckILInfo,
		       kIlInfo, numLayer, &end_f, kIl,
		       bytesPerLayerPck, layerStartOffset);
	  fprintf(protocol, "l :   %d\n", numLayer);
	  fprintf(protocol, "end:  %d\n", end_f);
	  for (i = 0; i < numLayer; i++) {
	    fprintf(protocol, "k :  %d \tbytes :  %d \n", kIl[i],
		    bytesPerLayerPck[i]);
	    /* Statistics */
	    if (bytesPerLayerPck[i]) {
	      totalLayerNumIl[i]++;
	      kIlMin[i] = MIN(kIl[i], kIlMin[i]);
	      kIlMax[i] = MAX(kIl[i], kIlMax[i]);
	      kIlAvg[i] += kIl[i];
	      bytesPerLayerPckMin[i] = MIN(bytesPerLayerPck[i],
					   bytesPerLayerPckMin[i]);
	      bytesPerLayerPckMax[i] = MAX(bytesPerLayerPck[i],
					   bytesPerLayerPckMax[i]);
	      bytesPerLayerPckAvg[i] += bytesPerLayerPck[i];
	    }
	  }
	}

	/*Statistics */
	totalNumIl++;
	packetSizeMin = MIN(packetSize, packetSizeMin);
	packetSizeMax = MAX(packetSize, packetSizeMax);
	packetSizeAvg += packetSize;
	nIlMin = MIN(n, nIlMin);
	nIlMax = MAX(n, nIlMax);
	nIlAvg += n;
      }

      /* Go on with next interleaver */
      iln = (int)(rsPckHeader.ilNumber);
      n = 0;
    }

    packetSize = rsPckHeader.bytesPerPck;

    /* Read packet data from input file */
    if ( (bytesinbuffer = fread(buffer +
				MIN(n,(kIlInfo)) * packetSize,
				1, packetSize, inputfilepointer)) < 
	 bytesinbuffer) {
      fprintf(protocol,"Bitstream syntax error: not enough bytes in packet .\n"
	      /*, packetheader.packetnumber*/);
      exit(3);
    }
  }

  /* Print statistics */
  fprintf(protocol, "\n\ntotalNumberIl: %d\n", totalNumIl);
  if (totalNumIl > 0) {
    /* packet size */
    fprintf(protocol, "pckSizeMin: %d\n", packetSizeMin);
    fprintf(protocol, "pckSizeMax: %d\n", packetSizeMax);
    fprintf(protocol, "pckSizeAvg: %g\n",
	    (double)(packetSizeAvg) / (double)(totalNumIl));
    /* N il */
    fprintf(protocol, "nIlMin: %d\n", nIlMin);
    fprintf(protocol, "nIlMax: %d\n", nIlMax);
    fprintf(protocol, "nIlAvg: %g\n", (double)(nIlAvg) / (double)(totalNumIl));
    /* kIl */
    fprintf(protocol, "kIlMin:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %d", kIlMin[i]);
    fprintf(protocol, "\n");
    fprintf(protocol, "kIlMax:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %d", kIlMax[i]);
    fprintf(protocol, "\n");
    fprintf(protocol, "kIlAvg:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %g", (double)(kIlAvg[i]) /
	      (double)(totalLayerNumIl[i]));
    fprintf(protocol, "\n");
    /* bytes per layer packet */
    fprintf(protocol, "bytesMin:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %d", bytesPerLayerPckMin[i]);
    fprintf(protocol, "\n");
    fprintf(protocol, "bytesMax:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %d", bytesPerLayerPckMax[i]);
    fprintf(protocol, "\n");
    fprintf(protocol, "bytesAvg:");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %g", (double)(bytesPerLayerPckAvg[i]) /
	      (double)(totalLayerNumIl[i]));
    fprintf(protocol, "\n");
    /* number of interleavers for each layer */
    fprintf(protocol, "totalNumberLayerIl: ");
    for (i = 0; i < maxLayers; i++)
      fprintf(protocol, " %d", totalLayerNumIl[i]);
    fprintf(protocol, "\n");
  }


  /* Close files */
  fclose(inputfilepointer);
  fclose(protocol);

  return 0;
}
