/***************************************************************************** 
 * Channel simulator for the Reed-Solomon-Decoder based on Newton's
 *         interpolation 
 *
 * Michael Link, LNT, Uni Erlangen, 1995-98
 *
 * Last changes: Jun 16, 1998: Markov Model for packet loss.
 *
 * compile with:  gcc -o channel channel.c rand32.c -lm
 *
 *****************************************************************************/

#include "codec.h"
#include "galois.h"
#include "rs.h"
#include "rand32.h"
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <time.h>

#include "bitOut.h"
#include "chacodec.h"


#define BSC           1
#define ERRORFILE     2
#define PACKETLOSS    3
#define DELETEPACKETS 4
#define MARKOV        5

#define WRITE    0
#define NOWRITE  1

#define PRINTRESULTS	 /*MS*/

#define MAX_PACKETSIZE   1024 /*MS*/

void usage (char *name)
{
	fprintf(stderr, "usage: %s  [-i InputFilename]  [-o OutputFilename]  CHANNEL \n\n", name);
	fprintf(stderr, "Possible CHANNEL values are:\n");
	fprintf(stderr, "  -bBitErrorRate: Insert bit errors into packet data with this rate (BSC).\n");
	fprintf(stderr, "  -eErrorFilename: Add data from this file to packet data.\n");
	fprintf(stderr, "  -pPacketLossRate: Omit packets with this rate.\n");
	fprintf(stderr, "  -mPacketLossRate AvgBurstLength: Markov model.\n");
	fprintf(stderr, "  -dFirstLostPacket LastLostPacket: Omit these packets in every block.\n");
	fprintf(stderr, "options: -i/-o: if these are omitted, stdin/stdout are used.\n");
	fprintf(stderr, "  -rsp: use R-S Interleaver packet stream\n");
	fprintf(stderr, "  -sSliceLength: Simply cut out 'SliceLength'-Bytes.\n\n");
	fprintf(stderr, "M. Link, LNT, 04/27/1998\n\n");

	exit(PARAMETER_ERROR);
}



/***************************************************************************** 
 *
 * main function: Channel
 *
 *****************************************************************************/
int main(int argc, char *argv[])
{
  char                        codeddataname[80], errordataname[80],
                              errorfilename[80];  /* filenames and pointers */
  char                        protocolname[80];
  FILE                        *inputfilepointer = stdin,
                              *outputfilepointer = stdout, *errorfilepointer;
  FILE                        *protocol; /*MS*/
#if 0
  struct fileheader_struct    fileheader;               /* file header */
#endif
  struct packetheader_struct  packetheader;             /* packet header */
  
  int                         packetSize;

  unsigned char               i;
  
  
  unsigned char               *buffer, errorbyte = 0;
  unsigned short              bytesinbuffer, bytenumber, errorbytesread;
  
  
  float                       biterrorrate = 0, packetlossrate = 0;   
  float                       avgburstlength = 0, transprob[2];
  unsigned short              firstlostpacket = 0, lastlostpacket = 0;
  int                         state = GOOD;

  unsigned char               kindoferror = 0;          /* flags */
  unsigned char               write = WRITE;

  unsigned short              bit;                  /* number of bit in byte */

  int                         slice = 0;

  int                         prs_f = 0;     /* Flag for R-S packet stream */
  PackRSFileHeader            packFileHeader;
  RSPacketHeader              rsPckHeader;

  
  /*---------------------------------------------------------------------------
   * Check command line arguments
   *-------------------------------------------------------------------------*/

  for (i=1; i<argc; i++) {
    if (argv[i][0] == '-') {
      switch (argv[i][1]) {

      case 'i' :

	/* Open input file: search *, *.rs, *.263.rs */
	sprintf(codeddataname, "%s", argv[++i]);
	if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	  sprintf(codeddataname, "%s.rs", argv[i]);
	  if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	    sprintf(codeddataname, "%s.263.rs", argv[i]);
	    if ( (inputfilepointer = fopen(codeddataname, "r")) == NULL ) {
	      fprintf(stderr,"Error in %s: can't open %s for reading.\n", argv[0], argv[i]);
	      exit(FILE_ERROR);
	    }
	  }
	}			  

	break;

      case 'o':

	/* Open output file for writing */
	if ( argv[++i][1] == '-' ) {
	  if ( codeddataname[0] != '\0' ) {
	    sprintf(errordataname, "%s.err", codeddataname); /* append ".err" for errored file */
	    if ( (outputfilepointer = fopen(errordataname, "w")) == NULL ) {
	      fprintf(stderr,"Error in %s: can't open %s for writing.\n",
		      argv[0], errordataname);
	      exit(FILE_ERROR);
	    }
	  } else
	    usage(argv[0]);
	} else {
	  sprintf(errordataname, "%s", argv[i]);
	  if ( (outputfilepointer = fopen(errordataname, "w")) == NULL ) {
	    fprintf(stderr,"Error in %s: can't open %s for writing.\n",
		    argv[0], errordataname);
	    exit(FILE_ERROR);
	  }
	  /*MS-Begin*/
	  protocolname[0] = '\0';
	  sprintf(protocolname, "pro_%s", argv[i]);
	  if ( (protocol = fopen(protocolname, "w")) == NULL ) {
	    fprintf(stderr,"Error in %s: can't open %s for writing.\n",
		    argv[0], protocolname);
	    exit(FILE_ERROR);
	  }
	  /*MS-End*/
	}

	break;

      case 'e' :

	sprintf(errorfilename, "%s", &argv[i][2]);
	if ( (errorfilepointer = fopen(errorfilename, "r")) == NULL ) {
	  fprintf(stderr,"Error in %s: can't open %s for reading.\n",
		  argv[0], errorfilename);
	  exit(FILE_ERROR);
	}
	kindoferror = ERRORFILE;

	break;

      case 'b' :

	biterrorrate = atof(&argv[i][2]);
	kindoferror = BSC;

	break;

      case 'p' :

	packetlossrate = atof(&argv[i][2]);
	kindoferror = PACKETLOSS;

	break;

      case 'm' :

	packetlossrate = atof(&argv[i][2]);
	avgburstlength = atof(argv[++i]);
	kindoferror = MARKOV;

	transprob[BAD] = 1/avgburstlength;
	transprob[GOOD] = packetlossrate/(1-packetlossrate)/avgburstlength;

	break;

      case 'd' :

	firstlostpacket = atoi(&argv[i][2]);
	lastlostpacket = atoi(argv[++i]);
	kindoferror = DELETEPACKETS;

	break;

      case 'r' :

	prs_f = 1;

	break;

      case 's' :

	slice = atoi(&argv[i][2]);

	break;

      default  :

	usage(argv[0]);

	break;
      }
    } else
      usage(argv[0]);
  }

  if ( !kindoferror ) {
    fprintf(stderr, "Define CHANNEL!\n");
    usage(argv[0]);
  }
  
  
  /*---------------------------------------------------------------------------
   * Read parameters from input file
   *-------------------------------------------------------------------------*/
  if (prs_f) {
    fread(&packFileHeader, sizeof(PackRSFileHeader), 1, inputfilepointer);

    packetSize = packFileHeader.bytesPerPck;

    /* Initialize packet buffer */
    if ((buffer = (unsigned char *)malloc(packetSize *
					  sizeof(unsigned char))) == NULL) {
      fprintf(stderr,"Error in %s: malloc returns NULL.\n", argv[0]);
      exit(MEMORY_ALLOCATION_ERROR);
    }

    /* Write parameters to output file header */ 
    fwrite(&packFileHeader, sizeof(PackRSFileHeader), 1, outputfilepointer);

    fread(buffer, sizeof(unsigned char), packFileHeader.maxLayers,
	  inputfilepointer);
    fwrite(buffer, sizeof(unsigned char), packFileHeader.maxLayers,
	   outputfilepointer);

#ifdef PRINTRESULTS
    fprintf(protocol, "RS-packed stream\n");
    fprintf(protocol, "%d bytes per packet; max %d packets per interleaver;\n",
	    packFileHeader.bytesPerPck, packFileHeader.nIlmax);
    fprintf(protocol, "k of il info %d; packetization mode %d\n",
	    packFileHeader.k_il_info, packFileHeader.pckMode);
#endif

  } else if (slice) {
    /* Plain bitstream, not packed. Cut out slices of length 'slice'-bytes */
    packetSize = slice;

    /* Initialize packet buffer */
    if ((buffer = (unsigned char *)malloc(packetSize *
					  sizeof(unsigned char))) == NULL) {
      fprintf(stderr,"Error in %s: malloc returns NULL.\n", argv[0]);
      exit(MEMORY_ALLOCATION_ERROR);
    }

  } else {
    /* File format:
     *
     * header:   short   bitspersymbol     ( short => highbyte lowbyte )
     *           short   n  
     *           short   K
     *           short   bytesperpacket            
     *
     * packet 0: short   packet number              \ packet header
     *           short   packet size in bytes       / 
     *           char    data... (packet size bytes)
     *
     * packet 1: ...
     */

#if 0
    fread(&fileheader, sizeof(char), sizeof(fileheader), inputfilepointer);

    /* Initialize packet buffer */
    if ((buffer = (unsigned char *)malloc(MAX_PACKETSIZE *
					  sizeof(unsigned char))) == NULL ) {
      fprintf(stderr,"Error in %s: malloc returns NULL.\n", argv[0]); /*MS*/
      exit(MEMORY_ALLOCATION_ERROR);
    }

    /* Write parameters to output file header */ 
    fwrite(&fileheader, sizeof(char), sizeof(fileheader), outputfilepointer); 
#endif
  }
	
	
#ifdef PRINTRESULTS	
  switch ( kindoferror ) {
  case BSC:
    fprintf(protocol,"Channel: Bit error rate: %f\n", biterrorrate);
    break;
  case PACKETLOSS:
    fprintf(protocol,"Channel: Packet loss rate: %f\n", packetlossrate);
    break;
  case MARKOV:
    fprintf(protocol,"Channel: Markov Model: Average packet loss rate = %f\n",
	    packetlossrate);
    fprintf(protocol,"                       Average burst length = %f\n",
	    avgburstlength);
    fprintf(protocol,"                       p_GB = %f, p_BG = %f\n",
	    transprob[GOOD], transprob[BAD]);
    break;
  case DELETEPACKETS:
    fprintf(protocol,"Channel: Delete packets %i - %i\n", firstlostpacket,
	    lastlostpacket);
    break;
  case ERRORFILE:
    fprintf(protocol,"Channel: Error file: %s\n", errorfilename);
    break;
  }
#ifdef PRINTRESULTS
  fprintf(protocol,"\n""-""=received packet, ""X""=lost packet\n");
#endif

#endif


  /* Initialize random number generator */
  SetRand32(time(NULL));

  /* Read packet header from input file */
  while (!feof(inputfilepointer)) {

    if (prs_f) {
      if (fread(&rsPckHeader, sizeof(RSPacketHeader), 1, inputfilepointer) ==0)
	break;
      packetSize = rsPckHeader.bytesPerPck;
    } else if (!slice) {
      if (fread(&packetheader, sizeof(packetheader), 1, inputfilepointer) !=
	sizeof(packetheader) )
	break;
      packetSize = packetheader.bytesinpacket;
    }

    /* Read packet data from input file */
    if ( (bytesinbuffer = fread(buffer, 1, packetSize, inputfilepointer)) < 
	 bytesinbuffer) {
      fprintf(protocol,"Bitstream syntax error: not enough bytes in packet .\n"
	      /*, packetheader.packetnumber*/);
      exit(FILE_ERROR);
    }

    switch ( kindoferror ) {

    case ERRORFILE:

      for ( bytenumber = 0 ; bytenumber<bytesinbuffer ; bytenumber++ ) {
	errorbytesread = fread(&errorbyte, sizeof(char), 1, errorfilepointer);

	if ( errorbytesread )
	  buffer[bytenumber] ^= errorbyte;
	else {
	  rewind(errorfilepointer);
	  errorbytesread = fread(&errorbyte, sizeof(char), 1,errorfilepointer);
	  if ( errorbytesread )
	    buffer[bytenumber] ^= errorbyte;
	  else {
	    fprintf(protocol,"Error in %s: Errorfile error.\n", argv[0]);
	    exit(FILE_ERROR);
	  }
	}
      }

      break;

    case BSC:

      for ( bit=0 ; bit<bytesinbuffer*8 ; bit++)
	if ( (float) Rand32(8*sizeof(unsigned long))/ULONG_MAX < biterrorrate )
	  buffer[bit/8] ^= 2 << (bit%8);

      break;

    case PACKETLOSS:

      if ( (float) Rand32(8*sizeof(unsigned long))/ULONG_MAX < packetlossrate )
	write = NOWRITE;
      else
	write = WRITE;

      break;

    case MARKOV:

      if ((float)Rand32(8*sizeof(unsigned long))/ULONG_MAX <transprob[state]) {
	state = 1-state;
	write = (state==GOOD) ? WRITE : NOWRITE;
      }

      /* 	  fprintf(stderr,"%1i", state); */
      /* 	  fprintf(stderr,"%1i", (state==GOOD)); */
      /* 	  if (write==WRITE) */
      /* 		fprintf(stderr,"w"); */
      /* 	  else */
      /* 		fprintf(stderr,"l"); */

      break;

    case DELETEPACKETS:

      printf("DELETEPACKETS must be adapted for use!!!"); /*MS*/
      exit(1); /*MS*/
#if 0 /*MS*/
      if ( (packetheader.packetnumber % fileheader.nrofcodesymbols >= firstlostpacket) &&
	   (packetheader.packetnumber % fileheader.nrofcodesymbols <= lastlostpacket) )
	write = NOWRITE;
      else
	write = WRITE;
#endif

      break;

    }


    /* Write errored packet to output file */
    if ( write == WRITE ) {

      if (prs_f) {
	/* Write packetheader */
	fwrite(&rsPckHeader, sizeof(RSPacketHeader), 1, outputfilepointer);
      } else if (!slice) {
	/* Write packetheader */
	fwrite(&packetheader, sizeof(char), sizeof(packetheader),
	       outputfilepointer); 
      }

      /* Write buffer */
      fwrite(buffer, 1, packetSize, outputfilepointer);

#ifdef PRINTRESULTS
      fprintf(protocol,"-");
#endif

    } else {

#ifdef PRINTRESULTS
      fprintf(protocol,"X");
#endif
    }
  }

#ifdef PRINTRESULTS
  fprintf(protocol,"\n\n");
#endif

  fclose(inputfilepointer);
  fclose(outputfilepointer);
  if ( kindoferror == ERRORFILE )
    fclose(errorfilepointer);

  return 0;
}


















