/**********************************************************************************
 * Galois Field library:
 *
 *  ==>  Galois Field basic calculus functions and transforms: 
 *  
 * GFinit
 * GFFree
 * mulGF
 * divGF
 * GFexp
 *
 * NewtonTransform
 * InverseNewtonTransform
 * NewtonInterpolation
 *
 *  ==>  Memory management functions:
 *  
 * StackInit
 * StackFree
 * StackCheck
 * MakeVector
 * PrintVector
 * PrintVectorAll
 * PrintVectorFormat
 *
 *
 * M. Link, LNT, Uni Erlangen, 1995-97 
 *
 * Last changes: Aug 28, 1997
 *
 ***********************************************************************************/ 

#include "codec.h"
#include "galois.h"
#include <stdio.h>
#include <stdlib.h>
 

/**********************************************************************************
 **********************************************************************************
 *
 *  ==>  Galois Field basic calculus functions. 
 *  
 **********************************************************************************
 ***********************************************************************************/


static SYMBOL *LOG, *EXP;           /* GF arithmetic tables, see GFInit */

static struct stack_struct  stack;  /* Workspace, see StackInit */


/*********************************************************************************** 
 * GFInit:   Initialize tables LOG and EXP
 * 
 * input:    l: number of bits per symbol 
 * modifies: GF tabels *LOG, *EXP 
 ***********************************************************************************/
void GFInit(short l)
{
  extern unsigned short    N;
  extern SYMBOL            **GFMULTAB; 
  extern SYMBOL            **GFDIVTAB; 
	
  register unsigned short  i, j;
  SYMBOL                   mask, element;
  
  /* Primitive polynomials for l = 0:16 (forget l=0, l=1, just simple indexing).
   * Binary coded without coefficient 1 at x^l, 
   * e.g. l=3: 3=011 stands for x^3 + 0*x^2 + 1*x^1 + 1*x^0  
   */
  const unsigned short primitive_polynomial[17] = {0,0,3,3,3,5,3,3,113,17,9,5,153,27,323,3,4107}; 
  
  /* Set N */
  N = (unsigned short) (1<<l) - 1;
  
  /* Initialize stack HERE, because N is number of rows and must be set before */
  StackInit();
  
  /* Allocate vectors from the stack */
  LOG = MakeVector();
  EXP = MakeVector();
  
  
  /* allocate memory for multiplication table */
  
  /* allocate memory for N+1 row pointers */
  if ( (GFMULTAB = (SYMBOL **) malloc( (N+1) * sizeof(SYMBOL*) )) == NULL ) {
	fprintf(stderr,"Error in GFInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* allocate memory for (N+1) x (N+1) SYMBOLs */
  if ( (GFMULTAB[0]=(SYMBOL *) malloc( (N+1) * (N+1) * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in GFInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* set row pointers */
  for( i=1 ; i<N+1 ; i++)
	GFMULTAB[i] = GFMULTAB[i-1] + (N+1);
  
  
  /* allocate memory for division table */
  
  /* allocate memory for N+1 row pointers */
  if ( (GFDIVTAB = (SYMBOL **) malloc( (N+1) * sizeof(SYMBOL*) )) == NULL ) {
	fprintf(stderr,"Error in GFInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* allocate memory for (N+1) x (N+1) SYMBOLs */
  if ( (GFDIVTAB[0]=(SYMBOL *) malloc( (N+1) * (N+1) * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in GFInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* set row pointers */
  for( i=1 ; i<N+1 ; i++)
	GFDIVTAB[i] = GFDIVTAB[i-1] + (N+1);
  
  
  
  
  /* mask for the MSB of SYMBOL */ 
  mask = (SYMBOL) 1 << (l-1);
  
  for ( i=0, element=1 ; i < N ; i++){
	EXP[i] = element;
	LOG[element] = i;
	if (element & mask)   
	  /* overflow will happen, so calculate element*2 mod primitive polynomial */
	  element = ((element << 1) ^ primitive_polynomial[l]) & (SYMBOL) N;
	else                   
	  /* no overflow, calculate element*2 */
	  element <<= 1;
  }
  
  EXP[N] = 0;
  LOG[0] = N;
  
  
  /* Calculate the multiplication table (look-up faster than mulGF function call!) */
  for ( i=0 ; i <= N ; i++ )
	for ( j=0 ; j <= N ; j++ )
	  GFMULTAB[i][j] = mulGF(i,j);
  
  /* Calculate the division table (look-up faster than divGF function call!) */
  for ( i=0 ; i <= N ; i++ ) {
	GFDIVTAB[i][0] = 0;
	for ( j=1 ; j <= N ; j++ )
	  GFDIVTAB[i][j] = divGF(i,j);
  }
  
  return;
}
	

/*********************************************************************************** 
 * GFFree:   Free tables 
 * 
 * input:    ---
 *
 * modifies: ---
 *
 ***********************************************************************************/
void GFFree(void)
{
	FreeVector(LOG);
	FreeVector(EXP);
	
	StackFree();

	free(GFMULTAB[0]);
	free(GFMULTAB);

	free(GFDIVTAB[0]);
	free(GFDIVTAB);

	return;
}



/*********************************************************************************** 
 * mulGF:   Multiplication of a and b in GF(2^l)
 *          A (N+1) x (N+1) multiplication table is faster, so this function is
 *          used only to create the table (see GFinit). The table look-up is done 
 *          via a define for GFmul (see galois.h)!
 * 
 * input:    a, b
 * output:   c = a*b
 *
 * Just adding the exponents of a and b mod 2^l-1 = N.  
 ***********************************************************************************/
SYMBOL mulGF(SYMBOL a, SYMBOL b)
{
  	extern unsigned short N;

	if (!(a && b))
		/* at least one value == 0 */
		return 0;
	else if ( (long int) (LOG[a] + LOG[b]) >= (long int) N )
		return EXP[ LOG[a] + LOG[b] - N ];
	else
		return EXP[ LOG[a] + LOG[b] ];
}

		

/*********************************************************************************** 
 * GFdiv:    Division  a/b  in GF(2^l)
 * 
 * input:    a, b
 * output:   c = a/b
 *
 * Just substracting the exponents of a and b mod 2^l-1 = N.  
 ***********************************************************************************/
SYMBOL divGF(SYMBOL a, SYMBOL b)
{
  	extern unsigned short N;

	if (!b){			/* b = 0 */
	    fprintf(stderr, "Error in GFdiv: division by zero.\n");
		return 0;  /* better than nothing */
	}
	else if (!a)     /* a = 0 */
		return 0;
	else if (LOG[a] >= LOG[b])
		return EXP[ LOG[a] - LOG[b] ];
	else
		return EXP[ N + LOG[a] - LOG[b] ];
}

		

/*********************************************************************************** 
 * GFexp:    Exponentation of a to the power of k in GF(2^l)
 * 
 * input:    a, k
 * output:   c = a^k
 *
 * Exponent of a times k mod 2^l-1 = N.  
 ***********************************************************************************/
SYMBOL GFexp(SYMBOL a, int k)
{
  	extern unsigned short N;

	if (a)
	    return EXP[ (SYMBOL) ((( (LOG[a] * k) % N) + N) % N)];
	else if (k)
		return 0;
	else 
		return 1;
}

		
	 
/*********************************************************************************** 
 * NewtonTransform:   Newton transform: calculate the Newton coefficients *A and the
 *                    new last column *AA from the (new) samples *a at (all) points *z,
 *                    using the last column *AA.
 * 
 * input:    a, z, AA
 * output:   ---
 *
 * modifies: *AA, *A
 *
 * NOTE:     a[N] is the number of new sample points,
 *           AA[N] is the number of points previously used,
 *           z[N] is the number of ALL sample points z[ 0 ... z[N]-1 ]:  
 *
 *           AA[N] + a[N] = z[N]
 *
 *           *a is filled up with zeros if necessary.
 *
 *           If A == NULL, A = NULL is returned: calculate AA only!
 *
 *           If AA == NULL, AA = NULL is returned: calculate A only!
 *
 ***********************************************************************************/
void NewtonTransform(SYMBOL *a, SYMBOL *z, SYMBOL *AA, SYMBOL *A)
{
    extern unsigned short N;
	extern SYMBOL         **GFMULTAB;
	extern SYMBOL         **GFDIVTAB;


	register SYMBOL  k,l;	
	SYMBOL *row, AA_store, sizeofAA;
	

	if (A == NULL)             /* don't return Newton cofficients */
		row = MakeVector();
	else
		row = A;               /* the last row will be the Newton coefficients A */


	if (AA == NULL)            /* don't return last column */
		sizeofAA = 0;
	else 
		sizeofAA = AA[N];
	

	/* copy *a to *row, fill up with zeros if necessary; this is the top row 0 */
	memcpy(row, a, a[N]*sizeof(SYMBOL)); 
	memset(&row[a[N]], 0, (z[N]-sizeofAA-a[N])*sizeof(SYMBOL));
	row[N] = z[N] - sizeofAA;   /* set size */


	/* The last coefficients of each row are the new last column. The old AA[0] 
	   is used in the next step, so store the new AA[0] first.     */
	AA_store = 0;      /* set to 0, true value a[a[N]-1] see below */
	

	/* calculate the rows 1 ... sizeofAA with constant length a[N] */
	for ( k=1 ; k <= sizeofAA ; k++ ){
		/* backwards through row => no store */
		for ( l = row[N]-1 ; l ; l--)
			row[l] = GFdiv( row[l-1] ^ row[l] , z[l+sizeofAA-k] ^ z[l+sizeofAA] );
		
		/* first element of row, use old last column */
		row[0] = GFdiv( AA[k-1] ^ row[0] , z[sizeofAA-k] ^ z[sizeofAA] );
		
		/* take new AA[k-1] calculated in the step before from the store */ 
		AA[k-1] = AA_store;
		/* store new AA[k] */
		AA_store = row[row[N]-1];
	}

	/* don't forget last stored coefficient */
	if (AA!=NULL)
		AA[sizeofAA] = AA_store;
		

	/* calculate the rows sizeofAA+1 ... z[N]-1 with decreasing length */
	for ( k=sizeofAA+1 ; k < z[N] ; k++ ){
		/* backwards through row => no store */
		for ( l = row[N]-1 ; l >= (k-sizeofAA) ; l--)
			row[l] = GFdiv( row[l-1] ^ row[l] , z[l+sizeofAA-k] ^ z[l+sizeofAA] );
		
		/* new AA[k] */
		if (AA!=NULL)
			AA[k] = row[row[N]-1];
	}

	/* first element of new last column, set to 0 earlier */
	if (AA!=NULL)
		AA[0] = a[a[N]-1];
		  

	/* set size of *A, *AA */
	if (A == NULL)
		FreeVector(row);
	else
		A[N] = z[N]-sizeofAA;
	
	if (AA != NULL)
		AA[N] = z[N];
				
	/* Now A = row points at the new Vector *A, 
	   AA points at the new Vector *AA. Ready! */

}



/*********************************************************************************** 
 * InverseNewtonTransform:   Inverse Newton transform: 
 *                           Calculate the samples *a at points *z from the Newton 
 *                           coefficients *A, using the last column *AA.
 * 
 * input:    A, z, AA
 * output:   ---
 *
 * modifies: *AA, *a
 *
 * NOTE:     z[N] is the number of ALL sample points z[ 0 ... z[N]-1 ]  
 *           AA[N] is the number of points previously used,
 *           A[N] is the number of new sample points:
 *
 *           AA[N] + A[N] <= z[N]
 *
 *           If AA == NULL, AA = NULL is returned.
 ***********************************************************************************/
void InverseNewtonTransform(SYMBOL *A, SYMBOL *z, SYMBOL *AA, SYMBOL *a)
{
  	extern unsigned short N;
	extern SYMBOL         **GFMULTAB;
	extern SYMBOL         **GFDIVTAB;

	register SYMBOL  k,d;	
	SYMBOL *row, AA_store, sizeofAA;
	


	row = a;                     /* the top row will be the samples a */

	   
	if (AA == NULL)
		sizeofAA = 0;
	else 
		sizeofAA = AA[N];
	
				  
	/* copy *A to *row, fill up with zeros if necessary; this is the bottom row 0 */
	memcpy(row, A, A[N]*sizeof(SYMBOL)); 
	memset(&row[A[N]], 0, (z[N]-sizeofAA-A[N])*sizeof(SYMBOL));
	row[N] = z[N] - sizeofAA;   /* set size */


	if ( row[N] == 0 ) return;

	

	/* last element of each row = element of the new last column */
	if (AA != NULL)
		AA[ z[N]-1 ] = row[row[N]-1];


   	/* calculate the rows 1 ... sizeofAA with constant length A[N] */
	/* d = counter of the diagonals bottom to top, here called rows!!! */
	for ( d=1 ; d <= sizeofAA ; d++){
		/* backwards through row => no store */
		for ( k = row[N]-1 ; k>0 ; k--)			
			row[k] = row[k-1] ^ GFmul(row[k] , z[sizeofAA+k] ^ z[d-1] );
		
		/* first element of row, use old last column */
		row[0] = AA[ sizeofAA-d ] ^ GFmul(row[0] , z[sizeofAA] ^ z[d-1] );
		
		/* new AA element */
		AA[ z[N]-d-1 ] = row[row[N]-1];
	}


	/* calculate the rows sizeofAA+1 ... z[N]-1 with decreasing length */
	for ( d=sizeofAA+1 ; d < z[N] ; d++ ){
		/* backwards through row => no store */
		for ( k = row[N]-1 ; k >= (d-sizeofAA) ; k--)
			row[k] = row[k-1] ^ GFmul(row[k] , z[sizeofAA+k] ^ z[d-1] );
		
		/* new AA element */
		if (AA != NULL)
			AA[ z[N]-d-1 ] = row[row[N]-1];
	}

	/* set size of *a, *AA */
	a[N] = z[N]-sizeofAA;
	if (AA != NULL)
		AA[N] = z[N];
 				
	/* Now a = row points at the new vector *a, AA points at the new vector *AA. Ready! */

}



/*********************************************************************************** 
 * FastNewtonInterpolation:  Newton's Interpolation, FAST Version without sorting
 *                           and function calls
 * 
 * input:    
 *
 * output:   ---
 *
 * modifies: 
 *
 ***********************************************************************************/
void FastNewtonInterpolation(SYMBOL *data, unsigned long *ind, SYMBOL *zold, SYMBOL K, SYMBOL *znew, SYMBOL nnew)
{
  extern unsigned short N;
  extern SYMBOL         **GFMULTAB;
  extern SYMBOL         **GFDIVTAB;
  
  register short        k, l;

  SYMBOL                *z, *row, *col;
  
  z = MakeVector();	
  row = MakeVector();	
  col = MakeVector();	
  
#ifdef PRINTRESULTS
  fprintf(stderr, "FastNewtonInterpolation: K = %i, nnew = %i\n", K, nnew);
  PrintVectorAll("zold", zold, K);
  PrintVectorAll("znew", znew, nnew);
#endif				
	
  /* Calculate last column of triangular Newton transform matrix with K old points */

  /* copy samples to first row and samplepoints to z */
  for ( k=0 ; k<K ; k++ ) {
	z[k] = zold[k];
	row[k] = data[ind[zold[k]]];
  }

  /* last element of row is first element of column */
  col[0] = row[K-1];

  /* calculate matrix row by row */
  for ( k=1 ; k < K ; k++ ){

	/* backwards through row => store new row to old row */
	for ( l = K-1 ; l >= k ; l--) 
	  row[l] = GFdiv( row[l-1] ^ row[l] , zold[l-k] ^ z[l] );
	
	/* last element of row is next element of column */
	col[k] = row[K-1];
  }

  

  /* Calculate nonzero elements of Newton transform matrix at nnew new points */

  /* append new samplepoints to z */
  memcpy(&z[K], znew, nnew*sizeof(SYMBOL));

  /* set last row (K-1) to last element of column */
  for ( l=0 ; l<nnew ; l++ )
	row[l] = col[K-1];

  /* calculate rows K-2 ... 0 with constant length */
  for ( k=K-2 ; k >= 0 ; k-- ){

	/* first element of row calculated with column element */
	row[0] = col[k] ^ GFmul( row[0] , z[K-k-1] ^ z[K] );

	/* forward through row => store new row to old row */
	for ( l=1 ; l<nnew ; l++ ) 
	  row[l] = row[l-1] ^ GFmul( row[l] , z[K+l-k-1] ^ z[K+l] );
  }

  /* copy first row to data samples */
  for ( l=0 ; l<nnew; l++)
	data[ind[znew[l]]] = row[l];
		
  FreeVector(col);
  FreeVector(row);
  FreeVector(z);
  
  return;
}




/*********************************************************************************** 
 * NewtonInterpolation:  Newton's Interpolation
 *                       Calculate the samples *anew at points *znew from the 
 *                       samples *aold at points *zold. The interpolation polynomial
 *                       is of degree K-1 (K coefficients).  
 * 
 * input:    aold, zold, K, znew, anew
 *
 * output:   ---
 *
 * modifies: *anew
 *           *II:   Last column of the interpolation polynomial (K coefficients)
 *
 ***********************************************************************************/
void NewtonInterpolation(SYMBOL *aold, SYMBOL *zold, SYMBOL K, SYMBOL *znew, SYMBOL *anew, SYMBOL *II)
{
  extern unsigned short N;
  extern SYMBOL         **GFMULTAB;
  extern SYMBOL         **GFDIVTAB;
  
  SYMBOL                *A, *AA, *areallynew, *zreallynew, *perm, *a, *z;
  SYMBOL                nnew, nold;
  register SYMBOL       i, j;
  
  
  zreallynew = MakeVector();
  areallynew = MakeVector();
  z = MakeVector();
  a = MakeVector();
  perm = MakeVector();
  A = MakeVector();	
  AA = MakeVector();
  

  /* search for points *zold which are in *znew,
	 put these in *z & *a,
	 put the really new points in *zreallynew & *areallynew 
	 */
  
  
  nnew = 0;
  nold = 0;
  
  memcpy(anew, znew, znew[N]*sizeof(SYMBOL));
  anew[N] = znew[N];
  
  for ( i=0 ; i < znew[N] ; i++ ){         /* search all new points */
	
	for ( j=0; (j < zold[N]) && (znew[i] != zold[j]) ; j++ );  /* stop if znew == zold */
	
	if (j < zold[N]) {                     /* znew == zold was found, this is an old point */
	  z[nold] = zold[j];                   /* use this point for the interpolation */
	  a[nold++] = aold[j];
	  anew[i] = aold[j];                   /* it's one of the result samples */
	} else {                               /* this is really a new point */
	  zreallynew[nnew] = znew[i];          /* interpolate at these points */
	  perm[nnew++] = i;                    /* the interpolated sample must be placed there later */
	}
		
  }
  zreallynew[N] = nnew;
  perm[N] = nnew;



  /* fill up *z & *a from length nold to length K with points from *zold which are not in *znew */

  for( j=0 ; (nold<K) && (j<zold[N]) ; j++ ) { /* search all old points until length nold == K */
	
	for ( i=0 ; (i<nold) && (zold[j] != z[i]); i++);           /* stop if z == zold */

	if ( i==nold ) {                        /* z == zold not found, this point was not used so far */
	  z[nold] = zold[j];                    /* use this point for the interpolation */
	  a[nold++] = aold[j];
	}
  
  }


  if ( (nold>=K) && (nnew>0) ) {                           /* ok, length K is reached */

	/* calculate Newton transform with K points*/
	z[N] = K;
	a[N] = K;
	NewtonTransform(a, z, AA, NULL);


	/* store as last column of the interpolation polynomial */
	if ( II ) {
	  for ( i=0 ; i<K ; i++)
		II[i] = AA[i];
	  II[N] = K;
	}
		
		
	/* append new points */

	memcpy(&z[K], zreallynew, nnew*sizeof(SYMBOL));
	z[N] = K+nnew;
	
	/* calculate new samples */
	A[N] = 0;

	InverseNewtonTransform(A, z, AA, areallynew);

	for ( i=0 ; i < nnew; i++)
	  anew[ perm[i] ] = areallynew[i];
		
  } else {     /* less than K old points: just return the received symbols sorted correctly */

	if ( II ) II[N] = 0;  /* no interpolation polynomial was calculated */

  }
	
   

  FreeVector(AA);
  FreeVector(A);
  FreeVector(perm);
  FreeVector(a);
  FreeVector(z);
  FreeVector(areallynew);
  FreeVector(zreallynew);
  
  return;
}




/**********************************************************************************
 **********************************************************************************
 *
 *  ==>  Memory management functions
 *  
 **********************************************************************************
 ***********************************************************************************/



/*********************************************************************************** 
 * StackInit: Allocate memory for a STACKSIZE x (N+1) matrix of SYMBOLs 
 *            and an index vector with STACKSIZE elements
 * 
 * input:    ---
 *
 * modifies: stack: stack.data[i][j] is the element in the ith row and jth 
 *                  column. 0 <= i < STACKSIZE, 0 <= i <= N.
 *
 * output:   ---
 *
 ***********************************************************************************/
void StackInit(void)
{
  extern unsigned short   N;
  
  register unsigned short i;
  

  /* allocate memory for STACKSIZE row pointers */
  if ( (stack.data = (SYMBOL **) malloc( STACKSIZE * sizeof(SYMBOL*) )) == NULL )	{
	fprintf(stderr,"Error in StackInit: 1st malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* allocate memory for STACKSIZE x (N+1) SYMBOLs */
  if ( (stack.data[0] = (SYMBOL *) malloc( STACKSIZE * (N+1) * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in StackInit: 2nd malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* set row pointers */
  for( i=1 ; i<STACKSIZE ; i++)
	stack.data[i] = stack.data[i-1] + (N+1);
  


  /* allocate memory for index vector */
  if ( (stack.free = (unsigned char *) malloc( STACKSIZE * sizeof(unsigned char) )) == NULL ) {
	fprintf(stderr,"Error in StackInit: 3rd malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  
  /* set all vectors FREE */
  for ( i=0 ; i<STACKSIZE ; stack.free[i] = i++ );
  
  /* set free index to first element of vector free */
  stack.index = 0;
#ifdef CHECKSTACKMAX
  stack.maxindex = 0;
#endif
  
  return;
}


/*********************************************************************************** 
 * StackFree
 * 
 * input:    ---
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void StackFree(void)
{
#ifdef CHECKSTACKMAX
  StackCheck();
#endif

  free(stack.data[0]);
  free(stack.data);
  
  free(stack.free);
  
  return;
}



/*********************************************************************************** 
 * StackCheck
 * 
 * input:    ---
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void StackCheck(void)
{

#ifdef CHECKSTACKMAX
	printf("StackCheck: %i registers in use, max. %i\n====================\n",stack.index,stack.maxindex);
#else
	printf("StackCheck: %i registers in use\n====================\n",stack.index);
#endif

	return;
}

	
/*********************************************************************************** 
 * MakeVector:  Allocate one line of the stack.
 *              No initialization, except vector[N] = 0.
 * 
 * input:    ---
 *
 * modifies: ---
 *
 * output:   vector
 *
 ***********************************************************************************/
SYMBOL *MakeVector()
{
  extern unsigned short         N;
  
  SYMBOL *vector;
  
  
  if ( stack.index < STACKSIZE ){	
	vector = stack.data[stack.free[stack.index++]] ;
	vector[N] = 0;
  } else {
	fprintf(stderr,"Error in MakeVector: stack full.\n");
	exit(STACK_FULL);
  }


#ifdef CHECKSTACKMAX
  if ( stack.index > stack.maxindex )
	stack.maxindex = stack.index;
#endif

  return vector;
}

		

/*********************************************************************************** 
 * FreeVector:  Free one line of the stack.
 * 
 * input:    vector
 *
 * modifies: stack
 *
 * output:   ---
 *
 ***********************************************************************************/
void FreeVector(SYMBOL *vector)
{
  extern unsigned short  N;
	
  unsigned char free, i;


  free = (vector-stack.data[0]) / (N+1);          /* line number in the stack */
	
  if ( free<STACKSIZE ) {                         /* vector lies within stack */
	
#ifdef CHECKFREE
	for ( i=stack.index ; i<STACKSIZE ; i++)      /* check: is this line really allocated ? */
	  if ( free == stack.free[i] ) {              /* line is free: error */
		fprintf(stderr,"Error in FreeVector:  vector is free.\n");
		exit(FREE_VECTOR_ERROR);
	  }	
#endif
	
	stack.free[--(stack.index)] = free;
  } else {
	fprintf(stderr,"Error in FreeVector:  address out of stack.\n");
	exit(FREE_VECTOR_ERROR);
  }
  
  return;
}



/*********************************************************************************** 
 * PrintVector: Print vector of length vector[N] with label *name. 
 * 
 * input:    name
 *           vector
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void PrintVector(char *name, SYMBOL *vector)
{
	extern unsigned short    N;
	register unsigned short  i;	
	
	printf("%4s",name);
	
	for ( i=0 ; i < vector[N] ; printf("%4d",vector[i++]) );
	printf("\n");
}



/*********************************************************************************** 
 * PrintVectorAll: Print vector of length n with label *name. 
 *
 * input:    name
 *           vector
 *           n
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void PrintVectorAll(char *name, SYMBOL *vector, unsigned long n)
{
	register unsigned short  i;
	
	printf("%4s",name);
	
	for ( i=0 ; i < n ; printf("%4d",vector[i++]) );
	printf("\n");
}


/*********************************************************************************** 
 * PrintVectorFormat: Print float vector of length n with format *format and label *name. 
 * 
 * input:    name
 *           format
 *           n
 *           vector
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void PrintVectorFormat(char *name, char *format, float *vector, unsigned long n)
{
	register unsigned long  i;
	
	printf("%4s",name);
	
	for ( i=0 ; i < n ; printf(format, vector[i++]) );
	printf("\n");
}





