/*
 ***************************************************
 *
 *	Module: rand32.c
 *
 *	Author: Udo Wachsmann
 *
 *	Created: 28.4.95
 *
 ***************************************************
 *	Routines to generate random values based on a 32-bit shift register
 ***************************************************
 */
 
#include "rand32.h"

/*
 ***************************************************
 *	Register of random generator 'Rand32()'
 ***************************************************
 */
static unsigned long Rand32Reg	= 0x11a868UL;

/*
 ***************************************************
 *	SetRand32() sets the register of the random generator 'Rand32()'
 *	to 'RegSetting'
 ***************************************************
 *	Dependencies:	none
 ***************************************************
 */
void
SetRand32(unsigned long RegSetting)
{
	Rand32Reg = RegSetting;
}


/*
 ***************************************************
 *	GetRand32() returns the current value of the register of the 
 *	random generator 'Rand32()'
 ***************************************************
 *	Dependencies:	none
 ***************************************************
 */
unsigned long
GetRand32(void)
{
	return (Rand32Reg);
}

/*************************************************************************
 *	Rand32()
 *  Zufallszahlengenerator
 *	Variable 'Rand32Reg' is defined static global in this module
 *************************************************************************/
/*
 * Autor:           Peter Schramm
 * Compiler:        ANSI-C
 * letzte Aenderung: 20. Juni 1993
 *
 * Pseudozufallszahlengenerator, auf 32-Bit-Schieberegister aufbauend
 *
 */

#define PRIMPOL  0x80200003L    /* Primitives Polynom: Grad 32; das Wort
                                   enthaelt die Koeffizienten c(32) [MSB] bis
                                   c(1) [LSB], der Koeffizient c(0) ist als 1
                                   vorausgesetzt. Diese 'Anzapfungen' wurden
                                   dem Buch 'Spread Spectrum Systems' von Dixon
                                   entnommen. */

/*
 * Erzeugung einer Ausschnitts einer PN-Folge (pseudo-noise) der Laenge 2^N-1
 * Das primitive Polynom wird in der Konstanten PRIMPOL definiert. Dabei sind
 * ausschliesslich die Potenzen N(MSB),...,1(LSB) angegeben.
 * hier: N=32 --> Laenge 2^32-1
 *
 * Das hier verwendete prim. Polynom wurde dem Buch von Dixon entnommen.
 *
 * Literatur:
 * - J.Huber: Galoisfelder, das mathematische Werkzeug der Kanalcodierung.
 *            Manuskript zum Seminar 'Algebraische Kanalcodierung'.
 *            LNT, Erlangen, 1992.
 * - R.C.Dixon: Spread Spectrum Systems. New York, 1976. (SSS 5)
 *	
 *************************************************************************
 *   Parameter
 *   - anz      Anzahl von Bits des auszugebenden Ausschnitts der PN-Folge
 *************************************************************************
 *   Resultat:
 *   - Bitfolge in einer unsigned long-Variable, Folge beginnt beim LSB
 ***************************************************
 *	Dependencies:	none
 *************************************************************************/
unsigned long
Rand32(short anz)
{
	unsigned long bit, out;
	short i;

	out=0;
	for (i=0; i<anz; i++)
	{
	   /* Shift and Add */
	   bit = Rand32Reg & 1;
	   Rand32Reg >>= 1;
	   if (bit)
	      Rand32Reg ^= PRIMPOL;
	   out ^= bit<<i;
	}
	return out;
}

/*
 ***************************************************
 *	GaussRand32() returns a gaussian distributed random variable
 ***************************************************
 *	Reference: 	Frieden B.R.; Probability, statistical optics and data 
 *					testing: a problem solving approach; Springer 1983
 ***************************************************
 *	in:	- gaussDeviation	Deviation of the gaussian random variable
 ***************************************************
 *	Dependencies:	math.h
 ***************************************************
 */
double
GaussRand32(double gaussDeviation)
{
	/* Independent identically distributed random variables */
	double	iidRand1, iidRand2;
	iidRand1 = DRand32();
	iidRand2 = DRand32();
	return (gaussDeviation*sqrt(-2*log(iidRand1))*cos(2*PI*iidRand2));
}

