/*********************************************************************************** 
 * Reed-Solomon-Codec based on Newton's interpolation 
 *
 * Encoder & Decoder
 *
 *
 * Michael Link, LNT, Uni Erlangen, 1995-97
 *
 * Last changes: Jun 27, 1997
 ***********************************************************************************/


#include "codec.h"
#include "galois.h"
#include "rs.h"
#include <stdio.h>
#include <stdlib.h>


/*********************************************************************************** 
 * RSEncode: Systematic RS-Encoder using Newton's interpolation
 * 
 * input:    *word: Information word, K symbols.
 *           *samplepoints: All sample points, N symbols.
 *
 * modifies: *word: Codeword, N symbols.
 *
 * output:   ---
 *
 * The information symbols are the first K symbols of the codeword => systematic.
 * The last N-K Newton coefficients of the codeword are equal to zero. This holds 
 * also for the GFFT coefficients. 
 * 
 * Encoding is done by calculating the first K Newton coefficients of the info 
 * symbols. Then, starting with the last column of this transform and the new
 * Newton coefficients set to zero, the redundancy symbols are calculated by
 * the inverse Newton transform an appended to the info symbols.
 *
 ***********************************************************************************/
void RSEncode(SYMBOL *word, SYMBOL *samplepoints)
{
  extern unsigned short  N;            /* max. codeword length */
  extern SYMBOL          **GFMULTAB;
  extern SYMBOL          **GFDIVTAB;
  
  register unsigned long i;
  
  SYMBOL numberofpoints;               /* codeword length = number of samplepoints */
  SYMBOL *newtoncoeff;                 /* Newton coefficients of the codeword */
  SYMBOL *lastcolumn;                  /* last column of triangular matrix */
  SYMBOL *redundancy;                  /* redundancy symbols */
  
  
  newtoncoeff = MakeVector();
  lastcolumn = MakeVector();
  redundancy = MakeVector();
  
  numberofpoints = samplepoints[N];
  samplepoints[N] = word[N];           /* use first K points */
  
  NewtonTransform(word, samplepoints, lastcolumn, newtoncoeff);
  
  newtoncoeff[N] = 0;                  /* set the last N-K Newton coeff. to zero */
  samplepoints[N] = numberofpoints;    /* use all points to calculate redundancy samples */
  
  InverseNewtonTransform(newtoncoeff, samplepoints, lastcolumn, redundancy);
  
  
  /* copy redundancy symbols to codeword */
  memcpy(&word[word[N]], redundancy, (numberofpoints-word[N])*sizeof(SYMBOL));
  word[N] = numberofpoints;
  
  FreeVector(newtoncoeff);
  FreeVector(lastcolumn);
  FreeVector(redundancy);
}





/*********************************************************************************** 
 * IndexSort: Create index vector to sort array arr[0...n-1] in descending order.
 * 
 * Taken from: "Numerical Recipes in C", pp.339, routine "indexx.c". 
 *             Modified for indices starting with 0.
 *
 * input:    n
 *           arr: the type of arr is defined in RELIABILITYTYPE, see rs.h!
 *     
 * modifies: index: sorted index vector 
 *
 ***********************************************************************************/
#define SWAP(a,b) itemp=(a);(a)=(b);(b)=itemp;
#define M 7
#define NSTACK 50
void IndexSort(unsigned short n, RELIABILITYTYPE *arr, SYMBOL *indx)
{
    /*
	int i, indxt, ir=n-1, itemp, j, k, l=0;
	int jstack=0, *istack; 
	*/

    short i, indxt, ir=n-1, itemp, j, k, l=0;
	short jstack=0;
	short istack[NSTACK];

	RELIABILITYTYPE a;
  

	for (j=0;j<n;j++) indx[j]=j;
  
	for (;;){
	    if (ir-l < M) {
		    for (j=l+1; j<=ir; j++) {
			    indxt=indx[j];
				a=arr[indxt];
				for (i=j-1; i>=0 ; i--) {
				    if (arr[indx[i]] >= a) break;
					indx[i+1]=indx[i];
				}
				indx[i+1]=indxt;
			}
			if (jstack == 0) break;
			ir=istack[jstack--];
			l=istack[jstack--];
		} else {
	        k=(l+ir) >> 1;
			SWAP(indx[k],indx[l+1]);
			if (arr[indx[l+1]] < arr[indx[ir]]) {
			    SWAP(indx[l+1],indx[ir])
			}
			if (arr[indx[l]] < arr[indx[ir]]) {
			    SWAP(indx[l],indx[ir])
			}
			if (arr[indx[l+1]] < arr[indx[l]]) {
		        SWAP(indx[l+1],indx[l])
			}
			i=l+1;
			j=ir;
			indxt=indx[l];
			a=arr[indxt];
			for (;;) {
			    do i++; while (arr[indx[i]] > a);
				do j--; while (arr[indx[j]] < a);
				if (j<i) break;
				SWAP(indx[i],indx[j])
			}
			indx[l]=indx[j];
			indx[j]=indxt;
			jstack +=2;
			if (jstack >= NSTACK) fprintf(stderr,"NSTACK too small in indexx.");
			if (ir-i+1 >= j-l) {
			    istack[jstack]=ir;
				istack[jstack-1]=i;
				ir=j-1;
			} else {
			    istack[jstack]=j-1;
				istack[jstack-1]=l;
				l=i;
			}
		}
	}
	
	/*
	free(istack);
	*/

}




/*********************************************************************************** 
 * RSDecode: Step by step RS-Decoder
 *  
 * input:    *receivedunsorted:  Received word, nreceived symbols, hard decision.
 *           *receivedpointsunsorted:  Corresponding sample points, nreceived symbols.
 *           *reliability: reliability measure for each received symbol.
 * 
 *           K:     number of information symbols. 
 *           nmin:  number of symbols to use in the first step
 *           nstep: number of additional symbols used in each step
 *
 *           CANDIDATES: flag: 0 => output only codeword found with minimum number of 
 *                                  erasures (=last codeword) in *(codewordstep[0])
 *                             1 => output matrix of (nreceived-nmin)/nstep+1 candidates:
 *                                  codewordstep[ne] is candidate cw found with 
 *                                  nmin+ne*nstep symbols.
 *
 *           *samplepoints:  sample points of the original codeword.
 *                           The first K points are the information points.
 *
 * modifies: **codewordstep: estimated codewords for step by step decoding
 *                           codewordstep[ne][i] is the ith symbol of the codeword 
 *                           found with nmin+ne*nstep symbols. If no codeword was found with 
 *                           this number symbols, then codewordstep[ne][N] == 0.
 *                            
 *                            
 *                            
 *                            
 *
 * output:   reached distance for step by step decoding << What's that?
 *           >> Starting with nmin >= K symbols, in each step nstep >= 2 more symbols are used
 *              to find a possible codeword. When the reliability is a proper measure
 *              for the symbol error reliability, and the symbols are sorted with
 *              descending reliability, the first symbols are more likely
 *              to be error-free, and so it's possible to find the corrrect
 *              codeword using n symbols, but not with n' > n symbols (too many errors
 *              in symbols n+1 ... n'). When n is the maximum number of used symbols
 *              for which a codeword was found, the so called 'reached distance' is
 *              n-K+1. Interpretation: this is the correct codeword, if there have 
 *              been less than (n-K)/2 errors in the n symbols used so far.
 *              
 *
 ***********************************************************************************/
char RSDecode(SYMBOL *receivedunsorted, SYMBOL *receivedpointsunsorted, RELIABILITYTYPE *reliability, SYMBOL K, SYMBOL nmin, SYMBOL nstep, unsigned char CANDIDATES, SYMBOL *samplepoints, SYMBOL **codewordstep)
{
  extern unsigned short N;
  extern SYMBOL         **GFMULTAB;                 
  extern SYMBOL         **GFDIVTAB;                 
  
  unsigned short nreceived;                           /* number of received points */
  unsigned short n;                                   /* number of points used in actual step */
  
  register unsigned short i,j,J;
  
  SYMBOL *received, *receivedpoints;                  /* sorted versions of ...unsorted */
  SYMBOL *sortindex, *receivedpointsindex;
  
  SYMBOL *W1, *W2;                                    /* EDA registers */
  SYMBOL *P1, *P2, *Pstore;
  SYMBOL g1, g2;
  SYMBOL factor;
  
  SYMBOL *znew, *rnew, *zeros, *z, *r;                /* EDA shortcuts and stores */
  
  SYMBOL *lastcolumn;                                 /* store for interpolation */
  
  SYMBOL *errorlocator;
  SYMBOL *estimate, *estimatepoints, *Estimate;       /* estimated correct symbols, points & NC */ 
  
  SYMBOL *correct, *correctpoints;                    /* last found correct symbols & points */
  
  SYMBOL distance, lastreacheddistance = 0;           /* step by step criterion */
  
  
  
  /* Check parameters */
  
  nreceived = receivedunsorted[N];
  
  /*	
  if ( nreceived < K ){
    fprintf(stderr, "Error in RSDecode: less than K received symbols.");
	exit(ARGUMENT_ERROR);
  }
  if ( nmin < K ){
    fprintf(stderr, "Error in RSDecode: cannot start with less than K symbols.");
	exit(ARGUMENT_ERROR);
  }
  if ( nstep < 2 ){
    fprintf(stderr, "Error in RSDecode: nstep set to 2 symbols.");
	exit(ARGUMENT_ERROR);
  }
  */
  
  /* Allocate memory */
  
  received = MakeVector();
  receivedpoints = MakeVector();
  
  estimate = MakeVector();
  estimatepoints = MakeVector();
  
  correct = MakeVector();
  correctpoints = MakeVector();
  
  lastcolumn = MakeVector();
  
  W1 = MakeVector();
  W2 = MakeVector();
  P1 = MakeVector();
  P2 = MakeVector();
  
  znew = MakeVector();
  rnew = MakeVector(); 
  
  zeros = MakeVector();
  memset( zeros, 0, (nreceived-K)*sizeof(SYMBOL));
  
  
  /* Use memory efficiently, but be careful here !!!     */
  z = receivedpoints;            /* shortcut used in EDA */
  r = received;                  /* shortcut used in EDA */
  sortindex = znew;
  errorlocator = estimate;
  Estimate = rnew;
				

  /*---------------------------------------------------------------- 
   * Sort received vectors in descending order of reliability
   *----------------------------------------------------------------*/ 
  
  IndexSort(nreceived, reliability, sortindex);			
  for ( i=0 ; i<nreceived ; i++ ) { 
	received[i] = receivedunsorted[sortindex[i]];
	receivedpoints[i] = receivedpointsunsorted[sortindex[i]];
	/*		receivedpointsindex[receivedpoints[i]] = sortindex[i]; /* to find the received values for a certain point later */
  }
  received[N] = nreceived;
  receivedpoints[N] = nreceived;
  
#ifdef PRINTRESULTS
/*   PrintVector("rw", receivedunsorted); */
/*   PrintVector("rws", received); */
#endif  
	
  /* prepare for interpolation: store received points as interpolation points to estimatepoints */
  memcpy(estimatepoints, receivedpoints, nreceived*sizeof(SYMBOL));
  estimatepoints[N] = nreceived;
  

  /*---------------------------------------------------------------- 
   * Initialize EDA registers P1, P2, W1, W2, g1, g2 
   *--------------------------------------------------------------------*/ 
  
  NewtonTransform(received, z, NULL, P2);                    /* Newton coefficients */
  memmove(P2, &P2[K], (nreceived-K)*sizeof(SYMBOL));         /* Syndrome coefficients */
  P2[N] = nreceived-K;

  P1[0] = 1;
  memset(&P1[1], 0, (nreceived-K-1)*sizeof(SYMBOL));
  P1[N] = nreceived-K;
  
  W1[0] = 0;
  W1[N] = 1;
  
  W2[0] = 1;
  W2[N] = 1;
  
  g1 = 1;
  g2 = 0;
  
  /* sample points of Newton coefficients */
  memcpy(znew, &z[K], (nreceived-K)*sizeof(SYMBOL));
  znew[N] = nreceived-K;
  

  /*---------------------------------------------------------------- 
   * Decode step by step, use nstep symbols per step 
   *----------------------------------------------------------------*/ 
  
  for ( n=nmin, J=0 ; n<=nreceived; n+=nstep ) {

	/*-------------------------------------------------------------------------- 
	 * Find estimate CW using first n symbols, nstep additional symbols per step
	 *--------------------------------------------------------------------------*/ 

	if ( n==K ) {          /* first step: no error correction; happens only if nmin==K */

	  /* use first K symbols as correct symbols */
	  memcpy(estimatepoints, receivedpoints, K*sizeof(SYMBOL));
	  estimatepoints[N] = K;
	  memcpy(estimate, received, K*sizeof(SYMBOL));
	  estimate[N] = K;
	  
	  
	  /* prepare lastcolumn for interpolation */
	  NewtonTransform(estimate, estimatepoints, lastcolumn, NULL);
	  
	  distance = 1;

	} else {
			
	  /*================================================================
	   * Find errors with Euclid's Division Algorithm (EDA)
	   *================================================================*/
	  
	  /*---------------------------------------------------------------- 
	   * Perform next nstep EDA steps
	   *----------------------------------------------------------------*/ 
	  
	  for ( ; J<n-K ; J++ ){		
		
		if ( P2[J] ){

		  /* force P1 to 0 */
		  factor = GFdiv(P1[J], P2[J]);
		  
		  for ( i=nreceived-K-1 ; i>J ; i-- )
			P1[i] = P1[i] ^ GFmul( factor, P2[i] );
		  P1[J] = 0;
		  
		  
		  /* W1 */
		  if ( W1[N] >= W2[N] )
			for ( i=0 ; i<W2[N] ; i++ )
			  W1[i] = W1[i] ^ GFmul( factor, W2[i] );
		  else{					
			for ( i=0 ; i<W1[N] ; i++ )
			  W1[i] = W1[i] ^ GFmul( factor, W2[i] );
			for ( i=W1[N] ; i<W2[N] ; i++ )
			  W1[i] = GFmul( factor, W2[i] );
			W1[N] = i;
		  }
		  
		  
		  /* shift P2 */
		  for ( i=nreceived-K-1 ; i>J ; i-- )
			P2[i] = P2[i-1] ^ GFmul( (znew[i] ^ znew[J]), P2[i] );
		  P2[J] = 0;
		  

		  /* shift W2 */
		  W2[ W2[N] ] = W2[ W2[N]-1 ];
		  for ( i=W2[N]-1 ; i ; i-- )
			W2[i] = W2[i-1] ^ GFmul( (z[i] ^ znew[J]), W2[i] );
		  W2[0] = GFmul( (z[0] ^ znew[J]), W2[0] );
		  W2[N]++;
		  
		  g2++;

		  if ( g2 > g1 ){             /* exchange registers   */
			Pstore = P1;            /* => exchange pointers */
			P1 = P2;
			P2 = Pstore;
			
			Pstore = W1;
			W1 = W2;
			W2 = Pstore;
			
			factor = g1;
			g1 = g2;
			g2 = factor;
		  }
		  
		} else {
		  
		  /* shift P1 */
		  for ( i=nreceived-1 ; i>J ; i-- )
			P1[i] = P1[i-1] ^ GFmul( (znew[i] ^ znew[J]), P1[i] );
		  P1[J] = 0;
		  
		  /* shift W1 */
		  W1[ W1[N] ] = W1[ W1[N]-1 ];
		  for ( i=W1[N]-1 ; i ; i-- )
			W1[i] = W1[i-1] ^ GFmul( (z[i] ^ znew[J]), W1[i] );
		  W1[0] = GFmul( (z[0] ^ znew[J]), W1[0] );
		  W1[N]++;
		  
		  g1++;
		}
	  }		
	  

	  /*-------------------------------------------------------------------- 
	   * Now W2 are the Newton coefficients of the error locator polynomial. 
	   * Calculate zeros of W2, these are the possibly erroneous symbols.
	   *--------------------------------------------------------------------*/ 
	  z[N] = n;
	  InverseNewtonTransform(W2, z, NULL, errorlocator);
	  z[N] = nreceived;
	  

	  /*---------------------------------------------------------------- 
	   * Sort the receivedpoints:
	   * + The non-zeros of errorlocator to the first elements of 
	   *   estimatepoints and the samples to estimate. 
	   * + The zeros to the last elements of estimatepoints.
	   *   Here they can be used for the interpolation!
	   *
	   * Check if the correct points after this step are the same points
	   * as after the last step. In this case don't interpolate again!
	   *----------------------------------------------------------------*/ 
	  
	  for ( i=0, j=0 ; i<n; i++ )
		if ( errorlocator[i] ){           /* Take care: identical vector, but always j < i !!! */
		  estimatepoints[j] = z[i];
		  estimate[j++] = r[i];             
		} else 
		  estimatepoints[n-1-(i-j)] = z[i];   /* Interpolate at these points later */
	  estimatepoints[N] = j;
	  estimate[N] = j;
	  
	  /* Now, the estimated correct points are in estimatepoints[0...K-1] & estimate,
	   * the estimated erroneous points are in estimatepoints[K...nreceived-1] */
	  


	  /*============================================================================
	   * Check if correct symbols are really a codeword 
	   *==========================================================================*/
	  if ( estimatepoints[N] >= (n+K)/2 ) {   /* There are enough correct symbols. */
		
		/* Calculate syndrome with possibly correct symbols, if it is all 0 => ok */
		Estimate[N] = 0;
		lastcolumn[N] = 0;
		NewtonTransform(estimate, estimatepoints, lastcolumn, Estimate);
		
		/* lastcolumn can be used to interpolate later!! */
		
		distance = n-K+1;
		for ( i=Estimate[N]-1 ; i>=K ; i--)    /* Check syndrome: */
		  if ( Estimate[i] ) {               /* syndrome != 0 */
			distance = 0;                  /* this is not a codeword */
			break;
		  }

	  } else
		distance = 0;                          /* less than (n+K)/2 correct points: this is not a codeword */
	  
	}

	
	/*============================================================================
	 * Interpolate the codeword from the estimated correct points
	 *==========================================================================*/

	if ( distance ) {

	  lastreacheddistance = distance;
	  
	  /* Interpolate only, if the new estimate will result in a new codeword different
	   * from the one calculated in a step before, stored in correctpoints. */
	  
	  if ( memcmp(estimatepoints, correctpoints, K*sizeof(SYMBOL)) ) {

		/*---------------------------------------------------------------- 
		 * Interpolation of the estimated codeword at the received points,
		 * using the estimated correct points (lastcolumn)
		 *----------------------------------------------------------------*/ 
		zeros[N] = nreceived-estimatepoints[N];
		estimatepoints[N] = nreceived;
		
		InverseNewtonTransform(zeros, estimatepoints, lastcolumn, correct);
				
		/* correct codeword is estimated correct symbols + interpolated symbols */
		memmove(&correct[nreceived-correct[N]], correct, correct[N]*sizeof(SYMBOL));
		memcpy(correct, estimate, (nreceived-correct[N])*sizeof(SYMBOL));
		correct[N] = nreceived;

		/* store correct points for later re-sorting, too */
		memcpy(correctpoints, estimatepoints, nreceived*sizeof(SYMBOL));
		correctpoints[N] = nreceived;
			  
	  } 
	  
	  if ( CANDIDATES ) {
		/*----------------------------------------------------------------------- 
		 * Interpolation of the estimated codeword at the original samplepoints:
		 * re-sort the estimated correct points in order of samplepoints.
		 *-----------------------------------------------------------------------*/ 
		NewtonInterpolation(correct, correctpoints, K, samplepoints, codewordstep[(n-nmin)/nstep], NULL);
	  }

	} else if ( CANDIDATES )

	  codewordstep[(n-nmin)/nstep][N] = 0;

  
  }  /*  for ( n=nmin, J=0 ; n<=nreceived; n+=nstep )  */

		
  FreeVector(received);
  FreeVector(receivedpoints);
  FreeVector(estimatepoints);
  FreeVector(estimate);
  FreeVector(lastcolumn);
  FreeVector(W1);
  FreeVector(W2);
  FreeVector(P1);
  FreeVector(P2);
  FreeVector(znew);
  FreeVector(rnew);
  FreeVector(zeros);
  


  if ( !CANDIDATES ) {
	  
	/*----------------------------------------------------------------------- 
	 * Interpolation of the estimated codewords at the original samplepoints:
	 * re-sort the estimated correct points in order of samplepoints.
	 *-----------------------------------------------------------------------*/ 
	if ( lastreacheddistance ) 
	  NewtonInterpolation(correct, correctpoints, K, samplepoints, codewordstep[0], NULL);
	else
	  codewordstep[0][N] = 0;

  }


  FreeVector(correct);
  FreeVector(correctpoints);
  
  return lastreacheddistance;
}	



/*********************************************************************************** 
 * RSCalculate: Calculate whether an RS-Decoder can find the correct codeword by
 *              counting the errors
 *  
 * input:    *codeword: Correct word 
 *           *received:  Received word, nreceived symbols, hard decision.
 *           *reliability: reliability measure for each received symbol.
 * 
 *           K:     number of information symbols. 
 *           nmin:  number of symbols to use in the first step
 *
 * modifies: *codewordGMD: codeword for optimum GMD decoding: the GMD criterion does
 *                         not make any errors.
 *                            
 *
 * output: --   
 *
 ***********************************************************************************/
void RSCalculate(SYMBOL *codeword, SYMBOL *received, RELIABILITYTYPE *reliability, SYMBOL K, SYMBOL nmin, SYMBOL *codewordGMD)
{
  extern unsigned short N;
  
  unsigned short nreceived;                        /* number of received points */
  register unsigned short n, nstart, nstop;        /* number of points used in actual step */
  
  SYMBOL *sortindex;

  SYMBOL errorcounter;

  
  nreceived = received[N];
  
  sortindex = MakeVector();

  /*---------------------------------------------------------------- 
   * Sort received vectors in descending order of reliability
   *----------------------------------------------------------------*/ 
  
  IndexSort(nreceived, reliability, sortindex);			
  
#ifdef PRINTRESULTS
  printf(" rws");
  for ( n=0 ; n<nreceived ; n++ )
	printf("%4i",received[sortindex[n]]);
  printf("\n");
#endif
	
  /*---------------------------------------------------------------- 
   * Count symbol errors in sorted received word
   *----------------------------------------------------------------*/ 

  errorcounter = 0;
  n = 0;
  nstart = 0;
  nstop = nmin;

  codewordGMD[N] = 0;

  while ( nstop<=nreceived ) {

	for ( n=nstart ; n<nstop ; n++ )
	  errorcounter += (codeword[sortindex[n]] != received[sortindex[n]]); 
	
	if ( errorcounter <= (n-K)/2 ) {   /* codeword can be found */
	  
	  memcpy(codewordGMD, codeword, nreceived);
	  codewordGMD[N] = nreceived;
	  break;
	}
	
	nstart = nstop;
	nstop += 2;
	
  } 

  /* "Optimales GMD" */
  
/*   if ( errorcounter <= (nreceived-K) ) {   /* codeword can be found */ 
	  
/* 	memcpy(codewordGMD, codeword, nreceived); */
/* 	codewordGMD[N] = nreceived; */
/*   } */



  FreeVector(sortindex);

  return;

}
	





/*********************************************************************************** 
 * InterleaverDesign: Setup interleaver for unequal error protection and internal
 *                    channel estimation.
 * 
 * input:    bitspersymbol
 *           numberofsegments
 *           numberoflevels
 *           infosymbolsoflevel
 *           columnsoflevel
 *           n
 * 
 * modifies: interleaver
 *
 * output:   ---
 *
 ***********************************************************************************/
void InterleaverDesign(unsigned char bitspersymbol, unsigned char nrofsegments, unsigned char nroflevels, SYMBOL *infosymbolsoflevel, unsigned short *columnsoflevel, unsigned short n, struct interleaver_struct *interleaver)
{
  register unsigned short i;


#ifdef PRINTRESULTS
/*   fprintf(stderr, "InterleaverDesign: %i segments with %i levels.\n", nrofsegments, nroflevels); */
/*   for ( i=0 ; i<nroflevels ; i++ ) */
/* 	   fprintf(stderr, "level %i: %i columns with k=%i.\n", i, columnsoflevel[i], infosymbolsoflevel[i]); */
#endif 

  /* set interleaver dimensions */

  if ( nrofsegments==0 ) {      /* no interleaver */

	interleaver->nrofsegments = 1;
	interleaver->nroflevels = 1;
	interleaver->nrofrows = n+1; 

  	interleaver->koflevel[0] = infosymbolsoflevel[0];
	interleaver->columnsoflevel[0] = 1;
	interleaver->nrofcolumns = 1;
	interleaver->totalinfosymbols = interleaver->koflevel[0];

	interleaver->bytesperpacket = 1;
	
  } else {

	interleaver->nrofsegments = nrofsegments;
	interleaver->nroflevels = nroflevels;
	interleaver->nrofrows = n+1; 

	interleaver->nrofcolumns = 0;
	interleaver->totalinfosymbols = 0;
	interleaver->bytesperpacket = 0;

	for ( i=0 ; i<nroflevels ; i++ ) {
	  interleaver->columnsoflevel[i] = columnsoflevel[i];
	  interleaver->koflevel[i] = infosymbolsoflevel[i];

	  interleaver->nrofcolumns += columnsoflevel[i];
	  interleaver->totalinfosymbols += infosymbolsoflevel[i] * columnsoflevel[i];
	}
	interleaver->nrofcolumns *= interleaver->nrofsegments;
	interleaver->totalinfosymbols *= interleaver->nrofsegments;
	
	interleaver->bytesperpacket =  (unsigned short) ceil( (float) interleaver->nrofcolumns * bitspersymbol / 8);

  }

  return;
}



/*********************************************************************************** 
 * InterleaverInit: Allocate memory for the interleaver designed by InterleaverDesign
 * 
 * input:    interleaver
 *
 * modifies: interleaver
 *
 * returns:  ---
 *
 ***********************************************************************************/
void InterleaverInit(struct interleaver_struct *interleaver)
{
  register unsigned short     row;

#ifdef PRINTRESULTS
/*    fprintf(stderr, "InterleaverInit: %i rows x %i columns.\n", interleaver->nrofrows, interleaver->nrofcolumns);  */
#endif 

  /* allocate memory for interleaver->nrofrows x interleaver->nrofcolumns data SYMBOLs */
  if ( (interleaver->data = (SYMBOL *) malloc( interleaver->nrofrows * interleaver->nrofcolumns * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }


  /* allocate memory for interleaver->nrofrows elements rowindex vector */
  if ( (interleaver->ri = (unsigned long *) malloc( interleaver->nrofrows * sizeof(unsigned long) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }
  /* set rowindex vector */
  for ( row=0 ; row<interleaver->nrofrows ; row++ )
	interleaver->ri[row] = (unsigned long) row * interleaver->nrofcolumns;
  

  /* allocate memory for interleaver->nrofrows elements samplepoints vector */
  if ( (interleaver->samplepoints = (SYMBOL *) malloc( interleaver->nrofrows * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }


  /* allocate memory for interleaver->nrofrows elements received packet numbers vector */
  if ( (interleaver->pnr = (SYMBOL *) malloc( interleaver->nrofrows * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }


  /* allocate memory for interleaver->nrofrows elements lost packet numbers vector */
  if ( (interleaver->pnl = (SYMBOL *) malloc( interleaver->nrofrows * sizeof(SYMBOL) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }


  /* allocate memory for interleaver->nrofcolumns elements reliability vector */
  if ( (interleaver->reliability = (RELIABILITYTYPE *) malloc( interleaver->nrofrows * sizeof(RELIABILITYTYPE) )) == NULL ) {
	fprintf(stderr,"Error in InterleaverInit: malloc returns NULL.\n");
	exit(MEMORY_ALLOCATION_ERROR);
  }

  interleaver->nrofpackets = 0;
  
  return;
}


/*********************************************************************************** 
 * InterleaverFree
 * 
 * input:    interleaver
 *
 * modifies: ---
 *
 * output:   ---
 *
 ***********************************************************************************/
void InterleaverFree(struct interleaver_struct interleaver)
{
	free(interleaver.data);
	free(interleaver.samplepoints);
	free(interleaver.pnr);
	free(interleaver.pnl);
	free(interleaver.reliability);

	return;
}








