#include <stdlib.h>
#include <unistd.h>	/* SEEK_SET on sunos4 */
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <defs.h>
#include "structs.h"
#include "Util.h"


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AllocPicture -- Allocate memory for a Picture structure
 *
 * Author:              K.S.
 *
 * Created:             5-Nov-97
 *
 * Purpose:             Allocates memory for a Picture structure, including 
 *                      memory for the actual image pixels. 
 * 
 * Arguments in:        int w              image width
 *                      int h              image height
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       Picture *          The allocated Picture structure
 *                                         if allocation possible;
 *                                         otherwise NULL.
 *
 * Example              pict = AllocPicture(w, h);
 *
 * Side effects:        -
 *
 * Description:         The memory for the three components y, u and v is
 *                      is allocated together in one piece.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
Picture *AllocPicture(int w, int h)
 /**********************************************************CommentEnd********/
{
  int     ysize = w * h;
  int     csize = ysize >> 2;   /* Color components u,v are quarter the size */
  int     size = ysize + 2 * csize;
  Picture *p = (Picture *)malloc(sizeof(Picture));


  p->w = w;
  p->h = h;
  p->ws = w;
  p->hs = h;
  p->y = (Byte *)malloc(size * sizeof(Byte));
  p->u = p->y + ysize;
  p->v = p->u + csize;
  return(p);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ClonePicture -- Allocate memory for a Picture acc. to the model picture
 *
 * Author:              K.S.
 *
 * Created:             5-Nov-97
 *
 * Purpose:             Allocates memory for a Picture structure.
 * 
 * Arguments in:        Picture *pict     model picture
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       Picture *        The allocated Picture structure.
 *
 * Example:             pict2 = ClonePicture(pict1);
 *
 * Side effects:        -
 *
 * Description:         Allocates memory for the Picture if possible, otherwise
 *                      NULL is returned.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
Picture *ClonePicture(Picture *pict)
 /**********************************************************CommentEnd********/
{
  Picture *p = AllocPicture(pict->w, pict->h);
  p->ws = pict->ws;
  p->hs = pict->hs;
  return(p);
}
 

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- FreePicture -- Free up all the memory associated with a Picture structure
 *
 * Author:              K.S.
 *
 * Created:             5-Nov-97
 *
 * Purpose:             Frees up all the memory associated with a
 *                      Picture structure
 * 
 * Arguments in:        -
 *
 * Arguments in/out:    Picture  *pict      Picture structure to be freed.
 *
 * Arguments out:       -
 *
 * Return values:       -
 *
 * Example:             FreePicture(pict);
 *
 * Side effects:        -
 *
 * Description:         -
 *
 * See also:            AllocPicture
 *
 * Modified:            -
 *
 *****************************************************************************/
void FreePicture(Picture *pict)
 /**********************************************************CommentEnd********/
{
  free(pict->y);

  free(pict);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ResetPicture -- Reset a picture to black
 *
 * Author:              K.S.
 *
 * Created:             29-May-97
 *
 * Purpose:             Resets a Picture structure,
 *                      y: 0, u,v: 128. 
 * 
 * Arguments in:        -
 *
 * Arguments in/out:    Picture *pict       
 *      
 * Arguments out:       -
 *
 * Return values:       -
 *
 * Example              ResetPicture(pict);
 *
 * Side effects:        -
 *
 * Description:         -
 *
 * See also:            -
 *      
 * Modified:            -
 *
 *****************************************************************************/
void ResetPicture(Picture *p)
 /**********************************************************CommentEnd********/
{
  int     ysize = p->w * p->h;
  int     csize = ysize >> 2;   /* Color components u,v are quarter the size */


  memset(p->y, 128, ysize);
  memset(p->u, 128, csize);
  memset(p->v, 128, csize);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- AllocResetPicture -- Allocate memory for a Picture structure and reset
 *
 * Author:              K.S.
 *
 * Created:             28-May-97
 *
 * Purpose:             Allocates memory for a Picture structure, including 
 *                      memory for the actual image pixels.
 *                      Reset the picture (y: 0; u,v: 128)
 * 
 * Arguments in:        int w              image width
 *                      int h              image height
 *
 * Arguments in/out:    -
 *      
 * Arguments out:       -
 *
 * Return values:       Picture *          The allocated Picture structure
 *                                         if allocation possible;
 *                                         otherwise NULL.
 *
 * Example              pict = AllocResetPicture(w, h);
 *
 * Side effects:        -
 *
 * Description:         The memory for the three components y, u and v is
 *                      is allocated together in one piece.
 *
 * See also:            -
 *      
 * Modified:            -
 *
 *****************************************************************************/
Picture *AllocResetPicture(int w, int h)
 /**********************************************************CommentEnd********/
{
  Picture *p = AllocPicture(w,h);
  ResetPicture(p);
  return(p);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CloneResetPicture -- Allocate memory for a Picture acc. to the model picture and reset it
 *
 * Author:              K.S.
 *
 * Created:             5-Nov-97
 *
 * Purpose:             Allocates memory for a Picture structure.
 *                      Sets y: 0; u,v: 128
 * 
 * Arguments in:        Picture *pict     model picture
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       Picture *        The allocated Picture structure.
 *
 * Example:             pict2 = ClonePicture(pict1);
 *
 * Side effects:        -
 *
 * Description:         Allocates memory for the Picture if possible, otherwise
 *                      NULL is returned.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
Picture *CloneResetPicture(Picture *pict)
 /**********************************************************CommentEnd********/
{
  Picture *p = AllocResetPicture(pict->w, pict->h);
  p->ws = pict->ws;
  p->hs = pict->hs;
  return(p);
}
 

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- SwapPictures -- Swaps two pictures
 *
 * Author:              K.S.
 *
 * Created:             13-Jan-98
 *
 * Purpose:             Swaps two Picture structures, i.e. the two pointers
 *                      are exchanged
 * 
 * Arguments in:        -
 *
 * Arguments in/out:    Picture  **pict1_p      pointer to first Picture.
 *                      Picture  **pict2_p      pointer to second Picture.
 *
 * Arguments out:       -
 *
 * Return values:       -
 *
 * Example:             SwapPictures(&pict1, &pict2);
 *
 * Side effects:        -
 *
 * Description:         -
 *
 * See also:            CopyPicture
 *
 * Modified:            -
 *
 *****************************************************************************/
void SwapPictures(Picture **pict1_p, Picture **pict2_p)
 /**********************************************************CommentEnd********/
{
  Picture *tmp;

  tmp = *pict1_p;
  *pict1_p = *pict2_p;
  *pict2_p = tmp;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- LinkPicture -- Link one picture to another
 *
 * Author:              K.S.
 *
 * Created:             13-Jan-98
 *
 * Purpose:             Links one picture to another Picture structure
 * 
 * Arguments in:        Picture  *pictIn      Picture structure.
 *
 * Arguments in/out:    Picture  *pictOut     Picture structure in which
 *                                            'pictIn' is linked.
 *
 * Arguments out:       -
 *
 * Return values:       -
 *
 * Example:             LinkPicture(pictIn, pictOut);
 *
 * Side effects:        -
 *
 * Description:         The linking is done by setting
 *                              pictOut->w = pictIn->w
 *                              pictOut->h = pictIn->h
 *                              pictOut->ws = pictIn->ws
 *                              pictOut->hs = pictIn->hs
 *                              pictOut->y = pictIn->y
 *                              pictOut->u = pictIn->u
 *                              pictOut->v = pictIn->v
 *
 * See also:            CopyPicture, SwapPictures
 *
 * Modified:            -
 *
 *****************************************************************************/
void LinkPicture(Picture *pictIn, Picture *pictOut)
 /**********************************************************CommentEnd********/
{
  pictOut->w = pictIn->w;
  pictOut->h = pictIn->h;
  pictOut->ws = pictIn->ws;
  pictOut->hs = pictIn->hs;
  pictOut->y = pictIn->y;
  pictOut->u = pictIn->u;
  pictOut->v = pictIn->v;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CopyPicture -- Copy a picture.
 *
 * Author:              K.S.
 *
 * Created:             7-Jan-98
 *
 * Purpose:             Copies al components of a Picture structure.
 * 
 * Arguments in:        Picture *pictIn     input picture
 *
 * Arguments in/out:    Picture *pictOut    output picture
 *
 * Arguments out:       -
 *
 * Return values:       -
 *
 * Example:             -
 *
 * Side effects:        -
 *
 * Description:         The pictures must have the same size.
 *
 * See also:            LinkPicture, SwapPictures
 *
 * Modified:            -
 *
 *****************************************************************************/
void CopyPicture(Picture *pictIn, Picture *pictOut)
 /**********************************************************CommentEnd********/
{
  int size = pictIn->w * pictIn->h;

  size += size / 2;
  memcpy(pictOut->y, pictIn->y, size * sizeof(Byte));
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- ReadPicture -- Read picture from file
 *
 * Author:           K.S.
 *
 * Created:          5-Nov-97
 *
 * Purpose:          Reads picture from file.
 *                   The file is specified by name and index.
 *                   The number of samples to be read is specified in picture.
 * 
 * Arguments in:     char          *name         filename of picture file.
 *                   int           fileType      SEP_RAW_TYPE or CAT_SEQ_TYPE.
 *                   int           index         index of the image file.
 *                                               If NO_INDEX, no index is
 *                                               appended.
 *
 * Arguments in/out: Picture       *pict  picture to receive the data from file
 *
 * Arguments out:    -
 *
 * Return values:    0  if error
 *                   1  if ok.
 *
 * Side effects:     -
 *
 * Example(s):       ReadPicture("name", SEP_RAW_TYPE, 2, pict);
 *                   would read data from the image files
 *                              "name002.y", "name002.u", "name002.v"
 *
 *                   ReadImage("name", CAT_SEQ_TYPE, 2, pict);
 *                   would read data from the image file
 *                              "name"
 *                   at offset of 2*25344*1.5 Bytes.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
int ReadPicture(char *name, int fileType, int index, Picture *pict)
 /**********************************************************CommentEnd********/
{
  char yname[200], uname[200], vname[200];
  int  returnValue = 1;
  int  size = pict->w * pict->h;
  int  offset;
  FILE *fp;


  switch (fileType) {
  case SEP_RAW_TYPE:
    if (index == NO_INDEX) {
      sprintf(yname, "%s.y", name);
      sprintf(uname, "%s.u", name);
      sprintf(vname, "%s.v", name);
    } else {
      sprintf(yname, "%s%03d.y", name, index);
      sprintf(uname, "%s%03d.u", name, index);
      sprintf(vname, "%s%03d.v", name, index);
    }
    /* Y */
    if ((fp = fopen(yname, "r")) == NULL)
      return(0);
    if (fread(pict->y, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    fclose(fp);
    size = size >> 2;   /* / 4 */
    /* U */
    if ((fp = fopen(uname, "r")) == NULL)
      return(0);
    if (fread(pict->u, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    fclose(fp);
    /* V */
    if ((fp = fopen(vname, "r")) == NULL)
      return(0);
    if (fread(pict->v, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    fclose(fp);
    break;
  case CAT_SEQ_TYPE:
    if ((fp = fopen(name, "r")) == NULL)
      return(0);
    /* Skip offset data */
    offset = size * 3 / 2 * index;
    if (fseek(fp, (long)offset, SEEK_SET) != 0) {
      return(0);
    }
    /* Y */
    if (fread(pict->y, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    size = size >> 2;   /* / 4 */
    /* U */
    if (fread(pict->u, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    /* V */
    if (fread(pict->v, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    fclose(fp);
    break;
  default:
    printf("File type >>%d<< not supported\n", fileType);
    return(0);
  }  

  return(returnValue);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- WritePicture -- Write picture to file
 *
 * Author:           K.S.
 *
 * Created:          9-Jan-97
 *
 * Purpose:          Writes picture to file.
 *                   The file is specified by name and index.
 *                   The number of samples to be read is specified in picture.
 * 
 * Arguments in:     char          *name         filename of picture file.
 *                   int           fileType      SEP_RAW_TYPE or CAT_SEQ_TYPE.
 *                   int           index         index of the image file.
 *                                               If NO_INDEX (= -1),
 *                                               no index is appended.
 *                   Picture       *pict         picture to write to file
 *
 * Arguments in/out: -
 *
 * Arguments out:    -
 *
 * Return values:    0  if error
 *                   1  if ok.
 *
 * Side effects:     -
 *
 * Example(s):       WritePicture("name", SEP_RAW_TYPE, 2, pict);
 *                   would write data to the image files
 *                              "name002.y", "name002.u", "name002.v"
 *
 *                   WriteImage("name", CAT_SEQ_TYPE, 2, pict);
 *                   would write data to the image file
 *                              "name"
 *                   The picture is written at a offset of 2*25344*1.5 Bytes.
 *                   If the file is shorter, the data is just appended.
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
int WritePicture(char *name, int fileType, int index, Picture *pict)
 /**********************************************************CommentEnd********/
{
  char yname[200], uname[200], vname[200];
  int  returnValue = 1;
  int  size = pict->w * pict->h;
  int  offset;
  FILE *fp;


  switch (fileType) {
  case SEP_RAW_TYPE:
    if (index == NO_INDEX) {
      sprintf(yname, "%s.y", name);
      sprintf(uname, "%s.u", name);
      sprintf(vname, "%s.v", name);
    } else {
      sprintf(yname, "%s%03d.y", name, index);
      sprintf(uname, "%s%03d.u", name, index);
      sprintf(vname, "%s%03d.v", name, index);
    }
    /* Y */
    if ((fp = fopen(yname, "w")) == NULL)
      return(0);
    if (fwrite(pict->y, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    size = size >> 2;   /* / 4 */
    /* U */
    if ((fp = fopen(uname, "w")) == NULL)
      return(0);
    if (fwrite(pict->u, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    /* V */
    if ((fp = fopen(vname, "w")) == NULL)
      return(0);
    if (fwrite(pict->v, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    break;
  case CAT_SEQ_TYPE:
    if ((fp = fopen(name, "a+")) == NULL)
      return(0);
    /* Skip offset data */
    if (index >= 0) {
      offset = size * 3 / 2 * index;
      if (fseek(fp, (long)offset, SEEK_SET) != 0)
	fseek(fp, 0, SEEK_END);
    }
    /* Y */
    if (fwrite(pict->y, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    size = size >> 2;   /* / 4 */
    /* U */
    if (fwrite(pict->u, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    /* V */
    if (fwrite(pict->v, sizeof(Byte), size, fp) != size)
      returnValue = 0;
    break;
  default:
    printf("File type >>%d<< not supported\n", fileType);
    return(0);
  }  

  if (fclose(fp) == EOF)
    return(0);
 
  return(returnValue);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PSNRPictures -- Measure PSNR between two pictures (only luminance).
 *
 * Author:              K.S.
 *
 * Created:             13-Nov-97
 *
 * Purpose:             Measures the PSNR between the luminance (Y) of two
 *                      Picture structures.
 * 
 * Arguments in:        Picture *pict1     input picture 1
 *                      Picture *pict2     input picture 2
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       float  psnr        PSNR between pict1->y and pict2->y.
 *
 * Example:             psnr = PSNRPictures(pict1, pict2);
 *
 * Side effects:        -
 *
 * Description:         The pictures must have the same size.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
float PSNRPictures(Picture *pict1, Picture *pict2)
 /**********************************************************CommentEnd********/
{
  const int size = pict1->w * pict1->h;
  unsigned long sum = 0;
  int tmp;
  int i;
  Byte *p1 = pict1->y;
  Byte *p2 = pict2->y;


  for (i = 0; i < size; i++) {
    tmp = (int)(*p1++) - (int)(*p2++);
    sum += SQR(tmp);
  }

  if (sum == 0)
    return(0.0);

  tmp = 255 * 255;

  return(10.0 * log10((float)size * (float)tmp / (float)sum));
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PSNRSubPictures -- Measure PSNR between two sub pictures.
 *
 * Author:              Uwe Horn
 *
 * Created:             26-Jun-98
 *
 * Purpose:             Measures the PSNR between subimages of two
 *                      Picture components.
 * 
 * Arguments in:        Byte *p1     input picture 1
 *                      Byte *p2     input picture 2
 *                      int  w       width of subimage
 *                      int  h       height of subimage
 *                      int  iw      width of full image component (pict->w).
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       float  psnr        PSNR between p1 and p2.
 *
 * Example:             psnr = PSNRSubPictures(pict1->u, pict2->u, 8, 8,
 *                                             pict1->w/2);
 *
 * Side effects:        -
 *
 * Description:         The pictures must have the same size.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
float PSNRSubPictures(Byte *p1, Byte *p2, int w, int h, int iw)
 /**********************************************************CommentEnd********/
{
  unsigned long sum = 0;
  int tmp;
  int i,j,index;

  for (j = 0; j < h; j++) {

    for (i = 0; i < w; i++) {

      index = (j*iw)+i;
    
      tmp = (int)(p1[index]) - (int)(p2[index]);
      sum += SQR(tmp);
    }
  }

  if (sum == 0)
    return(0.0);

  tmp = 255 * 255;

  return(10.0 * log10((float)(w*h) * (float)tmp / (float)sum));
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- FindFirstDifferenceBetweenPictures -- Find the first difference
 *
 * Author:              K.S.
 *
 * Created:             9-Jan-98
 *
 * Purpose:             Finds the first pel which is different between the two
 *                      Picture structures (Y, U, V).
 *                      If no differing pel is found, -1 is returned.
 * 
 * Arguments in:        Picture *pict1     input picture 1
 *                      Picture *pict2     input picture 2
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       int     index      index of the first differing pel.
 *                                         For U size(Y) is added.
 *                                         For V size(Y)+size(U) is added.
 *
 * Example:             i = FindFirstDifferenceBetweenPictures(pict1,pict2);
 *
 * Side effects:        -
 *
 * Description:         The pictures must have the same size.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
int FindFirstDifferenceBetweenPictures(Picture *pict1, Picture *pict2)
 /**********************************************************CommentEnd********/
{
  int size = pict1->w * pict1->h;
  int i, offset = 0;
  Byte *p1 = pict1->y;
  Byte *p2 = pict2->y;


  /* Y */
  for (i = 0; i < size; i++) {
    if ((*p1++) != (*p2++))
      return(i);
  }
  offset = size;
  size /= 4;
  /* U */
  p1 = pict1->u;
  p2 = pict2->u;
  for (i = 0; i < size; i++) {
    if ((*p1++) != (*p2++))
      return(offset + i);
  }
  /* V */
  offset += size;
  p1 = pict1->v;
  p2 = pict2->v;
  for (i = 0; i < size; i++) {
    if ((*p1++) != (*p2++))
      return(offset + i);
  }

  /* Found no differing pel */
  return(-1);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- FindFirstDifferenceBetweenSubPictures -- Find the first difference
 *
 * Author:              K.S.
 *
 * Created:             8-Jul-98
 *
 * Purpose:             Finds the first pel which is different between the two
 *                      Picture structures (Y, U, V).
 *                      Only the sub images (ws,hs) are compared.
 *                      If no differing pel is found, -1 is returned.
 * 
 * Arguments in:        Picture *pict1    input picture 1
 *                      Picture *pict2    input picture 2
 *
 * Arguments in/out:    -
 *
 * Arguments out:       -
 *
 * Return values:       int     index     index of the first differing pel.
 *                                        For U sizeSub(Y) is added.
 *                                        For V sizeSub(Y)+sizeSub(U) is added.
 *
 * Example:             i = FindFirstDifferenceBetweenSubPictures(pict1,pict2);
 *
 * Side effects:        -
 *
 * Description:         The pictures and sub pictures must have the same size.
 *
 * See also:            -
 *
 * Modified:            -
 *
 *****************************************************************************/
int FindFirstDifferenceBetweenSubPictures(Picture *pict1, Picture *pict2)
 /**********************************************************CommentEnd********/
{
  int skip = pict1->w - pict1->ws;
  int x, y;
  int i, offset = pict1->w * pict1->h;
  Byte *p1 = pict1->y;
  Byte *p2 = pict2->y;


  /* Y */
  for (i = 0, y = pict1->hs; y > 0; y--) {
    for (x = pict1->ws; x > 0; x--, i++) {
      if ((*p1++) != (*p2++))
	return(i);
    }
    p1 += skip;
    p2 += skip;
    i += skip;
  }
  skip /= 2;
  /* U */
  i = offset;
  p1 = pict1->u;
  p2 = pict2->u;
  for (y = pict1->hs / 2; y > 0; y--) {
    for (x = pict1->ws / 2; x > 0; x--, i++) {
      if ((*p1++) != (*p2++))
	return(i);
    }
    p1 += skip;
    p2 += skip;
    i += skip;
  }
  /* V */
  i = offset + offset / 2;
  p1 = pict1->v;
  p2 = pict2->v;
  for (y = pict1->hs / 2; y > 0; y--) {
    for (x = pict1->ws / 2; x > 0; x--, i++) {
      if ((*p1++) != (*p2++))
	return(i);
    }
    p1 += skip;
    p2 += skip;
    i += skip;
  }

  /* Found no differing pel */
  return(-1);
}
