#include <assert.h>

#include "defs.h"
#include "structs.h"


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- SampleUpMVField -- Sample up a motion vector field
 *
 * Author:           K.S.
 *
 * Created:          1-Sep-97
 *
 * Purpose:          Upsampling of a motion vector field to the same or a
 *                   higher spatial resolution.
 *                   The temporal resolution of the two MVFields have to
 *                   be specified by the input arguments.
 *
 * Arguments in:     MVField  in          input motion vector field
 *                   int      tempDistIn  temporal distance for input field.
 *                   int      tempDistOut temporal distance for output field.
 *
 * Arguments in/out: -
 *
 * Arguments out:    MVField  out         output motion vector field.
 *                                        The dimension must be the same
 *                                        or bigger than the input MVField.
 *                                        The proportion must be the same.
 *
 * Return values:    -
 *
 * Example:          SampleUpMVField(mvIn, 2, mvOut, 1);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
void SampleUpMVField(MVField in, int tempDistIn,
		     MVField out, int tempDistOut)
/***********************************************************CommentEnd********/
{
  int x, y, xx, yy, ii;
  int s = out.w / in.w;


  if ((out.w % in.w == 0) && (out.h % in.h == 0)) {
    assert((out.h / in.h) == s);

    for (y = 0; y < in.h; y++)
      for (x = 0; x < in.w; x++) {
	ii = x + y * in.w;
	for (yy = 0; yy < s; yy++)
	  for (xx = 0; xx < s; xx++) {
	    out.mode[x * s + xx + (y * s + yy) * out.w] = in.mode[ii];
	    out.mx[x * s + xx + (y * s + yy) * out.w] =
	      in.mx[ii] * s * tempDistOut / tempDistIn;
	    out.my[x * s + xx + (y * s + yy) * out.w] =
	      in.my[ii] * s * tempDistOut / tempDistIn;
	  }
      }
  } else {
    int sw = (out.w + in.w - 1) / in.w;
    int sh = (out.h + in.h - 1) / in.h;

    for (y = 0; y < out.h; y++) {
      for (x = 0; x < out.w; x++) {
	ii = x + y * out.w;
	out.mode[ii] = in.mode[x/sw+(y/sh)*in.h];
	out.mx[ii] = in.mx[x/sw+(y/sh)*in.h] * sw * tempDistOut/tempDistIn;
	out.my[ii] = in.my[x/sw+(y/sh)*in.h] * sh * tempDistOut/tempDistIn;
      }
    }
  }
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- MV2MVField -- Copy or sample up a motion vector field created by TMN
 *
 * Author:           K.S.
 *
 * Created:          8-Dec-97
 *
 * Purpose:          Upsampling of a motion vector field to the same or a
 *                   higher spatial resolution.
 *                   The temporal resolution of the two MVFields have to
 *                   be specified by the input arguments.
 *
 * Arguments in:     int MV[2][5][MBR+1][MBC+2]  input motion vector field
 *                   int modemap[MBR+1][MBC+2]   input mode map
 *                   int win                     width of input
 *                   int tempDistIn         temporal distance for input field.
 *                   int tempDistOut        temporal distance for output field.
 *
 * Arguments in/out: -
 *
 * Arguments out:    MVField  out         output motion vector field.
 *                                        The dimension must be the same
 *                                        or bigger than the input MVField.
 *                                        The proportion must be the same.
 *
 * Return values:    -
 *
 * Example:          MV2MVField(MV, modemap, 2, mvOut, 1);
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         -
 *
 * Modified:         -
 *
 *****************************************************************************/
void MV2MVField(int MV[2][5][MBR+1][MBC+2], int modemap[MBR+1][MBC+2], int win,
		int tempDistIn, MVField out, int tempDistOut)
/***********************************************************CommentEnd********/
{
  int x, y, xx, yy;
  int s = out.w / win;
  int hin = out.h / s;


  for (y = 0; y < hin; y++)
    for (x = 0; x < win; x++) {
      for (yy = 0; yy < s; yy++)
	for (xx = 0; xx < s; xx++) {
	  out.mode[x * s + xx + (y * s + yy) * out.w] = modemap[y+1][x+1];
	  out.mx[x * s + xx + (y * s + yy) * out.w] =
	    MV[0][0][y+1][x+1] * s * tempDistOut / tempDistIn;
	  out.my[x * s + xx + (y * s + yy) * out.w] =
	    MV[1][0][y+1][x+1] * s * tempDistOut / tempDistIn;
	}
    }
}
