#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#define DEFINE_GLOBALS
#include "Util.h"

#include "defs.h"
#include "structs.h"
#include "dispconf.h"
#include "options.h"
#include "config.h"
#include "display.h"

#include "common.h"
#include "bitOut.h"
#include "code.h"
#include "h263encoder.h"
#include "h263encoder.p"
#include "display.p"

#ifndef WITHOUT_GRABBER
#include "grabber.p"
#endif

/*#define PLOT_MVECTORS*/


static OptionsStruct Init_Options =
{
  NULL,
  USE_VIS,
  USE_DGA,
  USE_SHM,
  DISPLAY,
  0,		/* Default for cover_window: use own toplevel */
  THREAD_YUV,
  GRAY,
  PSEUDOCOLOR,
  EXPAND,
  SIZE_QCIF,
  0,		/* Default for fps */
  LOOP,
  AUDIO
};

OptionsStruct Options;


static void ExitWithUsage(char *msg, int ret)
{
#define O(txt, def, opt) fprintf(stderr, "  -%s. Default: -%s%s.\n", txt, def ? "" : "no", opt);
  fprintf(stderr, "Usage: %s [-[no]option ...] basename\n", Options.argv0);
#ifndef WITHOUT_DGA
  O("dga        Use Direct Graphics Access", USE_DGA, "dga");
#endif
  O("windowid   Use given window as parent", 1, " own toplevel");
  O("nodisplay  Don't show the decoded frames", DISPLAY, "display");
  O("expand     Double the size of the image", EXPAND, "expand");
  O("gray       Show video in grayscale", GRAY, "gray");
  O("noaudio    Disable audio", AUDIO, "audio");
/*  O("loop       Repeat the input stream infinitly", LOOP, "loop"); */
  O("pseudo     Show video in PseudoColor (8Bit)", PSEUDOCOLOR, "pseudo (TrueColor)");
  O("qcif       Show image in QCIF size", 1, SIZE_QCIF ? "qcif" : "cif");
#ifndef WITHOUT_SHM
  O("noshm      Don't use shared memory", USE_SHM, "shm");
#endif
#ifndef WITHOUT_THREADS
  O("threads    Spawn a thread for color conversion", THREAD_YUV, "threads");
#endif
#ifndef WITHOUT_VIS
  O("vis        Use UltraSparc Visual Instruction Set", USE_VIS, "vis");
#endif
  O("fps        If set, use given framerate", 1, "nofps (unlimited)");
  if (msg)
    fprintf(stderr, "\nHmmm: %s\n", msg);
  exit(ret);
}


void main(int argc, char *argv[]) {
/*   char       *basename = "/DATA/sequences/QCIF_CD/cost/carphone/cp"; */
  char          *bitstreamName = "bitstream";
  int           t;             /* Time */
  int           n0;            /* Number of coded frames */
  int           codingTime = 6;   /* 0...15 */
  int           percentIMBs = 0;
  int           *mbInd, *mbQuant;
  int           type = PICTURE_CODING_TYPE_INTRA;
  int           bits;
  unsigned int  totalBits = 0;
  Bitstr        *bs;
  Picture       *pict, *prevPict, *decPict, *prevDecPict;
  int           runtime;
  MVField       mvField;
  struct timeval tstart,tstop;
  int           rate = 5000;
  int           q = 15;
  int           step = 3;
  int           start = 0;
  int           stop = 100;
  int           grab = FALSE;
#ifdef MEASURE_PSNR
  float         psnr, totalPSNR = 0.0;
#endif

  Options = Init_Options; 	/* struct copy */
  Options.argv0 = *argv++;

  while(*argv && **argv == '-' && argv[0][1]) 
    {
      char *o = *argv++;
      int v = strncmp("no", o+1, 2) ? 1 : 0;

      o += v ? 1 : 3;
           if (!strcmp(o, "gray"))	Options.visgray = v;
      else if (!strcmp(o, "grey"))	Options.visgray = v;
      else if (!strcmp(o, "pseudo"))	Options.vis8 = v;
      else if (!strcmp(o, "vis"))	Options.use_vis = v;
      else if (!strcmp(o, "dga"))	Options.use_dga = v;
      else if (!strcmp(o, "shm"))	Options.use_shm = v;
      else if (!strcmp(o, "shmem"))	Options.use_shm = v;
      else if (!strcmp(o, "display"))	Options.display = v;
      else if (!strcmp(o, "thread"))	Options.thread_yuv = v;
      else if (!strcmp(o, "threads"))	Options.thread_yuv = v;
      else if (!strcmp(o, "expand"))	Options.visexpand = v;
      else if (!strcmp(o, "audio"))	Options.audio = v;
      else if (!strcmp(o, "qcif"))	Options.qcif = v;
      else if (!strcmp(o, "cif"))	Options.qcif = !v;
      else if (!strcmp(o, "fps"))	Options.fps = atoi(*argv++);
      else if (!strcmp(o, "window"))
        Options.cover_window = strtol(*argv++, NULL, 0);
      else if (!strcmp(o, "windowid"))
        Options.cover_window = strtol(*argv++, NULL, 0);
      else if (!strcmp(o, "id"))
        Options.cover_window = strtol(*argv++, NULL, 0);
/*    else if (!strcmp(o, "loop"))	Options.loop = v;	*/
      else 
        ExitWithUsage("Unknown commandline option.", 1);
    }

  if (!Options.fps)
    {
      if (Options.audio)
	fprintf(stderr, "Audio disabled, -fps missing.\n");
      Options.audio = 0;
    }

  if (!*argv)
    grab = TRUE;


  /* Allocate memory */
  bs = FopenBitstr(5000, bitstreamName, "w");
  pict = AllocPicture(QCIF_WIDTH, QCIF_HEIGHT);
  prevPict = ClonePicture(pict);
  decPict = ClonePicture(pict);
  prevDecPict = ClonePicture(pict);
  mbInd = (int *)malloc(QCIF_WIDTH / 16 * QCIF_HEIGHT / 16 * sizeof(int));
  mbQuant = (int *)malloc(QCIF_WIDTH / 16 * QCIF_HEIGHT / 16 * sizeof(int));
  mvField.mx = malloc(QCIF_WIDTH * QCIF_HEIGHT /16/ 8 * sizeof(short));
  mvField.my = malloc(QCIF_WIDTH * QCIF_HEIGHT /16/16 * sizeof(short));
  mvField.mode = malloc(QCIF_WIDTH * QCIF_HEIGHT/16/16* sizeof(short));

  mvField.w = QCIF_WIDTH / 16;
  mvField.h = QCIF_HEIGHT / 16;

  /* wrong: init_display should obey -qcif -cif flags */
  init_display(pict->w, pict->h);

  gettimeofday(&tstart,(struct timezone *)NULL);


  for (t = start, n0 = 0; (grab ? t < 1000 : (t < stop)); t += step, n0++) {
    printf("\rtime: %4d", t);
    fflush(stdout);
    if (grab) {
#ifndef WITHOUT_GRABBER
      grabPicture(pict);
#else
      fprintf(stderr, "Sorry, Encoder compiled WITHOUT_GRABBER support\n");
      exit(0);
#endif
    } else {
      if (!ReadPicture(*argv, SEP_RAW_TYPE, t, pict))
	break;
    }

#ifdef CONST_RATE
    if ((bits = EncodeH263(rate, codingTime, type, t, percentIMBs,
			   pict, prevPict, prevDecPict, NULL,
			   decPict, bs, mbInd, mbQuant, &mvField)) < 0) {
#else
    if ((bits = EncodeH263Q(q, codingTime, type, t, percentIMBs,
			    pict, prevPict, prevDecPict, NULL,
			    decPict, bs, mbInd, mbQuant, &mvField)) < 0) {
#endif
      fprintf(stderr, "Encoder failt\n");
      exit (-1);
    }
    printf("        rate: %5d", bs->ind);
    fflush(stdout);
    yuv2rgb(decPict);
#ifdef PLOT_MVECTORS
    CopyPicture(decPict, prevPict);
    PlotMVField(&mvField, 0, 1.0, prevPict);
    yuv2rgb(prevPict);
#endif

    CheckOutBuffer(bs);

#if 1
    type = PICTURE_CODING_TYPE_INTER;
#endif

#ifdef MEASURE_PSNR
    psnr = PSNRPictures(pict, decPict);
    totalPSNR += psnr;
    printf("         PSNR: %2.2fdB", (double)psnr);
#endif

    /* Update previous images */
    SwapPictures(&prevPict, &pict);
    SwapPictures(&prevDecPict, &decPict);

    totalBits += bits;
  }
  printf("\n");

  CheckOutBuffer(bs);

  /* Print some statistics */
  /* Naja, verbesserungsbeduerftig */
  gettimeofday(&tstop,(struct timezone *)NULL);
  runtime = 100*(tstop.tv_sec-tstart.tv_sec)
    + (tstop.tv_usec-tstart.tv_usec)/10000;
  printf("\n%d.%02d seconds, %d frames, %d.%02d fps\n",
	 runtime/100, runtime%100,
	 n0, ((10000*t+runtime/2)/runtime)/100,
	 ((10000*t+runtime/2)/runtime)%100);

  if (n0 > 0) {
    printf("%3d frames coded with %8d bits.\t%6d bpf\t%7d bps\n",
	   n0, totalBits, totalBits / n0, 30 / step * totalBits / n0);
#ifdef MEASURE_PSNR
    printf("Average PSNR: %2.2fdB\n", (double)totalPSNR / (double)n0);
#endif
  }

  /* Free memory */
  FcloseBitstr(&bs);
  FreePicture(pict);
  FreePicture(prevPict);
  FreePicture(decPict);
  FreePicture(prevDecPict);
  free(mbInd);
  free(mvField.mx);
  free(mvField.my);
  free(mvField.mode);

  exit_display();

  printf("\nDas war's\n");
}
