/*
 * jittr - Just In Time TRansport library
 *
 * This is about atoms. The code here does not know anything about
 * clients or network access.
 *
 * 3.3.95 jw.
 *
 * removed any knowledge about exec_context. (except for debugging info).
 * 18.7.96 jw.
 */
#include "jittr/atom.h"		/* brings dstring.h */
#include "jittr/parse.h"	/* for jarg_*() method declarations */

/* XXX: Fixme: this is jittr specific knowledge: */
extern int jittr_error __P((/* lousy varargs: fmt, a1, a2, a3, a4 */));

int atoms_allocated = 0;	/* total size of atoms list */
int atoms_index = 0;		/* index of the free atom to be used next */
int atoms_free = 0;		/* number of currently unused atoms */
atom **atoms = NULL;		/* the list */

static int       atom_create __P((void));
static int        atom_trace __P((int idx, int flag, void *call_data));
static int atom_manage_trace __P((int idx, int flag, void *dummy));
static int           set_cmd __P((int idx, int flag, void *append));
static int           get_cmd __P((int idx, int flag, void *dummy));
static int           del_cmd __P((int idx, int flag, void *dummy));
 
/*
 * Return the index of an unused atom in the list, its refcount is already 1.
 * The atom's dstring pointers may be zero.
 * As long as malloc succeeds, there are always unused atoms, if not, -1 is
 * returned. Atoms are not recycled immediatly after they are freed, to keep 
 * indices more unique. The list may grow. 
 * The global variable atoms_index contains the next value that atom_create 
 * will return. You are free to modify this value, if you want to create a 
 * specific atom. But the indicated atom must not exist when atom_create is 
 * called.
 */
static int
atom_create()
{
  int i, j;

  if ((atoms_free << 2) <= atoms_allocated || atoms_index >= atoms_allocated)
    {
      atom **new;

      /* 
       * we do not want to recycle old atoms immediatly. Grow the list
       * if this resource becomes low.
       */
      if (!(i = atoms_allocated << 1))
        i = 128;
      while (i <= atoms_index)
        i <<= 1;
      if (!(new = (atom **)calloc(sizeof(atom *), i+1)))
        return -1;
      new++;
      if (atoms)
        {
	  for (j = -1; j < atoms_allocated; j++)
	    new[j] = atoms[j];
	  free((char *)(--atoms));
	}
      atoms = new;
      atoms_free += i - atoms_allocated;
      atoms_allocated = i;
    }

  if (atoms[atoms_index])
    ASSERT(!atoms[atoms_index]->a.refcount);	/* atoms[index] in use? */
  if (!atoms[atoms_index] && 
      !(atoms[atoms_index] = (atom *)calloc(sizeof(atom), 1)))
    return -1;
  atoms[atoms_index]->a.refcount = 1;
  atoms_free--;
  if (atoms[atoms_index]->name)
    atoms[atoms_index]->name->buf[atoms[atoms_index]->name->length=0] = '\0';
  if (atoms[atoms_index]->value)
    atoms[atoms_index]->value->buf[atoms[atoms_index]->value->length=0] = '\0';
  ASSERT((atoms_free << 2) >= atoms_allocated);	/* one fourth is free */

  /* warp atoms_index to the next free atom. The list is a ring here */
  for (i = atoms_index + 1; i != atoms_index; i++)
    {
      if (i >= atoms_allocated)
        i = 0;
      if (!atoms[i] || !atoms[i]->a.refcount)
        break;
    }
  ASSERT(i != atoms_index);	/* No atoms? What said the above ASSERT()? */
  j = atoms_index;
  atoms_index = i;
  return j;
}

/*
 * Create an atom with name and value.
 * The given index is only a hint. Usually you should use -1.
 */
int
atom_new2(idx, name, nlen, value, vlen, flag)
int idx;
char *name;
int nlen;
char *value;
int vlen, flag;
{
  int i;
  if (idx >= 0)
    atoms_index = idx;
  if ((i = atoi(atom_get(NEW_ATOM)->buf)) < 0)
    return i;
  if (idx >= 0 && i != idx)
    jittr_error("atom_new: wanted %d, got %d, do you expect this?\n", idx, i, 0, 0);
  dstring_append(&atoms[i]->name, 0, name, nlen);
  dstring_append(&atoms[i]->value, 0, value, nlen);
  atoms[i]->a.flag = flag;
  return i;
}

int
atom_new(idx, name, value, flag)
int idx;
char *name, *value;
int flag;
{
  return atom_new2(idx, name, 0, value, 0, flag);
}

/*
 * Execution of an atom takes a parameter.
 * This is brought in by extern struct exec_context *exec_context;
 * Caller supplied it; caller shall reset it; it must be nonzero.
 * Most trace methods are satisfied with the knowledge that 
 * the jarg_* functions from parse.c know how to access the parameter.
 * Others (e.g. trace_cmd) need to know that
 * args = *(struct dstring **)exec_context; holds the command line arguments.
 * Very few (e.g. eval_cmd) directly operate on exec_context.
 *
 * XXX: data (our exec_context) is only acessed here for debugging.
 * We should not have any knowledge about the data pointer here.
 * Just pass it down to atom_trace.
 * 
 * atom_exec is very primitive now. It is just a public interface to
 * atom_trace, but much simpler than the other interfaces that have sideeffects
 * on the atom.
 */
int
atom_exec(idx, data)
int idx;
void *data;
{
  int r;

  ASSERT(data);

  if (!atoms[idx]->trace)
    {
      dstring *args = *(dstring **)data;

      jittr_error("atom_exec(%s): atom %s: nothing happens.\n", 
        args ? args->buf : "(NULL)", atom_name(idx), 0, 0);
      return -1;
    }
  debug2("atom_exec(%d):%s\n", idx, atom_name(idx));

  r = atom_trace(idx, TRACE_EXEC, data);
  return r;
}

/*
 * Returns the number of bytes in atom after append.
 * Returns negative when in trouble.
 */
int
atom_append(idx, offset, string, length)
int idx, offset, length;
char *string;
{
  int  i;

  if (idx >= atoms_allocated || !atoms[idx])
    {
      debug1("atom_append: creating atom nr. %d on the fly.\n", idx);
      if ((i = atom_new(idx, NULL, NULL, 0)) != idx)
        {
          debug1("atom_append: emergency create failed: got %d.\n", i);
          return -1;	/* huch? what did we create? */
	}
    }
  if (atoms[idx]->a.flag & A_READONLY)
    return jittr_error("# atom_append(%d) failed: A_READONLY", idx, 0, 0, 0);
  if ((i =  dstring_append(&atoms[idx]->value, offset, string, length)) >= 0)
    {
      if (atom_trace(idx, i ? TRACE_WRITE : TRACE_WRITE|TRACE_TRUNC, NULL) < 0)
        i = -1;
      /*
       * Ruins the top widget as recorded by x11_modmain:
       * atoms[idx]->value->data = (void *)atoms[idx]->value->length;
       */
    }
  return i;
}

/*
 * If the atom already has the desired value, do nothing.
 * Not even trigger callbacks.
 */
int
atom_update(idx, d)
int idx;
struct dstring *d;
{
  struct dstring *av;

  if (atom_exists(idx))
    if (d == (av = atoms[idx]->value) ||
	(av && d->length == av->length && !xbcmp(av->buf, d->buf, d->length)))
      return 0;			/* already up to date: nothing to do */
  /* XXX swapping dstrigns here would be more efficient */
  return atom_append(idx, 0, d->buf, d->length);
}

/* caller does not own the returned dstring value! */
dstring *
atom_get(idx)
int idx;
{
  if (!atom_exists(idx))
    return NULL;
  atom_trace(idx, TRACE_READ, NULL);
  if (!atoms[idx]->value)
     dstring_append(&atoms[idx]->value, 0, NULL, 0);
  return atoms[idx]->value;
}

dstring *
atom_get_name(idx)
int idx;
{
  if (!atom_exists(idx))
    return NULL;
  if (!atoms[idx]->name)
     dstring_appendf(&atoms[idx]->name, 0, "%d", idx);
  return atoms[idx]->name;
}

int
atom_trace_delete(tp)
struct trace **tp;
{
  struct trace *n;

#if 0
  /* We do not know the atom. So we cannot check for false deletes. Sigh */
  if (atoms[idx]->a.traced)
    debug1("Dangerous atom_trace_delete(%d): atom is being traced!\n", idx);
#endif

  if (!tp || !*tp)
    return 1;
  n = (*tp)->next;
  if ((*tp)->flag & TRACE_DATA_ALLOCED)
    free((char *)(*tp)->data);
  free((char *)(*tp));
  *tp = n;
  return 0;
}

/*
 * Trace callbacks may choose to prevent deletion.  -1 is returned then!
 * To force the deletion, pop traces first.
 */
int
atom_delete(idx)
int idx;
{
  int r;

  if (!atom_exists(idx))
    return 0;
  if (atoms[idx]->a.refcount > 1)
    {
      jittr_error("atom_delete: %s(%d) refcount = %d > 1, still removing...",
      	atom_name(idx), idx, atoms[idx]->a.refcount, 0);
    }
  if (atoms[idx]->a.flag & A_PERMANENT)
    return jittr_error("atom_delete(%d) failed: A_PERMANENT\n", idx, 0, 0, 0);
  r = atom_trace(idx, TRACE_UNSET, NULL);
  if (!atoms[idx])	/* they just did it. */
    return 0;
  if (r < 0)
    {
      debug1("atom_delete(%d): action forbidden by trace callback.\n", idx);
      return -1;
    }
  if (atoms[idx]->value)
    atoms[idx]->value->buf[atoms[idx]->value->length = 0] = '\0';

  while (!atom_trace_delete(&atoms[idx]->trace))
    ;

  if (atom_delete_hook)
    atom_delete_hook(atoms[idx]);

  if (atoms[idx]->a.flag & A_DATA_ALLOCED && atoms[idx]->data)
    free((char *)atoms[idx]->data);

  return atoms[idx]->a.refcount = 0;
}

/* 
 * This is quite a stupid one. FIXME: It searches linear.
 * Users are encouraged to memorize the index returned by this function.
 * -1 is returned, if the atom is not found. the third parameter should
 * be atoms_index. To verify if the same name exists multiple times, retry
 * a successful lookup but give now the previously returned index as a 
 * third parameter.
 *
 * This is an ugly hack FIXME:
 * The names "reply" and "stdout" always refer to atom 1. We need this, 
 * since the name of atom 1 is substituted by som client name most of the time.
 */
int
atom_lookup_atoms(name, length, idx)
char *name;
int length, idx;
{
  int i;

  ASSERT(name && length && idx >= 0 && idx < atoms_allocated);
  for (i = idx - 1; i != idx && i != atoms_index; i--)
    {
      if (i < 0)
        i = atoms_allocated - 1;
      if (atoms[i] && atoms[i]->a.refcount && atoms[i]->name &&
          length == atoms[i]->name->length && 
  	  !xbcmp(atoms[i]->name->buf, name, length))
	return i;
    }

  if (length == 6 && !xbcmp("stdout", name, 6))
    return 1;
  if (length == 5 && !xbcmp("reply", name, 5))
    return 1;

  return -1;
}

int (*atom_lookup_hook)() = NULL;
int (*atom_delete_hook)() = NULL;

/* 
 * Find an atom by index or by name.
 * >=0 (the found index) is returned on success, 
 * -1 is returned on a failed name (or hook function) lookup.
 * -2 is returned on a failed index lookup, and indicates that the
 * name parameter was a well formed numeric integer and no atom exists with 
 * that index.
 *
 * A usual idiom is 
 *  if ((idx = atom_lookup(s)) < 0)
 *    return jittr_error(...);
 */
int
atom_lookup(name)
char *name;
{
  return atom_lookup2(name, name ? strlen(name) : 0);
}

int
atom_lookup2(name, len)
char *name;
int len;
{
  int i = -1;
  int c;
  char *s;

  ASSERT(atoms[0]);

  if (!name)
    return -1;
  c = name[len];
  if (c)
    name[len] = '\0';
  i = strtol(name, &s, 0);
  if (c)
    name[len] = c;
  if (!atom_exists(i))		/* hack: atom 0 always exists */
    return s == name ? -1 : -2;
  
  if (s == name)		/* was not a numeric index */
    {
#if 0
      static struct dstring *buf = NULL;

      if (!buf || name != buf->buf)
        {
	  /* 
	   * XXX: copy the string, so that poking '\0' bytes in lookup_link
	   * does not crash.
	   */
          dstring_append(&buf, 0, name, len);
	  name = buf->buf;
	}
#endif
      if (atom_lookup_hook && (i = atom_lookup_hook(name, len)) >= 0)
	return i;

      i = atom_lookup_atoms(name, len, atoms_index);
    }

#ifdef DEBUGG
  debug3("atom_lookup2(%s, %d) = %d\n", name, len, i);
#endif
  return i;
}

int
atom_command(i, name, fn, data)
int i;
char *name;
int (*fn) __P((int flag, int idx, void * data));
void *data;
{
  int idx = i;
  ASSERT(name || idx >= 0);

  if (name && (idx = atom_new(i, name, NULL, A_PERMANENT)) != i && i >= 0)
    abort();
  return atom_trace_set(idx, TRACE_EXEC|TRACE_REPL, fn, data);
}

static char *help[] = 
{
  "help [pattern ...]",	
	"print entire command listing, or entries where all patterns appear",
  "append atom value",
  	"write value at end of atom",
  "del atom",
  	"delete an atom. Dangling links may remain",
  "set atom [-value] value",
	"change contents (may create)",
  "set atom -name name",
	"rename atom (may create)",
  "set atom -[no]readonly|-[no]permanent",
	"change mode flag (may create)",
  "get atom [-name|-mode|-index|-value] [prefix [suffix]]", 
	"return value or name of an atom",
  NULL
};

/*
 * set up the world of atoms.
 */
int 
atom_init()
{
  atoms_index = -1;
  (void)atom_create();		/* the inofficial way; returns -1 here */
  atoms[-1]->a.flag |= A_PERMANENT;
 
  /* trace flag masks from atom.h must not overlap: */
  ASSERT((TRACE_MODE_MASK+TRACE_WHEN_MASK+TRACE_EDIT_MASK+TRACE_DATA_MASK) ==
	 (TRACE_MODE_MASK|TRACE_WHEN_MASK|TRACE_EDIT_MASK|TRACE_DATA_MASK));

  atoms_index = NEW_ATOM;

  (void)atom_create();		/* atom_new() needs NEW_ATOM */
  atoms[NEW_ATOM]->a.flag |= A_PERMANENT;
  dstring_set(&atoms[NEW_ATOM]->name, "NewAtom", 0);
  atom_trace_set(NEW_ATOM, TRACE_MODE_MASK, atom_manage_trace, NULL);

  if (atom_new( REPLY_ATOM, "reply",  NULL, A_PERMANENT) !=REPLY_ATOM) abort();
  if (atom_new(STDERR_ATOM, "stderr", NULL, A_PERMANENT) !=STDERR_ATOM) abort();
  atom_trace_set(STDERR_ATOM, TRACE_WRITE, atom_simple_trace, (void *)stderr);

  atom_command(HELP_CMD_ATOM, "help", atom_help_cmd, (void *)0);
  atom_command(HELP_CMD_ATOM, NULL,   atom_help_cmd, (void *)help);
  atom_command( GET_CMD_ATOM, "get",        get_cmd, (void *)0);
  atom_command( SET_CMD_ATOM, "set",        set_cmd, (void *)0);
  atom_command( APP_CMD_ATOM, "append",     set_cmd, (void *)1);
  atom_command( DEL_CMD_ATOM, "del",        del_cmd, (void *)0);

  return 0;
}

/* ============ about traces ===================== */

/*
 * Return the address of the representative pointer to a trace
 * on atom idx, that matches fn and data. If fn == NULL, only data is matched.
 * Returns NULL if root is NULL. Returns the adress of a NULL pointer, if no
 * such trace. 
 * Bla bla. This function is so primitive and it does not even know how
 * to distinguish evil_cmd traces, we inline better versions in most cases.
 */
trace **
atom_trace_lookup(root, fn, data)
struct trace **root;			/* use here: &atoms[idx]->trace */
int (* fn) __P((int flag, int idx, void *data));
void *data;
{
  trace **tr;

  if (!root || !*root)
    return root;
  for (tr = root; *tr; tr = &(*tr)->next)
    {
      if ((!fn || (fn == (*tr)->fn)) && (data == (*tr)->data))
	break;
    }
  return tr;
}

/*
 * Append the trace to the end of the list, so that they are executed in 
 * chronological order later. If the bit TRACE_LATE is present in any 
 * trace callbacks, the new trace will be placed in the queue before the first
 * callback with that bit. This implements a stack.
 * The concepts of queue and stack may be spoiled if TRACE_REPL is specified.
 * In that case, the list is searched for an existing trace with same fn and
 * data pointer. This trace will be updated inplace.
 *
 * It is allowed to pass the bit TRACE_LAST here!
 *
 * Returns -1 if no atom or no memory,
 * returns 1, if only the flags of an existing trace were updated.
 * returns 0 if the trace was sucessfully establisehd.
 */
int
atom_trace_set(idx, flag, fn, data)
int idx, flag;
int (* fn) __P((int idx, int flag, void *data));
void *data;
{
  trace **tr, *n;

  ASSERT(fn);

#ifdef DEBUG
  /* we know that the traces on STDERR and HELP look dangerous */
  if (atoms[idx]->a.traced && idx > 3)
    debug1("Dangerous atom_trace_set(%d): atom is being traced!\n", idx);
#endif

  tr = &atoms[idx]->trace; 

  if ((flag & TRACE_REPL) && 
      !(tr = atom_trace_lookup(tr, fn, data)))
    return -1;

  if ((flag & TRACE_REPL) && *tr)
    {
      /* 
       * XXX: Fixme: TRACE_WHEN_MASK should also be updated. 
       * but that may require moving the trace... too difficult, too useless.
       */
      (*tr)->flag = ((*tr)->flag & ~TRACE_MODE_MASK) | (flag & TRACE_MODE_MASK);
#ifdef DEBUG
      if (((*tr)->flag & TRACE_WHEN_MASK) != (flag & TRACE_WHEN_MASK))
	debug1("TRACE_REPL: WHEN_MASK of trace on atom %d not updated.\n", idx);
#endif
      return 1;
    }
  else
    tr = &atoms[idx]->trace; 	/* switch back to normal semantic */

  while (*tr && !((*tr)->flag & TRACE_LATE))
    tr = &(*tr)->next; /* normal means FIFO until TRACE_LATE, thereafter LIFO */
  n = *tr;
  if (!(*tr = (trace *)calloc(sizeof(trace), 1)))
    return -1;
  (*tr)->fn = fn;
  (*tr)->flag = flag & ~TRACE_EDIT_MASK;
  (*tr)->data = data;
  (*tr)->next = n;
  return 0;
}

/*
 * Execute the trace callbacks for an atom. 
 * Trace callback functions should consider several parameters: 
 *  a) The atom they trace, especially its current value and 
 *  b) the current exec_context as established by jittr_process().
 *  c) the private tr->data element (never changed by the jittr world).
 *
 * Types of traces: (determined by flag)
 * TRACE_WRITE: 
 *	Happens after the append. exec_context->nwritten contains
 *      number of bytes of the final write. This trace is also called when 
 *	the contents was truncated to 0 bytes.
 *	Callback methods may choose to publish the new value, usually without
 *	triggering read callbacks.
 * TRACE_TRUNC:
 *	Happens after the truncate. exec_context->nwritten is 0.
 *	Callback methods may choose to publish the new value, usually without
 *	triggering read callbacks.
 * TRACE_READ:
 *	Happens before the value is sampled. Trace method may choose to alter 
 *      the value. They should consider whether or not to trigger write 
 *      callbacks while doing so.
 * TRACE_UNSET:
 *	Happens before the atom is zapped. If a trace method returns nonzero,
 *	the atom remains alive.
 * TRACE_EXEC:
 *	Just happens. Not directly related to a change in the atom's state.
 *	exec_context->what is the parameter list passed to 
 *      atom_exec(). Exec callback methods may consume the dstring containing
 *      the parameter list by advancing its data field. When they reach the 
 *	end of the parameter list, they should return positive, to prevent 
 *      further methods to operate. Error conditions while operating shall
 *	be returned negative. If a callback cannot operate with the given
 *	parameter list, it may issue a jittr_error() but should return 0 to 
 *      let other methods try their luck.
 * TRACE_LATE:
 *	Not really a type, but an additional attribute honoured by 
 *      atom_trace_set: no traces without this attribute will ever follow
 *	after the first trace with this attribute.
 * TRACE_LAST:
 *	A trace with this attribute prevents older traces from being executed.
 *      Older traces are simply ignored until this trace is removed from the
 *      queue again. The last trace method in the queue will be called with
 *      this bit ORed into its flag, regardless if it has this attribute or 
 *      not. Thus, the trace method can detect if more traces will be executed
 *	on the same atom immediatly. The trace method has no way to find out
 *	if there are suppressed traces (except, it may traverse the trace list
 *	manually).
 * TRACE_ONCE:
 *	A trace with this attribute will execute exactly once. It is destroyed
 *	after the first successful or unsuccessful invocation. If combined with
 *	TRACE_LAST, a queue of nonrepetitive actions can be constructed, so
 *	that every next READ/WRITE/EXEC trace will trigger the next action.
 *
 * All other elements of exec_context may be valid or NULL.
 * Traces may trigger more traces. Cycles are detected and avoided here.
 * Traces may add or remove to/from the atoms trace list.
 * Deleting the atom from within one of its traces is dangerous. atom_trace()
 * and atom_delete() will survive, but other functions may not.
 * 
 * atom_trace() returns 0, when all callback methods returned 0. If one of the
 * methods returns nonzero, atom_trace() immediatly returns this value without
 * executing further methods. If this value is negative, it is reported
 * as a jittr_error.
 *
 * If multiple trace callbacks are runable, they are invoked in the order
 * they were put in the queue. Except thost that are tagged "LATE".
 *
 * This is a hack to save some malloc and free cycles:
 * Exec methods may want to use a buffer for parsing their command line 
 * arguments. We asume that no two exec methods are active simultaneously and
 * take care that the global jarg buffer is unused here.
 * Exec methods must not rely on their call frame to reset the jarg buffer.
 *
 * call_data is currently unused. XXX: Fixme: call_data is the clean way to
 * pass an exec_context back into the jittr_world. Either add a fourth 
 * parameter to all trace callbacks, or make the second parameter a 
 * struct call_data pointer.
 */
static int
atom_trace(idx, flag, call_data)
int idx, flag;
void *call_data;
{
  trace *tr, **trp = &atoms[idx]->trace;
  struct trace trs;
  int r = 0, rr = 1;
  int callflag;

#ifdef DEBUGGG
  if (*trp)
    {
      debug3("atom_trace(%d(%s),%#02x) called", idx, atom_name(idx), flag);
      debug1("%s.\n", *trp ? "" : " (no trace)");
    }
#endif

  if (atoms[idx]->a.traced & flag)
    {
      /*
       * This allows that two atoms update each other via trace callbacks
       * without causing trouble. The call graph of trace callbacks can be 
       * extremly complicated. This code ensures that the same type of trace on
       * is never twice on the same atom.
       * Traces that use the scheduler or the I/O system do not pile up on 
       * the execution stack. Their recursion cannot be detected here.
       */
#ifdef DEBUGG
      debug3("cyclic trace callback (type %#02x) suppressed on atom %d(%s).\n",
	     flag, idx, atom_name(idx));
#endif
      return 0;
    }

  while ((tr = *trp))
    {
      callflag = 0;
      if (flag & tr->flag)
	{
	  /* we cannot dereference tr later, it may be freed */
	  trs = *tr;

	  /* non late traces must not follow late traces */
	  ASSERT(!tr->next || 
		 !(tr->flag & TRACE_LATE) ||
		 tr->next->flag & TRACE_LATE);
	  /* stacked exec traces ruin parsers that access global jarg */
#ifdef DEBUG
	  if (flag & TRACE_EXEC)
	    ASSERT(!jarg_reset());
#endif
	  callflag = flag | (tr->flag & TRACE_WHEN_MASK); 
	  atoms[idx]->a.traced |= flag;
	  rr = r = tr->fn(idx, callflag | (tr->next?0:TRACE_LAST), tr->data);
	  if (atoms[idx])
	    atoms[idx]->a.traced &= ~flag;
#ifdef DEBUG
	  if (flag & TRACE_EXEC)
	    jarg_reset();
#endif
	  if (r < 0)
	    jittr_error("atom_trace(%d, 0x%04x) returnd %d. ", 
			idx, callflag, r, 0);
	  if (r)
	    break;
	}
      /*
       * Carefully advance the pointer here. 
       * The trace is allowed to delete itself or the whole atom!
       */
      if (!atoms[idx] || !atoms[idx]->trace)
	break;
      /* XXX: Fixme: is it save to dereference trp now? */
      if (tr == *trp)		/* advance if it was still there */
	{
	  if (callflag & TRACE_ONCE)
	    atom_trace_delete(trp);	/* advances *trp */
	  else
	    trp = &tr->next;
	}
      else if (callflag & TRACE_ONCE)
	{
	  /* careful here: do not delete the wrong instance! */
	  struct trace **nt = &atoms[idx]->trace;
	  
	  while (*(nt = atom_trace_lookup(nt, trs.fn, trs.data)))
	    {
	      if (*nt == tr)
		{
		  atom_trace_delete(nt);
		  break;
		}
	    }
	}
      if (callflag & TRACE_LAST)
	break;
    }
#ifdef DEBUG
  if (flag == TRACE_EXEC && rr != r)
    debug2("atom_trace(TRACE_EXEC, %d(%s)): nothing happens.\n",
	  idx, atom_name(idx));
#endif
  return r;
}

/* ============ about standard trace callbacks ===================== */

/*
 * Trace callback methods are allowed to destroy the atom (with great care), 
 * they are also allowed to delete themselves or append new traces. No other
 * manipulation on the trace list is allowed.
 * Trace callbacks should return non-negative. Otherwise an error is posed.
 * See the comment at atom_trace() for details.
 */
int
atom_simple_trace(idx, flag, fp)
int idx, flag;
void *fp;
{
  fprintf((FILE *)fp, "simple %s: %s\n", atom_name(idx), atom_value(idx));
  fflush((FILE *)fp);
  dstring_append(&atoms[idx]->value, 0, NULL, 0);
  return 0;
}

/*
 * Operations performed by atom 0.
 *
 * - If you read this atom, you'll cause a new atom to be created and you
 *   read its id.
 * - If you write the atom, you should write the name or (preferred) id
 *   of another existing atom. This other atom will be deleted.
 *   If you write the id of a nonexisting atom, the next created atom will
 *   have this index.
 * - If you delete the atom, exit() will be called.
 * - exec is currently undefined. 
 *
 * XXX FIXME: we have no STDIN_ATOM. If we need one, this is the right place:
 * - exec of atom 0: with parameters deletes or creates named atoms.
 * - write: a text to be locally processed by jittr_interpret().
 * Change this by overloading :-)
 * Currently not done, because accidentay writes to atom 0 do happen. :-(
 */ 
static int
atom_manage_trace(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  ASSERT(idx == NEW_ATOM);
  if (flag & TRACE_READ)
    {
      /*
       * create a new atom, place its index in atom NEW_ATOM without
       * triggering write callbacks
       */
      dstring_appendf(&atoms[idx]->value, 0, "%d", atom_create(), 0, 0, 0);
    }
  if (flag & TRACE_WRITE)
    {
      char *s = atom_value(idx);
      int i;

      /* 
       * kill the atom who's index was written if an invalid index was 
       * written, arrange that the next create creates exacly this index.
       * A name of the atom may may be used instead of an index.
       */
      i = atom_lookup(s);
      if (i == NEW_ATOM || i == STDERR_ATOM)
	return -1;
      if (i < 0)	/* atom not found */
	{
	  /* 
	   * This may be usefull / may be dangerous.
	   * Return -1 without touching atoms_index,
	   * or document why it is needed here.
	   */
	  if (i == -1)
	    return jittr_error("%s: nonexistant name: %s. Try writing an existing name or an index!\n", atom_name(idx), s, 0, 0);
	  debug2("nonexistnt index %d written to '%s', preloads atom_new().\n",
		 atoi(s), atom_name(idx));
	  atoms_index = atoi(s);
	  return 0;
	}
      atom_delete(i);		/* this is an important feature */
      return 0;
    }
  if (flag & TRACE_UNSET)
    {
      debug1("Emergency exit: atom '%s' was destroyed.\n", atom_name(idx));
      exit(0);
    }
  return 0;
}

/* ============ atom command help text parser ================= */

/* 
 * cmd and descr are written 'linewise' into d
 * Each line starts with '#'. The cmd string starts at column 3.
 * If cmd is "." or if "./" is a prefix of cmd, then atom_name(idx) is printed
 * instead of the first character of cmd, otherwise idx is ignored.
 * The string descr starts at column DESCR_COL. If it extends over column
 * WRAP_COL, it is broken at the next ' ' character and continued at column
 * DESCR_COL in the next line. descr will wrap before reaching DESCR_COL,
 * if a '\n' character appears in the string.
 */
int
atom_help_fmt(d, idx, cmd, descr)
struct dstring **d;
int idx;
char *cmd, *descr;
{
  int pos;

#define DESCR_COL 27
#define WRAP_COL 70

  pos = dstring_append(d, -1, "#  ", 3) - 3; 
  if (!*cmd && !*descr)
    return dstring_append(d, -3, "", 0);	/* how a blank line is done */

  if (*cmd == '.' && (!cmd[1] || cmd[1] == '/' ||
		      cmd[1] == ' ' || cmd[1] == '\t' || cmd[1] == '\n'))
    {
      dstring_append(d, -1, atom_name(idx), 0);
      cmd++;
    }
  pos = dstring_append(d, -1, cmd, 0) - pos;

  ASSERT(descr);	/* or someone forgot a comma in the help pages */
  if (!*descr)
    return dstring_append(d, -1, "", 0);	/* how to do an empty descr */

  if (pos >= DESCR_COL)
    {
      dstring_append(d, -1, "\n#  ", 4);
      pos = 3;
    }
  while (pos++ < DESCR_COL)
    dstring_append(d, -1, " ", 1);

  if (!descr)
    return jittr_error("description for command '%s' missing.\n", cmd);

  while (*descr)
    {
      dstring_append(d, -1, descr++, 1);
      pos++;
      if (*descr == '\n' || (*descr == ' ' && pos > WRAP_COL))
        {
	  dstring_append(d, -1, "\n#  ", 4);
	  pos = 3;
	  while (pos++ < DESCR_COL)
	    dstring_append(d, -1, " ", 1);
	  descr++;
	}
    }
  return (*d)->length;
}

/* 
 * Atom_grep_trace() is temporarily placed onto REPLY_ATOM, to filter text.
 * It is a write callback triggered by the atom_append() calls near the end of
 * atom_help_cmd().
 */
static int
grep_trace(idx, flag, data)
int idx, flag;
void *data;
{
  char *pattern     = (char *)data;
  struct dstring *v = atoms[idx]->value;
  char *start       = v->buf + (int)(v->data);
  char *end         = v->buf + v->length;

  ASSERT(start >= v->buf && start <= end);
  *end = '\0';		/* just to be sure :-) */

  /* debug3("%s: grep for '%s' in \n%s\n", atom_name(idx), pattern, start); */
  if (!strstr(start, pattern))
    v->length = (int)(v->data);
  return 0;
}

/*
 * This is placed onto HELP_ATOM once and late, to do clean up after grep.
 * The value->data pointer is restored here to its previous contents.
 */
static int
remove_grep_trace(idx, flag, data)
int idx, flag;
void *data;
{
  struct trace **tr;

  if (atoms[REPLY_ATOM]->value)
    atoms[REPLY_ATOM]->value->data = data;
  
  tr = &atoms[REPLY_ATOM]->trace;
  while (*tr)
    if ((*tr)->fn == grep_trace)
      {
	debug1("remove_grep_trace: ZAP! grep '%s'\n", (char *)((*tr)->data));
        atom_trace_delete(tr);
      }
    else
      tr = &(*tr)->next;
  return 0;
}

/* 
 * May be stacked. The data pointer shall be the address of a
 * help strings array. the strings come in pairs:
 * first the command syntax synopsis, then the description text.
 * The array is expected to be delimited by a NULL pointer.
 * after each pair of strings, trace callbacks on the REPLY_ATOM
 * are triggered.
 *
 * If help is called with parameters, each word is a substring pattern, that is
 * matched on the output by a trace callback. This callback removes the newly 
 * added text, if it fails to match a single word. The value->data pointer of
 * the stdout atom is used to measure the text increment.
 *
 * When atom_help_cmd is called with a NULL data pointer, these traces are 
 * placed on REPLY_ATOM and their removal placed on the executing atom.
 * You need not call atom_help_cmd with a NULL data pointer. The very first
 * registration on HELP_ATOM already does that.
 *
 * Part of this code is duplicated in jmod_help_cmd().
 */
int
atom_help_cmd(idx, flag, data)
int idx, flag;
void *data;
{
  char **help = (char **)data;
  dstring **rp = &atoms[REPLY_ATOM]->value;

  if (!(flag & TRACE_EXEC))
    return 0;

  if (!help)
    {
      char *s = jarg_first_word(NULL);

      dstring_append(rp, 0, NULL, 0);		/* avoids a coredump */
      /* to be cooperative, we save and restore the value->data pointer */
      atom_trace_set(idx, TRACE_EXEC|TRACE_LATE|TRACE_UNIQ, remove_grep_trace,
        (*rp)->data);
      (*rp)->data = (void *)(*rp)->length;

      if (!s || !*s)
        atom_append(REPLY_ATOM, -1 ,"# Available commands in jittr core are:\n", 0);
      else
	do
	  atom_trace_set(REPLY_ATOM, TRACE_WRITE, grep_trace, (void *)s);
	while ((s = jarg_next_word(NULL)));
      return 0;
    }

  while (help && *help)
    {
      atom_help_fmt(rp, -1, help[0], help[1]);
      atom_append(REPLY_ATOM, -1, ".\n" + !*help[1], 2);
      (*rp)->data = (void *)(*rp)->length;
      help += 2;
    }
  return 0;
}

/* 
 * EXEC: write value into some other atom. 
 * Can also create/rename atoms. Should check for name collisions.
 *
 * Watch out for overloading by jcmd.c - If you change here, also look there!
 */
static int
set_cmd(idx, flag, append)
int idx, flag;
void *append;
{
  int slen, endopt = 0;
  char *s;

  if (!(s = jarg_first_word(&slen)))
    return jittr_error("usage: %s atom value\n", jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    {
      /* it is the name of an atom that does not exist */ 
      if ((idx = atom_new(idx == -1 ? idx : atoi(s), 
      			  idx == -1 ? s : NULL, NULL, 0)) < 0)
        return jittr_error("%s: failed to create '%s'.", jarg_name(), s, 0, 0);
    }
  
  s = jarg_next_word(&slen);
  while (!endopt && slen > 1 && *s == '-')  /* s == NULL implies slen == 0 */
    {
      if (!strncmp("-name", s, slen))
        {
	  if (!(s = jarg_next_word(&slen)))
	    return jittr_error("%s: -name: parameter missing.\n", 
	    	jarg_name(), 0, 0, 0, 0);
	  dstring_append(&atoms[idx]->name, append ? -1 : 0, s, slen);
	  return 1;
	}
      else if (!strncmp("-readonly", s, slen))
        atoms[idx]->a.flag |= A_READONLY;
      else if (!strncmp("-permanent", s, slen))
        atoms[idx]->a.flag |= A_PERMANENT;
      else if (!strncmp("-noreadonly", s, slen))
        atoms[idx]->a.flag &= ~A_READONLY;
      else if (!strncmp("-nopermanent", s, slen))
        atoms[idx]->a.flag &= ~A_PERMANENT;
      else if (!strncmp("-value", s, slen) || !strcmp("--", s))
        endopt++;
      else 
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      if (!(s = jarg_next_word(&slen)))
        return 1;
    }
    
  atom_append(idx, append ? -1 : 0, s, slen);
  return 1;
}

/* 
 * EXEC: reply the value or name of some other atom. 
 * optinal patterns may be provided. The first pattern is prepended to the 
 * result, the second pattern is appended. The first one defaults to "",
 * the second one to "\n".
 *
 * Watch out for overloading by jcmd.c - If you change here, also look there!
 */
static int
get_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring *d = NULL;
  int slen, endopt = 0, zapd = -1;
  char *s;

  if (!(s = jarg_first_word(&slen)))
    return jittr_error("usage: %s atom value\n", jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    return jittr_error("%s: atom '%s' not found.\n", jarg_name(), s, 0, 0);
  
  s = jarg_next_word(&slen);

  while (!endopt && slen > 1 && *s == '-')  /* s == NULL implies slen == 0 */
    {
      if (!strncmp("-name", s, slen))
	d = atom_get_name(idx);
      else if (!strncmp("-mode", s, slen))
	zapd = dstring_appendf(&d, 0, "-%sreadonly -%spermanent", 
	  atoms[idx]->a.flag & A_READONLY ? "" : "no",
	  atoms[idx]->a.flag & A_PERMANENT ? "" : "no", 0, 0);        
      else if (!strncmp("-index", s, slen) || !strncmp("-idx", s, slen))
	zapd = dstring_appendf(&d, 0, "%d", idx, 0, 0, 0);        
      else if (!strncmp("-value", s, slen) || !strcmp("--", s))
        { endopt++; if (zapd) free((char *)d); zapd = -1; d = NULL; }
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }

  if (!d)
    d = atom_get(idx);

  dstring_append(&atoms[REPLY_ATOM]->value, -1, s, slen);
  dstring_append(&atoms[REPLY_ATOM]->value, -1, d->buf, d->length);
  s = jarg_next_word(&slen);
  atom_append(REPLY_ATOM, -1, s ? s : "\n", s ? slen : 1);
  if (zapd >= 0)
    free((char *)d);
  return 1;
}

static int
del_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  char *s;
  
  if (!(s = jarg_first_word(NULL)))
    return jittr_error("%s: atom_name missing.\n", jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    return jittr_error("%s: atom '%s' not found.", jarg_name(), s, 0, 0);
  
  return atom_delete(idx) < 0 ? -1 : 1;
}
