/*
 * doio.c -- upper level network access.
 *
 * This file logically part of the net.c module. It was split due to file size.
 * The same rules apply.
 *
 * 4.1.96 jw.
 */
#include <unistd.h>		/* close, alarm */
#include <errno.h>		/* EBADF, errno */
#include <sys/types.h>		/* for sys/socket.h */
#include <sys/socket.h>		/* struct sockaddr */
#include <netinet/in.h>		/* struct sockaddr_in */
#include <arpa/inet.h>		/* inet_ntoa() */
#include <sys/time.h>		/* struct timeval */
#if defined(linux) || defined(hpux)
# include <sys/ioctl.h>		/* ioctl() */
# include <termios.h>		/* ioctl FIONREAD there? They must be kidding */
#else
# include <sys/filio.h>		/* ioctl FIONREAD */
#endif
#include "jittr/atom.h"		/* only for dstring datatype and ERR* macros */
#include "jittr/jittr.h"	/* only for xbcopy() & sizeof(struct cwp) */
#include "jittr/net.h"		/* also requires sockaddr & timeval ... */
#include "jittr/doio.h"		/* also needs sys/types */
#include "jittr/schedule.h"	/* only for sched_timerdiff() declaration */
#include "jittr/debug.h"	/* debug() macros */

struct client *zombies = NULL;		/* list of all turned clients */
static int max_client_id = 0;		/* highest id number issued */
static int fd_client_max = -1;		/* highest fd number in list */

struct client *fd_client[FD_SETSIZE];	/* made by mkfdset() */
dstring *jittr_ioerr;			/* where to dump error messages */

/* ====================== fildescriptor I/O section ==================== */
#define FD_ENTER(fd, fdset, cl) do {	\
  FD_SET((fd), (fdset));		\
  if ((fd) > fd_client_max)	 	\
    fd_client_max = (fd);		\
  while (cleared <= (fd))		\
    fd_client[cleared++] = NULL;	\
  fd_client[(fd)] = (cl);		\
  fd_count++;				\
} while (0)

/*
 * Walk through the client list and prepare *rp and *wp for a select()
 * system call. The fd_client[] is updated, and fd_client_max is set to
 * the highest filedescriptor number found.
 * Filedescriptors are only recorded in *wp, if the clients outbound buffer 
 * is not empty. This is because entries in *wp usually cause select()
 * to return immediatly.
 * The total count of filedescriptors placed in the fd_sets is returned.
 * Double entries are counted twice.
 * We usually listen on all filedescriptors. Even those from outbound 
 * connections. A special connection type CTYPE_WFD is the only exception to 
 * avoid double prompts and empty reads on ttys, where full duplex I/O is 
 * possible on both stdin and stdout.
 *
 * Hack: As a side effect, mkfdset(), directly writes data from outbound
 * buffers to UDP connections that do not have a filedescriptor.
 */
static int
mkfdset(rp, wp)
fd_set *rp, *wp;
{
  struct client *cl;
  int cleared = 0;
  int fd, fd_count = 0;

  fd_client_max = -1;
  FD_ZERO(rp);
  FD_ZERO(wp);

  for (cl = clients; cl; cl = cl->next)
    {
      if (cl->o)
        {
	  struct dstring *d = cl->o->obuf;

	  fd = cl->o->fd;
	  if (dstring_length(d))
	    {
#if 0
	      debug2("mkfdset: %d bytes output pend fd=%d.\n", d->length, fd);
#endif
	      if (fd < 0)
		{
		  if (IS_CTYPE_TCP(cl->o->type))
		    /* tcp clients don't come back, do they? */
		    TurnClient(ClientById(NULL, cl->id), 0);
		  else if (cl->o->type == CTYPE_UDP)
		    {
		      int l;

		      debug2("mkfdset: UdpSend to: %s: %s\n",  
		      	cl->name->buf, d->buf);
		      if ((l=UdpSend((struct sockaddr_in *)&cl->o->sa, d->buf, 
		      		     (d->length > 1024) ? 1024 : d->length,
				     NULL)) < 0)
		        {
			  debug3("mkfdset UdpSend(%s)=%d: %d bytes zapped.\n",
			    cl->name->buf, l, d->length);
			  d->length = 0;
			}
		      else
			{
			  debug2("mkfdset UdpSend(%s)=%d.\n", cl->name->buf, l);
			  if ((d->length -= l) > 0)
			    xbcopy(d->buf + l, d->buf, d->length);
			}
		    }
		  else
		    {
		      debug2("mkfdset: %s is disconnected: %d bytes zapped.\n",
			cl->name->buf, d->length);
		      d->length = 0;
		    }
		}
	      else
		FD_ENTER(fd, wp, cl);	/* When we have something to say. */
	    }

	  if ((fd >= 0) && (cl->o->type != CTYPE_WFD))
	    FD_ENTER(fd, rp, cl);	/* But we always listen for a reply. */
	}

      if (cl->i && cl->i != cl->o)
        {
          fd = cl->i->fd;
	  /* this is odd, but possible with two full duplex connections */
	  if (dstring_length(cl->i->obuf))
	    {
	      if (fd < 0)
	        {
		  debug2("mkfdset: %s is disconnected: %d bytes ZAPPED.\n",
		    cl->name->buf, cl->i->obuf->length);
		  cl->i->obuf->length = 0;
		}
	      else
		FD_ENTER(fd, wp, cl);	
	    }
	  if (fd >= 0)
	    FD_ENTER(fd, rp, cl);
	}
    }
  if ((fd = server.tcp_port_fd) >= 0)
    FD_ENTER(fd, rp, INVALID_CLIENT);
  if ((fd = server.udp_port_fd) >= 0)
    FD_ENTER(fd, rp, INVALID_CLIENT);
  return fd_count;
}
#undef FD_ENTER

static void
assert_unused_fd(fd, where)
int fd;
char *where;
{
  if (fd > fd_client_max)
    return;

  ASSERT(fd_client[fd] != INVALID_CLIENT);
  if (fd_client[fd])
    {
      if (fd_client[fd]->i && fd_client[fd]->i->fd == fd)
        fd_client[fd]->i->fd = -1;
      if (fd_client[fd]->o && fd_client[fd]->o->fd == fd)
        fd_client[fd]->o->fd = -1;
      fd_client[fd] = NULL;
      debug2("%s: ASSERT(!fd_client[%d]) avoided. Old client killed! Gulp.\n", 
        where ? where : "assert_unused_fd", fd);
    }
}

/* a file descriptor is closed. Do cleanup in the jittr world. */
int
ConnShutdown(conn)
struct conn *conn;
{
  int r = 0;
  char *name = NULL;

  debug1("ConnShutdown: closing fd %d\n", conn ? conn->fd : -1);
  if (!conn || conn->type == CTYPE_INVALID || conn->fd < 0)
    return -1;
  if (fd_client[conn->fd] && fd_client[conn->fd]->name)
    {
      name = fd_client[conn->fd]->name->buf;
      debug2("ConnShutdown: %s client %s\n",
        fd_client[conn->fd]->o == conn ? "to" : "from", name);
    }
  if (conn->type == CTYPE_FD || conn->type == CTYPE_WFD ||
      conn->type == CTYPE_UDP)
    {
      r = close(conn->fd);
#ifdef DEBUG
      if (!fd_client[conn->fd])
        debug1("ConnShutdown: ASSERT(fd_client[%d]) avoided ...\n", conn->fd);
#endif
      fd_client[conn->fd] = NULL;
      conn->fd = -1;
    }
  if (IS_CTYPE_TCP(conn->type))
    r = PortShutdown(conn->fd, fd_client, &jittr_ioerr);
  if (dstring_length(conn->ibuf))
    {
      debug1("ConnShutdown: nuking %d bytes of input\n", conn->ibuf->length);
      conn->ibuf->length = 0;
    }
  
  /* 
   * The quit_cmd() can zap a conn, while jittr_process() has the 
   * obuf redirected. Thus we must check here for DEADBEEF.
   */
  if ((conn->obuf != (dstring *)0xDEADBEEF) && dstring_length(conn->obuf))
    {
      debug1("ConnShutdown: dumping %d bytes to STDERR\n", conn->obuf->length);
      dstring_append(&jittr_ioerr, -1, "Connection to ", 0);
      dstring_append(&jittr_ioerr, -1, name, 0);
      dstring_append(&jittr_ioerr, -1, " closed. UNSENT TEXT FOLLOWS:\n", 0);
      dstring_append(&jittr_ioerr, -1, conn->obuf->buf, conn->obuf->length);
      dstring_append(&jittr_ioerr, -1, "END OF UNSENT TEXT\n", 0);
    }
  return r;
}

/*
 * Walk through the list of filedescriptors established by mkfdset()
 * and probe each entry. Dead entries are removed propperly.
 * If multiple entries are bad, only one is removed.
 * Returns zero if nothing to remove.
 */
static int
zap_bad_fd()
{
  fd_set r, w;
  struct timeval tv;
  int i;

  for (i = 0; i <= fd_client_max; i++)
    {
      if (!fd_client[i])
	continue;
      FD_ZERO(&r);
      FD_ZERO(&w);
      FD_SET(i, &r);
      FD_SET(i, &w);
      tv.tv_sec = tv.tv_usec = 0;
      if (select(FD_SETSIZE, &r, &w, (fd_set *)0, &tv) < 0)
	{
	  if (errno == EINTR)
	    continue;
	  /* zap a bad client connection */
	  ASSERT((fd_client[i]->o && fd_client[i]->o->fd == i) || 
		 (fd_client[i]->i && fd_client[i]->i->fd == i));
	  if (fd_client[i]->o && fd_client[i]->o->fd == i)
	    fd_client[i]->o->fd = -1;
	  if (fd_client[i]->i && fd_client[i]->i->fd == i)
	    fd_client[i]->i->fd = -1;
	  debug2("Zapped bad client fildedescriptor: %d of client %s.\n",
	  	i, fd_client[i]->name->buf);
	  fd_client[i] = NULL;
	  return 1;
	}
    }
  if (server.tcp_port_fd >= 0)
    {
      FD_ZERO(&r);
      FD_SET(server.tcp_port_fd, &r);
      tv.tv_sec = tv.tv_usec = 0;
      if (select(FD_SETSIZE, &r, (fd_set *)0, (fd_set *)0, &tv) < 0 && 
          errno != EINTR)
	return server.tcp_port_fd = -1;
    }
  if (server.udp_port_fd >= 0)
    {
      FD_ZERO(&r);
      FD_SET(server.udp_port_fd, &r);
      tv.tv_sec = tv.tv_usec = 0;
      if (select(FD_SETSIZE, &r, (fd_set *)0, (fd_set *)0, &tv) < 0 && 
          errno != EINTR)
	return server.udp_port_fd = -1;
    }
  return 0;
}

static int
default_prompt(p)
struct prompt *p;
{
  p->type = PT_COMMENT;
  dstring_appendf(&p->prompt_ok,  0, "%s:%d OK> ",  Hostname(), getpid(), 0, 0);
  dstring_appendf(&p->prompt_bad, 0, "%s:%d BAD> ", Hostname(), getpid(), 0, 0);
  dstring_append(&p->prompt_more, 0, "MORE> ", 0);
  return 0;
}

/*
 * Prompt for input on the connection "conn" by choosing one of the
 * prompts in "pr" based on bit-mask "why". Encoding of the prompt is due 
 * to "pr->type". Bits not interpreted here, are returned.
 *
 * FIXME: XXX: if there is still data in the inbound buffer, we should not 
 * prompt. Do we need a new bit in why therefore, or can jittr_do_prompt() 
 * decide that on its own?
 */
int
jittr_do_prompt(pr, conn, why)
struct prompt *pr;
struct conn *conn;
int why;
{
  struct dstring *prompt = pr->prompt_ok;

  if (why & CIN_VOID_MASK) 
    prompt = pr->prompt_bad;
  else if (why & CIN_MORE_MASK)
    prompt = pr->prompt_more;
 if (conn && pr->type != PT_DISABLED)
   {
     int l = conn->obuf ? conn->obuf->length - 1: -1;

     if (pr->type == PT_COMMENT && *prompt->buf != '#')
       dstring_append(&conn->obuf,  -1, "#", 1);
     dstring_append(&conn->obuf,  -1, prompt->buf, prompt->length);
     if (pr->type == PT_COMMENT)
       {
         while (++l < conn->obuf->length - 1)
	   if (conn->obuf->buf[l] == '\n')
	     conn->obuf->buf[l] = '\t';
	 if (conn->obuf->buf[l] != '\n')
           dstring_append(&conn->obuf,  -1, "\n", 1);
       }
   }
 return why & ~(CIN_VOID_MASK|CIN_MORE_MASK);
}

/*
 * DoIo() is responsible for outbound buffers being emptied and
 * inbound buffers being filled in our client structures. It also calls
 * a client notification function that should assemble inbound packets
 * and dispatch further offline processing. No real information 
 * processing should be done before DoIo() returns. DoIo() almost always passes
 * pointers to buffers around. Data is only copied when buffers must merge.
 *
 * DoIo() promises to return 0 when the the stardate given in *tv has been
 * reached.
 *
 * Thus any cl_input_notify_fn() must not use up significant processing time.
 * default_cl_Input_notify_fn() is the inbound buffer handler. It is 
 * responsible for doing something clever with the message strings from 
 * the standard clients. Clients may also come with their own input handlers.
 *
 * Gettimeval_fn() is called to fetch the current stardate.
 * DoIo() repetitively stores the current stardate in *tv.
 * Returns negative on select() error.
 *
 * jittr_make_client() and jittr_register_fd() duplicate code from DoIo(). 
 * Keep in sync!
 */
int
DoIo(tv, gettimeval_fn, default_cl_input_notify_fn)
struct timeval *tv; 
int (* gettimeval_fn) __P((struct timeval *now));
int (* default_cl_input_notify_fn) __P((void *cl, char *buf, int len));
{
  fd_set r, w;
  struct timeval when, t;
  struct conn *conn;
  struct dstring **bp;
  int more, nsel, served, fd;
  long l;

  more = 1;
  when = *tv;

  for (;;)
    {
      mkfdset(&r, &w); /* in the loop: zap_bad()/write() may change the set */
      gettimeval_fn(tv);
      /*
       * The timeout may be wrong as callbacks can schedule new events.
       * Such a clients notifier return value should be 0 (as stored in 'more')
       * In this case we return to our caller, who recalculates the timeout.
       */
      if (!more || sched_timerdiff(&t, &when, tv))
        break;

      if ((nsel = select(FD_SETSIZE, &r, &w, (fd_set *)0, &t)) < 0)
	{
	  if (errno == EIO)
	    {
	      zap_bad_fd();
	      continue;
	    }
	  if (errno == EINTR)
	    continue;
	  debug1("DoIo() emergency exit taken: select: errno %d.\n", errno);
	  return -1;
	}
      else if (!nsel)
        {
	  gettimeval_fn(tv);
          break;			/* timed out, nothing happened */
	}
      
      served = 0;

      /* output first */
      for (fd = 0; fd <= fd_client_max; fd++)
        {
	  if (!FD_ISSET(fd, &w))
	    continue;
	  ASSERT((fd_client[fd]->o && fd_client[fd]->o->fd == fd) || 
		 (fd_client[fd]->i && fd_client[fd]->i->fd == fd));
	  if (fd_client[fd]->o && fd_client[fd]->o->fd == fd)
	    bp = &fd_client[fd]->o->obuf;
	  else
	    bp = &fd_client[fd]->i->obuf;	/* odd, but still ... */
	  ASSERT(*bp);

	  if (fd_client[fd]->iotype & IOT_DONT_WRITE)
	    {
	      debug3("Discarding %d bytes for '%s' fd %d, marked DONT_WRITE\n",
	      	(*bp)->length, fd_client[fd]->name->buf, fd);
	      (*bp)->length = 0;
	    }
	  else
	    {
	      debug2("write(%d, ..., %d)\n", fd, (*bp)->length);
	      if ((l = write(fd, (*bp)->buf, (*bp)->length)) > 0)
		{
		  if (((*bp)->length -= l) > 0)
		    xbcopy((*bp)->buf + l, (*bp)->buf, (*bp)->length);
		  else if (fd_client[fd]->close_when_empty)
		    {
		      debug1("client %s is nuked with close_when_empty\n",
		      	fd_client[fd]->name->buf);
		      jittr_nuke_client(fd_client[fd], 
		    		        fd_client[fd]->close_when_empty);
		    }
		  served++;
		}
	      if (l < 0)
	        {
		  jittr_error("DoIo: write error fd %d, errno %d\n", fd, errno, 0, 0);
		  if (fd_client[fd]->o->obuf == (*bp))
		    ConnShutdown(fd_client[fd]->o);
		  else
		    ConnShutdown(fd_client[fd]->i);
                  /* XXX Fixme: or better TurnClient() == shutdown both? */
		}
	    }
	}

      /* input from the tcp socket? */
      /* 
       * Done before the shortcut, as no inbound data is expected here,
       * but clients connect attempts should be satisfied fast. If we have to
       * block them, block them while writing, not while opening.
       */
      if ((fd = server.tcp_port_fd) >= 0 && FD_ISSET(fd, &r))
        {
	  dstring *errp = NULL;

	  if ((fd = TcpPortAccept(&errp)) < 0)
	    {
	      debug3("DoIo: tcp accept fd %d failed: %s(errno %d)\n", 
		     server.tcp_port_fd, (errp ? errp->buf : ""), errno);
	    }
	  else
	    {
	      struct sockaddr_in si;
	      struct client **cl = NULL;
	      int len;

	      assert_unused_fd(fd, "DoIo after TcpPortAccept");
#ifdef DEBUG
	      if (*ClientByFd(NULL, fd))
		debug1("DoIo: ASSERT(!*ClientByFd(NULL, %d)) avoided.\n", fd);
#else
	      ASSERT(!*ClientByFd(NULL, fd));
#endif
	      len = sizeof(si);
	      if (getpeername(fd, (struct sockaddr *)&si, &len))
	        {
		  debug("DoIo: cannot getpeername after accept.");
		  continue;
		}
	    
	      /* has he visited us before? */
	      if (!*(cl = ClientByPeer(NULL, (struct sockaddr *)&si)))
		{
		  /* No, assemble a new client and give him the fd */
		  if (!((*cl) = 
		      (struct client *)calloc(1, sizeof(struct client))))
		    {
		      debug("DoIo: malloc failed, tcp accept ignored.\n");
		      shutdown(fd, 2);
		    }
		  else
		    {
		      /* Common code with jittr_make_client(). Keep in sync! */
		      (*cl)->next = NULL;
		      (*cl)->id = ++max_client_id;

		      default_prompt(&(*cl)->prompt);

		      (*cl)->i = &(*cl)->_i;
		      (*cl)->i->fd = fd;
		      (*cl)->i->type = CTYPE_TCP_OR_TELNET;
		      (*cl)->i->sa = *((struct sockaddr *)&si);
		      if (server.crypted_pw && *server.crypted_pw)
		        (*cl)->i->read_only = 1;
		      else
		        (*cl)->i->read_only =
		          CheckNetmask(server.netmask_write, &si);
		      (*cl)->i->ibuf = NULL;
		      (*cl)->i->obuf = NULL;
		      (*cl)->o = (*cl)->i;	/* have nothing better */
		      (*cl)->iotype = 0;

		      conn2dstr(&(*cl)->name, 0, (*cl)->i);
		    }
		}
	      else
		{
		  /*
		   * if we had no inbound connection, we sure did 
		   * identify him by the outbound connection. Then we
		   * have two cases to distinguish:
		   * If the outbound fildescriptor is open, 
		   * we set up a new inbound connection to the same
		   * peer. Otherwise, we make *o full duplex and assign
		   * the fildescriptor.
		   */
		  if (!(*cl)->i)
		    {
		      if (IS_CTYPE_TCP((*cl)->o->type) && (*cl)->o->fd < 0)
			{
			  (*cl)->i = (*cl)->o;
			  (*cl)->i->fd = fd;
			}
		      else
			{
			  if ((*cl)->o == &(*cl)->_i)
			    (*cl)->i = &(*cl)->_o;
			  else
			    (*cl)->i = &(*cl)->_i;
			  (*cl)->i->fd = fd;
			  (*cl)->i->type = CTYPE_TCP_OR_TELNET;
			  (*cl)->i->sa = *((struct sockaddr *)&si);
			  if (server.crypted_pw && *server.crypted_pw)
			    (*cl)->i->read_only = 1;
			  else
		            (*cl)->i->read_only =
			      CheckNetmask(server.netmask_write, &si);
			  (*cl)->i->ibuf = NULL;
			  (*cl)->i->obuf = NULL;
			}
		    }
		  else if ((*cl)->i == (*cl)->o)
		    {
		      /* 
		       * It was full duplex. If already connected, we have 
		       * to split the connections, and keep the outbound
		       * half alive.
		       */
		      if (IS_CTYPE_TCP((*cl)->o->type) && (*cl)->o->fd < 0)
			(*cl)->i->fd = fd;
		      else
			{
			  if ((*cl)->o == &(*cl)->_i)
			    (*cl)->_o = (*cl)->_i;	/* struct copy */
			  else if ((*cl)->o == &(*cl)->_o)
			    (*cl)->_i = (*cl)->_o;	/* struct copy */
			  else
			    ASSERT(0);	/* a malloced connection? */
			  (*cl)->o = &(*cl)->_o;
			  (*cl)->i = &(*cl)->_i;
			  (*cl)->i->fd = fd;
			  (*cl)->i->type = CTYPE_TCP_OR_TELNET;
			  (*cl)->i->sa = *((struct sockaddr *)&si);
			  if (server.crypted_pw && *server.crypted_pw)
			    (*cl)->i->read_only = 1;
			  else
		            (*cl)->i->read_only =
			      CheckNetmask(server.netmask_write, &si);
			  (*cl)->i->ibuf = NULL;
			  (*cl)->i->obuf = NULL;
			}
		    }
		  else
		    {
		      /* 
		       * Update/reset the inbound connection.
		       * Even if we identified him by the outbound peer,
		       * we disturb the old inbound connection, as we never
		       * know if output is currently in progress.
		       * ConnMatchPeer() may or may not help to improve
		       * this code.
		       */
		      if ((*cl)->i->type != CTYPE_INVALID && (*cl)->i->fd >= 0)
			ConnShutdown((*cl)->i);
		      (*cl)->i->fd = fd;
		      (*cl)->i->type = CTYPE_TCP_OR_TELNET;
		      (*cl)->i->sa = *((struct sockaddr *)&si);
		      if (server.crypted_pw && *server.crypted_pw)
			(*cl)->i->read_only = 1;
		      else
			(*cl)->i->read_only =
			  CheckNetmask(server.netmask_write, &si);
		      if ((*cl)->i->ibuf) (*cl)->i->ibuf->length = 0;
		      if ((*cl)->i->obuf) (*cl)->i->obuf->length = 0;
		    }
		}
	    
	      /* Greetings... nice to meet you, who are you? */
	      bp = &(*cl)->o->obuf;
	      ASSERT(DATE_CMD_ATOM == 24);
	      dstring_append(bp, -1, "id -s ", 6);
	      dstring_append(bp, -1, server.name->buf, server.name->length);
	      dstring_append(bp, -1, "; id -q; date ", 14);
	      dstring_append(bp, -1, NULL, 50);
	      gettimeval_fn(tv);
	      l = sched_format_tv((*bp)->buf + (*bp)->length, 50, tv, NULL);
	      (*bp)->length += l;

	      /* 
	       * if the client does not live on the same host as we do,
	       * ask him to perform clock synchronisation. Otherwise we already
	       * mentioned the current time to better support him, if he is
	       * human.
	       */
	      if (Str2Si("localhost",            0, NULL)->sin_addr.s_addr == 
	          Str2Si(inet_ntoa(si.sin_addr), 0, &si)->sin_addr.s_addr)
	        dstring_append(bp, -1, "\n", 1);
	      else
	        dstring_append(bp, -1, " {24 -e -s } { {24 -s }}\\n\n", 27);
	      jittr_do_prompt(&(*cl)->prompt, (*cl)->o, 0);

	      debug1("Should do for client %s: \n", (*cl)->name->buf);
	      debug("	trace stderr {get . \"6 2 \\\"\" \"\\\"\\n\"}\n");
	    }

	  if (errp)
	    {
	      dstring_append(&jittr_ioerr, -1, errp->buf, errp->length);
	      free((char *)errp);
	    }
	}
      
      if (served >= nsel)
        continue;	/* shortcut, may or may not work */

      if (served)	/* if we did write, check if our time is up */
	{
	  gettimeval_fn(tv);
	  if (sched_timerdiff(&t, &when, tv))
	    {
	      debug("DoIo: center exit taken.\n");
	      break;
	    }
	}
      
      /* input now */
      for (fd = 0; fd <= fd_client_max; fd++)
	{
	  struct client *cl;

	  /* false alert or server socket */
	  if (!FD_ISSET(fd, &r) || (cl = fd_client[fd]) == INVALID_CLIENT)
	    continue;
	  if (!cl)
	    continue;
	  /* 
	   * Careful here: A newly created client's filedescriptor may fall
	   * in the range 0..fd_client_max. But we asume that it's bit isn't
	   * yet set anywhere in the FD_SET's r or w. So he does not come
	   * here.
	   */

	  ASSERT((cl->o && cl->o->fd == fd) || (cl->i && cl->i->fd == fd));

	  if (!cl->i || cl->i->fd != fd)
	    {
	      debug1("DoIo: reading from outbound fd %d\n", fd);
	      conn = cl->o;
	    }
	  else
	    conn = cl->i;
	  ASSERT(conn);		/* both i and o ZERO? */
	  bp = &conn->ibuf;

	  if (ioctl(fd, FIONREAD, &l) || !l)	/* XXX: fixme */
	    l = ESTIMATED_PACKET_SIZE;
	  /* make sure the buffer is big enough */
	  dstring_append(bp, -1, NULL, l); 

	  if (cl->iotype & IOT_DONT_READ)
	    {
	      ASSERT(cl->input_notify_fn &&
		     cl->input_notify_fn != default_cl_input_notify_fn);
#ifdef DEBUGG
	      debug1("DoIo: '%s' has IOT_DONT_READ, notify_fn should ...\n",
		      cl->name->buf);
#endif
	    }
	  else if ((l = read(fd, (*bp)->buf + (*bp)->length, l)) < 0)
	    {
	      if (errno == EINTR || errno == EAGAIN || errno == EWOULDBLOCK)
		continue;
	      debug2("DoIo: read fd %d failed: errno %d\n", fd, errno);
	      continue;
	    }
#ifdef DEBUGG
	  debug2("DoIo: read %d bytes from fd %d\n", (int)l, fd);
#endif
	  if (l || (cl->iotype & IOT_DONT_READ))
	    {
	      int a = (cl->iotype & IOT_DONT_READ) ? 0 : l;

	      if (!cl->input_notify_fn)
		cl->input_notify_fn = default_cl_input_notify_fn;
	      (*bp)->buf[(*bp)->length += a] = '\0';
	      if (server.access_fn && server.access_fn(&server, cl,
	      			      (*bp)->buf + (*bp)->length - a, l))
		l = 0;	/* flag destruction */
	      else
		more = jittr_do_prompt(&cl->prompt, conn,
			      cl->input_notify_fn((void *)cl, 
				      (*bp)->buf + (*bp)->length - a, l));
	    }
	  /* 
	   * When read returns 0 it causes the file to be closed.
	   * The notify handler is called to give it a second chance 
	   */
	  if ((!l && !(cl->iotype & IOT_DONT_READ)) || (more & CIN_CLOSE_MASK))
	    {
	      /* what shall we do with a drunken sailor? */
	      if (cl->o && cl->o != conn)
		{
		  dstring_append(&cl->o->obuf, -1, "# input connection ", 0);
		  dstring_append(&cl->o->obuf, -1, cl->name->buf, 0);
		  dstring_append(&cl->o->obuf, -1, " closed.\n", 0);
		}
	      
	      /* give cl->input_notify_fn() a chance to destruct cl */
	      if (!cl->input_notify_fn ||
	          cl->input_notify_fn == default_cl_input_notify_fn ||
		  !cl->input_notify_fn((void *)cl, NULL, 0))
	        ConnShutdown(conn);
	    }
	  more = !(more & CIN_SCHED_MASK);
	  served++;
	}
      
      /* input from the udp socket? */
      if ((fd = server.udp_port_fd) >= 0 && FD_ISSET(fd, &r))
        {
	  static struct dstring *buf;
	  struct sockaddr_in si;
	  struct client **cl;

          ASSERT(!buf || !buf->length);
	  if (ioctl(fd, FIONREAD, &l))
	    l = ESTIMATED_PACKET_SIZE;
	  /* make sure the buffer is big enough */
	  dstring_append(&buf, 0, NULL, l); 
	  if ((l = UdpRecv(buf->buf, l, &si, &jittr_ioerr)) < 0)
	    {
#ifdef DEBUG
	      if (errno != EINTR && errno != EWOULDBLOCK)
	        debug2("DoIo: read udp fd %d failed: errno %d\n", fd, errno);
#endif
	    }
	  else if (l > 0)
	    {
	      buf->buf[buf->length = l] = '\0';

	      /* Is the sender already known? */
	      if (!*(cl = ClientByPeer(NULL, (struct sockaddr *)&si)))
	        {
		  /* No, assemble a new client and give him the buffer */
		  if (!((*cl) = 
		      (struct client *)calloc(1, sizeof(struct client))))
		    {
		      debug("DoIo: malloc failed, udp packet discarded.\n");
		      buf->length = 0;	
		      break;
		    }
		
		  /* Common code with jittr_make_client(). Keep in sync! */
		  (*cl)->next = NULL;
		  (*cl)->id = ++max_client_id;

		  (*cl)->prompt.type = PT_DISABLED;

		  (*cl)->i = &(*cl)->_i;
		  (*cl)->i->fd = -1;
		  (*cl)->i->type = CTYPE_UDP;
		  (*cl)->i->sa = *(struct sockaddr *)&si;
		  if (server.crypted_pw && *server.crypted_pw)
		    (*cl)->i->read_only = 1;
		  else
		    (*cl)->i->read_only =
		      CheckNetmask(server.netmask_write, &si);
		  (*cl)->i->obuf = NULL;
		  (*cl)->i->ibuf = buf;
		  (*cl)->iotype = 0;
		  (*cl)->o = (*cl)->i;	/* we may want to reply */
		  buf = NULL;

		  conn2dstr(&(*cl)->name, 0, (*cl)->i);
		  bp = &(*cl)->i->ibuf;
		}
	      else
	        {
                  /* 
		   * Client known. Put the data in clients inbound buffer.
		   * If clients buffer was empty, just swap buffers.
		   * Otherwise append to the buffer.
		   */

		  /*
		   * see if we identified him by an outbound peer. If so, we 
		   * know that it is full duplex, as he started talking
		   */
		  if (!(*cl)->i)
		    (*cl)->i = (*cl)->o;

		  bp = &(*cl)->i->ibuf;

		  if (!dstring_length(*bp))
		    {
		      struct dstring *swp = *bp;

		      *bp = buf;
		      buf = swp;
		    }
		  else
		    {
		      dstring_append(bp, -1, buf->buf, l);
		      buf->length = 0;
		    }
		}
	      if (!(*cl)->input_notify_fn)
	        (*cl)->input_notify_fn = default_cl_input_notify_fn;
	      more = jittr_do_prompt(&(*cl)->prompt, (*cl)->o, 
			(*cl)->input_notify_fn((void *)*cl, 
				(*bp)->buf + (*bp)->length - l, l));
	      more = !(more & CIN_SCHED_MASK);
	      served++;
	    }
	}
#ifndef GREEDY_INTERPRETER
      if (served)
        break;
#endif
    }
  return 0;
}

/*
 * a new filedescriptor enters the jittr world. 
 * The caller must arrange storage space and client pointer for the connection.
 *
 * DoIo() duplicates this code. Keep in sync!
 */
int
jittr_register_fd(fd, cl, conn)
int fd;
struct client *cl;
struct conn *conn;
{
  ASSERT(!*ClientByFd(NULL, fd));

  conn->fd = fd;
  conn->type = CTYPE_FD;  /* the client assertion must be above this line */
  conn->ibuf = NULL;
  conn->obuf = NULL;

  if (fd < 0)
    return conn->type = CTYPE_INVALID;	/* in case it smells (or if UDP) */

  assert_unused_fd(fd, "jittr_make_client: jittr_register_fd");

  /*
   * fd_client[] is updated again by mkfdset(), but the initial prompt
   * wants to know where to go, before DoIo() starts rotating.
   */
  fd_client[fd] = cl;
  return 0;
}

/*
 * jittr_make_client: Create a client with a given name and bidirectional
 * filedescriptor.  If name is NULL, a name is generated based  on
 * Hostname(), getpid() and the filedescriptor. Comment prompting with
 * some stupid default prompts (based on the name) is enabled.  If you do
 * not like this, or want that a seperate output fildescriptor shall be used,
 * just poke into the returned structure (jittr_register_fd() helps here).
 *
 * Caution: DoIo() duplicates this code in two places. Keep in sync!
 *
 * To prepare a udp client come here with a negative fd.
 */
struct client *
jittr_make_client(name, fd)
char *name;
int fd;
{
  struct client **cl;

  if (*(cl = ClientByFd(NULL, fd)))
    {
      dstring_appendf(&jittr_ioerr, -1, "jittr_make_client %s fd=%d already open as %s\n", name ? name : "<NULL>", fd, (*cl)->name->buf, 0);
      return 0;
    }

  /* Now assemble a new client and give him the fd */
  if (!((*cl) = (struct client *)calloc(1, sizeof(struct client))))
    {
      debug1("jittr_make_client: malloc failed, fd %d ignored.\n", fd);
      dstring_appendf(&jittr_ioerr, -1, "jittr_make_client: malloc failed, %s fd=%d ignored.\n", name ? name : "<NULL>", fd, 0, 0);
      return NULL;
    }

  (*cl)->next = NULL;
  (*cl)->id = ++max_client_id;

  if (name)
    dstring_append(&(*cl)->name, 0, name, 0);
  else
    dstring_appendf(&(*cl)->name, 0, "fd%d:%s:pid%d",
      fd, Hostname(), getpid(), 0);

  /* Assume fd is open RDWR */
  (*cl)->o = &(*cl)->_i;
  jittr_register_fd(fd, (*cl), (*cl)->i = &(*cl)->_i);
  (*cl)->iotype = 0;	/* DoIo() shall read and write */

  default_prompt(&(*cl)->prompt);
  return *cl;
}

/*
 * lookup or create a client by name or by peer that it connects to.
 * The peer or name is described by the string s.
 * Returns NULL when in trouble.
 *
 * Caution: you cannot use the obuf of the returned client, if the client is
 * your own connection and you are called by jittr_process.
 *
 * InetClient() can be forced to lookup/create a CTYPE_UDP or a CTYPE_TCP
 * connection by specifying one of these flags as a third parameter. If you
 * don't care, specify 0.
 *
 * Peer names are parsed like this:
 * 		[[UDP:]hostname]:[port]
 */
struct client *
InetClient(name, create_timeout, ctype)
char *name;
int create_timeout, ctype;
{
  struct client *c;
  struct dstring *errp = NULL;
  char *s = name;

  if (!s)
    return NULL;

  if (!(c = *ClientByName(NULL, s)))
    {
      int ch;
      char *pp = NULL;
      int udp = 0, len, fd = -1, p = ParsePeer(s, JITTR_PORT, &pp);
      struct sockaddr_in si;

      if (!strncasecmp("tcp:", s, 4))
        if (ctype == CTYPE_UDP)
	  return NULL;
	else
	  s += 4;
      if (!strncasecmp("udp:", s, 4))
        if (ctype == CTYPE_TCP)
	  return NULL;
	else
	  {
	    udp++;
	    s += 4;
	  }
      
      if (pp)
        { 
	  ch = *pp;
	  *pp = '\0';
	}
      if (!(c = *ClientByPeer(NULL, (struct sockaddr *)Str2Si(s, p, &si))))
	{
	  if (!create_timeout)
	    return NULL;

	  if (!udp)
	    {
	      debug2("InetClient: connecting to host=%s, port=%d ...\n", s, p);
	      if ((fd = TcpPortConnect(s, p, create_timeout, &errp)) < 0)
		{
		  dstring_appendf(&jittr_ioerr, -1, "TcpClient: Connect to %s,%d: %s.\n", s, p, errp ? errp->buf : "haeh?", 0);
		  return NULL;
		}

	      len = sizeof(si);
	      if (getpeername(fd, (struct sockaddr *)&si, &len))
	        return NULL;
	    }

	  if (pp)
	    *pp = ch;

	  if (!(c = jittr_make_client(name, fd)))
	    return NULL;

	  c->i->type = udp ? CTYPE_UDP : CTYPE_TCP;
	  c->i->sa = *((struct sockaddr *)&si);
	  if (server.crypted_pw && *server.crypted_pw)
	    c->i->read_only = 1;
	  else
	    c->i->read_only = CheckNetmask(server.netmask_write, &si);
	  c->prompt.type = PT_DISABLED;
	  conn2dstr(&c->name, 0, c->i);
	}
    }
#ifdef DEBUG
  else
    {
      /* be paranoid: check if name is uniq */
      if (c->next && *ClientByName(&c->next, s))
	{
	  debug("TcpClient: client name is not uniq, oldest one chosen.");
	}
    }
#endif

  return c;
}

/*
 * TurnClient() checks if a client is a zombie. That means, it is not 
 * connected. Note that conn structures may exist that are not connected.
 * E.g. CTYPE_UDP. If force is nonzero, existing connections are closed.
 *
 * If the client is connected, -1 is returned and nothing happened.
 * Otherwise the client is assigned a negative id and is unlinked from the
 * clients list. Pointers to such a client should be detected and removed.
 *
 * XXX FIXME: We cannot even free client structures, because there are pointers
 * everywhere and there is no refcount.  We simply form a chained list of the
 * turned clients.
 */
int
TurnClient(cl, force)
struct client **cl;
int force;
{
  struct client *c = *cl;

  if (!c)
    return 0;
  
  if (force)
    {
      ConnShutdown(c->i);
      ConnShutdown(c->o);
    }
  else if ((c->i && c->i->type != CTYPE_INVALID && c->i->fd >= 0) ||
           (c->o && c->o->type != CTYPE_INVALID && c->o->fd >= 0))
    {
      debug1("TurnClient: Cannot turn %s: he is still alive\n", c->name->buf);
      return -1;	
    }
  c->id = -1 - c->id;

  debug1("TurnClient: old %s is fleeing.\n", c->name->buf);
    
  (*cl) = c->next;
  c->next = zombies;
  zombies = c;
  /* He is dead, Jim */
  return 0;
}
