/*
 * jittr - Just In Time TRansport library
 *
 * dstring.c -- a string class, that manages its own memory allocation.
 *
 * 3.3.95 jw.
 */
#include "jittr/dstring.h"	/* struct dstring, also ASSERT() from debug.h */

/*
 * Store a string of characters in a dstring structure whose pointer is found 
 * at *dsp.
 * To use binary strings, you should specify a length, if length is 0, 
 * strlen is called. A NULL string is always of lenght 0, regardless what
 * the length argument claims. 
 * Returns -1, if malloc fails, else the length of the entire new dstring.
 * If offset is 0, the previously contents of dsp is completely
 * overridden. If offset is positive, that many characters are
 * preserved at the beginning of the old string, before the new strign starts.
 * If offset is negative, one less characters than its absolute value
 * will be overwritten at the end of the old string. If offset is -1, 
 * the new string is appended without any overlap.
 *
 * if *dsp is NULL, it is treated exactly like a dstring of length 0.
 *
 * A special case usage is a NULL string with a positive length argument:
 * This checks and makes sure that there is enough space allocated
 * to hold offset + length + 1 bytes. Length remains unchanged, buffer contents
 * too, except that the following condition is assured.
 * 
 * There is always one unused byte at the end of the buffer. Thus the 
 * assignement  (*dsp)->buf[(*dsp)->length] = '\0'; is a safe way to 
 * create a NULL-terminated string. This is already done by dstring_append.
 */
int
dstring_append(dsp, offset, string, length)
dstring **dsp;
int offset, length;
char *string;
{
  int l;

  if (!string && !length)
    string = "";		/* will be a truncate */
  if (!length)
    length = strlen(string);
  ASSERT(dsp);
  l = dstring_length(*dsp);
  ASSERT(!(*dsp) || (l <= (*dsp)->allocated));	/* dstring corrupt */
  if (offset < 0)
    offset += ++l;
  ASSERT(offset >= 0 && offset <= l);

  if (!*dsp || offset + length > (*dsp)->allocated)
    {
      dstring *new;
      int i;

      i = offset + length + sizeof(dstring);
      i = (((i - 1) | 0x7f) + 1);	/* round up to 64 */
      if (!(new = (dstring *)calloc(i, 1)))
        return -1;
      new->allocated = i - sizeof(dstring);
      new->length = 0;
      if (*dsp)
        {
	  /* 
	   * if data points inside the own buffer, let it point to the 
	   * same place in the new buffer.  
	   */
	  if ((char *)(*dsp)->data - (*dsp)->buf >= 0 && 
	      (char *)(*dsp)->data - (*dsp)->buf <= (*dsp)->length)
	    new->data = (void *)((char *)(*dsp)->data - (*dsp)->buf + new->buf);
	  else
	    new->data = (*dsp)->data;
          if ((new->length = offset))
	    xbcopy((*dsp)->buf, new->buf, offset);
	  free((char *)(*dsp));
	}
      *dsp = new;
    }
  ASSERT(*dsp && offset + length <= (*dsp)->allocated);
  if (string)
    {
      xbcopy(string, (*dsp)->buf + offset, length);
      (*dsp)->length = offset + length;
    }
  (*dsp)->buf[(*dsp)->length] = '\0';
  return (*dsp)->length;
}

/*
 * Write text with backslash quoting. If parse_word() interpolates the result
 * within double quotes, it will return exactly the original contents of s.
 * This Method is used for shipping arbitrary binary data through the net.
 * See parse_needsq() if you are not sure when to call dstring_appendq().
 */
int 
dstring_appendq(d, off, s, l)
struct dstring **d;
int off;
char *s;
int l;
{
  int i, n = 0;		/* gcc shut up */
  unsigned char *p, buf[6]; 
  extern char *index();

  if (!s || !l)
    return dstring_append(d, off, s, l);
  if (off != -1)
    n = dstring_append(d, off, "", 0);
 
  xbcopy("\\ \\x", buf, 4); 
  for (i = 0; i < l; i++)
    {
      *(p = buf+1) = s[i];
      n = 1;

      if (*p == '"' || *p == '\\') 
        {
	  p = buf;
	  n = 2;
	}
      else if  (*p < ' ' || (*p >= 127 && !index("", *p)))
        {
	  buf[4] = "0123456789abcdef"[*p >> 4];
	  buf[5] = "0123456789abcdef"[*p & 15];
	  p = buf + 2;
	  n = 4;
	}
      n = dstring_append(d, -1, (char *)p, n);
    }
  return n;
}

/* 
 * dstring_appendn() is ideal for printing numeric values.
 * It is faster than dstring_appendf() below, because it avoids a 
 * temproary buffer. You may also use it for printing strings with
 * less than 12 characters.
 *
 * If pat is NULL, it defaults to "%d".
 *
 * CAUTION: Be careful with %c printing. '\0' bytes won't work.
 *	Do not use with double arguments. Try dstring_appendg then.
 */
int 
dstring_appendn(d, off, pat, n)
struct dstring **d;
int off;
char *pat;
int n;
{
  int l;

  if (!pat) pat = "%d";

#ifdef DEBUG
  if (strstr(pat, "%c") && !n)
    debug2("WARNING: dstring_appendn(d, %d, \"%s\", 0) will not work.\n", off, pat);
#endif

  if ((l = dstring_append(d, off, NULL, strlen(pat) + 10)) < 0)
    return l;
  if (off < 0) off += ++l;
  sprintf((*d)->buf + off, pat, n);
  return (*d)->length = off + strlen((*d)->buf + off);
}

/* 
 * dstring_appendg() is for printing numeric floating point values.
 * It is faster than dstring_appendf() below, because it avoids a 
 * temproary buffer. You may also use it for printing strings with
 * less than 12 characters.
 *
 * If pat is NULL, it defaults to "%g".
 *
 * CAUTION: Be careful with %c printing. '\0' bytes won't work.
 */
int 
dstring_appendg(d, off, pat, f)
struct dstring **d;
int off;
char *pat;
double f;
{
  int l;

  if (!pat) pat = "%g";

#ifdef DEBUG
  if (strstr(pat, "%c") && !f)
    debug2("WARNING: dstring_appendg(d, %d, \"%s\", 0) will not work.\n", off, pat);
#endif

  if ((l = dstring_append(d, off, NULL, strlen(pat) + 20)) < 0)
    return l;
  if (off < 0) off += ++l;
  sprintf((*d)->buf + off, pat, f);
  return (*d)->length = off + strlen((*d)->buf + off);
}

/*
 * this function returns an over-estimate of a printf format string with 
 * optionally one argument. *fmtp is advanced to the '\0'-byte or to the
 * next % sequence.
 */
#if 1
# define fmt_size(fmtp, argp) 1024
#else
static int fmt_size(fmtp, argp)
char **fmtp;
void **argp;
{
  int n;
  char *fmt = *fmtp;

  if (!fmt)
    return 0;

  /* advance to the % character */
  while (*fmt)
    {
      if (*fmt == '%')
        {
	  switch (*(++fmt))
	    {
	    case '\0':
	      *fmtp = fmt;
	      return n;
	    case '-': 	/* flags */
	    case '+':
	    case ' ':
	    case '#':
	    case '0':

	      *fmtp = fmt;
	      return n;
	    case '%':	/* FALLTHROUGH */
	    default:
	      fmt++;
	      n++;
	      continue;
	    }
	}
      fmt++;
      n++;
    }
  *fmtp = fmt;
  return n;
}
#endif

/* 
 * A primitive printf like dstring_append. 
 * Can handle up to 4 arguments and any number of bytes at once.
 * Beware: no prototype and first arg is a pointer-pointer.
 * Beware: no varargs! pad with 0 to keep purify silent!
 */
int
dstring_appendf(dsp, off, fmt, arg1, arg2, arg3, arg4)
dstring **dsp;
int off;
char *fmt, *arg1, *arg2, *arg3, *arg4;
{
  int l, n = 0;
  char *f = fmt;

  n += fmt_size(&f, &(void *)arg1);
  n += fmt_size(&f, &(void *)arg2);
  n += fmt_size(&f, &(void *)arg3);
  n += fmt_size(&f, &(void *)arg4);

  l = dstring_append(dsp, off, NULL, n);
  if (off < 0) off += ++l;
  sprintf((*dsp)->buf + off, fmt, arg1, arg2, arg3, arg4);
  return (*dsp)->length = off + strlen((*dsp)->buf + off);
}
