/*
 * gettime.c --
 * system independant interface a la gettimeofday.
 *
 * this module exports:
 * 	int gettimeval(struct timeval *)
 * 	int gettimezone(struct timezone *)
 *
 * jw. 9.3.95
 *
 * XXX: it should also export
 *	time_t timegmt(struct tm *);
 * which is the reverse of the wellknown gmtime(), which is currently inlined
 * in jlib/schedule.c:sched_parse_tv().  Sun and linux have timegm() which is
 * exactly what timegmt() shall do; on other systems timegmt() can be
 * implemented by adjusting the return value of mktime().
 */
/* #define STANDALONE */

/* ======================= portablility definitions ================== */
#if defined(_SEQUENT_)
# define NEED_USCLKC
# define HAVE_GETUSCLK
#endif

#if defined(sun) || defined(__osf__) || defined(sgi) || defined(linux) || defined(m68k) || defined(m88k) || defined(_IBMR2) || defined(__hpux) || defined(__NetBSD__) || defined(__bsdi__) || defined(__ksr__)
# define NEED_SYS_TIME
# define HAVE_GETTIMEOFDAY
#endif

#if defined(sun) && defined(__GNUC__)
/* they do not declare the beast anywhere: */
int gettimeofday();
#endif

#if defined(_SEQUENT_) || defined(sgi) || defined(sun) || defined(m88k) || defined(m68k)
# define NEED_SYS_TYPES
# define NEED_SYS_TIMES
# define NEED_SYS_PARAM
# define NEED_SYS_TIME
# define HAVE_TIMES
# define HAVE_TIME
#endif

#if defined(sgi)
# define NEED_SYS_TIME
# define HAVE_GETITIMER
#endif

/* ================== end of port defs. Beware: Algorithms below ==== */

#ifdef NEED_SYS_TYPES
# include <sys/types.h>		/* struct timeval */
#endif
#ifdef NEED_SYS_TIMES
# include <sys/times.h>		/* struct tms */
#endif
#ifdef NEED_SYS_PARAM
# include <sys/param.h>		/* HZ */
#endif
#ifdef NEED_LIMITS
# include <limits.h>		/* CLK_TCK */
#endif
#ifdef NEED_USCLKC
# include <usclkc.h>		/* GETUSCLK() */
#endif
#ifdef NEED_SYS_TIME
# include <sys/time.h>		/* struct timeval */
#endif
 
#ifdef HAVE_GETTIMEOFDAY
# ifndef HAVE_GETTIMEVAL
#  define HAVE_GETTIMEVAL
int gettimeval(tp)
struct timeval *tp;
{
  struct timezone tz;
  return gettimeofday(tp, &tz);	/* localtime */
}
# endif /* HAVE_GETTIMEVAL */

# ifndef HAVE_GETTIMEZONE
#  define HAVE_GETTIMEZONE
int gettimezone(tzp)
struct timezone *tzp;
{
  struct timeval tv;
  return gettimeofday(&tv, tzp);
}
# endif /* HAVE_GETTIMEZONE */
#endif /* HAVE_GETTIMEOFDAY */

#if defined(HAVE_TIMES) && defined(HAVE_TIME)
# ifndef HAVE_GETTIMEVAL
#  define HAVE_GETTIMEVAL
/*
 * This implementation may be late up to one second, but the offset
 * is constant between multiple calls. 
 * The usec value is accurate to 1/HZ seconds.
 */
int gettimeval(tp)
struct timeval *tp;
{
  static time_t tsec = 0;
  static clock_t ticks = 0;
  clock_t c;		
  struct tms dummy;

  c = times(&dummy);	/* Clock ticks since system start.
  			 * With 100 Hz and a signed long, this will wrap 
			 * after 15 years of uptime. We accept that.
			 */
  
  if (!tsec)
    {
      time(&tsec);		/* UTC */
      ticks = c;
    }
  
  c -= ticks;
  tp->tv_sec = c / HZ;
  tp->tv_usec = c - HZ * tp->tv_sec;
  tp->tv_usec *= (1000000/HZ);
  tp->tv_sec += tsec;
  return 0;
}
# endif /* HAVE_GETTIMEVAL */
#endif /* TIME && TIMES */
 
#ifdef HAVE_GETITIMER
# ifndef HAVE_GETTIMEVAL
#  define HAVE_GETTIMEVAL
/*
 * Please don't use this function unless you know what's happening:
 * this messes up alarm(), SIGALRM and other use of setitimer(0, ...)
 */
int gettimeval(tp)
struct timeval *tp;
{
  static int started = 0;
  struct itimerval it;
  int r;
  
  if (!started)
    {
      if (setitimer(2, &it, NULL))
        perror("setitimer");
    }
  if ((r = getitimer(2, &it)))
    return r;

  tp->tv_sec = it.it_interval.tv_sec;
  tp->tv_usec = it.it_interval.tv_usec;
  return 0;
}
# endif /* HAVE_GETTIMEVAL */
#endif /* HAVE_GETITIMER */

#ifdef HAVE_GETUSCLK
# ifndef HAVE_GETTIMEVAL
/* #define HAVE_GETTIMEVAL */
/* Sequent special hardware microsecond clock: to be implemented */
# endif /* HAVE_GETTIMEVAL */
#endif /* HAVE_GETUSCLK */

#ifndef HAVE_GETTIMEVAL
int gettimeval() { return -1; }
#endif
#ifndef HAVE_GETTIMEZONE
int gettimezone() { return -1; }
#endif

/* ==================== End of algorithms. Selftest below ============== */

#ifdef STANDALONE
int
main(ac, av)
int ac;
char **av;
{
  struct timeval tp;
  int i, j;

#ifdef INT_TSTAMP
  if (gettimeval(&tp))
    return 1;
  j = tp.tv_usec / 1000;
  i = 1000 * tp.tv_sec + j;
#ifdef __ksr__
  i = (i & ~15) | (getpid() & 15);
#endif
  printf("%d\n", i & ~(1 << (sizeof(int) * 8 - 1)));
#else
  i = gettimeval(&tp); printf("%d: %d s, %d usecs\n", i, tp.tv_sec, tp.tv_usec);
  sleep(1);
  for (j = 0; j < 10; j++)
    {
      i = gettimeval(&tp); printf("%d: %d s, %d usecs\n", i, tp.tv_sec, tp.tv_usec);
      for(i = 0; i < 200000; i++) sleep(0);
    }
#endif
  return 0;
}
#endif /* standalone */
