/*
 * jcmd.c -- jittr standard commands library (Core Methods)
 *
 * atom.c has some command primitives that do not need to know
 * what clients are. Here are the commands that do.
 *
 * 13.1.96 jw.
 */
#include <unistd.h>		/* getpid() */
#include <sys/types.h>  	/* for sys/socket.h */
#include <sys/socket.h> 	/* struct sockaddr for net.h */
#include <sys/time.h>   	/* struct timeval for net.h */
#include "jittr/atom.h"		/* atoms */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"	/* struct exec_context for eval_cmd(), etc. */
#include "jittr/net.h"		/* struct client */
#include "jittr/doio.h"		/* TcpClient() */
#include "jittr/schedule.h"	/* struct sched_action for late_stdio_trace */

static int            tell_cmd __P((int idx, int flag, void *dummy));
static int            eval_cmd __P((int idx, int flag, void *context));
static int           trace_cmd __P((int idx, int flag, void *dummy));
static int         ext_get_cmd __P((int idx, int flag, void *dummy));
static int         ext_set_cmd __P((int idx, int flag, void *append));
static int    late_stdio_trace __P((int idx, int flag, void *fp));
static int late_printf_process __P((void **av));

/*
 * tell client command-text [suffix]
 *
 * suffix defaults to "\n", which is what I usually forget.
 * tell is a convenicence function, equivalent to a forced redirection like
 *
 * 	>!client append 1 command-text\n
 *
 * tell can also be used to send udp packets, if the client name is an address
 * of the following form:
 *
 *	tell UDP:faui42:7447 {set /jsatd /nokia_sat/off}\n
 */
static int
tell_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  int slen;
  char *s = jarg_first_word(&slen);
  struct client *c = InetClient(s, 1, 0);

  if (!c || !c->o) 
    return jittr_error("%s: cannot talk to TCP client %s: %s.\n", jarg_name(),
      s ? s : "(Null)", c ? "no outbound connection" : " no such client", 0);

  if (exec_context && exec_context->who == c)
    return jittr_error("%s: phone yourself (%s)? Please try 'append stdout'.\n",
    	jarg_name(), c->name->buf, 0, 0);
    
  s = jarg_next_word(&slen); dstring_append(&c->o->obuf,-1, s, slen);
  s = jarg_next_word(&slen); dstring_append(&c->o->obuf,-1, s?s:"\n", s?slen:1);

  return 1;
}

/*
 * chmod_cmd implements the rename and chmod atoms.
 * This command is not available for readonly clients.
 */
static int
chmod_cmd(idx, flag, rename)
int idx, flag;
void *rename;
{
  int slen;
  char *s = jarg_first_word(&slen);
  struct client *who = NULL;

  if (!s)
    return jittr_error("%s: atom name missing.", jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    {
      /* it is the name of an atom that does not exist */ 
      if ((idx = atom_new(idx == -1 ? idx : atoi(s), 
      			  idx == -1 ? s : NULL, NULL, 0)) < 0)
        return jittr_error("%s: failed to create '%s'.", jarg_name(), s, 0, 0);
    }

  if (exec_context)
    who = exec_context->who;
  if (who && who->i && who->i->read_only)
    return jittr_error("%s: -name ignored, %s is read_only\n",
      jarg_name(), who ? who->name->buf : "client", 0, 0);
  
  s = "-name"; slen = 5;
  if (!rename)
    s = jarg_next_word(&slen);

  while (slen > 1 && *s == '-')  /* s == NULL implies slen == 0 */
    {

      if (!strncmp("-name", s, slen))
        {
	  if (!(s = jarg_next_word(&slen)))
	    return jittr_error("%s: -name parameter missing.",
	    	jarg_name(), 0, 0, 0);
	  dstring_append(&atoms[idx]->name, 0, s, slen);
	  return 1;
	}
      else if (!strncmp("-readonly", s, slen))
        atoms[idx]->a.flag |= A_READONLY;
      else if (!strncmp("-permanent", s, slen))
        atoms[idx]->a.flag |= A_PERMANENT;
      else if (!strncmp("-noreadonly", s, slen))
        atoms[idx]->a.flag &= ~A_READONLY;
      else if (!strncmp("-nopermanent", s, slen))
        atoms[idx]->a.flag &= ~A_PERMANENT;
      else 
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      if (!(s = jarg_next_word(&slen)))
        return 1;
    }
    
  return 0;
}

/* 
 * EXEC: write value into some other atom. 
 * Can also create/rename atoms.
 * This overloads the set_cmd from atom.c, because atom.c does not know
 * anything about the jittr command syntax. If you change anything 
 * check both functions!
 * Usually (i.e. from within the interpreter) the jittr core commands are
 * protected against overloading.  See jittr_jcmd_init() for the trick.
 *
 * The difference is this:
 *
 * 1) Here we record the command timestamp and executing client in the atoms
 *    data field.
 * 2) We filter out set commands that are older or same age than the atom 
 *    contents. In that case callbacks are triggered with unchanged value.
 * 3) If the client has its read_only flag set, neither name, value nor 
 *    timestamp can be modified.
 * 4) set -name has been transferred to the rename command, set -re|-perm
 *    have been transferred to the chmod command, to make set argument parsing 
 *    faster in the mainstream case.
 */
static int
ext_set_cmd(idx, flag, append)
int idx, flag;
void *append;
{
  int slen;
  char *s;
  int vlen;
  char *v;
  struct timeval tv;
  struct client *who = NULL;

  if (!(s = jarg_first_word(&slen)))
    return jittr_error("usage: %s atom value\n", jarg_name(), 0, 0, 0);

  if (!(v = jarg_next_word(&vlen)))
    return jittr_error("%s: value for '%s' missing.\n", jarg_name(), s, 0, 0);

  if ((idx = atom_lookup2(s, slen)) < 0)
    {
      /* it is the name of an atom that does not exist */ 
      if ((idx = atom_new2(idx == -1 ? idx : atoi(s), 
      			   idx == -1 ? s : NULL,
			   idx == -1 ? slen : 0,
			   NULL, 0, 0)) < 0)
        return jittr_error("%s: failed to create '%s'.", jarg_name(), s, 0, 0);
    }
  
    
  jittr_gettimeval(&tv);

  if (exec_context)
    {
      who = exec_context->who;
      if (exec_context->when)
        tv = *exec_context->when;	/* structure copy */
    }
  
  return jittr_atom_set(idx, (int)append, v, vlen, &tv, who, 1);
}

/*
 * This is the (hidden) key for network aware atom modification.
 * The atom idx recieves the text s. The text's timestamp is tv.
 * The text's origin is who. tv is vitally important. The atom's timestamp is
 * compared against tv. If the atom is already younger, the text is discarded.
 *
 * If a client's read_only flag is set, the effect is just as if the message's
 * tv were a "very old" timestamp.
 *
 * Note: If the atom is older and the text is identical, trace callbacks must
 * be triggered. This implements the unary datatype.
 *
 *           Message      ||        Atom            
 *      Value   |Timestamp||  Value  |Timestamp|   Action
 *    ----------+---------++---------+---------+--------------
 *      same    |  older  ||  keep   |  keep   |    none
 *      same    |  equal  ||  keep   |  keep   |    none
 *      same    |  newer  ||  keep   | update  | if noisy call traces
 *     differs  |  older  ||  keep   |  keep   | call traces
 *     differs  |  equal  || update  | update  | call traces
 *     differs  |  newer  || update  | update  | call traces
 *
 * Negative slen means: use strlen.
 */
int
jittr_atom_set(idx, append, s, slen, tv, who, noisy)
int idx, append;
char *s;
int slen;
struct timeval *tv;
struct client *who;
int noisy;
{
  struct atom_data *a = jittr_atom_data(idx);
  struct dstring *v = atoms[idx]->value;
#ifdef DEBUG
  static int debug_jittr_atom_set = 0;	/* enable this with a debugger */
#endif

  if (slen < 0)
    slen = strlen(s);

  if (who && who->i && who->i->read_only)
    {
      ASSERT(APP_CMD_ATOM == 6);
      ASSERT( STDERR_ATOM == 2);
      atom_append(REPLY_ATOM, -1, "6 2 ", 4);
      atom_append(REPLY_ATOM, -1, server.name->buf, server.name->length);
      atom_append(REPLY_ATOM, -1, ": readonly.\n\r", 13);
    }

  if ((append && !slen) ||
      (!append && v && (slen == v->length) && !xbcmp(v->buf, s, slen)))
    {
      if ((!who || !who->i->read_only) &&
          ((tv->tv_sec > a->sec) ||
           ((tv->tv_sec == a->sec) && (tv->tv_usec > a->usec))))
	{
#ifdef DEBUG
	  if (debug_jittr_atom_set)
	    debug2("jittr_atom_set: %d same value, new timestamp: update%s.\n", 
	      idx, noisy ? " and shout" : " timestamp silently");
#endif
	  a->who = who;
	  a->sec  = tv->tv_sec;
	  a->usec = tv->tv_usec;
	  if (noisy)
	    atom_append(idx, -1, NULL, 0);
	  return 1;
	}
#ifdef DEBUG
      if (debug_jittr_atom_set)
        {
	  debug1("jittr_atom_set: client->read_only=%d\n", 
	    (who && who->i) ? who->i->read_only : -1);
	  debug1("jittr_atom_set: %d same old value: ignored\n", idx);
	}
#endif
      return 1;
    }

  if ((who && who->i && who->i->read_only) ||
      (tv->tv_sec < a->sec) || 
      (a->sec && (tv->tv_sec == a->sec) && (tv->tv_usec < a->usec)))
    {
#ifdef DEBUG
      if (debug_jittr_atom_set)
	{
	  debug1("jittr_atom_set: client->read_only=%d\n", 
	    (who && who->i) ? who->i->read_only : 0);
	  debug3("jittr_atom_set: late command discarded: %s %s '%s'\n",
	    jarg_name(),  atom_name(idx), s);
	  debug2("jittr_atom_set:    atom time: %d.%06d\n", 
	    (int)a->sec, (int)a->usec);
	  debug2("jittr_atom_set: command time: %d.%06d\n", 
	    (int)tv->tv_sec, (int)tv->tv_usec);
	}
#endif

      /* 
       * Trigger callbacks. This is needed, to tell the originator of the
       * outdated set command that he is being ignored, and he should switch
       * back to the previous value. Same thing applies to read_only clients.
       */
      if (who && who->i && who->i->read_only)
        {
	  /* do not trigger callbacks for readonly clients */
	  return 1;
	}
      atom_append(idx, -1, NULL, 0);
      return 1;
    }

#ifdef DEBUG
  if (debug_jittr_atom_set && (tv->tv_sec==a->sec) && (tv->tv_usec==a->usec))
    debug3("jittr_atom_set: %d: same time, different value: '%s' was '%s'\n", 
      idx, s, v ? v->buf : "");
#endif

  a->who = who;
  a->sec  = tv->tv_sec;
  a->usec = tv->tv_usec;
  atom_append(idx, append ? -1 : 0, s, slen);
  return 1;
}

/*
 * Cause write callbacks on the atom idx, without changing value or timestamp.
 * This is useful for propagating an old setting within the client. 
 */
int
jittr_atom_tickle(idx)
int idx;
{
  struct atom_data *a = jittr_atom_data(idx);
  struct timeval tv;

  tv.tv_sec  = a->sec;
  tv.tv_usec = a->usec;		
  a->usec--;		/* fake its own timestamp older */
  debug2("jittr_atom_tickle(%d): broadcasting old value '%s'\n",
  	idx, atom_value(idx));
  return jittr_atom_set(idx, 1, "", 0, &tv, a->who, 1);
}

static struct timeval atom_tv = {0, 0};

static int atom_gettv(tv)
struct timeval *tv;
{
  tv->tv_sec  = atom_tv.tv_sec;
  tv->tv_usec = atom_tv.tv_usec;
  return 0;
}

/* 
 * EXEC: reply the value or name of some other atom.
 * optinal patterns may be provided. The first pattern is prepended to the 
 * result, the second pattern is appended. The first one defaults to "",
 * the second one to "\n". The atom can be ommitted here and defaults to ".".
 * This overloads the get_cmd from atom.c, because atom.c does not know
 * anything about the jittr command syntax. If you change anything 
 * check both functions!
 *
 * The difference is this: 
 *
 * 1) It will overload the system time method temporarily. The effect is that
 *    \@ expansion as done by the jarg_option parser will reflect the atom
 *    change time while we are here.
 *
 * 2) If a prefix argument is present, (even if it evaluates as the empty
 *    string,) The atom-value is written with all ''\\', '"', and special
 *    characters escaped by '\\'.
 *
 * 3) option -stat makes the atom_data entries available.
 * 
 * Perhaps I should place a one-time write trace on the REPLY_ATOM.
 * This trace is triggered by writing the value, and will encapsulate
 * it with prefix, propper escaping and suffix. XXX: That could vastly
 * improve the concept of prefix and suffix command line parameters. But how
 * would that trace know the age of the atom as needed for \@ expansion?
 */
static int
ext_get_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring *d = NULL, **rp = &atoms[REPLY_ATOM]->value;
  int slen, endopt = 0, zapd = -1;
  char *s = jarg_first_word(&slen);
  int (* saved_gettimeval) __P((struct timeval *now));

  if ((idx = atom_lookup(s ? s : ".")) < 0)
    return jittr_error("%s: atom '%s' not found.\n", jarg_name(), s, 0, 0);
  
  atom_tv.tv_sec   = jittr_atom_data(idx)->sec;
  atom_tv.tv_usec  = jittr_atom_data(idx)->usec;
  saved_gettimeval = jittr_gettimeval;
  jittr_gettimeval = atom_gettv;

  s = jarg_next_word(&slen);

  while (!endopt && slen > 1 && *s == '-')  /* s == NULL implies slen == 0 */
    {
      if (!strncmp("-name", s, slen))
	d = atom_get_name(idx);
      else if (!strncmp("-mode", s, slen))
	zapd = dstring_appendf(&d, 0, "-%sreadonly -%spermanent", 
	  atoms[idx]->a.flag & A_READONLY ? "" : "no",
	  atoms[idx]->a.flag & A_PERMANENT ? "" : "no", 0, 0);        
      else if (!strncmp("-index", s, slen) || !strncmp("-idx", s, slen))
	zapd = dstring_appendf(&d, 0, "%d", idx, 0, 0, 0);        
      else if (!strncmp("-stat",  s, slen) || !strncmp("-who", s, slen))
        {
	  struct client *c = jittr_atom_data(idx)->who;

	  if (c)
	    {
	      dstring_appendn(&d, 0, "%d ", c->id);
	      dstring_append(&d, -1, c->name->buf, c->name->length);
	      zapd = 1;
	    }
	  else
	    {
	      debug1("get %d -stat: no client. HELP!!!! FIXME.\n", idx);
	      dstring_append(&d, 0, NULL, 0);
	      zapd = 1;
	    }
	}
      else if (!strncmp("-value", s, slen) || !strcmp("--", s))
        { endopt++; if (zapd) free((char *)d); zapd = -1; d = NULL; }
      else
        {
	  jittr_gettimeval = saved_gettimeval;
	  return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
	}
      s = jarg_next_word(&slen);
    }

  if (!d)
    d = atom_get(idx);

  dstring_append(rp, -1, s, slen);
  if (s)
    dstring_appendq(rp, -1, d->buf, d->length);
  else
    dstring_append( rp, -1, d->buf, d->length);
  s = jarg_next_word(&slen);
  atom_append(REPLY_ATOM, -1, s ? s : "\n", s ? slen : 1);
  if (zapd >= 0)
    free((char *)d);
  jittr_gettimeval = saved_gettimeval;
  return 1;	/* don't call to the original get_cmd, which is still there */
}

/*
 * If this is the eval command, we provide a NULL context.
 * Then the client himself is active. 1 is returned, to indicate that
 * the command was sucessfully processed.
 * If this is a trace method planted on some atom, context provides the
 * execution context of the planter. We are active on his behalf.
 * In this case 0 is returned to indicate that other trace methods are 
 * welcome.
 *
 * Context (coming from trace_cmd()) should have the current atom as its
 * where field. This is how jittr trace callbacks are defined. We do not
 * check that here.  
 * Eval breaks up word boundaries. This is done deliberatly to conform to
 * the commonly known definition of eval. It requires extra parsing here,
 * which is also useful if you want to place multiple commands in one eval.
 *
 * The default command "get ." is scheduled, when no command is specified.
 */
static int
eval_cmd(idx, flag, context)
int idx, flag;
void *context;
{
  struct exec_context *c = context ? (struct exec_context *)context : exec_context;
  dstring *cmd = NULL;
  dstring *args = c->what;
  int i, slen = 0, rest = 0;
  char *p = NULL, *s = NULL;
  
  if (!args)
    return jittr_error("eval without args (atom idx %d).", idx, 0, 0, 0);
 
  /*
   * Here we have to copy the buffer. Two reasons:
   * 1) jittr_interpret steals it, when it constructs an action.
   * 2) we want to do another stage of parsing here. (not needed??)
   * 3) A parser needs to look at each byte anyway, the price includes copying.
   */
  p = (char *)args->data;
  rest = args->length - (p - args->buf);
  while ((s = parse_word(&cmd, &p, &rest, &slen, NULL)))
    dstring_append(&cmd, -1, " ", 1);
  if (dstring_length(cmd))
    dstring_append(&cmd, -2, "\n", 1);
  else
    dstring_append(&cmd, 0, "get .\n", 6);
  if ((i = jittr_interpret(&cmd, c->who, &c->where)) != CIN_SCHED_MASK)
    {
      /*
       * jittr_interpret can tell us that it does not want to see that command
       * again. E.g. if it detects, that we want to reply to a dead client.
       * If so, it will return CIN_VOID_MASK. If we are from within a trace
       * callback, we will try to remove the trace then.
       */
      if ((i & CIN_VOID_MASK) && context)
        {
	  debug("eval_cmd removes callback that reports to dead client.\n");
	  atom_trace_delete(atom_trace_lookup(&atoms[idx]->trace, 
	  			eval_cmd, context));
	}
      else
        {
	  jittr_error("eval(%d): '%s' returns 0x%02d.\n", idx, 
	  	(char *)args->data, i, 0);
	  if (i & CIN_VOID_MASK)
	    jittr_error("eval: command corrupt, discarded.\n", 0, 0, 0, 0);
	  if (i & CIN_MORE_MASK)
	    jittr_error("eval: quoting error after parsing.\n", 0, 0, 0, 0);
	}
    }
  if (cmd)
    free((char *)cmd);
  return context ? 0 : 1;
}

/*
 * Accept any combinations of the letters '-rwtuexl',
 * and set *opsp accordingly. Return -1 and set nothing, if unknown
 * letters appear.
 */
static int
trace_ops_gobble(s, slen, opsp)
char *s;
int slen, *opsp;
{
  int ops = 0;

  while (*s)
    switch (*s++)
      {
      case '-': break;
      case 'r': ops |=  TRACE_READ; break;
      case 'w': ops |=  TRACE_WRITE; break;
      case 't': ops |=  TRACE_TRUNC; break;
      case 'u': ops |=  TRACE_UNSET; break;
      case 'e': ops |=  TRACE_EXEC; break;
      case 'x': ops |=  TRACE_EXEC; break;
      case 'l': ops |=  TRACE_LATE; break;
      default: return 1;	/* bad ops */
      }
  *opsp = ops;
  return 0;
}

/*
 * trace idx -list|-info [-client name] [-r|w|u[l]] [prefix [suffix]]
 * trace idx -delete [-client name] [-r|w|u[l]] command
 * trace idx [-add] [-r|w|u[l]] [command]
 *
 * Commands are executed with a current working point of idx.
 * If -r|w|u is not specified, it defaults to all for the list and del
 * subcommands and to -w for -add.
 * The modifier 'l' can be appended to indicate TRACE_LATE.
 * The command (with -add and -delete) is optional and defaults to "get ."
 *
 * As jittr_interpret() is a prefix parser, all remaining words (as found
 * in *(struct dstring **)exec_context) are completely unparsed. This
 * function parses options off the beginning of the text, but it does not
 * parse the remaining command.  The command is placed in the scheduled
 * action verbatim. Thus no extra quoting is required to protect the
 * command. I don't know if this is good or bad. But unquoted semicolons
 * and newlines will not even arrive here.  The delete option needs a
 * verbatim match, whitespace must not vary.
 *
 */
static int
trace_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int what = 0;		/* 0:add, 1:list, 2:delete */
  int ops = 0;		/* ORed TRACE_WRITE, TRACE_READ, TRACE_UNSET */
  struct trace *tr;
  struct client *cl_opt = NULL;
  int rlen, slen, endopt = 0;
  struct exec_context *context = NULL;
  char *r, *s;
  
  if (!(s = jarg_first_word(&slen)))
    return jittr_error("%s: atom name missing.", jarg_name(), s, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    return jittr_error("%s: atom '%s' not found.", jarg_name(), s, 0, 0);
  
  r = *jarg_rawp(&rlen);		/* save parser context */
  while ((s = jarg_next_word(&slen)))
    {
      if (endopt || slen < 2 || *s != '-')
	break;
      if (!strncmp("-list", s, slen) || 
	  !strncmp("-info", s, slen))		what = 1;
      else if (!strncmp("-delete", s, slen))        what = 2;
      else if (!strncmp("-add", s, slen))           what = 0;
      else if (!strncmp("-write", s, slen))    ops |= TRACE_WRITE;
      else if (!strncmp("-read", s, slen))     ops |= TRACE_READ;
      else if (!strncmp("-unset", s, slen))    ops |= TRACE_UNSET;
      else if (!strncmp("-truncate", s, slen)) ops |= TRACE_TRUNC;
      else if (!strncmp("-exec", s, slen))     ops |= TRACE_EXEC;
      else if (!strncmp("-late", s, slen>2?slen:3))     ops |= TRACE_LATE;
      else if (!strncmp("-last", s, slen>2?slen:3))     ops |= TRACE_LATE;
      else if (!trace_ops_gobble(s, slen, &ops));
      else if (!strncmp("-client", s, slen))
	{
	  s = jarg_next_word(&slen); 
	  if (s[0] == '.' && s[1] == '\0')
	    {
	      if (!exec_context)
		return jittr_error("%s: '-client .' without exec_context.\n",
			jarg_name(), 0, 0, 0);
	      cl_opt = exec_context->who;
	    }
	  if (!cl_opt) cl_opt = *ClientByName(NULL, s);
	  debug1("trace -client '%s'\n", cl_opt->name->buf);
	}
      else if (!strcmp("--", s))
	endopt++;
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      r = *jarg_rawp(&rlen);		/* save parser context */
    }

  if (!ops) 
    ops = what ? TRACE_MODE_MASK : TRACE_WRITE;
  
  if (cl_opt && !what)
    return jittr_error("trace -client only supported with -delete or with -list\n", 0,0,0,0);

  if (what == 1)	/* list */
    {
      int suflen;
      struct dstring *d;
      char *suffix = jarg_next_word(&suflen);
      /* s, as left over by the option parser is the prefix */

      for (tr = atoms[idx]->trace; tr; tr = tr->next)
        {
	  context = (struct exec_context *)tr->data;
	  d = (struct dstring *)&context[1];

	  if (!(tr->flag & ops) || (cl_opt && cl_opt != context->who))
	    continue;

	  dstring_append(rp, -1, s, slen);

	  /* If we did not specify a client, show them all */
	  if (!cl_opt && flag & TRACE_DATA_CONTEXT)	
	    dstring_appendf(rp, -1, "-client %d ",
	    	context->who->id, 0, 0, 0);
	  dstring_appendf(rp, -1, "-%c%c%c%c",
		tr->flag & TRACE_READ  ? 'r' : '-',
		tr->flag & TRACE_WRITE ? 'w' : '-',
		tr->flag & TRACE_TRUNC ? 't' : '-',
		tr->flag & TRACE_UNSET ? 'u' : '-');
	  dstring_appendf(rp, -1, "%c%c ",
		tr->flag & TRACE_EXEC ? 'x' : '-',
		tr->flag & TRACE_LATE ? 'l' : '-', 0, 0);

	  if (tr->flag & TRACE_DATA_CONTEXT)
	    {
	      int l; 

	      r = (char *)d->data;
	      if ((l = d->length - (r - d->buf)) > 0)
	        {
		  /* test parse the string, to see what eval_cmd will see */
		  dstring *cmd = NULL;
		  int ql = l;
		  char *q = r;

		  parse_word(&cmd, &q, &ql, NULL, NULL);
		  if (!dstring_length(cmd)) l = 0;
		  if (cmd) free((char *)cmd);
		}
	      
	      /* 
	       * If we have a command string print it in double quotes.
	       * r is the raw buffer and needs no extra level of quoting.
	       *
	       * Here is a tiny hack: Without a command, print the default
	       * "get ." but without quotes. This is a hint for the user
	       * that it is a default and not explicitly said. It makes a
	       * difference, when he wants to delete the trace.
	       */
	      if (l > 0)
	        {
		  dstring_append(rp, -1, "\"", 1);
	          dstring_append(rp, -1, r, l);
		  dstring_append(rp, -1, "\" #", 3);
		}
	      else
	        dstring_append(rp, -1, "get . #", 7);
	      dstring_append(rp, -1, context->who->name->buf, 
	        context->who->name->length);
	    }
	  else
	    dstring_appendf(rp, -1, "[0x%08x 0x%08x]",
	    	(long)tr->fn, (long)tr->data, 0, 0);

	  atom_append(REPLY_ATOM, -1, suflen?suffix:"\n", suflen?suflen:1);
	}
      return 1;
    }

  if (what == 2)	/* delete */
    {
      struct trace *ntp, **tp = &atoms[idx]->trace;
      struct dstring *d;
      int ast = -1;


      /*
       * The following code has been removed by the human peep hole optimizer.
       * We back up to the unparsed parser text of s by simply using r.
       */
#if 0
      *jarg_rawp(NULL) = r;	/* restore parser context */
      r = *jarg_rawp(&rlen);	/* retrieve unparsed text */
#endif

      /* 
       * A prefix match is a nice idea, but if the text prefix contains
       * opening braces, we must put closing braces somewhere, otherwise
       * the parser does not accept the word. We allow any number of
       * closing braces after the asterix, and still consider the text
       * before the asterix as a prefix match.
       */
      for (ast = rlen - 1; ast >= 0; ast--)
        {
	  if (r[ast] != '}')
	    {
	      if (r[ast] != '*')
	        ast = -1;
	      break;
	    }
	}
	    
      while ((tr = *tp))
        {
	  context = (struct exec_context *)tr->data;
	  d = (struct dstring *)&context[1];
	  ntp = tr->next;

	  if (!(tr->flag & ops) || !(tr->flag & TRACE_DATA_CONTEXT) ||
	      (cl_opt && cl_opt != context->who))
	    {
	      tp = &tr->next;
	      continue;
	    }

	  s = (char *)d->data;
	  slen = d->length - (s - d->buf);

	  if ((rlen == slen && !xbcmp(s, r, rlen)) ||	/* exact match */
	      (ast >= 0 && slen >= rlen &&
	       !xbcmp(s, r, ast)) || 			/* prefix match */
	      (ast == 0 && rlen == 1 && slen == 0))	/* miniature match */
	    {
	      debug("should consider TRACE_WRITE and/or -client here\n");
	      atom_trace_delete(tp);
	      *tp = ntp;
	    }
	  else
	    tp = &tr->next;
	}
      return 1;
    }
  
  *jarg_rawp(NULL) = r;		/* restore parser context */

  if (idx == 1)
    return jittr_error("trace_cmd: cannot trace stdout: too dangerous today\n", 
    	0, 0, 0, 0);

  if (!(context = (struct exec_context *)calloc(1, sizeof(struct exec_context)
        + sizeof(struct dstring) + (exec_context->what ? exec_context->what->length : 0))))
    return jittr_error("trace_cmd %d -add: malloc failed.", idx, 0, 0, 0);
  context->where.atom = idx;
  context->who = exec_context->who;
  context->what = (struct dstring *)&context[1];
  context->what->data = (void *)context->what->buf;
  context->when = NULL;		/* ext_set_cmd() shall call jittr_gettimeval */
  if (exec_context->what)
    {
      xbcopy(exec_context->what->buf, context->what->buf, exec_context->what->length);
      context->what->length = exec_context->what->length;
      context->what->data = (void *)(context->what->buf + 
      				(r - exec_context->what->buf));
    }
  atom_trace_set(idx, ops | TRACE_DATA_CONTEXT | TRACE_DATA_ALLOCED,
  	eval_cmd, (void *)context);
  return 1;
}
  
/* 
 * late_stdio_trace is different from other traces as it cares about the 
 * list of traces:
 * a) If it is not at the end, it moves itself at the end. Other instances 
 *    of itself may follow (e.g. with different filepointers).
 * b) After printing and if no trace follows, it schedules immediate 
 *    truncation of the atom, assuming that all the actions that need the 
 *    contents of the atom have already been done or were scheduled earler.
 *
 * Thus: If one of the traces has flow control problems, it should copy out
 * the current contents to some private location.  I currently don't care
 * about flow control.
 */
static int
late_stdio_trace(idx, flag, fp)
int idx, flag;
void *fp;
{
  struct sched_action act;

  if (!(flag & TRACE_WRITE))
    return 0;
  if (!(flag & TRACE_LATE))	/* fool, you forgot to queue me late! */
    {
      atom_trace_delete(atom_trace_lookup(&atoms[idx]->trace, late_stdio_trace, fp));
      atom_trace_set(idx, TRACE_WRITE|TRACE_LATE, late_stdio_trace, fp);
      return 0;
    }

  /* initialisation a la sched_make_action */
  ASSERT(SCHED_ARGC > 2);
  act.argv[0] = (void *)fp;
  act.argv[1] = (void *)idx;
  act.argv[2] = NULL;
  act.fn = late_printf_process;
  act.flag = 0;
  act.next = NULL;

  if (flag & TRACE_LAST)
    {
      struct timeval now;
      struct sched_timing ti;

      jittr_gettimeval(&now);
      TV2TIMING(&now, &ti);
      act.argv[2] = act.argv[0];
      sched_enter_event(&ti, &act, 0);
    }
  else
    late_printf_process(act.argv);
  return 0;
}

static int
late_printf_process(av)
void **av;
{
  char buf[51];
  struct timeval tv;

  if (!atoms[(int)av[1]] || !atoms[(int)av[1]]->value ||
      !atoms[(int)av[1]]->value->length)
    return 0;	/* nothing to do here. */

  
  jittr_gettimeval(&tv); tv.tv_usec = 0;
  sched_format_tv(buf, 50, &tv, NULL);
  fprintf((FILE *)av[0], "%s %d: %s: %s\n", buf, (int)getpid(),
    atom_name((int)av[1]), atom_value((int)av[1]));
  if (av[2])
#if 0		/* checks for traces, too late now, unnecessary slowdown */
    atom_append((int)av[1], 0, NULL, 0);
#else
    dstring_append(&atoms[(int)av[1]]->value, 0, NULL, 0);
#endif
  return 0;
}

static char *help[] =
{
  "chmod atom [-[no]readonly|-[no]permanent",
  	"change mode flag of an atom. set does no longer do that",
  "rename atom name",
  	"change name of an atom. set does no longer do that",
  "get atom -stat [prefix [suffix]]",
  	"returns index and name of the client who modified the atom last",
  "",	"Within the prefix/suffix strings of the get command, \\@ evaluates to the atom timestamp. If the optional argument <prefix> is present, the returned value is backslash escaped",
  "eval command-string",
  	"remove extra quoting and run commands",
  "trace atom [-add] [-rwxtl [command-string]]",
	"Arrange that command is executed whenever the atom is read/written/executed. Command defaults to {get .} and -rwx defaults to {-w}",
  "trace atom -list [-client name|.] [-rwxtl] [prefix [suffix]]",
  	"List trace callbacks of an atom, including those written in C",
  "trace atom -delete [-client name|.] [-rwxtl] [command-string|command-prefix*]",
  	"remove a trace that was set with the 'trace ... -add' command",
  "tell client command-text [suffix]",
  	"connect to a client (or a new peer) and send text. Suffix defaults to '\\n'",
  "",	"",
  "<general jittr word syntax>",
  	"Jittr commands are built from words in a syntax similar to John Ousterhout's TCL with a few additions known from Larry Wall's Perl. Quoting characters are \\, \"\", '' and {}. \\@ expands to the timestamp of the current context. \\x{4845 58} allows for multiple hex-characters",
  "<general jittr command syntax>",
  	"The jittr command parser reads lines of the following form: ``@timestamp+repeat >redirection command parameters delimiter'', which can be as complicated as: @14:45h+30m*4*.5 >&26 will cause a command to execute at 14:45h, 15:15h, at 15:30h, and at 15:37h30s UTC, while appending its output each time to atom 26",
  "",	"",
  NULL
};

int
jittr_jcmd_init()
{
  atom_trace_delete(atom_trace_lookup(
          &atoms[STDERR_ATOM]->trace,       atom_simple_trace, (void *)stderr));
  atom_trace_set(STDERR_ATOM,   TRACE_WRITE|TRACE_LATE, late_stdio_trace, (void *)stderr);

  /* Convert GET/SET_CMD_ATOM into a one time stack. */
  (*atom_trace_lookup(&atoms[GET_CMD_ATOM]->trace,NULL,NULL))->flag|=TRACE_LATE;
  (*atom_trace_lookup(&atoms[SET_CMD_ATOM]->trace,NULL,NULL))->flag|=TRACE_LATE;

  atom_command(  HELP_CMD_ATOM, NULL,   atom_help_cmd, (void *)help);
  atom_command(   GET_CMD_ATOM, NULL,     ext_get_cmd, (void *)0);
  atom_command(   SET_CMD_ATOM, NULL,     ext_set_cmd, (void *)0);
  atom_command(   APP_CMD_ATOM, NULL,     ext_set_cmd, (void *)1);
  atom_command(  EVAL_CMD_ATOM, "eval",      eval_cmd, (void *)0);
  atom_command( TRACE_CMD_ATOM, "trace",    trace_cmd, (void *)0);
  atom_command(  TELL_CMD_ATOM, "tell",      tell_cmd, (void *)0);
  atom_command( CHMOD_CMD_ATOM, "chmod",    chmod_cmd, (void *)0);
  atom_command(RENAME_CMD_ATOM, "rename",   chmod_cmd, (void *)1);

  debug("jittr_jcmd_init: done.\n");
  return 0;
}
