/*
 * jittr - Just In Time TRansport library
 *
 * This is about timestamps, sockets and (shared) atoms.
 *
 * 3.3.95 jw.
 */
#include <sys/types.h>	/* for sys/socket.h */
#include <sys/socket.h>	/* struct sockaddr for net.h */
#include <netinet/in.h>	/* struct sockaddr_in */
#include <sys/time.h>	/* struct timeval for net.h */
#include <errno.h>	/* for errno */

#include "jittr/atom.h"
#include "jittr/parse.h"	/* parse_check(), parse_word() */
#include "jittr/jittr.h"
#include "jittr/action.h"	/* struct jittr_action */
#include "jittr/net.h"		/* struct client for jittr.h */
#include "jittr/doio.h"		/* TcpClient() */
#include "jittr/schedule.h"	/* struct sched_timing for jittr_interpret() */
#include "jittr/version.h"	/* how old is this code? */


/* global pointers, sigh! */
struct exec_context *exec_context = NULL;
int (* jittr_gettimeval) __P((struct timeval *now));

static int jittr_input __P((void *cli, char *bufp, int len));

/* ================ action handling ===================== */

/*
 * called by sched_process().
 * argv is a pointer to sched_action.argv[SCHED_ARGC], 
 * but is interpreted here as struct jittr_action. sched_process() takes 
 * care to free malloced memory in its sched_action.argv[] array.
 *
 * Note: You must not make any (permanent) changes to the action!
 *   No string parsing, no buffer trimming, no data-pointer movement, nothing!
 *   The action may be repetitive and should look identical when it comes
 *   again. Watch out for the assertions below. You have been warned.
 *
 * Currently, we asume that the jittr machine is idle most of the time.
 *
 * We do not need to consider this: (do we?)
 *   If we take a long time to process this action, we should call
 *   sched_timeout() regularily. If our time is up, we must return. The
 *   execution can be continued by another action that we choose to
 *   schedule from here. Our own (re)schedule plan is available at
 *   (struct sched_event *)argv->time.
 */
int
jittr_process(argv)
void **argv;
{
  static struct exec_context _exec_context;   /* yes, we are single-threaded */
  static int in_jittr_process = 0;
  struct jittr_action *ja = (struct jittr_action *)argv;
  struct atom *ora = atoms[REPLY_ATOM];
  struct dstring *orv = ora->value;
  int c = -1, r = 0;

  ASSERT(!in_jittr_process);	/* because _exec_context is static */

  if (!atoms[ja->atom] || atoms[ja->atom]->a.refcount < 1)
    {
      jittr_error("Atom %d(%s) no longer existing. Cannot execute '... %s ... %s'.",
		  ja->atom, atom_name(ja->atom), ja->args->buf, 
		  (char *)ja->args->data);
      return -1;
    }

  /* 
   * Redirect REPLY by pointer.
   * Could also be done by tracing, but pointers are faster. 
   * If redirection target is an atom, we'll link the whole atom, so that
   * trace callbacks are triggered. If it is a client, linking its obuf is 
   * sufficient. Linking in the name is tricky, as the name of the client
   * may well be modified by jittr commands. We don't do that.
   */
  if (dstring_length(orv))	/* we already have sth. to reply? Strange. */
    {
      jittr_error("jittr_process: left over REPLY: '%s'\n", orv->buf, 0, 0, 0);
      *orv->buf = '\0';
      orv->length = 0;
    }

  if (ja->type == RT_CLIENT)
    {
      if (ja->reply.client && ja->reply.client->o)
        {
	  atoms[REPLY_ATOM]->value = ja->reply.client->o->obuf;
	  ja->reply.client->o->obuf = (struct dstring *)0xDEADBEEF; /*I'll be back*/
	}
      else
        {
	  debug("jittr_process: replyto client is deaf, dump to stderr\n");
	  ja->type = RT_ATOM;
          atoms[REPLY_ATOM] = atoms[ja->reply.atom = STDERR_ATOM];
	  atoms[REPLY_ATOM]->a.refcount++;
	}
    }
  else
    {
      ASSERT(ja->type == RT_ATOM);
      atoms[REPLY_ATOM] = atoms[ja->reply.atom];
      if (!atoms[REPLY_ATOM])
        {
	  jittr_error("jittr_process: replyto atom %d is dead, dump to stderr\n",
	  	      ja->reply.atom, 0, 0, 0);
	  atoms[REPLY_ATOM] = atoms[STDERR_ATOM];
	}
      atoms[REPLY_ATOM]->a.refcount++;
    }

  /* carefully stomp on the string delimiter, if nonzero string arg */
  if (ja->args && ((char *)ja->args->data - ja->args->buf < ja->args->length))
    {
      c = ja->args->buf[--ja->args->length];
      ja->args->buf[ja->args->length] = '\0';
    }

  in_jittr_process++;
  _exec_context.what = ja->args;
  _exec_context.who = ja->who;
  _exec_context.where = ja->cwp;	/* structure copy */
  _exec_context.when = &((struct sched_event *)argv)->time.when;
  _exec_context.which = ja->atom;
  exec_context = &_exec_context;
  r = atom_exec(ja->atom, (void *)exec_context);
  exec_context = NULL;
  in_jittr_process--;

  if (ja->args && ((char *)ja->args->data - ja->args->buf <= ja->args->length))
    {
      /* did you try to change the action structure? */
      ASSERT(c != -1 && ja->args->buf[ja->args->length] == '\0');
      ja->args->buf[ja->args->length++] = c;
    }
  else
    {
      ASSERT(c == -1); /* no, you must not change the action structure! */
    }

  if (ja->type == RT_CLIENT)
    {
      /* while the pointers are linked, their originals must not be touched */
      ASSERT(ja->reply.client->o->obuf == (struct dstring *)0xDEADBEEF);
      /*
       * Move back the pointer from reply atom to obuf. It is very 
       * likely that value is changed now.
       */
      ja->reply.client->o->obuf = atoms[REPLY_ATOM]->value;
      atoms[REPLY_ATOM]->value = orv;
    }
  else
    {
      atoms[REPLY_ATOM]->a.refcount--;
      atoms[REPLY_ATOM] = ora;
    }
  return r;
}

/*
 * The modules doio.o and net.o have no hardcoded information about the jittr
 * world. They only contain knowledge to interface with "the network" as
 * presented by the operating system and present it in form of clients which 
 * are structures built around dstring I/O buffers.
 *
 * All other "interfacing" is by function calls into doio.o/net.o and by
 * following references found in the function call parameters. 
 * This is why jittr_input is a callback for DoIo. 
 */
static int 
do_io(tv, gettimeval_fn)
struct timeval *tv;
int (* gettimeval_fn) __P((struct timeval *now));
{
  int r;

  if ((r = DoIo(tv, gettimeval_fn, jittr_input)) < 0)
    return jittr_error("do_io(): select: %s\n", strerror(errno), 0, 0, 0);
  return r;
}

/*
 * jittr_input() is called each time a clients inbound buffer has been
 * appended to. bufp must point inside the clients inbound buffer and 
 * indicates the start of the newly appended data. bufp+len must be identical
 * to the end of the buffer structure as indicated by its own length member.
 * Whenever a complete jittr packet is found in the buffer, jittr_interpret is
 * called to construct an action that is scheduled.
 * Telnet commands are intercepted here by jittr_telnet_check()/PurgeTelent().
 * This should be some generic input decoding mechanism, (e.g. via a function
 * pointer coming with the connection structure). But I don't need it now.
 *
 * Returns whatever jittr_interpret returns, unless 
 * a) the buffer is empty: returns 0
 * b) the input decoder wants to close the connection: returns CIN_CLOSE_MASK
 */
static int
jittr_input(cli, bufp, len)
void *cli;
char *bufp;
int len;
{
  struct client *cl = (struct client *)cli;
  struct conn *conn;
  struct dstring *buf;
  int bytes, r = 0;

  if (!bufp || !len)
    return 0;		/* considererd o.k. */
  
  /* check if bufp points into one of the two inbond buffers of the client */
  buf = *jittr_ptr2ds(bufp, DSHINT_CLIENT, cli);
  if (buf != (conn = cl->i)->ibuf) conn = cl->o;
  ASSERT(buf == conn->ibuf);

  /* try to find out if he talks telnet with us by looking at his first write */
  if ((conn->type == CTYPE_TCP_OR_TELNET) && (len > 3) &&
      ((bytes = jittr_telnet_check(bufp, len)) != 0))
    conn->type = (bytes > 0) ? CTYPE_TELNET : CTYPE_TCP;

  /* if we are not sure, that he is plain tcp, we intercept telnet protocol */
  if (conn->type == CTYPE_TELNET || conn->type == CTYPE_TCP_OR_TELNET)
    {
      if ((bytes = jittr_purge_telnet(bufp, len)) < 0)
        {
	  /*
	   * CTRL-C has two meanings:
	   * If there is more in the buffer, it means 'discard the buffer'.
	   * if there is nothing else in the buffer, it means 'close this
	   * connection'.
	   */ 
	  if (bufp > buf->buf)
	    {
	      /* IAC WONT TIMING_MARK */
              dstring_append(&conn->obuf, -1, "\377\374\006", 3);
	      return buf->length = 0;
	    }
	  return CIN_CLOSE_MASK;
	}
      if (!bytes)
	return buf->length = 0;
      if (bytes < len)
        {
	  /*
	   * no rest allowed, thus nothing to copy.
	  xbcopy(buf->buf + bufp + len, buf->buf + bufp + bytes, 
	  	 buf->length - len - (bufp - buf->buf));
	   */
	  buf->length -= len - bytes;
          len = bytes;
	}
    }

#ifdef WE_KNOW_THAT_JITTR_INTERPRET_IS_BUGFREE
  r = jittr_interpret(&conn->ibuf, cl, &cl->cwp);
#else
  {
    struct exec_context *old_ec = exec_context;

    r = jittr_interpret(&conn->ibuf, cl, &cl->cwp);
    ASSERT(exec_context == old_ec);	/* bad return detected */
  }
#endif

#ifndef HAVE_A_BETTER_SOLUTION
  {
    /*
     * This does not belong here: 
     * Network error reporting should be processed by a 
     * repetitive scheduled action.
     */
    extern dstring *jittr_ioerr;

    if (dstring_length(jittr_ioerr))
      {
	jittr_ioerr->buf[jittr_ioerr->length] = '\0';
	jittr_error("I/O trouble: %s", jittr_ioerr->buf, 0, 0, 0);
	jittr_ioerr->length = 0;
	r |= CIN_SCHED_MASK;
      }
  }
#endif
  return r;
}

/* 
 * Interpret a buffer that came from cl, who works at cwp.
 * The buffer may contain multiple commands. All are interpreted.
 * The buffer may or may not disappear while creating actions; either
 * *bufp == NULL or (*bufp)->length == 0 when jittr_interpret returns.
 * This is an option: There is a definition for GREEDY_INTERPRETER near the 
 * beginning of net.h; toggle it and you toggle the greedyness of
 * jittr_interpret(), this changes the parsing of multiple commands per data
 * chunk.  Both ways have advantages and disadvantages:
 *
 *  if jittr_interpret() is greedy, all commands are converted into actions, 
 *     before any of them is executed. If there is a cd command within, it will
 *     not have any effect on the other commands in the same chunk.
 *  if jittr_interpret() is not greedy, only one command is converted at each
 *     call. Thus we do not keep the above promise. (*bufp)->length may still
 *     be positive, when we return. The disadvantage is, that we have to
 *     change the structure of the DoIo() loop too. Otherwise dangling 
 *     input would remains unprocessed, until more text arrives. 
 *
 * Returns a bit-mask describing the contents of the buffer as defined in net.h:
 * CIN_SCHED_MASK true:	actions scheduled.
 * CIN_MORE_MASK true:	command delimiter missing.
 * CIN_VOID_MASK true:	command illegal. (e.g. report to dead client).
 *
 * CAUTION: Comments are handled here like ordinary commands.
 * This will confuse people that comment out lines containing a ';'.
 */
int
jittr_interpret(bufp, cl, cwp)
dstring **bufp;
struct client *cl;
struct cwp *cwp;
{
  struct dstring *buf = *bufp;
  int len;	/* size of first complete command in buf */
  int bytes;	/* size of the current word */
  struct timeval now;
  struct sched_timing ti;
  struct jittr_action *ja;
  struct sched_action act;
  struct exec_context tmp_con, *old_e_c = exec_context;
  char *s, *p;
  int i, j, scheduled = 0;

  /*
   * If bufp does not contain a '\n' character, return now.
   * Otherwise consider the buffer up to this character:
   * Check for an incomplete last word, or if the trailing '\n' character is 
   * escaped. If so, return now.
   * If everything is o.k. the lenght of the first complete command in buf is
   * measued.
   *
   * We have to start this at the real start of the buffer. Saving parser
   * context and continuing from bufp would make this faster.
   */
  if ((len = parse_check(p = buf->buf, buf->length, NULL)) < 0)
    {
#ifdef DEBUG
      if (len == -1)
        debug("jittr_interpret: quoting correct, but delimiter missing\n");
#endif
      return CIN_MORE_MASK;	/* syntax analysis: buffer incomplete */
    }

#if 0
  debug2("jittr_interpret: %s says %s\n", cl->name->buf, p);
#endif

  /*
   * Transform the buffer into an action and place it in the queue.
   * This includes special treatment of the first word:
   *  if it is a '#' character, discard this action.
   *  if it is a '@' character followed by a time spec,
   *   translate it into a timestamp and skip it.
   *  if it is (now) a '>' character followed by an address,
   *   this is a client's name or a peer for reply messages. 
   *   If the peer starts with a '!', and no such client exists, one is
   *   created on the fly.
   *  but if this '>' character is followed by an '&', then we write the
   *   reply into the thereafter named atom.
   *  if it is (now) a '[' character, we expect to read a default prefix and
   *   suffix word, that is stored in the exec-context.
   *
   * To do this, we parse the first word and leave buf->data to point 
   * at any remaining arguments, still in the buffer, then we steal the
   * buffer and construct an action. If there were any remaining data between
   * len and buf->length, we must construct another buffer containing the
   * rest and place it back in conn->ibuf; If so, we start again.
   */
  p = buf->buf;
  len++;	/* include the seperating '\n' character */

  /* initialize default timing */
  jittr_gettimeval(&now);
  TV2TIMING(&now, &ti);

  /* initialize default action */
  ASSERT(sizeof(struct jittr_action) <= sizeof(act.argv));
  ja = (struct jittr_action *)&act.argv[0];
  ja->type = RT_CLIENT;
  ja->reply.client = cl;
  ja->who = cl;
  ja->cwp.atom = cwp->atom;	/* how about remote working points? */
  ja->args = NULL;
  

  /* initialize a current context to help atom_lookup */
  tmp_con.what = NULL;
  tmp_con.who = cl;
  tmp_con.where = *cwp;
  exec_context = &tmp_con;

  while ((s = parse_word(NULL, &p, &len, &bytes, NULL)))
    {
      switch (*s)
        {
	case '@':
	  j = *(++s);		/* skip the '@' */
	  if ((i = sched_parse_timing(&s, &ti, &now)) < 0)
	    {
	      jittr_error("jittr_interpret: parse timing: error code %d.\n",
		i, 0, 0, 0);
	      break;
	    }
#ifdef DEBUG
	  if (s > p)	/* cannot happen: parse_word() placed a '\0' byte. */
	    debug("jittr_interpret: int'l error: greedy sched_parse_timing.\n");
	  if (*s && s < p)	/* we did not make it to the '\0' byte? */
	    debug1("jittr_interpret: garbage after timing '%s' skipped.\n", s);
#endif
	  if (j != '+' && cl->off.sign)	/* adjust his timestamps */
	    {
	      if (cl->off.sign > 0)	/* he lives in the past */
	        {
		  ti.when.tv_sec += cl->off.tv.tv_sec;
		  if ((ti.when.tv_usec += cl->off.tv.tv_usec) >= 1000000)
		    {
		      ti.when.tv_sec++;
		      ti.when.tv_usec -= 1000000;
		    }
		}
	      else			/* he live in the future */
	        {
		  ti.when.tv_sec -= cl->off.tv.tv_sec;
		  if ((ti.when.tv_usec -= cl->off.tv.tv_usec) < 0)
		    {
		      ti.when.tv_sec--;
		      ti.when.tv_usec += 1000000;
		    }
		}
	    }
	  continue;

        case '>':
	  /* reply to client or append to atom */
	  bytes--;
	  if (*++s == '&')		/* '>' is now skipped */
	    {
	      int idx;

	      bytes--;
	      s++;			/* '&' is now skipped */
	      /* 
	       * reply by appending to an atom.
	       * If it is an append to REPLY_ATOM, behave as if no 
	       * reply-to was specified.
	       */
	      if ((idx = atom_lookup(s)) < 1)
		{
		  idx = STDERR_ATOM;
		  debug1("atom '%s' not valid, replying to 'stderr'\n", s);
		}
	      debug2("jittr_interpret: reply to atom (%d) '%s'\n", idx, atom_name(idx));
	      if (idx == REPLY_ATOM)
	        {
		  debug("reply spec REPLY_ATOM ignored.\n");
		}
	      else
	        {
		  /* store the replyto in the action */
		  ja->type = RT_ATOM;
		  ja->reply.atom = idx;
		}
	    }
	  else
	    {
	      int force = 0;	/* create client if needed */

	      if (*s == '!')
	        {
		  s++;
		  force++;
		}
	      if (*s) /* parse a client name or addr now. and store it */
	        {
		  struct client *c = InetClient(s, force, 0);

		  if (c)
		    {
		      ja->type = RT_CLIENT;
		      ja->reply.client = c;
		    }
		  else
		    {
		      jittr_error("jittr_interpret: %s: %s. Reply to stderr.\n",
		    	s, "unknown client specification", 0, 0);
		      ja->type = RT_ATOM;
		      ja->reply.atom = STDERR_ATOM;
		    }
		}
	    }
	  continue;

	case '[':
	  jittr_error("jittr_interpret: %s: Cannot interpret format. %s.\n",
	    s, "Please specify prefix (and suffix) at the end of the line",0,0);
	  break; 	/* XXX-Fixme: command ignored */

	
	case '#':
	  break;	/* just plain nuthin' */
	
	default:
	  buf->data = p;
	  /* command word has been parsed. Construct action now */
	  /* zero: a final check, last exit before schedule... */
	  if ((ja->type == RT_CLIENT) && (ja->reply.client->id < 0))
	    {
	      scheduled = CIN_VOID_MASK;
	      break;
	    }

	  /* first: find the executing atom. 's' has its name */
	  if ((ja->atom = atom_lookup(s)) < 0)
	    {
	      jittr_error("jittr_interpret(%d): %s: unknown command\n", 
	      	cl->id, s, 0, 0);
	      /* not really discarded. We'll dump it to stderr */
	      /* but what does executing stderr mean? */
	      debug("executing stderr instead of unknown command.\n");
	      ja->atom = STDERR_ATOM;
	    }

	  /* second: steal the buffer, if it has parameters */
	  if (len)
	    {
	      *bufp = NULL;		/* steal buffer */
	      ja->args = buf;		/* may be too long ... */
	    }

	  /* third: complete the action */
	  act.fn = jittr_process; 
	  act.next = NULL;
	  /*
	   * Due to structure layout, act.argv[0] corresponds to ja.args
	   * and should be eventually passed to free(). As the timing spec may
	   * well be repetitive, only the schedule level code can handle this 
	   * correctly, jittr_process() can't.
	   */
	  act.flag = ja->args ? 1 : 0;

	  /* forth: combine action and timing to create an event */
	  if ((i = sched_enter_event(&ti, &act, 0)) < 0)
	    jittr_error("sched_enter_event '%s %d %s' failed.", buf->buf, 
	    		ja->atom, (char *)buf->data, 0);
#ifdef DEBUG
	  if (i)
	    {
	      char t[50];

	      sched_format_timing(t, 49, &ti, &now);
	      debug2("event Nr. %d scheduled in %s.\n", i, t);
	    }
#endif
	  scheduled = CIN_SCHED_MASK;
	  break;
	}
      break;	/* yes, the continues above were gotos in disguise */
    }

  exec_context = old_e_c;	/* restore context */

  /* see if there was more data in the buffer */
  if ((bytes = (buf->buf + buf->length) - (p + len)) > 0)
    {
      if (*bufp)
        {
	  /* a comment or empty or parameterless command followed by more */
	  ASSERT(buf == *bufp);
	  xbcopy(p+len, buf->buf, bytes);
	  buf->length = bytes;
	}
      else
        {
	  /* 
	   * A real command (action triggered) followed by more.
	   * So, copy back the rest to prepare a restart.
	   */
	  dstring_append(bufp, 0, p+len, bytes);
	  ASSERT(buf == ja->args);
	  buf->length -= bytes;	/* ... ja->args no longer too long */
	}
      /* restart all over with the rest of the buffer. */
#ifdef GREEDY_INTERPRETER
      return jittr_interpret(bufp, cl, cwp) | scheduled;
#else
      return scheduled;
#endif
    }
  else if (*bufp)
    (*bufp)->length = 0;
	
  return scheduled;
}

/*
 * This is the glue between jxcmd.o and jmod.o which both do not
 * (need to) know what a struct client is (and what a 0xDEADBEEF is).
 */
int
jittr_program_export(client_name, s, slen, t, tlen, idx)
char *client_name, *s, *t;
int slen, tlen, idx;
{
  struct client *c = TcpClient(client_name, server.connect_timeout);

  if (!c || !c->o) 
    return jittr_error("%s: cannot talk to client %s: %s.\n", jarg_name(), 
    	client_name, c ? "no outbound connection" : " no such client", 0);

  if (exec_context && exec_context->who == c)
    return jittr_error("%s: why do you phone yourself (%s)?.\n",
    	"jittr_program_export", c->name->buf, 0, 0);

  return jittr_program_export_dstring(&c->o->obuf, s, slen, t, tlen, idx);
}

/*
 * This function does two things:
 * a) It changes the name (and link path) of atom idx to be name. (if exists)
 * b) it changes the name of the application (inside struct server).
 *
 * Note: 
 * Calling jittr_change_name while clients are registered may yield confusion!
 */
int
jittr_change_name(idx, name)
int idx;
char *name;
{
  char *s, *t, *old;
  int l, nlen = strlen(name);

  if (*name == '/') name++;
  if (strchr(name, '/'))
    return jittr_error("jittr_change_name: Name '%s' must not contain '/' characters.\n", name, 0, 0);

  if (atom_exists(idx))
    {
      old = atom_name(idx);
      if (*old == '/') old++;
      l = jittr_do_link(0, old, strlen(old), NULL, 0, 0);	/* unlink */
      dstring_append(&atoms[idx]->name, 0, "/", 1);
      dstring_append(&atoms[idx]->name, 1, name, 0);
      if (!l) jittr_do_link(0, name, nlen, atom_name(idx), 0, 0);
    }

  s = t = Hostname();
  if (!atoi(s))
    while (*(++s) != '.' && *s)
      ;

  dstring_append(&server.name, 0, name, nlen);
  dstring_append(&server.name, -1, "@", 1);
  dstring_append(&server.name, -1, t, s - t);
  dstring_appendn(&server.name, -1, (server.tcp_port_fd < 0) ? ":NONE" : ":%d", 
  	server.tcp_port);

  return 0;  
}


/* =========== sample usage, you may have better ideas ============= */

/*
 * simple initialisation. only server ports are attempted here.
 * We count a while, to find a free port number.
 * TODO: if the server ports fail on the given port number, we should connect
 * to the existing local server port and announce ourselves. 
 */

/* first: all non-networking stuff */
int 
jittr_init1(av0)
char *av0;
{
  dstring *errp1 = NULL;
  char *s, *t;

  jittr_gettimeval = gettimeval;
  /* 
   * Do not change the sequence of the first four init proccedures! 
   * They build on each other and assert fixed command indices.
   */
  atom_init();			/* atom management */
  jittr_jcmd_init();		/* core commands */
  jittr_link_init();		/* atom hierarchy */
  jittr_jmod_init();		/* module loader */
  jittr_session_init();		/* session manager */
  jittr_mbuiltin_init();	/* loads static modules. */
  jittr_user_init();		/* additinal. indices depend on static mods */
  atom_new(-1, "jittr_version", VERSION, A_READONLY|A_PERMANENT);

  s = t = Hostname();
  if (!atoi(s))
    while (*(++s) != '.' && *s)
      ;

  jittr_change_name(-1, av0);

#if 0
  NetmaskDefault(&server.netmask,     0xffff, t, &errp1);
  NetmaskDefault(&server.netmask_write, 0xff, t, &errp1);
#else
  dstring_append(&server.netmask,       0, NULL, 0);
  dstring_append(&server.netmask_write, 0, NULL, 0);
#endif
  server.connect_timeout = 10;
  if (dstring_length(errp1))
    jittr_error("init: NetmaskDefault(..., %s)\n", t, 0, 0, 0);
  
  if (errp1) free((char *)errp1);
  return 0;
}

int 
jittr_init2(port)
int port;
{
  dstring *errp1 = NULL;
  dstring *errp2 = NULL;
  int choose = port ? 0 : 1;
  int first = choose ? JITTR_PORT : port;
  int last = choose ? (JITTR_PORT + 32) : port;
  int tfd = -1;

  jittr_gettimeval = gettimeval;

  if (!port) port = JITTR_PORT;
    
  server.tcp_port_fd = server.udp_port_fd = -1;
  while (ListenPort((choose?0:SO_REUSEADDR)|SOCK_STREAM, port, &tfd, &errp1) ||
  	 ListenPort((choose?0:SO_REUSEADDR)|SOCK_DGRAM, port, NULL, &errp2))
    {
      if (tfd >= 0) 
        PortShutdown(tfd, fd_client, NULL);
      if (++port > last)
        break;
      if (errp1) errp1->length = 0;
      if (errp2) errp2->length = 0;
      tfd = -1;
    }
  if (dstring_length(errp1))
    jittr_error("init: ListenPort(%d...%d) SOCK_STREAM: %s (%s).\n", 
    	first, last, errp1->buf, strerror(errno));
  debug1(port > last ? "SOCK_STREAM failed.\n" : "SOCK_STREAM=%d.\n", port);

  if (dstring_length(errp2))
    jittr_error("init: ListenPort(%d...%d) SOCK_DGRAM: %s (%s).\n", 
    	first, last, errp2->buf, strerror(errno));
  debug1(port > last ? "SOCK_DGRAM failed.\n" : "SOCK_DGRAM=%d.\n", port);

  if (errp1) free((char *)errp1);
  if (errp2) free((char *)errp2);

  return (port > last) ? -1 : 0;
}

int
jittr_init(av0, port)
char *av0;
int port;
{
  jittr_init1(av0);
  jittr_init2(port);
  jittr_change_name(-1, av0);	/* update port number in name */
  return 0;
}

#ifdef HAVE_LONGJMP
jmp_buf jittr_resurrect;
#endif

/* 
 * Main event loop calls alternatively do_io() and sched_timeout(),
 * while looping over sched_process() in between.
 * jittr_loop returns, when do_io() has trouble, or when the 
 * scheduler is empty and atom 0 is no longer marked A_PERMANENT.
 * The quit_cmd() makes atom 0 nonpermanenet.
 */
int
jittr_loop()
{
  struct timeval when, now;
  int i, j;
#define T 20

#ifdef HAVE_LONGJMP
  setjmp(jittr_resurrect);
#endif
  debug("jittr_loop() started.\n");

  jittr_gettimeval(&when);
  for (;;)
    {
      do
	{
	  i = sched_process(&j, &when);
#ifdef DEBUGG
	  if ((i > 0) && (j != 1))
	    debug2("jittr_loop: event Nr. %d returned %d\n", i, j);
#endif
	}
      while (i > 0);
      now = when;	/* updated by do_io() */
      i = sched_timeout(&when, NULL);	/* get absolute timestamp */
#ifdef DEBUGG
      if (i)
	{
	  char t[50];

	  sched_format_tv(t, 49, &when, &now);
	  debug2("jittr_loop: next event: Nr. %d in %s\n", i, t);
	}
#endif
      if (!i)
	{
/*
	  extern dstring *jittr_ioerr;
	  dstring_append(&jittr_ioerr, -1, "relax, just testing... ", 0);
*/
#if 1
	  static h = 0;
	  debug1("\r%c", "|/-\\"[h++]);
	  if (h > 3) h = 0;
#else
	  debug1("jittr_loop: wait %d sec. in do_io().\n", T);
#endif
	  jittr_gettimeval(&when);			/* is this needed? */
	  if (atoms[NEW_ATOM]->a.flag & A_PERMANENT)
	    when.tv_sec += T;
	  else
	    when.tv_usec += 100000;	/* otherwise select won't start */
	}
      if (do_io(&when, jittr_gettimeval) < 0)	/* need absolute timestamp */
        {
	  debug("do_io: select() error, exiting main loop\n");
          break;					/* select() error */
	}

      if (!(atoms[NEW_ATOM]->a.flag & A_PERMANENT))
        {
	  /* he pulled the ring. Tick, tack. tick tack... */

          if (sched_peek_event(NULL, NULL))
	    {
	      struct sched_action *ap;
	      int i = 0;

	      /* see if some jittr_process things are here. */
	      while ((i = sched_peek_event(NULL, &ap)) && 
	             (ap->fn != jittr_process))
	        ;
	      if (i)
	        continue;	/* waitaminute ... */
	    }
	  debug("NEW_ATOM no longer PERMANENT, exiting main loop\n");
	  break;		/* booom! */
	}
    }

  /* rarely reached */
  debug("deleting NEW_ATOM\n");
  atom_delete(NEW_ATOM);
  /* NOTREACHED */
  return 0;
}

#ifdef STANDALONE
/* ============ about nothing useful, except selftests ============== */
int
main(ac, av)
int ac;
char **av;
{
  int idx;

#ifdef DEBUG
/*
  OPENDEBUG("/tmp/jittr.dbg");
  */
  debugfp = stderr;
#endif

  atom_init();

  idx = atom_new(-1,  NULL, NULL, 0);
  dstring_set(&atoms[idx]->name, "testing_atoms", 0);
  dstring_set(&atoms[idx]->value, "Oh", 0);
  dstring_set(&atoms[idx]->value, "This atom has a rather long value", 0);
  dstring_set(&atoms[STDERR_ATOM]->value, "Initial error message.\n", 0);
  jittr_error("testing error %s\n", "logger", 0, 0, 0);
  idx = atom_lookup("NewAtom");
  idx = atom_lookup_atoms("NewAtom", 7, idx);
  atom_append(atom_lookup("testing_atoms"), 10, "is shorted", 0);

  {
    char buf[100];
    int l, n;
    char *p;

    sprintf(buf, "%s\n", av[1] ? av[1] : "\tand this    { \"} chop\n");
    l = strlen(p = buf);
    n = parse_check(p, l, NULL);
    printf("parse_check(\"%s\", %d, 0) = %d\n", p, l, n);

    if (n >= 0)
      l = ++n;
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
    printf("'%s'\n", parse_word(NULL, &p, &l, &idx, NULL));
  }


  atoms[NEW_ATOM]->a.flag &= ~A_PERMANENT;
  atom_delete(NEW_ATOM);
  return 0;
}
#endif /* STANDALONE */
