/*
 * jmod.c -- loadable modules for jittr
 *
 * This file references the external symbol mbuiltins. If you do not know what
 * that is, then provide a dummy 'struct mbuiltin mbuiltins[] = {{0}};'
 *
 * 28.3.96 jw
 *
 * modload & moddup can now jmod_push on existing atoms.
 * 05.12.97, jw: jittr_mount learns type 'l'.
 */

#include <math.h>		/* for strtod() */
#include <stdlib.h>		/* for strtod() */
#include "jittr/atom.h"		/* __P */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"
#include "jittr/modload.h"	/* jmod_shl_t */
#include "jittr/mbuiltin.h"	/* struct mbuiltin *mbuiltins */

/* ============================ jittr specific section ==================== */

/*
 *
 * A module is a shared library, residing in the unix filesystem or in
 * builtin code. A module can be mounted ontop of any atom. If the atom
 * does not exist, it is created.
 * 
 * Loaded modules are in one of two states: Initialized or deinitialised.
 * A newly loaded module is deinitialized. The initializer of the module is
 * set as an exclusive exec trace callback on its atom. Thus all previous
 * exec traces are disabled while this trace exists.  The deinitializer is
 * set as a delete trace callback on its atom. Its purpose is also
 * unloading the module.
 * 
 * Executing the atom may require parameters. E.g. the name of a serial
 * line and possibly its attribute. These parameters are recorded as the
 * value of the atom and the initializer is called. 
 * The initializer usually creates atoms representing the state of the
 * device which is instantiated by the module. These atoms should be linked
 * as a subtree to the current atom to allow multiple instances of the same
 * module.  If the module was already initialized, executing the atom will
 * implicitly deinitialize it, before initializing it anew.
 * 
 * modload [atom] <filename>
 * If the atom is omited, filename stripped from any path components is
 * used to create an atom of that name. The atom is flaged readonly and
 * permanent. Multiple modules can be stacked onto the same atom.  But
 * modload deinitializes the underlying module, before pushing a new one.
 * 
 * modunload <atom>
 * Unload the topmost module from an atom. This deinitializes the module
 * instance mounted on an atom and removes its initializer. The underlying
 * module (if any) is still deinitialized but can be accessed again.
 * 
 * modstat
 * lists all atoms which have modules. Each with a list of their module
 * names.  For the topmost module the initializer parameters are shown, if
 * initialized.  
 *
 * libload <filename> [symbol ...]
 * In case a module fails to resolve symbols from some strange library,
 * this library can be added through libload. No initializer will be called,
 * but the libraries sybols will be available to the module. Optionally
 * mention a few symbols to let the dynamic linker know they are referenced.
 * We do not expect modules to rely on this feature. Well written modules
 * should be self-contained (except for references into libjittr.a).
 */

struct module
{
  struct module *next;		/* global module chain */
  jmod_shl_t handle;		/* shared library handle of the module */
  int atom_idx;			/* where it was loaded to */
  char *filename;		/* where it was loaded from */
  char *modname;		/* its name & version number */
  char **modhelp;		/* jittr style help text */
  char **modstruct;		/* instance building jittr command fragments */
  struct dstring *param;	/* saved copy of the modmain argument string */
  int (*mod_main_fn) __P((int atom_idx, int flag, void **private));
  int (*mod_exit_fn) __P((int atom_idx, void **private));
  void *private;
};

/*
 * new modules must come at the beginning of this chained list, 
 * so that modules on an atom appear to be a stack.
 */
static struct module *mod_root = NULL;

/* forward declarations */
static int modpop_trace __P((int, int, void *));
static int modmain_trace __P((int, int, void *));

/*
 * Get a pointer for the module handle on atom at.
 * The any-atom is at = -1; the any handle is NULL.
 */
static struct module **
jmod_lookup(rp, at, handle)
struct module **rp;
int at;
jmod_shl_t handle;
{
  if (!rp)
    rp = &mod_root;
  while (*rp)
    {
      if (((at >= 0 && at == (*rp)->atom_idx)) ||
          ((handle != NULL && handle == (*rp)->handle)) ||
	  (at < 0 && !handle))
	break;
      rp = &((*rp)->next);
    }
  return rp;
}

/*
 * jmod_pop() returns 0, if the atom idx got rid of its toplevel module, if 
 * handle is NULL. Otherwise hadle is used to lookup which of the atoms modules
 * are to be removed.
 * Errors may happend during close, but this does not affect the return value.
 * jmod_pop() returns negative, if there are no modules on atom idx.
 */
static int
jmod_pop(idx, handle)
int idx;
jmod_shl_t handle;
{
  struct module **mpp = jmod_lookup(NULL, idx, handle);
  struct module *m;

  if (!mpp || !*mpp)
    return 1;		/* nothing pops here */
  m = *mpp;
  *mpp = m->next;
  if (m->mod_exit_fn)
    m->mod_exit_fn(idx, &(m->private));

  /* if the module has been dup'ed, we must not close it */
  if (!*jmod_lookup(NULL, 0, m->handle))
    {
      dstring *err = NULL;

      if (m->filename && jmod_dlclose(m->handle, &err))
        jittr_error("%s: warning: jmod_dlclose(%s): %s.\n",
	  jarg_name(), m->modname, err->buf, 0);
      /*
       * We must continue here, even if we have problems;
       * because we did (and had to) call mod_exit_fn before.
       */
    }
  atom_trace_delete(atom_trace_lookup(&atoms[idx]->trace, modmain_trace, 
  		    (void *)m));
  atom_trace_delete(atom_trace_lookup(&atoms[idx]->trace, modpop_trace,
  		    (void *)m));
  
  if (m->filename) free((char *)m->filename);
  if (m->param)    free((char *)m->param);
  free((char *)m);
  return 0;
}

static int
modpop_trace(idx, flag, data)
int idx, flag;
void *data;
{
  struct module *m = (struct module *)data;
  
  ASSERT(idx == m->atom_idx);	/* someone got a moddup wrong? */
  debug2("modpop_trace calling '%s' on atom %s\n", m->modname, atom_name(idx));
  return jmod_pop(idx, m->handle);
}


/* 
 * Initially we don't execute the modmain method directly, because
 * we need a few preparations: exec params are recorded and the private
 * pointer is initialized to NULL.
 *
 * If the modules modmain method wishes to remain active, it should call
 * atom_trace_set(idx, TRACE_..., ZZZ_modmain, (void *)m);
 * 
 */
static int
modmain_trace(idx, flag, data)
int idx, flag;
void *data;
{
  struct module *m = (struct module *)data;
  char **args;
  int n;
  
  ASSERT(idx == m->atom_idx);	/* someone got a moddup wrong? */
  debug3("modmain_trace (flag=%d) calling '%s' on atom %s\n", 
    flag, m->modname, atom_name(idx));
  
  if ((flag & TRACE_EXEC) && (args = jarg_rawp(&n)))
    dstring_append(&m->param, 0, *args, n);
  atom_trace_delete(atom_trace_lookup(&atoms[idx]->trace, modmain_trace, data));
  m->private = NULL;
  return m->mod_main_fn(idx, flag, &m->private);
}

/*
 * This one is different than others: it collects help from the modules.
 * It mimics the behaviour of and duplicates code from atom_help_cmd().
 */
static int
jmod_help_cmd(idx, flag, data)
int idx, flag;
void *data;
{
  char **help = (char **)data;
  struct module **mp = NULL;
  dstring **rp = &atoms[REPLY_ATOM]->value;

  atom_help_cmd(idx, flag, data);

  atom_append(REPLY_ATOM, -1 , mod_root ? 
    "#\n# Available commands in jittr modules are:\n" : 
    "#\n# No modules loaded.\n", 0);

  while (*(mp = jmod_lookup(mp, -1, NULL)))
    {
      help = (*mp)->modhelp;

      while (help && *help)
        {
	  atom_help_fmt(rp, (*mp)->atom_idx, help[0], help[1]);
	  atom_append(REPLY_ATOM, -1, ".\n", 2);
	  (*rp)->data = (void *)(*rp)->length;
	  help += 2;
	}
      mp = &(*mp)->next;
    }
  return 0;
}

static int
jmod_push(idx, m)
int idx;
struct module *m;
{
  struct module *mp;

  /* deinitialze atom's top module, if any */
  if ((mp = *jmod_lookup(NULL, idx, NULL)) && mp->mod_exit_fn)
    {
      debug3("hiding existing '%s' under new '%s' on atom %s\n", 
      	mp->modname, m->modname, atom_name(idx));
      mp->mod_exit_fn(idx, &mp->private);
    }

  m->atom_idx = idx;
  m->next = mod_root;
  mod_root = m;

  /* 
   * With the combination LATE & LAST, we have formed a stack:
   * Only the trace that was pushed last, is executed.
   */
  atom_trace_set(idx, (TRACE_MODE_MASK & ~TRACE_UNSET)|TRACE_LATE|TRACE_LAST,
    modmain_trace, (void *)m);
  atom_trace_set(idx, TRACE_UNSET, modpop_trace, (void *)m);
  return 0;
}

int
jittr_init_builtin(b)
struct mbuiltin *b;
{
  struct module *m;
  char *p = (char *)b->atomname;
  int idx = 0;		/* have no exec_context, cannot do jittr_getcwd(); */

  if (p && (idx = jittr_mknod(0, p, A_PERMANENT)) < 0)
    return -1;

  if (!(m = (struct module *)calloc(sizeof(struct module), 1)))
    return jittr_error("jittr_init_builtin: %s: out of memory!\n", p, 0, 0);

  m->atom_idx = idx;
  m->filename = NULL;		/* a builtin module has no filename */
  m->handle = (jmod_shl_t)b;	/* we need a handle to recognize dups */
  m->mod_main_fn = b->modmain;
  m->mod_exit_fn = b->modexit;
  m->modname = (char *)b->modname;
  m->modhelp = (char **)b->modhelp;
  m->modstruct = (char **)b->modstruct;
  if (!m->modname)
    m->modname = strdup(p);
  return jmod_push(idx, m); 
}

int jittr_argc = 0;
char **jittr_argv = NULL;

/*
 * Idx gives the index of the arg that shall be removed from jittr_argv.
 * This is done, by copying all subsequent args pointers to one lower position.
 */
int
jittr_shift_argv(idx)
int idx;
{
  if (idx < 0 || idx >= jittr_argc)
    return -1;
  while (++idx < jittr_argc)
    jittr_argv[idx-1] = jittr_argv[idx];
  jittr_argv[--jittr_argc] = NULL;
  return jittr_argc;
}

int
jittr_store_argv(av0, acp, avp)
char *av0;
int *acp;
char ***avp;
{
  struct module **mp = NULL;

  jittr_argc = *acp;
  jittr_argv = *avp;

  while (*(mp = jmod_lookup(mp, -1, NULL)))
    {
      if (*mp != *jmod_lookup(NULL, (*mp)->atom_idx, NULL))
        continue;	/* only initialize uncovered modules */
      /* hello module, *private is write-flag, that means get your argv now! */
      if (atom_append((*mp)->atom_idx, 0, av0, 0) < 0) 
        return -1;	/* critical error in module instantiation */
      mp = &(*mp)->next;
    }
  *avp = jittr_argv;
  return *acp = jittr_argc;
}

/*
 * Mount the data structure described in modstruct argv on atom idx.
 *
 * each line has the following layout:
 *
 * argv[0]: integer, (not char *)! byte offset of location in iarr, where to
 * 	    return (int)atom_idx. Not written, if (int)argv[0] negative.
 * argv[1]: pathname of the thing
 * argv[2]: type of the thing
 * argv[3]: value string of the thing. parse_word() applies.
 *
 * the following subtypes are recognize:.
 *   'a' atom. with all directoy links to make things look nice.
 *       has the subtypes n: no inheritance of flags.
 *                        r: readonly,
 *                        p: permanent.
 *			  1: add a value link .widget -> button
 *   'v' value, a simple data link. no subtypes.
 *   'l' link, a simple atom link by name. no subtypes.
 */
int
jittr_mount(idx, argv, iarr)
int idx;
char **argv;
void *iarr;
{
  struct dstring *d = NULL;
  char *p = NULL;	/* can't parse inplace, because of const argv[] */
  int slen = 0;
  int line = 1;
  int a, r = 0;

  while (argv && argv[1])
    {
      if (!argv[2] || (!strchr(argv[2], 'a') && !argv[3]))
        {
	  if (d) free((char *)d);
          return jittr_error("jittr_mount(%d): Line %d: bad triple '%s'.%s",
			     idx, line, argv[1], " Comma missing?\n");
        }

      p = argv[3];
      if (d) d->data = (void *)d->buf;
      slen = p ? strlen(p) : 0;
      parse_word(&d, &p, &slen, NULL, NULL);
      if (strchr(argv[2], 'a'))
        {
	  a = jittr_mknod(idx, argv[1], 0);
	  if (atom_exists(a))
	    {
	      atom_append(a, 0, d->buf, d->length);
	      atoms[a]->a.flag = strchr(argv[2], 'n') ? 0 : atoms[idx]->a.flag;
	      if (strchr(argv[2], 'r')) atoms[a]->a.flag |= A_READONLY;
	      if (strchr(argv[2], 'p')) atoms[a]->a.flag |= A_PERMANENT;
	      if (strchr(argv[2], '1'))
		jittr_do_link(a, ".widget", 7, "button", 6, 1);
	      
	      /* record the atom index, if desired */
	      if (((int)*argv >= 0) && iarr)
	        *(int *)((char *)iarr + (int)*argv) = a;
	    }
	}
      else if (strchr(argv[2], 'v'))
	r |= jittr_link(idx, argv[1], strlen(argv[1]), d->buf, d->length, 1);
      else if (strchr(argv[2], 'l'))
        {
	  static dstring *n = NULL;

	  if (d->buf[0] == '.' && d->buf[1] == '/')
	    {
	      dstring_append(&n, 0, atom_name(idx), 0);
	      dstring_append(&n, -1, d->buf+1, d->length - 1);

	      r |= jittr_link(idx, argv[1],strlen(argv[1]), n->buf,n->length,0);
	    }
	  else
	    r |= jittr_link(idx, argv[1],strlen(argv[1]), d->buf,d->length,0);
	}
      else
        {
	  free((char *)d);
          jittr_error("jittr_mount(%d): Line %d: Type '%s' %s: ",
		idx, line, argv[2], "is unknown. Please use 'a' or 'v'");
	  return jittr_error("name=\"%s\", type=\"%s\", value=\"%s\"\n",
	  	argv[1], argv[2], argv[3]);
	} 
      if (d)
        d->length = 0;
      argv += 4;
      line++;
    }
  return r;
}

/*
 * jittr_check_numeric() implements the atom datatypes .state and .min/.max
 * If the atom idx does not have one of these type flags, 1 is returned without
 * touching *np. Otherwise the value is interpreted as follows:
 * If the value makes sense, its numerical equivalent is written into the atom
 * and (if np is nonzero) into *np. 0 is returned then. If out of range, the
 * returned value is bound within the .min/.max limits. The value of .min may
 * be greater or smaller than the value of .max. Thus inverted sliders are
 * supported.
 * If in trouble, jittr_check_numeric() returns with jittr_error.
 */ 
int
jittr_check_numeric(idx, np, dp)
int idx, *np;
double *dp;
{
  char *p, *r, *s;
  int num, l;
  double d;

  if ((r = jittr_link_resolve(idx, ".state", 6, NULL, NULL)))
    {
      p = atom_value(idx);
      while (*p == ' ' || *p == '\t')
        p++;
      num = strtol(p, &s, 0);
      while (*s == ' ' || *s == '\t')
        s++;
      if (*s)	/* not a pure integer */
	{
	  dstring *st = NULL;
	  int sl, pl = strlen(r) + 1; /* include '\0'-byte */

	  l = strlen(s = p);
	  for (num = 0, p = r; *p; num++)
	    {
	      parse_word(&st, &p, &pl, &sl, NULL);
	      if ((l == sl) && !strncasecmp(s, st->buf, l))
		{
		  p = st->buf;
		  break;
		}
	      st->data = (void *)st->buf;
	    }
	  if (!*p)
	    {
	      if (st) free((char *)st);
	      for (p = r; *p && (*p != ' '); p++)
	        ;
	      jittr_error("jittr_check_numeric: %s: bad value %s. Should be ",
	      	atom_name(idx), atom_value(idx), 0, 0);
	      dstring_append(&atoms[idx]->value, 0, r, p - r);
	      return jittr_error("numeric or one of {%s}. Resetting to %s.\n", 
	        r, atom_value(idx), 0, 0);
	    }
	  if (st) free((char *)st);
	  dstring_appendn(&atoms[idx]->value, 0, NULL, num);
	}
      if (np) *np = num;
      if (dp) *dp = (double)num;
      return 0;
    }

  if ((r = jittr_link_resolve(idx, ".max", 4, NULL, NULL)))
    {
      double mi, ma;

      d = strtod(atom_value(idx), &s);
      if (*s)
        return jittr_error("jittr_check_numeric: %s: bad value %s. %s.\n",
		atom_name(idx), atom_value(idx), 
		dp ? "Must be integer or floating point" : "Must be integer", 
		0);

      ma = strtod(r, NULL);
      mi = (r = jittr_link_resolve(idx, ".min", 4, NULL, NULL)) ? strtod(r, NULL) : 0.0;

      if ((d != ma) && ((d > ma) ^ (mi > ma)))
        {
	  debug3("atom %d: value %g out of range. set to %g.\n", idx, d, ma);
	  dstring_appendg(&atoms[idx]->value, 0, NULL, d = ma);
	}

      if ((d != mi) && ((d < mi) ^ (ma < mi)))
        {
	  debug3("atom %d: value %g out of range. set to %g.\n", idx, d, mi);
	  dstring_appendg(&atoms[idx]->value, 0, NULL, d = mi);
	}
      if (np) *np = (int)((d < 0.0) ? (d - .5) : (d + .5));
      if (dp) *dp = d;
      return 0;
    }

  /* 
   * An atom without hints containing a pure numeric value is a number.
   */
  d = strtod(atom_value(idx), &s);
  if (*s) return 1;		/* no, it is not */
  if (dp) *dp = d;
  if (np) *np = (int)((d < 0.0) ? (d - .5) : (d + .5));
  return 0;
}

/*
 * This is pretty straight forward:
 * Foreach path in argv, foreach element in each path, lookup the atom and
 * delete it. But don't delete ".", only strip off all links.
 *
 * This is not really a umount. This kills a hierarchy.
 */
int
jittr_umount(idx)
int idx;
{
  void *ptr = jittr_opendir(idx, JDIR_SAVE);
  struct dstring **dp;
  char *name;
  int a;

  while ((dp = jittr_readdir(ptr, &name)))
    {
      if (!*dp)
        continue;
      (void)jittr_link_resolve(idx, name, strlen(name), &a, NULL);
      if (!atom_exists(a))
        continue;
      jittr_umount(a);
      if (a)
        atom_delete(a);
    }
  jittr_closedir(ptr);
  /*
   * XXX Fixme: 
   * I fear it is unsecure to remove hash entries while inside
   * Tcl_NextHashEntry()... Have to read tcl doku.
   */
  jittr_error("jittr_umount: toplevel link removal not implemented.\n");

  return 0; 
}


/*
 * jmod_export() writes commands to d, that construct a partial data structure
 * as described by its other arguments.
 *
 * The commands that the Importer receives should be construction commands
 * and trace commands. As these trace commands are assembled by the
 * exporter, indices can be used.
 * the exporter can send echo commands that will contain the index of the 
 * newly created partner atom when it bounces back to the exporter.
 * these echos are trace commands...
 *
 * 
 *      We start with the trivial things as export -tree.  We send to our gui:
 * 		mknod /pbut/action {get 26 "set } { \"" \"\n}\n
 *      this creates the atom there, and requests the partner to ask back 
 *      for its value. If the mknod command failed, he will never ask back.
 *      For brevity (and speed), we can use numeric indices instead of the 
 *      command names and abbreviate all options to one letter.
 *      	4	get
 *      	5	set
 *      	9	trace
 *      	10	tell
 *      	13	mknod
 *      	14	link
 *      	20	export
 *
 * 13 /pbut/action {4 26 "5 } { \"" \"\n}\n
 * 	
 *      As a reply we will see the command
 *      	4 26 "5 37 \"" \"\n
 *      To which we will reply with the contents of the exported atom:
 *      	5 37 "..."
 *      	
 *
 *      With -link we need to estabslish all the directly linked constant 
 *      data values and two traces per atom. A link is done like this:
 *
 * 14 -d {.75-.2 .6-.1 .4 .2} /pbut/button/b6/.location
 * 
 *      Now the traces. First his trace. Knowing that our /jpbutd/action is
 *      index 26, we send him this, to directly create his trace callback:
 *
 * 9 /jpbutd/action {4 . "@\@ 5 26 \"" \"\n}
 *
 *      Second, the outbound direction: still only knowing our 26 we send him 
 *
 * 4 /jpbutd/action -i '9 26 {4 . "@\@ 5 ' ' \"" \"\n}'\n
 *
 *      which will bounce back a command containing the number 37 of his atom:
 *      	9 26 {4 . "@\@ 5 37 \"" \"\n}
 *      receiving this, our trace is ready. When his atom is set, we'll receive
 *      	@15:07h6.443676s,6/28/1996 5 26 "..."
 *      and when ours is set, he'll be notified:
 *      	@15:11h45.739595s,6/28/1996 5 37 "..."
 */
static int
jmod_export(d, pass, source, dest, type, lval)
struct dstring **d;
int pass;
char *source, *dest, *type, *lval;
{
  int idx;

  ASSERT( LINK_CMD_ATOM == 14);
  ASSERT(MKNOD_CMD_ATOM == 13);
  ASSERT(TRACE_CMD_ATOM ==  9);
  ASSERT(  SET_CMD_ATOM ==  5);
  ASSERT(  GET_CMD_ATOM ==  4);
  
  switch (pass)
    {
    case 0:		/* -tree */
      if (strchr(type, 'a'))
        {
	  idx = atom_lookup(source);

 	  /*	13 /pbut/action {4 26 "5 } { \"" \"\n}\n 	*/

	  dstring_append(d, -1, "13 ", 3);
	  dstring_append(d, -1, dest, 0);
	  dstring_appendn(d,-1, " {4 %d \"5 } { \\\"\" \\\"\\n}\\n\n", idx);
	}
      break;
    case 1:		/* -links */
      if (strchr(type, 'a'))
        {
	  idx = atom_lookup(source);

	  if (!(atoms[idx]->a.flag & A_READONLY))
	    {
	      /*	9 /jpbutd/action {4 . "@\@ 5 26 \"" \"\n}	*/

	      dstring_append(d, -1, "9 ", 2);
	      dstring_append(d, -1, dest, 0);
	      dstring_appendn(d,-1, " {4 . \"@\\@ 5 %d \\\"\" \\\"\\n}\n", idx);
	    }

 	  /*	4 /jpbutd/action -i '9 26 {4 . "@\@ 5 ' ' \"" \"\n}'\n	*/

	  dstring_append(d, -1, "4 ", 2);
	  dstring_append(d, -1, dest, 0);
	  dstring_appendn(d,-1, " -i '9 %d {4 . \"@\\@ 5 ' ' \\\"\" \\\"\\n}'\\n\n", idx);
	}
      else if (strchr(type, 'v'))
        {
 	  /*	14 -d {.75-.2 .6-.1 .4 .2} /pbut/button/b6/.location	*/
	  ASSERT(lval);	/* sorry, cannot read a link by fullname */
	  dstring_append(d, -1, "14 -d \"", 7);
	  dstring_appendq(d,-1, lval, 0);
	  dstring_append(d, -1, "\" ", 2);
	  dstring_append(d, -1, dest, 0);
	  dstring_append(d, -1, "\n", 1);
	}
      else
        return jittr_error("jittr_export: type '%s' not impl: %s '%s' '%s'.\n",
		type, source, type, lval ? lval : "");
      break;
    case 2:  /* - realtree */
    case 3:  /* - reallinks */
      {
	struct dstring *rdp, **dp;
	void *ptr;
	char *name;
	char sbuf[1024], dbuf[1024];
	idx = atom_lookup(source);
	ptr = jittr_opendir(idx, JDIR_SORT_IDX | JDIR_SAVE | JDIR_AUTO);
	while ((dp = jittr_readdir(ptr, &name)))
	  {
	    if (!(rdp = *dp))
	      {
		continue; /* jw's paranoid check. I'm it, too? */
	      }
	    if(!strcmp(name, ".")) /* could this happen? */
	      continue;
	    switch ((int)rdp->data)
	      {
		case JLINK_UNRESOLVED: /* ? */
		  break;
		case JLINK_UNRESOLVABLE:  /* non-atom link */
		  /* We can afford this overhead here and prevent *
		   * double code since this function shouldn't be *
		   * called VERY often during a session		  */
		  if(pass==2)
		    {
		      break;
		    }
		  dstring_append(d, -1, "14 -d \"", 7);
		  if(parse_needsq(rdp->buf, rdp->length))
		    {
		      /* "on off" -> "{on off}" */
		      /* I'm not sure if this is right for -1 ... -SB 09/17/97*/
		      dstring_append(d,-1, "{", 1);
		      dstring_appendq(d,-1, rdp->buf, 0);
		      dstring_append(d,-1, "}", 1);
		    }
		  else
		    {
		      dstring_appendq(d,-1, rdp->buf, 0);
		    }
		  dstring_append(d, -1, "\" ", 2);
		  dstring_append(d, -1, dest, 0);
		  dstring_append(d, -1, "/", 1);
		  dstring_append(d, -1, name, 0);
		  dstring_append(d, -1, "\n", 1);
		  break;
		default:  /* JLINK_IS_RESOLVED(d->data):  atom index */
		  if(pass==2)
		    {
		      /*	13 /pbut/action {4 26 "5 } { \"" \"\n}\n 	*/
		      dstring_append(d, -1, "13 ", 3);
		      dstring_append(d, -1, dest, 0);
		      dstring_append(d, -1, "/", 1);
		      dstring_append(d, -1, name, 0);
		      dstring_appendn(d,-1, " {4 %d \"5 } { \\\"\" \\\"\\n}\\n\n", (int)rdp->data);
		    }
		  else
		    {
		      if (!(atoms[idx]->a.flag & A_READONLY))
			{ /*	9 /jpbutd/action {4 . "@\@ 5 26 \"" \"\n}	*/

			  dstring_append(d, -1, "9 ", 2);
			  dstring_append(d, -1, dest, 0);
			  dstring_append(d, -1, "/", 1);
			  dstring_append(d, -1, name, 0);
			  dstring_appendn(d,-1, " {4 . \"@\\@ 5 %d \\\"\" \\\"\\n}\n", (int)rdp->data);
			}
		      /*  4 /jpbutd/action -i '9 26 {4 . "@\@ 5 ' ' \"" \"\n}'\n	*/

		      dstring_append(d, -1, "4 ", 2);
		      dstring_append(d, -1, dest, 0);
		      dstring_append(d, -1, "/", 1);
		      dstring_append(d, -1, name, 0);
		      dstring_appendn(d,-1, " -i '9 %d {4 . \"@\\@ 5 ' ' \\\"\" \\\"\\n}'\\n\n", (int)rdp->data);
		    }
		  /* Go Down */
		  sprintf(sbuf, "%s/%s", source, name);
		  sprintf(dbuf, "%s/%s", dest, name);
		  jmod_export(d, pass, sbuf, dbuf, type, NULL); 
		  break;
	      }
	  }
	jittr_closedir(ptr);
      }
      break;
    default:
      ASSERT(0);	/* Pardon? Did you change export_cmd()? */
    }
  return 0;
}

/* 
 * jittr_program_export_dstring() constructs a program in d, that when sent to a
 * partner causes him to export the atom hierarchy s back to us. We will see it
 * under the name t, and will create widgets on arrival by executing cmd_atom.
 *
 * Used only by jxcmd.c:x11_cmd(), but does not belong into X11 related code.
 */ 
int
jittr_program_export_dstring(d, s, slen, t, tlen, cmd_atom)
struct dstring **d;
char *s, *t;
int slen, tlen, cmd_atom;
{
  ASSERT(EXPORT_CMD_ATOM == 20);
  ASSERT(   APP_CMD_ATOM ==  6);
  ASSERT(     REPLY_ATOM ==  1);

  dstring_append(d, -1, "\n20 ", 4);
  dstring_append(d, -1, s, slen);
  dstring_append(d, -1, " -t ", 4);
  dstring_append(d, -1, t, tlen);
  dstring_append(d, -1, " {6 1 {20 ", 10);
  dstring_append(d, -1, s, slen);
  dstring_append(d, -1, " -l ", 4);
  dstring_append(d, -1, t, tlen);
  dstring_appendn(d, -1, " {%d ", cmd_atom);
  dstring_append(d, -1, t, tlen);
  dstring_append(d, -1, "}\\n}\\n}\\n\n", 10);

  return 1;
}

/*
 * export atom [-tree|-link] [prefix [return-command]]
 *
 * When options -tree or -link are specified, return-command defaults to
 * none, otherwise it defaults to "export atom -l prefix\n"
 */
static int
export_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen, line, autocmd = idx;
  int pass = 0;		/* 0 == export tree, 1 == export links & traces */
  char *p = jarg_first_word(NULL);
  char *s = jarg_next_word(&slen);
  char sbuf[1024], dbuf[1024];
  char *source_root, *dest_root, **argv;
  struct module *m;
  
  if (!p)
    return jittr_error("Use: %s atom [-l|-t|-realtree|-reallinks] [pre [cmd]] (parameters missing)", 
      jarg_name(), 0, 0, 0);

  if (s && slen > 1 && 
      (!strncmp("-tree", s, slen) || !strncmp("-links", s, slen)
	    || !strncmp("-realtree", s, slen) || !strncmp("-reallinks", s, slen)))
    {
      /* XXX overhead */
      pass = (s[1] == 't') ? 0 : (s[1] == 'l') ? 1 : (!strncmp("-realtree", s, slen)) ? 2 : 3;
      s = jarg_next_word(&slen);
      autocmd = 0;
    }

  /* Find the module structure that is mounted on the atom. */

  if ((idx = atom_lookup(p)) < 0)
    return jittr_error("%s: no such atom: %s\n", jarg_name(), p, 0, 0);

  source_root = atom_name(idx);
  if (*source_root != '/')
    return jittr_error("%s: fullname of %s does not start with '/': %s.\n",
    	jarg_name(), p, source_root, 0);
  dest_root = s ? s : source_root;

  ASSERT(autocmd == EXPORT_CMD_ATOM || autocmd == 0);	/* me or nothing */
  ASSERT(REPLY_ATOM == 1);

  if ((pass < 2) && !(m = *jmod_lookup(NULL, idx, NULL)))
    {
      /* nothing mounted here, simply export the single atom */
      jmod_export(rp, pass&1, source_root, dest_root, "a", NULL);
      /* keep in sync with last lines if export_cmd() */
      if ((s = jarg_next_word(&slen)))
	dstring_append(rp, -1, s, slen);
      else if (autocmd)
	dstring_appendf(rp, -1, "%d 1 {%d %d -l %s {}}\\n\n", 
    		    APP_CMD_ATOM, autocmd, idx, dest_root);
      return 1;
    }

  /*
   * Export only prints text. It does not change anything directly.
   * It prints one of two sorts of text:
   * With the -tree option (or without any option) it prints a description
   * of the atom hierarchy of the module.
   * With the -link option, it prints commands that cause mutual trace 
   * callbacks to be established, if the receiver understands them.
   *
   * After printing, the return command is echoed. The purpose of the
   * return command is to get feedback that everything is done.
   *
   * Example usage: 
   * A curious shell sends to another:
   *
   *	20 jmodtest /remote {6 1 {20 jmodtest -l /remote {x11 /remote}\n}\n}\n
   * 
   * a gui sends this to the jpbutd daemon:
   *
   * 	20 /jpbutd -t /pbut {6 1 {20 /jpbutd -l /pbut {x11 /pbut}\n}\n}\n
   *
   * This causes a fully functional duplicate of the jpbutd module structure to
   * appear in the gui. The daemon and the gui will have trace callbacks 
   * established, so that all read and writes on the atoms are synchronized.
   *
   */

  if(pass>1)
    {
      /* needed to get '.' at start, too */
      jmod_export(rp, pass&1, source_root, dest_root, "a", NULL);
      jmod_export(rp, pass, source_root, dest_root, "a", NULL);
    }
  else /* pass <2 */
    {
      jmod_export(rp, pass, source_root, dest_root, "a", NULL);
      /*
       * Take its modstruct description and print a sequence of jittr commands
       * which can create the same tree, rooted at prefix in the reciever of the
       * commands. If no prefix is specified, the atom name is used.
       */ 
      argv = m->modstruct;
      line = 1;

      while (argv && argv[1])
	{
	  if (((argv[1][0] != '.') && (argv[1][0] != '/')) || !argv[2])
	    return jittr_error("export_cmd(%d): Line %d: bad triple '%s'.%s",
			       idx, line, argv[1], " Comma missing?\n");

	  if (!strncmp("./", argv[1], 2))
	    {
	      sprintf(sbuf, "%s%s", source_root, argv[1] + 1);
	      sprintf(dbuf, "%s%s", dest_root, argv[1] + 1);
	      jmod_export(rp, pass, sbuf, dbuf, argv[2], argv[3]); 
	    }
	  else
	    jmod_export(rp, pass, argv[1], argv[1], argv[2], argv[3]); 

	  argv += 4;
	  line++;
	}
    }
  if ((s = jarg_next_word(&slen)))
    dstring_append(rp, -1, s, slen);
  else if (autocmd)
    dstring_appendf(rp, -1, "%d 1 {%d %d -l %s {}}\\n\n", 
    		    APP_CMD_ATOM, autocmd, idx, dest_root);
  return 1;
}

static int
modload_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  char buf[30];
  jmod_shl_t h;
  struct module *m;
  struct dstring *err = NULL;
  dstring **rp = &atoms[REPLY_ATOM]->value;
  char *p = jarg_first_word(NULL);	/* atom or path */
  char *q = jarg_next_word(NULL);	/* path or NULL */
  char *s = q ? q : p;			/* path */
  char *f = s;				/* file basename extraced from s */

  if (!s)
    return jittr_error("%s: load what? (parameter missing)", 
      jarg_name(), 0, 0, 0);

  if (s == p)	/* if we have only one parameter, we have no atom name */
    p = NULL;	

  /* collect all resources, to see if we have a chance to proceeed later */
  if (jmod_dlopen(s, &h, &err))
    return jittr_error("%s: jmod_dlopen(%s) failed: %s\n", jarg_name(),
      s, err->buf, 0);

  if (!(s = strdup(s)))
    return jittr_error("%s: out of mem.\n", jarg_name(), jmod_dlclose(h, NULL),
      0, 0);

  /* construct basename */
  q = f;
  while (*f++)
    if (f[-1] == '/')
      q = f;
  f = q;
  while (*q)
    if (*q++ == '.')
      q[-1] = '\0';

  if (!p)	/* if we have no atom name, use basename as atom name */
    p = f;

  /* make sure atom p exists */
  if ((idx = atom_lookup(p)) < 0 &&
      (idx = jittr_mknod(jittr_getcwd(), p, 0)) < 0)
    {
      free(s);
      return jittr_error("%s: mknod for '%s' failed.\n", 
	jarg_name(), p, jmod_dlclose(h, NULL), 0);
    }

  if (!(m = (struct module *)calloc(sizeof(struct module), 1)))
    {
      free(s);
      return jittr_error("%s: out of memory!\n", jarg_name(), 
	jmod_dlclose(h, NULL), 0, 0);
    }

  /* build the module structure and link it */
  m->handle = h;
  m->atom_idx = idx;
  m->filename = s;
  m->modname = NULL;
  
  p = buf; q = f;
  while (*q)
    *p++ = *q++;
  *p++ = '_';

  sprintf(p, "modmain");
  if (jmod_dlsym(buf, 1, h, (void *)&m->mod_main_fn, &err))
    {
      sprintf(buf, "ZZZ_modmain"); p = buf+4;
      if (jmod_dlsym(buf, 1, h, (void *)&m->mod_main_fn, &err))
	{
	  sprintf(buf, "modmain"); p = buf;
	  if (jmod_dlsym(buf, 1, h, (void *)&m->mod_main_fn, &err))
	    {
	      free((char *)m->filename);
	      free((char *)m);
	      return jittr_error("%s: symbol 'modmain' missing in %s: %s.\n", 
		jarg_name(), s, err->buf, 0);
	    }
	}
    }

  sprintf(p, "modexit");
#if 0
  if (jmod_dlsym(buf, 1, h, (void *)&m->mod_exit_fn, &err))
    jittr_error("%s: warning: %s has no 'modexit': %s.\n",
	jarg_name(), s, err->buf, 0);
#else
  jmod_dlsym(buf, 1, h, (void *)&m->mod_exit_fn, &err);
#endif
  sprintf(p, "modname");   jmod_dlsym(buf, 0, h, (void *)&m->modname, &err);
  sprintf(p, "modhelp");   jmod_dlsym(buf, 0, h, (void *)&m->modhelp, &err);
  sprintf(p, "modstruct"); jmod_dlsym(buf, 0, h, (void *)&m->modstruct, &err);
  if (!m->modname)
    m->modname = m->filename;
  
  (void)jmod_push(idx, m);
  dstring_appendf(rp, -1, "# %d", idx);
  atom_append(REPLY_ATOM, -1, "\n", 1);
  return 0;
}

/* 
 * Dear module! 
 * Please don't be surprised, if you modmain() method is called
 * with a different atom some day. This means, that another instance of 
 * yourself is being created. Don't worry about instance specific data.
 * You'll have a new private pointer then, where all this can be stored.
 */ 
static int
moddup_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  char *p = jarg_first_word(NULL);
  char *q = jarg_next_word(NULL);
  struct module *mp, *m;
  
  if (!p || !q)
    return jittr_error("%s usage: source destination (parameters missing)", 
      jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(p)) < 0)
    return jittr_error("%s: no such atom: %s\n", jarg_name(), p, 0, 0);
  if (!(mp = *jmod_lookup(NULL, idx, NULL)))
    return jittr_error("%s: no module on atom %s\n", jarg_name(), p, 0, 0);

  if ((idx = atom_lookup(q)) < 0 &&
      (idx = jittr_mknod(jittr_getcwd(), q, 0)) < 0)
    return jittr_error("%s: mknod for '%s' failed.\n", jarg_name(), q, 0, 0);

  if (!(m = (struct module *)calloc(sizeof(struct module), 1)))
    return jittr_error("%s: malloc failed.\n", jarg_name(), 0, 0, 0);

  *m = *mp;	/* structure copy */
  m->atom_idx = -1;
  m->filename = mp->filename ? strdup(mp->filename) : NULL; /* NULL==builtin */
  m->private = NULL;
  m->param = NULL;

  (void)jmod_push(idx, m);
  dstring_appendf(rp, -1, "# %d", idx);
  atom_append(REPLY_ATOM, -1, "\n", 1);
  return 0;
}

static int
modunload_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  char *p;

  if (!(p = jarg_first_word(NULL)))
    return jittr_error("%s: atom name missing.", jarg_name(), 0, 0, 0);

  if ((idx = atom_lookup(p)) < 0)
    return jittr_error("%s: no such atom: %s\n", jarg_name(), p, 0, 0);

  if (jmod_pop(idx, NULL))
    return jittr_error("%s: no module on atom %s\n", jarg_name(), p, 0, 0);

  return 0;
}

static int
modstat_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  struct module **mp = NULL;
  char *p;

  idx = -1;
  if ((p = jarg_first_word(NULL)) && (idx = atom_lookup(p)) < 0)
    return jittr_error("%s: no such atom: %s\n", jarg_name(), p, 0, 0);

  while (*(mp = jmod_lookup(mp, idx, NULL)))
    {
      dstring_appendf(rp, -1, "# %d {%s} ", (*mp)->atom_idx,
        (*mp)->modname, 0, 0);
      if ((*mp)->param)
	dstring_append(rp, -1, (*mp)->param->buf, (*mp)->param->length);
      atom_append(REPLY_ATOM, -1, "\n", 1);
      mp = &(*mp)->next;
    }
  return 0;
}

static char *help[] = 
{
  "modload [atom] file",	"load a jittr module (uninitialized)",
  "modunload atom",		"remove a module",
  "moddup source_atom dest_atom",	"copy a module from mod_atom onto atom",
  "modstat [atom]",		"print info about some or all loaded modules",
  "export atom [-tree|-link|-realtree|-reallinks] [prefix [return-command]]",
  	"print jittr commands to construct the module mounted on atom. Without options and without return-command, -tree is executed and -link is programmed as a return-command. '-realtree' and '-reallinks' comply to '-tree' and '-link' except that they don't return the module-structure but parse the relly existing tree",
  NULL
};

int
jittr_jmod_init()
{
  atom_command(     HELP_CMD_ATOM, NULL,       jmod_help_cmd, (void *)help);
  atom_command(  MODLOAD_CMD_ATOM, "modload",    modload_cmd, (void *)0);
  atom_command(MODUNLOAD_CMD_ATOM, "modunload",modunload_cmd, (void *)0);
  atom_command(   MODDUP_CMD_ATOM, "moddup",      moddup_cmd, (void *)0);
  atom_command(  MODSTAT_CMD_ATOM, "modstat",    modstat_cmd, (void *)0);
  atom_command(   EXPORT_CMD_ATOM, "export",      export_cmd, (void *)0);
  debug("jittr_jmod_init: done.\n");
  return 0;
}

int
jittr_mbuiltin_init()
{
  struct mbuiltin *b = mbuiltins;

  while (b && b->modmain)
    {
      debug2("jittr_init_builtin: planting builtin module '%s' on atom '%s'.\n",
        b->modname  ? b->modname   : "unnamed",
	b->atomname ? b->atomname : ".");
      jittr_init_builtin(b);
      b++;
    }
  debug("jittr_mbuiltin_init: done.\n");
  return 0;
}

/* Stefan tells his vim: ts=8 sw=2
*/
