/*
 * jqueue.c -- a generic queueing mechanism for jittr devices
 *
 * This was modelled after the jvcc1/queue.c experiment.
 *
 * 23.5.97, jw
 */

#include "jittr/dstring.h"	/* dstring, xbcopy() */
#include "jittr/jqueue.h"	/* struct jittr_queue */

static int default_send_fn __P((jittr_queue *q, dstring *head, void *data));
static int jittr_queue_sched __P((jittr_queue *q));

static int
default_send_fn(q, d, dummy)
struct jittr_queue *q;
dstring *d;
void *dummy;
{
  debug2("default_send_fn: writing %d bytes to client %s\n", d->length, q->name->buf);
  dstring_append(q->obufp, -1, d->buf, d->length);
  d->length = 0;
  return 0;
}

/*
 * this should use the scheduler to place the send_fn.
 */
static int jittr_queue_sched(q)
jittr_queue *q;
{
  debug("jittr_queue_sched, direct send without scheduler\n");
  /*
   * if wait_sched_id is still not processed, do nothing.
   */
  ASSERT(q->send_fn);
  return q->send_fn(q, q->wait, q->send_fn_data);
}

/*
 * Construct an empty jittr queue and returns its pointer.
 * Default send_fn is a simple append to the clients obuf.
 */
jittr_queue *jittr_queue_init(char *name, struct client *cl, dstring **obuf)
{
  jittr_queue *q;

  if (!(q = (jittr_queue *)calloc(sizeof(jittr_queue), 1)))
    return NULL;
  
  q->alloced = 16;
  if (!(q->q = (dstring **)calloc(sizeof(dstring *), q->alloced)))
    {
      free((char *)q);
      return NULL;
    }

  dstring_append(&q->name, 0, name, 0);
  q->obufp = obuf;
  q->cl = cl;
  q->send_fn = default_send_fn;

  return q;
}

/*
 * destructs a jittr queue (even if non-empty).
 * if additional data pointers into allocated memory exist, flags should be set
 * accordingly or the should be freed before entering here.
 * The pointer of the queue is set to NULL.
 */
int jittr_queue_free(qq)
jittr_queue **qq;
{
  int i;

  if ((*qq)->q)
    {
      for (i = 0; i < (*qq)->alloced; i++)
	if ((*qq)->q[i])
	  {
	    if (((*qq)->flags & JQUEUE_FLAG_DS_DATA_ALLOC) && (*qq)->q[i]->data)
	      free((char *)((*qq)->q[i]->data));
	    free((char *)((*qq)->q[i]));
	  }
      free((char *)((*qq)->q));
      (*qq)->q = NULL;
      (*qq)->alloced = 0;
    }

  if ((*qq)->wait)
    {
      if (((*qq)->flags & JQUEUE_FLAG_DS_DATA_ALLOC) && (*qq)->wait->data)
	free((char *)((*qq)->wait->data));
      free((char *)((*qq)->wait));
      (*qq)->wait = NULL;
    }

  if (((*qq)->flags & JQUEUE_FLAG_CMP_DATA_ALLOC) && (*qq)->cmp_fn_data)
    free((char *)((*qq)->cmp_fn_data));

  if (((*qq)->flags & JQUEUE_FLAG_SEND_DATA_ALLOC) && (*qq)->send_fn_data)
    free((char *)((*qq)->send_fn_data));

  free((char *)*qq);
  *qq = NULL;
  return 0;
}

/* 
 * Remove empty entries in the queue. 
 * Allocated memory is preserved by recycling dstrings.
 * Returns the number of empty entries that were removed.
 */
int jittr_queue_squeeze(q)
jittr_queue *q;
{
  int i, j, s = 0;

  for (i = 0; i < q->next; i++)
    {
      dstring *d = q->q[i];

      if (!d || !d->length)
        {
	  /* rotate all follwoing entries forward */
	  for (j = i; j < q->next - 1; j++)
	    q->q[j] = q->q[j+1];
	  q->q[j] = d;
	  q->next--;
	  s++;
	}
    }
  return s;
}

/*
 * put a command on the wire, if possible.
 * jittr_queue_squeeze is called, then send_fn.
 */
int jittr_queue_try(q)
jittr_queue *q;
{
  dstring *d = q->wait;

  if (!q->next)
    {
      debug("jittr_queue_try: queue empty\n");
      return 0;			/* queue empty */
    }

  if (d && d->length)
    {
      debug("jittr_queue_try: cannot send, old cmd still waiting\n");
      return 0;			/* cmd still pending, fool! */
    }
  
  q->wait = q->q[0];
  q->q[0] = d;

  jittr_queue_squeeze(q);	/* head is empty. advance all */

  return jittr_queue_sched(q);
}

/* 
 * Add the dstring found at *dp to the queue. The pointer at *dp is replaced by
 * an empty (or NULL) dstring that was a placeholder in the queue.  Thus the
 * the parameter's memory is consumed by jittr_queue_add_ds.
 *
 * cmp_fn is called and its return value decides how entries are unified.
 * >0 no action, =0 remove an old entry, <0 ignore new entry.
 *
 * jittr_queue_try is called.
 */
int jittr_queue_add_ds(q, dp)
jittr_queue *q;
dstring **dp;
{
  dstring *d;

  if (!dp || !(d = *dp))		/* nothing to do */
    return 0;
  
  ASSERT(dp != &q->q[q->next]);		/* need another location to swap ptr */
   
  while (q->next >= q->alloced)
    {
      dstring **p;
      int n = q->alloced + 32;

      if (!(p = (dstring **)calloc(sizeof(dstring *), n)))
        return -1;
      
      xbcopy((char *)q->q, (char *)p, q->alloced * sizeof(dstring *));
      q->alloced = n;
      free((char *)q->q);
      q->q = p;
    }
  *dp = q->q[q->next];
  q->q[q->next++] = d;

  /* added. */

  if (q->cmp_fn)
    {
      int r, i = q->next - 2;
      
      while (i >= 0)
        {
	  if (!(r = q->cmp_fn(q, d, q->q[i], q->cmp_fn_data)))
	    {
	      q->q[i]->length = 0;	/* make old invalid */
	      debug1("jittr_queue_add_ds: old entry at %d became invalid\n", i);
	    }
	  else if (r < 0)
	    {
	      d->length = 0;		/* make new invalid */
	      debug1("jittr_queue_add_ds: new entry ignored because of entry %d\n", i);
	      break;
	    }
	  i--;
	}
    }

  return jittr_queue_try(q);
}

/*
 * just like jittr_queue_add_ds, but does create/recycle a dstring inside the
 * queue, rather than consuming one. Text from buf is copied. 
 */
int jittr_queue_add(q, buf, len, data)
jittr_queue *q;
char *buf; 
int len;
void *data;
{
  /* a buffer that contains a memory fragment belonging to the queue */
  static dstring *d = NULL;
  
  dstring_append(&d, 0, buf, len);
  if (d)
    {
      if ((q->flags & JQUEUE_FLAG_DS_DATA_ALLOC) && d->data)
        free((char *)(d->data));
      d->data = data;
    }

  return jittr_queue_add_ds(q, &d);
}

/*
 * the client sent a positive acknowledge for the last entry that was put on
 * the wire.
 *
 * The entry is removed and the next entry is tried.
 */
int jittr_queue_ack(q)
jittr_queue *q;
{
  debug("jittr_queue_ack\n");

  /* if wait_sched_id is valid, we have a retry pending. zap it */
  if (q->wait)
      q->wait->length = 0;
  return jittr_queue_try(q);
}

/*
 * The client sent a negative acknowledge or a timeout occured.
 *
 * We retry the current command. Possibly modified by nak_fn.
 */
int jittr_queue_nak(q)
jittr_queue *q;
{
  debug("jittr_queue_nak\n");
  
  jittr_queue_sched(q); /* retry same entry */
  return 0;  
}
