/*
 * link.c -- atom hierarchies for the jittr library
 *
 * most of these are used in jcmd.c to implement jittr commands.
 *
 * 6.6.96 jw.
 *
 * link.c does no longer know what an exec_context is.
 * 18.7.96 jw.
 */
#include "jittr/atom.h"		/* atoms */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"	/* JLINK_* types */
#include "jittr/hash.h"		/* struct Tcl_HashTable */

static int       lookup_link __P((char *name, int len));
static int hash_jittr_delete __P((struct atom *));
static int          do_mknod __P((char *path, int len, int flag));

static int          link_cmd __P((int idx, int flag, void *is_unlink));
static int         mknod_cmd __P((int idx, int flag, void *dummy));

struct sort_el
{
  struct Tcl_HashEntry *e;
  struct dstring *d;
  char *n;
};

/*
 * jittr atoms are directories. Here is the proof:
 * jittr_opendir, jittr_readdir, jittr_closedir to implement the dir_cmd.
 */
struct jittr_dir
{
  int idx;			/* the atom (used for debugging only) */
  int flags;			/* Or of JDIR_* bits */
  struct Tcl_HashTable *h;	/* the atoms hash table */
  struct Tcl_HashEntry *e;	/* last entry seen */
  ClientData val;		/* value temporarily removed from e */
  struct Tcl_HashSearch hs;
  struct sort_el *s;		/* array of sorted entries */
  int s_idx;			/* current enumeration index in s */
};

static int jdir_sort_link(e1, e2)
void *e1, *e2;
{
  return strcmp(((struct sort_el *)e1)->n, ((struct sort_el *)e2)->n);
}

static int jdir_sort_idx(e1, e2)
void *e1, *e2;
{
  void *d1 = ((struct sort_el *)e1)->d->data;
  void *d2 = ((struct sort_el *)e2)->d->data;

  if (!JLINK_IS_RESOLVED(d1) || !JLINK_IS_RESOLVED(d2))
    {
      if (JLINK_IS_RESOLVED(d2))
	return -1;
      if (JLINK_IS_RESOLVED(d1))
	return 1;
      return 0;
    }
  return (int)d1 - (int)d2; 
}

void *
jittr_opendir(idx, flags)
int idx, flags;
{
  struct jittr_dir *j;
  struct Tcl_HashTable *h;

  if (!(h = jittr_atom_data(idx)->h))
    return NULL;
  
  if (!(j = (struct jittr_dir *)calloc(1, sizeof(struct jittr_dir))))
    {
      jittr_error("jittr_opendir(%d): malloc failed.\n", idx, 0, 0, 0);
      return NULL;
    }
  j->e = NULL;
  j->idx = idx;
  j->flags = flags;
  j->h = h;

  if (flags & JDIR_SORT_MASK)
    {
      struct dstring **dp;
      char *name;

      debug2("sorting %d entries of atom %d.\n", h->numEntries, idx);
      if (!(j->s = (struct sort_el *)calloc(sizeof(struct sort_el), h->numEntries)))
        return NULL;
      
      j->flags &= ~JDIR_SORT_MASK;
      while ((dp = jittr_readdir((void *)j, &name)))
	{
	  j->s[j->s_idx].e = j->e;
	  j->s[j->s_idx].d = *dp;
	  j->s[j->s_idx++].n = name;
	  if (j->flags & JDIR_SAVE)
	    j->val = NULL;	/* trick: readdir shall not put back link */
	}
      j->flags = flags;

      ASSERT(j->s_idx == h->numEntries);

      qsort((void *)j->s, h->numEntries, sizeof(struct sort_el), 
            (flags & JDIR_SORT_IDX) ? jdir_sort_idx : jdir_sort_link);
      j->s_idx = 0;
    }
  return (void *)j;
}

/*
 * jittr_readdir returns the adress of a dstring structure associated with
 * the link. Until the next call, or until a call to jittr_closedir()
 * this dstring is removed from the hash table, so that cyclic links are 
 * impossible.
 *
 * Stop calling jittr_readdir() after it returned NULL once. If you call again,
 * the enumeration starts all over again.
 *
 * If jittr_readdir() returns a pointer to NULL, you just detected a cyclic
 * link frustration trap. Although *namep is valid then too.
 */
struct dstring **
jittr_readdir(ptr, namep)
void *ptr;
char **namep;
{
  struct jittr_dir *j = (struct jittr_dir *)ptr;
  
  if (!j)
    return NULL;

  if (j->flags & JDIR_SORT_MASK)
    {
      if (j->s_idx >= j->h->numEntries)
        {
	  j->s_idx = 0;
	  if (namep) 
	    *namep = NULL;
          return NULL;
	}
      if (namep)
        *namep = j->s[j->s_idx].n;
      return &j->s[j->s_idx++].d;
    }

  if (!j->e)
    j->e = Tcl_FirstHashEntry(j->h, &j->hs);
  else
    {
      Tcl_SetHashValue(j->e, j->val); /* put back link (after frustration) */
      j->e = Tcl_NextHashEntry(&j->hs);
    }
  if (!j->e)
    {
      if (namep) *namep = NULL;
      return NULL;			/* end of directory */
    }
  j->val = Tcl_GetHashValue(j->e);
  if (j->flags & JDIR_SAVE)
    Tcl_SetHashValue(j->e, NULL);		/* frustrate cyclic links */

  if (j->flags & JDIR_AUTO)			/* update (UN)RESOLVED links */
    {
      struct dstring *d = (struct dstring *)j->val;
      int ri;

      if (d)
	switch ((int)d->data)
	  {
	  case JLINK_UNRESOLVABLE:
	    break;
	  case JLINK_UNRESOLVED:
	    ri = atom_lookup(d->buf);
	    if (JLINK_IS_RESOLVED(ri))
	      d->data = (void *)ri;
	    break;
	  default:
	    if (!atom_exists((int)d->data))
	      d->data = (void *)JLINK_UNRESOLVED;
	  }

      /* 
       * If we have the index, users are encouraged to use it instead of
       * the value buffer string.
       * Nevertheless, we update the value buffer string, in case the atom 
       * gets renamed and becaue simple code may exists that does not lookup
       * the atom name by index (e.g. dir_cmd).
       */
      if (atom_exists((int)d->data))
	{
	  d = atom_get_name((int)d->data);
	  dstring_append((struct dstring **)&j->val, 0, d->buf, d->length);
	  d = (struct dstring *)j->val;
	}
    }
  
  if (namep) 
    *namep = Tcl_GetHashKey(j->h, j->e);

  return (struct dstring **)&j->val;
}

int
jittr_closedir(ptr)
void *ptr;
{
  struct jittr_dir *j = (struct jittr_dir *)ptr;
  
  if (!j)
    return -1;

  if (j->flags & JDIR_SORT_MASK)
    {
      int i;

      for (i = 0; i < j->h->numEntries; i++)
	Tcl_SetHashValue(j->s[i].e, j->s[i].d); /* put back link */
      free((char *)j->s); 
    }
  else
    {
      if (j->e)
        Tcl_SetHashValue(j->e, j->val);	/* put back link (after frustration) */
    }
  j->e = NULL;
  free((char *)ptr);
  return 0;
}

/*
 * link {[-atom] {idx|path}|-data value} [alias|.]",
 * unlink alias
 *
 * Construct / destruct an atom hierarchy
 *
 */
static int
link_cmd(idx, flag, is_unlink)
int idx, flag;
void *is_unlink;
{
  int dlen, slen, link_type = JLINK_UNRESOLVED;
  char *dest;
  char *s = jarg_first_word(&slen);

  if (slen > 1 && (!strncmp("-atom", s, slen) || !strncmp("-data", s, slen)))
    {
      link_type = (s[1] == 'a') ? JLINK_UNRESOLVED : JLINK_UNRESOLVABLE;
      s = jarg_next_word(&slen);
    }
  if (!s)
    return jittr_error("%s: %slink what? (%s required).",
    	jarg_name(), is_unlink ? "un" : "", 
	is_unlink ? "link-name" : "source (and optionally destination)", 0);

  if (!(dest = jarg_next_word(&dlen)) || !strcmp(dest, "."))
    {
      char *p;

      dest = p = s;	/* no alias provided. link in under original name */

      while(*p)
        if (*p++ == '/') 
	  dest = p;
      dlen = slen - (dest - s);
    }
  
  if ((idx = jittr_getcwd()) < 0)
    return jittr_error("%s: no exec_context?\n", jarg_name(), 0, 0, 0);

  return (jittr_link(idx, dest, dlen, is_unlink ? NULL : s, slen, link_type == JLINK_UNRESOLVABLE) < 0) ? -1 : 1;
}

/* 
 * If pointer is NULL, we unlink.
 * If raw is nonzero, pointer is a value string, else pointer is an 
 * atom name and len is 0 or the length of the name.
 * link_name must be without path components.
 * jittr_do_link() returns with jittr_error() or 0.
 */
int
jittr_do_link(idx, link_name, nlen, pointer, len, raw)
int idx;
char *link_name;
int nlen;
char *pointer;
int len, raw;
{
  struct Tcl_HashTable *h;
  struct Tcl_HashEntry *e;
  dstring *d = NULL;
  char *p, *ee = link_name + nlen;
  int n;

  for (p = link_name; p < ee; p++)
    if (*p == '/')
      return jittr_error("jittr_do_link: cannot handle '/' in link_name '%s'\n",
      	link_name, 0, 0, 0);
  
  if (!(h = jittr_atom_data(idx)->h))
    {
      h = (struct Tcl_HashTable *)malloc(sizeof(struct Tcl_HashTable));
      Tcl_InitHashTable(h, TCL_STRING_KEYS);
      jittr_atom_data(idx)->h = h;
    }

  if (!pointer)
    {
      if (!(e = Tcl_FindHashEntry2(h, link_name, nlen)))
        return jittr_error("jittr_do_link: cannot unlink %s: no such link.\n",
		link_name, 0, 0, 0);
      free((char *)Tcl_GetHashValue(e));	/* was a dstring */
      Tcl_DeleteHashEntry(e);
      return 0;
    }

  /* we are link */
  e = Tcl_CreateHashEntry2(h, link_name, nlen, &n);
  if (!n)	/* harvest memory of previous link value */
    d = (struct dstring *)Tcl_GetHashValue(e);
  dstring_append(&d, 0, pointer, (len || raw) ? len : strlen(pointer));
  d->data = (void *)(raw ? JLINK_UNRESOLVABLE : JLINK_UNRESOLVED);
  Tcl_SetHashValue(e, (void *)d);
  return 0;
}

/*
 * lookup_link() resolves a path.
 * It implements the following naming conventions for jittr atoms:
 *  1) The name "." is special and refers to the current working point.
 *  2) The names "/" and "" refer to atom 0.
 *  3) Names starting with "/" cause path traversal from root on, a lookup 
 *     by fullname is never done here.
 *  4) The component ".." is not supported here. Too complicated for tonight.
 *
 * Lookup_link() really looks up a single link component.
 * Components are extracted and resolved piecewise.
 * It is not guaranteed that it ends up at the same atom as jittr_mkpath().
 *
 * lookup_link() returns the atom index, or JLINK_UNRESOLVABLE, which is
 * negative.
 *
 * Note: 
 * We can afford to do item 3) of the above list. But the reason why is not
 * trivial. We have no seperate entry point to do fullname lookup then.  Even
 * when called with a fullname, atom_lookup() will start here:
 *       lookup_link() then calls 
 *         jittr_link_resolve() for each component. jittr_link_resolve() 
 *           temporarily removes the link from the atom and calls 
 *           atom_lookup() to see, if it can be resolved. atom_lookup(),
 *             first tries 
 *             lookup_link(), which calls 
 *               jittr_link_resolve() again. But this time, our link isn't 
 *                 there. 
 *           Thus we collaps back into the (second) atom_lookup()
 *           which resolves the fullname into an index. Now we return to
 *         jittr_link_resolve() which puts back the link. And returns the 
 *         index of the component to 
 *       lookup_link(), which continues with the next component.
 *
 * Is that an entry for the obfuscated algorithm contest?
 */
static int
lookup_link(name, len)
char *name;
int len;
{
  int idx;
  int cwd;
  char *ee;

  if (!name || !*name || (*name == '/' && len == 1))
    return 0;

  if ((cwd = jittr_getcwd()) < 0)
    return -1;			/* sorry atom_lookup, we cannot help */

#if 0
  debug2("lookup_link(%s, %d)\n", name, len);
#endif

  ee = name + len;

  if (!jittr_link_resolve(cwd, name, len, &idx, NULL) &&
      (idx == JLINK_UNRESOLVABLE))
    {
      char *e, *s;
      
      e = s = name;

      while ((e < ee) && *e != '/') e++;	/* chop it into components */

      if (e < ee)			/* if it is all one piece, forget it */
	for (;;)
	  {
	    if (!*s || (e == s))
	      {
		if (s == name) idx = 0;
	      }
#if 0
	    /* 
	     * if we cannot to allow real ".." links in the atoms, comment in
	     * this block, otherwise jittr_link_resolve will resolve them
	     * nicely if present.   Disable this feature, if it causes
	     * confusion. I have no concept and no use for it.
	     */
	    else if (!strcmp(s, ".."))
	      {
	        *e = c; 		/* so that name prints out correctly */
		debug1("lookup_link: cannot handle .. in '%s'.\n", name);
		idx = JLINK_UNRESOLVABLE;
		break;
	      }
#endif
	    else if (!jittr_link_resolve(cwd, s, e - s, &idx, NULL) || 
	    	     !atom_exists(idx))
	      break;

	    cwd = idx;			/* next round */
	    if (e == ee)
	      break;
	    e++;
	    s = e;
	    while (*e && *e != '/') e++;
	  }
    }
  return idx;
}

/*
 * Return the value found at the end of a link. Name is the null terminated
 * link component, which must not countain '/' characters. If idxp is nonzero,
 * additional information is placed at *idxp: The atom index if it is an atom
 * or the link type (which is negative), if not. If there is an error, NULL is
 * returned and information is stored at *idxp.
 *
 * The following types of linkes are known here:
 *
 *  JLINK_UNRESOLVABLE: (a stupid misnomer, but I won't change it now)
 *	The link is the value. Nothing needs to be resolved. The value string
 *	is returned. *idxp is set to JLINK_UNRESOLVABLE.
 *  JLINK_UNRESOLVED:
 *      An attempt is made to lookup the atom that this link points to.
 *      If successful, the link type is changed to JLINK_RESOLVED, see there.
 *      If unsuccessful, NULL is returned and *idxp is set to JLINK_UNRESOLVED.
 *  JLINK_RESOLVED:
 *	The atom is checked. If it no longer exists, the link type is changed to
 *	JLINK_UNRESOLVED, continue there. Otherwise, the atoms index is 
 *	wirtten int *idxp and the value of the atom is returned.
 *	*NOT* triggering read callbacks!
 *  The special link name ".", does not really exist, but points to the
 *	current atom; behaviour as with JLINK_RESOLVED.
 *
 * Note: Below is a cyclic link frustration trap. I did not realize what it
 *       is *really* worth, until I singlestepped through a fullname lookup.  
 *	 See the comment at lookup_link().
 */
char *
jittr_link_resolve(idx, name, nlen, idxp, lenp)
int idx;
char *name;
int nlen;
int *idxp, *lenp;
{
  struct Tcl_HashTable *h;
  struct Tcl_HashEntry *e;
  dstring *d;

  if (idxp) *idxp = JLINK_UNRESOLVABLE;

  if (!atom_exists(idx))
    return NULL;

  if ((nlen == 1) && (*name == '.'))
    {
      if (idxp) *idxp = idx;
      if (lenp && atoms[idx]->value)
        *lenp = atoms[idx]->value->length;
      return atom_value(idx);
    }

  if (!(h = jittr_atom_data(idx)->h) ||
      !(e = Tcl_FindHashEntry2(h, name, nlen)) ||
      !(d = (struct dstring *)Tcl_GetHashValue(e)))
    return NULL;

  if (JLINK_IS_UNRESOLVED(d->data))
    {
      Tcl_SetHashValue(e, NULL);	/* frustrate cyclic links */
      idx = atom_lookup(d->buf);
      Tcl_SetHashValue(e, (void *)d); 	/* link back after frustration */
      if (idxp) *idxp = JLINK_UNRESOLVED;
      if (JLINK_IS_RESOLVED(idx))
        d->data = (void *)idx;
      else
	return NULL;
    }

  if (JLINK_IS_RESOLVED(d->data))
    {
      /*
       * When an atom is deleted, it is not unlinked;
       * and atom indices may be reused some day.
       * So we better remember a dead link, whenever we see one.
       */
      if (!atom_exists((int)d->data))
        {
	  d->data = (void *)JLINK_UNRESOLVED;
	  if (idxp) *idxp = JLINK_UNRESOLVED;
	  return NULL;
	}
      else
        {
	  if (idxp) *idxp = (int)d->data;
	  if (lenp && atoms[(int)d->data]->value)
	    *lenp = atoms[(int)d->data]->value->length;
	  return atom_value((int)d->data);
	}
      /* NOTREACHED */
    }

  if (lenp) *lenp = d->length;
  return d->buf;	/* JLINK_UNRESOLVABLE */
}

/*
 * split off the last component, and see if the first part is a well formed
 * path to an atom. The second part is examined by jittr_link_resolve.
 *
 * If name is only one path component, it must be either an atom fullname  or a
 * link in the current working point.
 */
char *
jittr_link_value(name, nlen, idxp, lenp, pp)
char *name;
int nlen;
int *idxp, *lenp, *pp;
{
  int i;
  char *s = NULL, *e, *ee = name + nlen;

  if (idxp) *idxp = JLINK_UNRESOLVED;
  if (lenp) *lenp = JLINK_UNRESOLVED;
  if (pp)   *pp   = JLINK_UNRESOLVED;

  /*
   * trivial case: its the name or index of an atom:
   * What we really need here is the numeric lookup and the call to
   * atom_lookup_atoms(). The call to atom_lookup_hook is redundant, because we
   * do again below. atom_lookup_hook may be quite expensive. In total, 
   * jittr_link_value() may call lookup_link() up to 5 times for the same
   * value. sigh.
   */
  if (atom_exists(i = atom_lookup2(name, nlen)))
    {
      if (idxp) *idxp = i;
      if (lenp && atoms[i]->value)
        *lenp = atoms[i]->value->length;
      return atom_value(i);
    }

  /* if not, try to split off the path */
  for (e = name; e < ee; e++)
    if (*e == '/')
      s = e;

  if (!s)
    {
      /*
       * Not an atom name, not splitable, hmm. 
       * Maybe a link in the current atom? 
       */
      i = jittr_getcwd();
      if (pp) *pp = i;
      return jittr_link_resolve(i, name, nlen, idxp, lenp);
    }

  if (!atom_exists(i = atom_lookup2(name, s - name)))
    return NULL;	/* oh bugger! some path components are corrupt */
  s++;
  if (pp) *pp = i;
  return jittr_link_resolve(i, s, ee - s, idxp, lenp);
}

static int
hash_jittr_delete(a)
struct atom *a;
{
  struct Tcl_HashTable *h;
  struct Tcl_HashEntry *e;
  struct Tcl_HashSearch hs;

  if (!a->data || !(h = ((struct atom_data *)(a->data))->h))
    return 0;
  
  for (e = Tcl_FirstHashEntry(h, &hs); e; e = Tcl_NextHashEntry(&hs))
    free((char *)Tcl_GetHashValue(e));

  Tcl_DeleteHashTable(h);
  free((char *)h);
  ((struct atom_data *)(a->data))->h = NULL;
  return 0;
}

/*
 * do_mknod() returns the atom index of the newly linked atom.
 * The absolute path to the new atom is expected in buf, the first len bytes
 * are valid. The contents of buf is modified temporarily.
 * This one does recursion, to build all missing path components and links.
 *
 * Atoms and links are created here, if not present.
 */
static int
do_mknod(buf, len, flags)
char *buf;
int len, flags;
{
  int at, idx;
  char *p = buf+len-1;

  if (len < 2)
    return 0;

  ASSERT(*buf == '/');

  if ((at = atom_lookup2(buf, len)) < 0)
    at = atom_new2(-1, buf, len, NULL, 0, flags);
  else
    atoms[at]->a.flag |= flags;

  while (p > buf && *p != '/')
    p--;
  idx = (p > buf) ? do_mknod(buf, p - buf, flags) : 0;
  (void)jittr_do_link(idx, p + 1, (buf + len) - (p + 1), buf, len, 0);
  return at;
}

/*
 * jittr_link() is like do_mknod, except for two details:
 * 1) In do_mknod(), buf denotes the path of all links as well as the 
 * name of the final atom. Here, in jittr_link() no final atom is created.
 * buf is the path of all links, but the final link does not point to an atom.
 * its value is target.
 * 2) jittr_link() returns nonzero for failure, which usually means that the
 * desired link already exists and is a link to a canonically named atom.
 *
 * jittr_link() is like jittr_do_link(), exept for one detail:
 * 3) jittr_link() correctly resolves path components.
 *
 * nonzero raw means that the target is UNRESOLVABLE i.e. not an atom.
 */
int
jittr_link(idx, path, plen, target, tlen, raw)
int idx;
char *path;
int plen;
char *target;
int tlen, raw;
{
  char buf[1024];
  int len = jittr_mkpath(idx, path, buf);
  char *p = buf+len;

  ASSERT(len < sizeof(buf));

  if (*buf != '/')
    return jittr_error("jittr_link: fullname %s of %s should start with '/'.\n",
    			buf, path, 0, 0);

  if (len < 2)
    return jittr_error("jittr_link: Please let \"/\" remain an atom.\n", 
    			0, 0, 0, 0);
  *p-- = '\0';
  while ( p > buf && *p != '/')
    p--;
  
  if (raw && atom_lookup(buf) >= 0)
    return jittr_error("jittr_link: %s is an atom name. Remove it, then try again\n", buf, 0, 0, 0);

  idx = do_mknod(buf, p - buf, atoms[idx]->a.flag);
  *p = '/';

  return jittr_do_link(idx, p + 1, (buf + len) - (p + 1), target, tlen, raw);
}

/* 
 * absolute path construction
 * idx is the current working point, 
 * buf is a temporary space for path construction
 * path is an absolute or relative path.
 * 
 * If a relative path is resolved using the current working point and the
 * current working point does not have a fullname starting with '/', no 
 * absolute path can be constructed. The caller should check if (*buf == '/')
 * when an absolute name is essential.
 *
 * No atoms are made here.
 */
int
jittr_mkpath(idx, path, buf)
int idx;
char *path, *buf;
{
  char *p, *e;
  int c;

  ASSERT(buf && idx >= 0 && atoms[idx]);

  p = path; e = buf;

  if (*p == '/')
    {
      idx = 0;
      p++;
    }
  else if (idx)
    {
      xbcopy(atom_name(idx), e, atoms[idx]->name->length);
      e += atoms[idx]->name->length;
    }
  e[0] = '/';

  /* 
   * path reduction on the fly, while appending:
   *  "//", "/./", "/foo/../" are alle reduced to "/".
   * The delimiting '\0' byte also counts as a trailing '/' here.
   */
  for (;;)
    {
      if ((c = (*(++e) = *p++)) && *e != '/')
        ;
      else if (e-1 >= buf && e[-1] == '/')
        e--;
      else if (e-2 >= buf && e[-1] == '.' && e[-2] == '/')
        e -= 2;
      else if (e-3 >= buf && e[-1] == '.' && e[-2] == '.' && e[-3] == '/')
        {
	  e -= 4;
	  while (e > buf && *e != '/')
	    e--;
	  if (e < buf)
	    e = buf;
	}
      if (!c)
        break;
    }
  *e = '\0';
  return e - buf;
}

/* 
 * this one is very simple now: 
 * Atoms are created and linked to establish the path, the atom at the end
 * is named as the fullpath and it contains the empty string.
 */
int
jittr_mknod(idx, path, flags)
int idx, flags;
char *path;
{
  char buf[1024];
  int len = jittr_mkpath(idx, path, buf);
  
  if (!*buf == '/')
    return jittr_error("jittr_mknod: fullpath '%s' does not start with '/'.\n",
    	buf, 0, 0, 0);
  return do_mknod(buf, len, flags);
}

static int
mknod_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen;
  char *p, *s;

  if (!(s = jarg_first_word(&slen)))
    return jittr_error("%s: parameter 'path' required.\n", jarg_name(), 0,0,0);
  if ((idx = jittr_getcwd()) < 0)
    return jittr_error("%s: no exec_context?\n", jarg_name(), 0, 0, 0);

  if ((idx = jittr_mknod(idx, s, 0)) < 0)
    return jittr_error("%s: jittr_mknod(%d, %s) failed.\n", 
    		       jarg_name(), idx, s, 0);

  p = jarg_next_word(NULL);
  s = jarg_next_word(NULL);
  
  dstring_appendf(rp, -1, "%s%d%s", p ? p : "# ", idx, s ? s : "\n", 0);
  return 1;
}

static char *help[] =
{
  "link [-atom] path [alias|.]", "",
  "link -data value [path/]linkname",
	"link another atom or a constant value into the current working point atom",
  "unlink path",
	"undo a link (do *not* use del instead!)",
  "mknod path [prefix [suffix]]",
	"create all atoms leading up to path and return its index embedded in prefix and suffix. Prefix defaults to {# }, suffix defaults to \\n",
  NULL
};

int
jittr_link_init()
{
  atom_lookup_hook = lookup_link;
  atom_delete_hook = hash_jittr_delete;

  atom_command(  HELP_CMD_ATOM, NULL, atom_help_cmd, (void *)help);
  atom_command( MKNOD_CMD_ATOM, "mknod",  mknod_cmd, (void *)0);
  atom_command(  LINK_CMD_ATOM, "link",    link_cmd, (void *)0);
  atom_command(UNLINK_CMD_ATOM, "unlink",  link_cmd, (void *)1);

  debug("jittr_link_init: done.\n");
  return 0;
}
