/*
 * misc.c -- various general purpuse functions. split from jittr.c
 *
 * 19.7.96 jw.
 * 31.3.97 jw: -cstopb added to SttyMode().
 * 13.7.97 jw: modem control bits added to SttyMode().
 * 22.11.97, jw: 9600,75	obaud,ibaud setting added to SttyMode().
 */
#include <unistd.h>		/* close() */
#include <fcntl.h>		/* O_RDWR, O_NOCTTY, O_NDELAY */
#ifdef linux
# include <sys/ioctl.h>		/* ioctl() */
#endif
#include <sys/types.h>          /* for sys/socket.h */
#include <sys/socket.h>         /* SOCK_STREAM */
#include <sys/time.h>           /* struct timeval for net.h */
#ifdef sun
# include <stdio.h>
# ifdef FILE
#  include <strings.h>		/* bzero()@sunos4 */
# else
#  ifdef _SYS_VA_LIST_H		/* included by stdio.h@solaris5.5 */
#   include <strings.h>		/* bzero()@solaris 5.5 */
#  else
#   include <string.h>		/* memset()@solaris 5.4 */
void bzero();
#  endif
# endif
# include <sys/ttold.h>		/* sun has TIOCEXCL here */
# ifndef POSIX
#  define POSIX
# endif
#endif

#include <errno.h>		/* strerror(), errno */
#include <sys/termios.h>	/* struct termios */
#include <pwd.h>		/* struct passwd */
#include <grp.h>		/* struct group */

#include "jittr/atom.h"		/* dstring */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"	/* struct atom_data */
#include "jittr/net.h"		/* struct client */
#include "jittr/doio.h"		/* jittr_make_client() */
#include "jittr/schedule.h"	/* sched_peek_event() */
#include "jittr/action.h"	/* jittr_process() */
#include "jittr/attr.h"		/* only for jittr_get_attr() dummy reference */

/* ============ about the I/O protocol ================== */

/* telnet command codes */
#define TELNET_IAC		((unsigned char)0xff)
#define TELNET_DONT		((unsigned char)0xfe)
#define TELNET_DO		((unsigned char)0xfd)
#define TELNET_WONT		((unsigned char)0xfc)
#define TELNET_WO		((unsigned char)0xfb)
#define TELNET_LONG_INTRO	((unsigned char)0xfa)
#define TELNET_INTR		((unsigned char)0xf4)
#define TELNET_LONG_TERM	((unsigned char)0xf0)
 
/*
 * Try to guess if the telnet protocol is used. 
 * We base our decision on TELNET_IAC and \n \r characters appearing in the
 * buffer.  1 is returned when we saw a TELNET_IAC. -1 is returned if we detect
 * a string that violates telnet encoding. Currently we only check if there is
 * a '\n' character without a '\r'. If we find nothing special, 0 is returned.
 */
int
jittr_telnet_check(pp, len)
char *pp;
int len;
{
  unsigned char *s = (unsigned char *)pp;
  int c = -1;

  while (len-- > 0)
    {
      switch (*s)
        {
	case '\n':
	  if (c != -1 && c != '\r')
	    return -1;	/* Ah, plain \n without any traces of telnet slang! */
	  break;
	case TELNET_IAC: 
	  return 1; 	/* You typed the IAC by hand? I don't believe you */
	case '\r':
	  if (!len)
	    return 0;
	  if (!s[1])
	    return 1;	/* Only telnet would place a \0 byte after \r... */
	  if (s[1] != '\n')
	    return -1;	/* Telnet must have \n or \0 after \r. */
	}
      c = (unsigned char)*s++;	/* please don't get negative... */
    }
  return 0;		/* no decision yet... */
}

/*
 * Clean out telnet escape sequences, used by jittr_input(). 
 * If buffer fragments split up telnet commands, this code breaks,
 * as we have no state informtion in the conn structur.
 *
 * This is a hog. In 99% of all cases, this code will copy the data 
 * into itself doing effectively nothing, but wasting cpu.
 *
 * Returns length of cleaned buffer. Returns -1, if TELNET_INTR seen.
 */
int
jittr_purge_telnet(pp, len)
char *pp;
int len;
{
  unsigned char *s, *p;
  int l = 0;

  for (s = p = (unsigned char *)pp; len > 0; len--)
    {
      if (*s == TELNET_IAC)
	{
	  if (len < 2)
	    return l;
	  switch (*++s)		/* skip the IAC */
	    {
	    case TELNET_IAC:
	      if (--len <= 0)
		return l;
	      *p++ = *s++;	/* pass one IAC through */
	      l++;
	      break;
	    case TELNET_DONT:
	    case TELNET_DO:
	    case TELNET_WONT:
	    case TELNET_WO:
	      if ((len -= 2) < 0)
		return l;
	      s += 2;
	      break;
	    case TELNET_LONG_INTRO:
	      while (*++s != TELNET_LONG_TERM)
		if (--len <= 0)
		  return l;
	      break;
	    case TELNET_INTR:
	      /*
	       * This is not quite true. The ^C character sends the following
	       * 5 byte sequence:  ff f4 ff fd 06
	       */
	      return jittr_error("TELNET_INTR requested.", 0, 0, 0, 0);
	    default:
	      debug1("jittr_purge_telnet: IAC 0x%02x unknown. Passed through.\n", *s);
	    }
	}
      else
	{
	  *p++ = *s++;
	  l++;
	}
    }
  return l;
}

/* ============ about misc. goodies ===================== */

/* this is how atom_lookup(".") is implemented: */
int
jittr_getcwd()
{
  if (!exec_context)
    {
#ifdef DEBUGG
      /* 
       * that is very noisy, because atom_lookup_hook() uses jittr_getcwd()
       * to probe for an exec_context. link.c is deliberately not exec_context
       * aware.
       */
      debug("jittr_getcwd() without exec_context. MAEEEEEP!\n");
#endif
      return -1;
    }

  /* 
   * I am curious. Let us see where this explodes:
   * ASSERT(exec_context->who->cwp.atom == exec_context->where.atom);
   * It does when running a trace callback that does atom_lookup().
   */
  return exec_context->where.atom;
}

/*
 * Return a pointer to the atom data structure associated with each atom.
 * If the structure does not yet exist, it is allocated and initialized now.
 * If the atom is invalid NULL is returned. 
 *
 * In OO-Terms: 
 * Does that mean that the atom is subclassed at runtime and on-demand?
 */
struct atom_data *
jittr_atom_data(idx)
int idx;
{
  struct atom_data *d = NULL;
  struct atom *a = atoms[idx];

  if (!atom_exists(idx) || !a || (d = (struct atom_data *)a->data))
    return d;
  a->data = (void *)calloc(sizeof(struct atom_data), 1);
  a->a.flag |= A_DATA_ALLOCED;
  return (struct atom_data *)a->data;
}

int
jittr_error(fmt, arg1, arg2, arg3, arg4)
char *fmt;
void *arg1, *arg2, *arg3, *arg4;
{
  static int in_jittr_error = 0;
  char buf[2*ESTIMATED_PACKET_SIZE+1024];	/* always too short */

  sprintf(buf, fmt, arg1, arg2, arg3, arg4);
#ifdef DEBUG
  if (*buf)
    debug2("jittr_error: %s%s", buf, buf[strlen(buf)-1] == '\n' ? "" : "\n");
#endif
  if (in_jittr_error)
    {
      debug("jittr_error double trouble (supressed)\n");
      return -1;
    }
  in_jittr_error++;
  atom_append(STDERR_ATOM, -1, buf, 0);
  in_jittr_error--;
  return -1;
}

#define COLS 16
#define LLEN (9 + (5*COLS-1)/2 + 2 + COLS)

void
jittr_hex_line(dp, l, nz)
struct dstring **dp;
char *l;
int nz;
{ 
  static char z[LLEN+LLEN+1];
  static int zero_seen = 0;
 
  strcpy(l + strlen(l), "\n"); 
  if (!nz && zero_seen == 1)
    strcpy(z, l);
 
  if (nz || !zero_seen++)
    { 
      if (nz)
        { 
          if (zero_seen == 2)
            dstring_append(dp, -1, z, 0);
          if (zero_seen > 2)
            dstring_append(dp, -1, "*\n", 2);
          zero_seen = 0;
        }
      dstring_append(dp, -1, l, 0);
    }
}

static char hexx[] = "0123456789abcdef";

int 
jittr_hexdump(dp, buf, len, prefix)
struct dstring **dp;
char *buf;
int len;
char *prefix;
{
  char ll[LLEN + LLEN + 1], *l = ll;
  int e, c, p = 0;
  int n = 0;
  int nonzero = 0;

  if (!prefix || (strlen(prefix) > LLEN)) prefix = "";

  while (len-- > 0)
    {
      e = *buf++;

      if (p == 0)
        { 
          sprintf(l, "%s%07x:", prefix, n);
	  l += strlen(prefix);
          for (c = 8; c < LLEN; l[c++] = ' ');
        }
      l[9 + (5 * p) / 2] = hexx[(e >> 4) & 0xf];
      l[10 + (5 * p) / 2] = hexx[ e       & 0xf];
      l[11 + (5 * COLS - 1) / 2 + p] = (e > 31 && e < 127) ? e : '.';
      if (e)
        nonzero++;
      if (++p == COLS)
        { 
          l[11 + (5 * COLS - 1) / 2 + p] = '\0';
          jittr_hex_line(dp, ll, nonzero);
	  l = ll;
          nonzero = 0;
          n += COLS;
          p = 0;
        }
    }
  if (p)
    { 
      l[11 + (5 * COLS - 1) / 2 + p] = 0;
      jittr_hex_line(dp, ll, nonzero);
    }
  return 0;
}
#undef COLS
#undef LLEN

/* 
 * Each line written is optionally preceded with the string prefix 
 * and additional comment identifies start end end of the hexdump
 */
int
jittr_hex_error(buf, len, prefix)
char *buf;
int len;
char *prefix;
{ 
  dstring_append(&atoms[STDERR_ATOM]->value, -1, "# HEXDUMP follows:\n", 19);
  jittr_hexdump(&atoms[STDERR_ATOM]->value, buf, len, prefix);
  atom_append(STDERR_ATOM, -1, "# END HEX\n", 10);
  return 0;
}

/* ======================= tty section ========================== */

/*
 * mostly taken from screen/tty.c and adapted...
 */

struct baud_values
{
  int idx;	/* the index in the bsd-is padding lookup table */
  int bps;	/* bits per seconds */
  int sym;	/* symbol defined in ttydev.h */
};

/*
 * On hpux, idx and sym will be different. 
 * Rumor has it that, we need idx in D_dospeed to make tputs
 * padding correct. 
 * Frequently used entries come first.
 */
static struct baud_values btable[] =
{
#ifdef B9600
	{	13,	9600,	B9600	},
#endif
#ifdef B19200
	{	14,	19200,	B19200	},
#endif
#ifdef EXTA
	{	14,	19200,	EXTA	},
#endif
#ifdef B38400
	{	15,	38400,	B38400	},
#endif
#ifdef EXTB
	{	15,	38400,	EXTB	},
#endif
#ifdef B7200
	{	13,	7200,	B7200	},
#endif
#ifdef B4800
	{	12,	4800,	B4800	},
#endif
#ifdef B3600
	{	12,	3600,	B3600	},
#endif
#ifdef B2400
	{	11,	2400,	B2400	},
#endif
#ifdef B1800
	{	10,	1800,	B1800	},
#endif
#ifdef B1200
	{	9,	1200,	B1200	},
#endif
#ifdef B900
 	{	9,	900,	B900	},
#endif
#ifdef B600
 	{	8,	600,	B600	},
#endif
#ifdef B300
 	{	7,	300, 	B300	},
#endif
#ifdef B200
 	{	6,	200, 	B200	},
#endif
#ifdef B150
 	{	5,	150,	B150	},
#endif
#ifdef B134
 	{	4,	134,	B134	},
#endif
#ifdef B110
 	{	3,	110,	B110	},
#endif
#ifdef B75
  	{	2,	75,	B75	},
#endif
#ifdef B50
  	{	1,	50,	B50	},
#endif
#ifdef B0
   	{	0,	0,	B0	},
#endif
#ifdef B57600
	{	16,	57600,	B57600	},
#endif
#ifdef B115200
	{	17,	115200,	B115200	},
#endif
#ifdef B230400
	{	18,	230400,	B230400	},
#endif
#ifdef B460800
	{	19,	460800,	B460800	},
#endif
	{	-1,	-1,	-1	}
};

/*
 * baud may either be a bits-per-second value or a symbolic
 * value as returned by cfget?speed() 
 */
static struct baud_values *
lookup_baud(baud)
int baud;
{
  struct baud_values *p;

  for (p = btable; p->idx >= 0; p++)
    if (baud == p->bps || baud == p->sym)
      return p;
  return NULL;
}

/*
 * change the baud rate in a mode structure.
 * ibaud and obaud are given in bit/second, or at your option as
 * termio B... symbols as defined in e.g. suns sys/ttydev.h
 * -1 means don't change.
 */
static int
SetBaud(m, ibaud, obaud)
struct termios *m;
int ibaud, obaud;
{
  struct baud_values *ip, *op;

  if ((!(ip = lookup_baud(ibaud)) && ibaud != -1) ||
      (!(op = lookup_baud(obaud)) && obaud != -1))
    return -1;

#ifdef POSIX
  debug("SetBaud: using POSIX cfsetispeed, cfsetospeed\n");
  if (ip) cfsetispeed(m, ip->sym);
  if (op) cfsetospeed(m, op->sym);
#else /* POSIX */
  if (ip)
    {
#  ifdef IBSHIFT
      m->c_cflag &= ~(CBAUD << IBSHIFT);
      m->c_cflag |= (ip->sym & CBAUD) << IBSHIFT;
#  else /* IBSHIFT */
      if (ibaud != obaud)
        return -1;
#  endif /* IBSHIFT */
    }
  if (op)
    {
      m->c_cflag &= ~CBAUD;
      m->c_cflag |= op->sym & CBAUD;
    }
#endif /* POSIX */
  return 0;
}

struct modem
{
#ifndef TIOCM_SCAR
# define TIOCM_SCAR (TIOCM_CD << 16)
#endif
  int bis;	/* bits to be set */
  int bic;	/* bits to be cleared */
};

/* this is from screen/tty.c */
/* parse commands from opt and modify m */
static int
SttyMode(t, m, opt)
struct termios *t;
struct modem *m;
char *opt;
{
  static const char sep[] = " \t:;,";
  char *obaud = NULL;

  if (!opt || !*opt)
    return 0;

  while (*opt)
    {
      while (index(sep, *opt)) opt++;
      if (*opt >= '0' && *opt <= '9')
        {
	  if (!obaud)
	    obaud = opt;
	  debug2("SetBaud(%d, %d)\n", atoi(obaud), atoi(opt));
	  if (SetBaud(t, atoi(opt), atoi(obaud)))
	    return -1;
	}
      else if (!strncmp(opt, "cs7", 3))
        {
	  t->c_cflag &= ~CSIZE;
	  t->c_cflag |= CS7;
	}
      else if (!strncmp(opt, "cs8", 3))
	{
	  t->c_cflag &= ~CSIZE;
	  t->c_cflag |= CS8;
	}
      else if (!strncmp(opt, "cstopb", 6))	t->c_cflag |= CSTOPB;
      else if (!strncmp(opt, "-cstopb", 7))	t->c_cflag &= ~CSTOPB;

      else if (!strncmp(opt, "istrip", 6))	t->c_iflag |= ISTRIP;
      else if (!strncmp(opt, "-istrip", 7))	t->c_iflag &= ~ISTRIP;
      else if (!strncmp(opt, "ixon", 4))	t->c_iflag |= IXON;
      else if (!strncmp(opt, "-ixon", 5))	t->c_iflag &= ~IXON;
      else if (!strncmp(opt, "ixoff", 5))	t->c_iflag |= IXOFF;
      else if (!strncmp(opt, "-ixoff", 6))	t->c_iflag &= ~IXOFF;

      else if (!strncmp(opt, "softcar", 7))	m->bis |= TIOCM_SCAR;
      else if (!strncmp(opt, "scar", 4))	m->bis |= TIOCM_SCAR;
      else if (!strncasecmp(opt, "dtr", 3))	m->bis |= TIOCM_DTR;
      else if (!strncasecmp(opt, "rts", 3))	m->bis |= TIOCM_RTS;
      else if (!strncasecmp(opt, "cts", 3))	m->bis |= TIOCM_CTS;
      else if (!strncasecmp(opt, "dsr", 3))	m->bis |= TIOCM_DSR;
      else if (!strncasecmp(opt, "car", 3))	m->bis |= TIOCM_CAR;
      else if (!strncasecmp(opt, "cd", 2))	m->bis |= TIOCM_CD;

      else if (!strncmp(opt, "-softcar", 8))	m->bic |= TIOCM_SCAR;
      else if (!strncmp(opt, "-scar", 5))	m->bic |= TIOCM_SCAR;
      else if (!strncasecmp(opt, "-dtr", 4))	m->bic |= TIOCM_DTR;
      else if (!strncasecmp(opt, "-rts", 4))	m->bic |= TIOCM_RTS;
      else if (!strncasecmp(opt, "-cts", 4))	m->bic |= TIOCM_CTS;
      else if (!strncasecmp(opt, "-dsr", 4))	m->bic |= TIOCM_DSR;
      else if (!strncasecmp(opt, "-car", 4))	m->bic |= TIOCM_CAR;
      else if (!strncasecmp(opt, "-cd", 3))	m->bic |= TIOCM_CD;

      else
        return -1;
      while (*opt && !index(sep, *opt)) opt++;
    }
  return 0;
}

static int
jittr_do_init_tty(port, opts)
char *port;
char *opts;
{
  struct termios tio;	/* its tty line settings */
  struct modem mo;	/* modem hardware settings */
  int fd;
  int one = 1;
  int zero = 0;

#ifndef O_NDELAY
# define O_NDELAY O_NONBLOCK
#endif
  if ((fd = open(port, O_RDWR | O_NOCTTY | O_NDELAY)) == -1)
    return NULL;
  fcntl(fd, F_SETFL, O_NDELAY);

#ifdef I_POP
  debug("jittr_do_init_tty I_POP\n");
  while (ioctl(fd, I_POP, (char *)0) >= 0)
    ;
#endif

#ifdef TIOCEXCL
  debug("jittr_init_tty TIOCEXCL\n");
  if (isatty(fd) && ioctl(fd, TIOCEXCL, (char *) 0))
    {
      jittr_error("jittr_do_init_tty: ioctl(%s, TIOCEXCL, 0) failed.\n",
      	port, 0, 0, 0);
      close(fd);
      return NULL;
    }
#else
  debug("jittr_do_init_tty without TIOCEXCL !!!!!!!!!!!!!!!!!\n");
#endif
  bzero((char *)&tio, sizeof(tio));
  tio.c_cflag = CS8 | CSTOPB | CLOCAL | CREAD; /* | CRTSCTS; */
  tio.c_iflag = IGNBRK | IGNPAR;
  tio.c_lflag = tio.c_oflag = 0;
  tio.c_cc[VMIN] = 1;
  tio.c_cc[VTIME] = 10;		/* does not seem to work. I_POP above? */
  mo.bis = mo.bic = 0;
  if (SttyMode(&tio, &mo, opts))
    {
      jittr_error("SttyMode(%s) failed for fd=%d.\n", opts, fd, 0, 0);
      close(fd);
      return NULL;
    }

  tcsetattr(fd, TCSANOW, &tio);

#ifdef TIOCSSOFTCAR
  if ((mo.bic & TIOCM_SCAR) && ioctl(fd, TIOCSSOFTCAR, (char *)&zero))
    {
      jittr_error("ioctl(TIOCSSOFTCAR) failed for fd=%d.\n", opts, fd, 0, 0);
      close(fd); return NULL;
    }
  if ((mo.bis & TIOCM_SCAR) && ioctl(fd, TIOCSSOFTCAR, (char *)&one))
    {
      jittr_error("ioctl(TIOCSSOFTCAR) failed for fd=%d.\n", opts, fd, 0, 0);
      close(fd); return NULL;
    }
#endif /* TIOCSSOFTCAR */

  mo.bic &= ~TIOCM_SCAR;
  mo.bis &= ~TIOCM_SCAR;

  if (mo.bic && ioctl(fd, TIOCMBIC, (char *)&mo.bic))
    {
      jittr_error("ioctl(TIOCMBIC) failed for fd=%d.\n", opts, fd, 0, 0);
      close(fd); return NULL;
    }
  if (mo.bis && ioctl(fd, TIOCMBIS, (char *)&mo.bis))
    {
      jittr_error("ioctl(TIOCMBIS) failed for fd=%d.\n", opts, fd, 0, 0);
      close(fd); return NULL;
    }
  return fd;
}

struct client *
jittr_init_tty(atom_index, port, opts)
int atom_index;
char *port;
char *opts;
{
  int fd = jittr_do_init_tty(port, opts);
  struct client *cl;

  if (!(cl = jittr_make_client(port, fd)))
    {
      jittr_error("jittr_make_client failed for fd=%d.\n", fd, 0, 0, 0);
      close(fd);
      return NULL;
    }

  dstring_append(&cl->init_port, 0, port, 0);
  dstring_append(&cl->init_opts, 0, opts, 0);
  dstring_append(&cl->name, -1, "@", 1);
  dstring_append(&cl->name, -1, atom_name(atom_index), 0);
  cl->prompt.type = PT_DISABLED;	/* prompts may confuse the device */
  cl->iotype = 0;			/* DoIo() shall READ and WRITE */
  return cl;
}

int
jittr_remake_client(cl)
struct client *cl;
{
  int fd;
  
  debug("jittr_remake_client\n");
  fd = jittr_do_init_tty(cl->init_port->buf, cl->init_opts->buf);
  cl->o->fd = fd;
  /* mkfdset() will soon update our fd_client[] array */
  return fd;
}

/*
 * CAUTION: This lookup function is only partially implemented!
 */
struct dstring **
jittr_ptr2ds(ptr, hint, data)
char *ptr;
enum ds_location_hint hint;
void *data;
/* struct ds_location **retp; */
{
  static struct dstring *nul = NULL;
  struct client *cl = (struct client *)data;

  ASSERT(cl && (hint == DSHINT_CLIENT));	/* sorry, not more impl. */
    
  if (dstring_covers_pointer(cl->i->ibuf, ptr))      return &cl->i->ibuf;
  else if (dstring_covers_pointer(cl->i->obuf, ptr)) return &cl->i->obuf;
  else if (dstring_covers_pointer(cl->o->ibuf, ptr)) return &cl->o->ibuf;
  else if (dstring_covers_pointer(cl->o->obuf, ptr)) return &cl->o->obuf;
  else if (dstring_covers_pointer(cl->name, ptr))    return &cl->name;
  else if (dstring_covers_pointer(cl->prompt.prompt_ok, ptr))
  	return &cl->prompt.prompt_ok;
  else if (dstring_covers_pointer(cl->prompt.prompt_bad, ptr)) 
  	return &cl->prompt.prompt_bad;
  else if (dstring_covers_pointer(cl->prompt.prompt_more, ptr)) 
  	return &cl->prompt.prompt_more;

  return &nul;
}

/*
 * This is dangerous. Use at own risk.
 * Is your name "Garbage-Collector"?, Do you know what you do?
 *
 * Why are you so certain that there is no pointer to this client structure
 * anywhere? You aren't? Then use only TurnClient().
 */
int
jittr_nuke_client(cl, parent)
struct client *cl, *parent;
{
  extern struct client *zombies;	/* from doio.c */

  int i, id;
  struct client *n = zombies;
  struct sched_timing *t;
  struct sched_action *a;

  /* 
   * step 1: move him from active clients list to zombie list. 
   */
  if (TurnClient(ClientById(NULL, cl->id), 1))
    return -1;
  ASSERT((zombies == cl) && (n == cl->next));

  /* 
   * step 2: check all atoms. In atom_data there is a last modified stamp;
   * This is changed to point to the http-server 'client' if it was him.
   *
   * Caution: This code assumes that all allocated atom data pointers
   *          point to a struct atom_data.
   */
  for (i = atoms_index - 1; i != atoms_index; i--)
    {
      struct atom_data *ad;

      if (i < 0)
        i = atoms_allocated - 1;
      if (!atom_exists(i) || !(atoms[i]->a.flag & A_DATA_ALLOCED))
        continue;
      ad = (struct atom_data *)atoms[i]->data; 
      if (ad && ad->who == cl)
        {
          ad->who = parent;
	  debug2("jittr_nuke_client: %s inherits atom %d\n", parent->name->buf, i);
	}
    }
  
  /*
   * Step 3: walk through the scheduler queue and a) remove all commands that
   * originate from him. This is needed to stop looping commands.
   * b) redirect all commands output that would write to him to neverland
   */
  id = sched_peek_event(NULL, NULL);
  id = sched_peek_event(&t, &a);
  while (id > 0)
    {
      struct jittr_action *ja = (struct jittr_action *)a->argv;

      if ((a->fn == jittr_process) &&
          ((ja->who == cl) ||
           ((ja->type == RT_CLIENT) && (ja->reply.client == cl))))
        {
	  debug2("jittr_nuke_client: removing cmd id%d '%s' from scheduler\n", id,
	    atom_name(ja->atom));
	  sched_unlink_event(id);
	  id = sched_peek_event(NULL, NULL);
	}
      
      id = sched_peek_event(&t, &a);
    }

  /*
   * Step 4: exec_context may also contain a client pointer.
   */
  if (exec_context && exec_context->who == cl)
    {
      debug1("jittr_nuke_client: exec_context inherited by %s\n", parent->name->buf);
      exec_context->who = parent;
    }

  /* 
   * Last most dangerous step: We don't expect a pointer to this client 
   * to be anywhere any longer. 
   */
#define JFREE(a)	if (a) { free((char *)(a)); (a) = NULL; }
  JFREE(cl->name)
  JFREE(cl->init_port)
  JFREE(cl->init_opts)
  JFREE(cl->prompt.prompt_ok)
  JFREE(cl->prompt.prompt_bad)
  JFREE(cl->prompt.prompt_more)
  if (cl->i && cl->i != &cl->_i && cl->i != &cl->_o)
    {
      JFREE(cl->i->ibuf)
      JFREE(cl->i->obuf)
      JFREE(cl->i)
    }
  if (cl->o && cl->o != &cl->_i && cl->o != &cl->_o)
    {
      JFREE(cl->o->ibuf)
      JFREE(cl->o->obuf)
      JFREE(cl->o)
    }
  debug1("jittr_nuke_client(%08x) A BLAST OF DISINTEGRATION HITS.\n", (unsigned int)cl);
  JFREE(cl)
#undef JFREE
  zombies = n;

  return 0;
}

/*
 * To be called by any daemon initialisation code, where an s-bit was needed
 * and the filesystem will be accessed later.
 *
 * jittr_setuid(NULL, NULL) is a common usage.
 */
int
jittr_setuid(uid, gid)
char *uid, *gid;
{
  int new_uid = 65534;		/* defaults to nobody */
  struct passwd *ppp;
  int new_gid = 65534;		/* defaults to nobody */
  struct group *ggg;

  if (uid && *uid >= '1' && *uid <= '9') 
    new_uid = atoi(uid);
  else if ((uid && (ppp = getpwnam(uid))) || (ppp = getpwnam("nobody")))
    new_uid = ppp->pw_uid;

  if (gid && *gid >= '1' && *gid <= '9') 
    new_gid = atoi(gid);
  else if ((gid && (ggg = getgrnam(gid))) || (ggg = getgrnam("nobody")))
    new_gid = ggg->gr_gid;

  if (setgid(new_gid))
    return jittr_error("%s: unix setgid(%d) failed: (errno=%d) %s\n", 
      jarg_name(), new_gid, errno, strerror(errno));

  if (setuid(new_uid))
    return jittr_error("%s: unix setuid(%d) failed: (errno=%d) %s\n", 
      jarg_name(), new_uid, errno, strerror(errno));
  
  return 0;
}

/*
 * This is odd behaviour of the solaris /usr/ccs/bin/ld: global symbols is not
 * available to shared modules, unless it is referenced at least once in the
 * main code. The SunOS linker caused no problems.
 *
 * This is critical for jhttpd.so, which likes to see jittr_get_attr().
 * XXX: Workaround: we force in the symbol here.
 */
int
jittr_get_attr_stub(i)
int i;
{
  return jittr_get_attr(i);
}

#if defined(sun) && !defined(FILE) && !defined(_SYS_VA_LIST_H)
/* this is Solaris 5.4, wich does not have bzero() */
void bzero(void *s, int n) { memset(s, 0, n);}
#endif

/*
 * ChopByteSync finds the first nbits bits of pat in the file referenced by fp.
 * Start of the sync pattern pat is expected to be byte aligned.
 * all material from the current file position up to and including the bytes
 * containing pat are appended to *d.
 * to read an unlimited amount of bytes use 0 for max.
 *
 * returns nonzero at EOF.
 */
int ChopByteSync(d, fp, max, pat, nbits)
dstring **d;
FILE *fp;
int max;
unsigned char *pat;
int nbits;
{
  unsigned char *s, *p = pat;
  int c, need = nbits;
  int nbytes = (nbits + 7) >> 3;

  if (!*d) dstring_append(d, 0, NULL, 8192);

  if (max) max++;

  for (;;)
    {
      if (((c = getc(fp)) == EOF) || !--max)
	{
	  (*d)->buf[(*d)->length] = '\0';
	  return -1;
	}
      if ((*d)->allocated <= (*d)->length) dstring_append(d, -1, NULL, 8192);
      *(s = (unsigned char *)((*d)->buf + (*d)->length++)) = c;

      if ((*d)->length < nbytes)
        continue;
	
      s += 1-nbytes;
      
      while (need >= 8)
        {
	  if (*s == *p)
	    {
	      s++; p++;
	      need -= 8;
	    }
	  else
	    break;
	}
      if (!need || ((need < 8) && ((*s >> (8-need)) == (*p >> (8-need)))))
        break;
      p = pat;
      need = nbits; 
    }
  (*d)->buf[(*d)->length] = '\0';
  return 0;
}
