/*
 * modload.c -- portable loadable modules support
 *
 * Copyright (c) 1996
 *      Juergen Weigert (jnweiger@informatik.uni-erlangen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 ****************************************************************
 *
 * Per repeated request of a single person. Module loading for jittr consists
 * of two parts: module.c and jmod.c. It is split up, so that this file
 * is jittr independent and can be reused with other projects.
 *
 * CAUTION: With GNU_DLD, opening multiple modules with identical symbols 
 * simultaneously may yield collisions.
 *
 * 1.3.96 jw
 * 20.6.96 jw
 *
 * linker command lines if you want to use loadable modules:
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * SUNOS4:	/usr/bin/cc -o m m.c
 * SOLARIS:	/opt/SUNWspro/bin/cc -o m m.c -ldl
 * linux:	gcc -rdynamic -o m m.c -ldl
 * hpux:	/bin/cc -o m m.c -ldld
 * sgi:   	/usr/bin/cc -o m m.c
 * GNU_DLD:     cc -static -o m m.c -DGNU_DLD -I../dld -L../dld -ldld
 *
 *
 * compiler & linker command lines to create a loadable module:
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * SUNOS4:	/usr/bin/cc -g -pic -c mm.c
 *		/bin/ld -o mm.so -assert pure-text mm.o
 * SOLARIS:	/opt/SUNWspro/bin/cc -Kpic -c mm.c
 *		/usr/ccs/bin/ld -G -o mm.so mm.o
 * linux:	gcc -shared -fpic -c mm.c
 *		ld -shared -o mm.so mm.o
 * hpux:	/bin/cc +z -c mm.c
 *		/bin/ld -b -o mm.so mm.o
 * sgi:  	/usr/bin/cc -c mm.c
 *		/usr/bin/ld -shared mm.o -o mm.so
 * GNU-dld:     cc -c mm.c
 *              GNU-dld does not know about shared libraries. Thus a module
 *              cannot call functions from them. which may result in an not
 *              executable modmain(). As a fix, you have to link the main
 *              program statically.
 *            
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This file exports the following symbols:
 *
 * int jmod_dlopen __P((char *path, jmod_shl_t *handlep, dstring **errp));
 * int jmod_dlsym __P((char *name, jmod_shl_t *h, void **symp, dstring **errp));
 * int jmod_dlclose __P((jmod_shl_t handle));
 * char *jmod_make __P((char *tag, int item));
 * char *jmod_av0_for_dld;
 */
#include <stdio.h>	/* for detecting SOLARIS */
#include <unistd.h>	/* for getcwd() */
#include <stdlib.h>	/* for getenv() */
#include <errno.h>	/* for perror() */
#include <fcntl.h>	/* O_RDWR */
#include <stdio.h>	/* stderr  */

#include "jittr/dstring.h"	/* for struct dstring & ERR() macros */
#include "jittr/modload.h"	/* for jmod_shl_t */

#if defined(sun) && !defined(FILE) && !defined(SOLARIS)
# define SOLARIS 1	/* stdio.h defines FILE on sos4, but typedefs on sos5 */
#endif

/* with gcc, please link with -rdynamic so that modules see this symbol */
int saved_stderr_fd = 2; /* modules' _init() function  may want to see it */
#define LINUX_DLOPEN_SHUT_UP 1	/* it uses ldd code, which babbles on stderr */

char *jmod_av0_for_dld = NULL;

# ifdef GNU_DLD_before_326
/* they only have dld_perror(), which pollutes stderr. We duplicate it here. */
static char *dld_errlist[] = {
"DLD_ENONE       0  /* no error */",
"DLD_ENOFILE     1  /* cannot open file */",
"DLD_EBADMAGIC   2  /* bad magic number */",
"DLD_EBADHEADER  3  /* failure reading header */",
"DLD_ENOTEXT     4  /* premature eof in text section */",
"DLD_ENOSYMBOLS  5  /* premature end of file in symbols */",
"DLD_ENOSTRINGS  6  /* bad string table */",
"DLD_ENOTXTRELOC 7  /* premature eof in text relocation */",
"DLD_ENODATA     8  /* premature eof in data section */",
"DLD_ENODATRELOC 9  /* premature eof in data relocation */",
"DLD_EMULTDEFS   10 /* multiple definitions of symbol */",
"DLD_EBADLIBRARY 11 /* malformed library archive */",
"DLD_EBADCOMMON  12 /* common block not supported */",
"DLD_EBADOBJECT  13 /* malformed input file (not rel or archive) */",
"DLD_EBADRELOC   14 /* bad relocation info */",
"DLD_ENOMEMORY   15 /* virtual memory exhausted */",
"DLD_EUNDEFSYM   16 /* undefined symbol */"
};

extern int dld_errno;

static char *
dld_strerror(code)
{
  if (code < 1 || code > sizeof (dld_errlst)/sizeof (char *))
    return "DLD_EUNKNOWN ?? /* unknown error */";
  else
    return dld_errlist[code];
}
# endif

/*
 * returns 0, if object at path loaded. *handlep contains a handle then.
 * returns nonzero otherwise, *errp contains a complaint then.
 */
int
jmod_dlopen(path, handlep, errp)
char *path;
jmod_shl_t *handlep;
dstring **errp;
{
  jmod_shl_t h;
# if !defined(GNU_DLD) && defined(linux) && defined(LINUX_DLOPEN_SHUT_UP)
  int new_stderr_fd;
# endif
# if !defined(GNU_DLD) && (defined(linux) || defined(SOLARIS) || defined(sgi))
  char buf[1024], *p;
  int l = 0;

  ASSERT(path && handlep);
  /*
   * on linux, sgi and solaris, dlopen() needs full path
   * or it searches only in /etc/ld.so.cache
   */
  if (*path != '/')
    {
      p = getenv("PWD");
      if (p && *p)
        strcpy(buf, p);
      else
        getcwd(buf, 1000);
      l = strlen(buf);
      buf[l++] = '/';
    }
  strcpy(buf+l, path);
# else
  ASSERT(path && handlep);
# endif

# ifndef GNU_DLD
#  ifdef linux
#   if LINUX_DLOPEN_SHUT_UP
  /* 
   * On linux, dlopen() babbles on filedescriptor 2, when in trouble.
   * That is stupid and undocumented.
   * Did they forget that they have dlerror() for that purpose?
   */
  saved_stderr_fd = dup(2);
  close(2);
  if ((new_stderr_fd = open("/dev/null", O_RDWR)) != 2)
    {
      dup2(new_stderr_fd, 2);
      close(new_stderr_fd);
    }
  h = dlopen(buf, RTLD_NOW);
  dup2(saved_stderr_fd, 2);	/* may already be done by modules' _init(); */
  close(saved_stderr_fd);
#   else
  h = dlopen(buf, RTLD_NOW);
#   endif
  if (!h)
    ERRFMTRET(1, "%s: %s\n", buf, dlerror(), 0, 0);
#  endif /* linux */

#  ifdef sun
  /*
   * SUNOS4's dlopen() takes the file also from the current working directory
   * to make life easier. But it looks like the module's _init() and _fini()
   * are never called, although the man page promises that. Am I linking 
   * wrong?
   *
   * SOLARIS invokes _init() and _fini().
   * Dlsym on SUNOS4 is tolerant against leading underscores. It finds the 
   * symbol with or without the leading '_' character. That is funny and
   * undocumented. On SOLARIS it is different.
   *
   * ATTENTION:
   * SUNOS4's dlopen() exits the process, if the loaded module cannot resolve
   * all of its symbols. The man page only says: 'If dlopen fails, it will
   * return a null pointer.' I can't help it.
   */
#   ifdef SOLARIS
  if (!(h = dlopen(buf, RTLD_NOW)))
#   else
  if (!(h = dlopen(path, 1)))
#   endif
    ERRFMTRET(1, "%s: %s\n", path, dlerror(), 0, 0);
#  endif /* sun */

#  ifdef sgi
  /*
   * the man page promises that dlopen searches the current directory too,
   * but it does not. We have to provide the full pathname again. sigh!
   */
  if (!(h = dlopen(buf, RTLD_NOW)))
    ERRFMTRET(1, "%s: %s\n", buf, dlerror(), 0, 0);
#  endif /* sgi */

#  ifdef hpux
  /* 
   * shl_load() finds the module in the current working directory
   * but does not trigger _init() or _fini() functions.
   */
  if (!(h = shl_load(path, BIND_IMMEDIATE, NULL)))
    ERRFMTRET(errno, "%s: %s\n", path, strerror(errno), 0, 0);
    /* yes, shl_load() sets errno */
#  endif /* hpux */
# else /* GNU_DLD */
  {
    static int onw_symbols_loaded = 0;
    int r, look_in_path = 1;
    char *p;

    if (!onw_symbols_loaded && !jmod_av0_for_dld)
      ERRFMTRET(1, "%s\n", "GNU dld needs argv[0] in jmod_av0_for_dld", 0,0,0);

    if (!onw_symbols_loaded)
      {
	for (p = jmod_av0_for_dld; *p; p++)
	  if (*p == '/')
	    look_in_path = 0;
	p = look_in_path?dld_find_executable(jmod_av0_for_dld):jmod_av0_for_dld;
	if ((r = dld_init(p)))
	  ERRFMTRET(r, "%s: %s\n", p, dld_strerror(dld_errno), 0, 0);
	if (look_in_path)
	  free(p);
      }

    if ((r = dld_link(path)))
      ERRFMTRET(r+2, "%s: %s\n", p, dld_strerror(dld_errno), 0, 0);
    h = (jmod_shl_t)strdup(path);
  }
# endif /* GNU_DLD */

  *handlep = h;
  return 0;
}

int
jmod_dlsym(name, wantfn, h, symp, errp)
char *name;
int wantfn;
jmod_shl_t *h;
void **symp;
dstring **errp;
{
  void *sym;	/* if wantfn, this is a (int (*)()) */

  ASSERT(name && symp);
#ifndef GNU_DLD
# if defined(linux) || defined(sun) || defined(sgi)
  /* we cannot distinguish executables from other symbols here. sigh. */
  if (!(sym = (void *)dlsym(h, name)))
    ERRFMTRET(2+wantfn, "%s: %s\n", name, dlerror(), 0, 0);
# endif /* sgi || sun || linux */

# ifdef hpux
  if (shl_findsym(&h, name, wantfn ? TYPE_PROCEDURE : TYPE_UNDEFINED, (void *)symp))
    ERRFMTRET(2, "%s: %s\n", name, strerror(errno ? errno : ENOSYS), 0, 0);
    /* this ENOSYS should be ENOSYM, but that is used */
# endif /* hpux */
#else /* GNU_DLD */
  if (wantfn)
    {
      int r;

      if (!dld_function_executable_p(name))
	{
	  char **u = dld_list_undefined_sym();

	  ERR(name); ERR(": function not executable. ");
	  ERRNL(u&&dld_undefined_sym_count?"Undefined symbols are:":"Hmmm...");
	  for (r = 0; r < dld_undefined_sym_count; r++)
	    ERRNL(*u++);
	  if (u)
	    free((char *)u);
	  return -1;
	}
      sym = (void *)dld_get_func(name);
    }
  else
    sym = (void *)dld_get_symbol(name);

  if (!sym)
    ERRFMTRET(2, "%s: %s\n", name, dld_strerror(dld_errno), 0, 0);
#endif /* GNU_DLD */

  *symp = sym;
  return 0;
}

int
jmod_dlclose(h, errp)
jmod_shl_t h;
struct dstring **errp;
{
#ifdef GNU_DLD
  int r;
#endif

  ASSERT(h);
#ifndef GNU_DLD
# if defined(linux) || defined(sun) || defined(sgi)
  if (dlclose(h))
    ERRFMTRET(1, "dlclose: %s\n", dlerror(), 0, 0, 0);
# endif /* sgi || sun || linux */

# ifdef hpux
  if (shl_unload(h))
    ERRFMTRET(1, "shl_unload: %s\n", strerror(errno ? errno : ENOSYS), 0, 0, 0);
    /* this ENOSYS should be ENOSYM, but that is used */
# endif /* hpux */
#else /* GNU_DLD */
  if ((r = dld_unlink_by_file((char *)h, 0)))
    ERRFMTRET(r+2, "dld_unlink_by_file: %s: %s\n", (char *)h,
      dld_strerror(dld_errno), 0, 0);
  free((char *)h);
#endif /* GNU_DLD */
  return 0;
}

#ifdef STANDALONE
int
main(ac, av)
int ac;
char **av;
{
  dstring *err = NULL;
  jmod_shl_t h;
  int (*fn)();

  jmod_av0_for_dld = av[0];
  if (jmod_dlopen(av[1], &h, &err) ||
      jmod_dlsym("modmain", 1, h, (void *)&fn, &err))
    fprintf(stderr, "modload_main: %s", err->buf), exit(errno);
  return (*fn)(ac);
}
#endif
