/*
 * net.c -- lower level network access.
 *
 * All communication partners in the jittr world are equal as it is a
 * symmetric protocol. The term 'client' refers to any 'remote instance'
 * of a jittr atom server that interacts with us. Memory occupied by client
 * structures shall never be freed, as pointers are used in the fd_client[] 
 * array and jittr_action structures.
 *
 * Note that this module shall remain completely self contained. We collect 
 * here only an interface to "the network" as presented by the operating 
 * system. Currently only Unix TCP is really functional, UDP and AF_UNIX
 * code is incomplete and untested.
 * No knowledge about atoms or jittr is written down here. Processing of 
 * input and timing is done through callback function pointers. 
 *
 * 29.04.95 jw.
 * 24.08.95 jw.
 *
 * Return errors are appended to provided dstring pointer.
 * 27.12.95 jw.
 *
 * Support for non-selectable unix device drivers added.
 * 24.06.96 jw.
 *
 * Removed references to jittr_error() and fd_client[].
 * Made net.c useable outside of libjittr.
 * 06.09.96 jw.
 */
#include <unistd.h>		/* close, alarm */
#include <errno.h>		/* EBADF, errno */
#include <signal.h>		/* SIGPIPE, SIGALRM, SIG_IGN */
#include <sys/types.h>		/* for sys/socket.h */
#include <sys/socket.h>		/* struct sockaddr */
#include <netinet/in.h>		/* struct sockaddr_in */
#include <sys/un.h>		/* struct sockaddr_un */
#include <arpa/inet.h>		/* inet_addr() */
#include <netdb.h>		/* gethostbyname(name) */
#include <sys/time.h>		/* struct timeval */
#include <sys/wait.h>		/* WIFEXITED() macros, union wait */
#include <sys/stat.h>		/* struct stat for conn2dstr() */

#ifdef IP_ADD_MEMBERSHIP
#define MULTICAST
#endif

#include "jittr/atom.h"		/* only for dstring datatype and ERR* macros */
#include "jittr/jittr.h"	/* only for xbcopy() & sizeof(struct cwp) */
#include "jittr/net.h"		/* also requires sockaddr & timeval ... */
#include "jittr/doio.h"		/* fd_client[] */
#include "jittr/debug.h"	/* debug() macros */

#ifdef __SVR4
#include <sys/stropts.h>	/* for I_POP */
#endif

#ifdef SOLARIS
# include <sys/utsname.h>	/* uname()@solaris */
#endif

struct client *clients = NULL;	/* list of all known clients */

struct server server;		/* current state of server sockets */
				/* not static, because of doio.o, user.o */

/* keep this in sync with enum con_type in net.h */
char *con_type_name[] = {
  "inval", "TCP  ", "Tel/T", "Telnet", "UDP", "FD:w ", "FD:rw", "buf"
};

/* ===================== the lowest level code: message passing =========== */

/* 
 * portable lookup of own Hostname
 */
char *
Hostname()
{
  static char h[256];
#if defined(SOLARIS) || (defined(SYSV) && !defined(ISC))
  struct utsname utsnam;

  uname(&utsnam);
  strncpy(h, utsnam.nodename, 255);
#else
  gethostname(h, 255);
#endif
  return h;
}

/*
 * Given hostname (name string or digit string) and port in native host byte 
 * order, constructs a canonical sockaddr in either a passed memory area, or
 * (if the third argument is NULL) in an internal static area.
 * If failed, NULL is returned.
 *
 * Canonification includes mapping of a localhost spec to a real name.
 * The empty string "" also denotes the local host.
 * If hostname is NULL, the returned sockaddr is suitable for bind(), 
 * as it will accept connections for any address.
 */
struct sockaddr_in *
Str2Si(host, port, sip)
  char *host;
  int port;
  struct sockaddr_in *sip;
{
  struct hostent *hp;
  static struct sockaddr_in ssip;

  if (!sip)
    sip = &ssip;

  sip->sin_addr.s_addr = INADDR_ANY;
  sip->sin_port = htons(port);
  sip->sin_family = AF_INET;

  if (!host)
    return sip;

  if (!*host || !strcmp(host, "localhost") || !strcmp(host, "127.0.0.1"))
    host = Hostname();

  if ((sip->sin_addr.s_addr = inet_addr(host)) == -1)
    {
      if (!(hp = gethostbyname(host)))
        return NULL;
    }
  else
    {
      /* canonify, what inet_addr returned */
      if (!(hp = gethostbyaddr((char *)&sip->sin_addr.s_addr,
      				 sizeof(sip->sin_addr.s_addr), AF_INET)))
        return NULL;
    }
  /* any lousy bcopy() or memcpy() will do here */
  xbcopy((char *)hp->h_addr,(char *)&sip->sin_addr.s_addr, hp->h_length);
  sip->sin_family = hp->h_addrtype;
  return sip;
}

/*
 * Both pattern and addr must be of same byte order and size.
 * '\0' and '\377' bytes in pattern are accepted as wildcards.
 * Nonzero is returned if they don't match.
 */
static int
CheckNetmaskAddr(pattern, addr, l)
char *pattern, *addr;
int l;
{
  while (--l >= 0)
    {
      if (pattern[l] && (pattern[l] != '\377') && (pattern[l] != addr[l]))
        return 1;
    }
  return 0;
}

/*
 * The string p is matched against all the names and addresses found in 
 * the hostentry *h, if h is nonzero. p is also matched agaist the single 
 * address *a if a is nonzero.
 * ChecNetmaskString returns nonzero if no match was found.
 */
static int
CheckNetmaskString(p, h, a)
char *p;
struct hostent *h;
struct in_addr *a;
{
#if 0
  struct in_addr t;	/* inet_addr() does not return a structure. Hmm */
#else
  unsigned long t;
#endif
  char **nv;

  if (!p || !*p)
    return -1;			/* aehem. match what??? */

  if (*p < '0' || *p > '9')	/* it is a name, not a number */
    {
      if (!h)
        return -1;		/* need hostentry to match names */
      if (!strcasecmp(p, h->h_name))
        return 0;		/* official match. o.k. */
      if (!(nv = h->h_aliases))
        return -1;		/* no other names to match, sorry */
      while (*nv)
        if (!strcasecmp(p, *nv++))
	  return 0;		/* gotcha! o.k. */
      return 1;			/* sorry */
    }

  if (((t = inet_addr(p)) == -1) && strcmp(p, "255.255.255.255"))
    return -1;			/* cannot do address compare */

  if (h && (nv = h->h_addr_list) && (sizeof(t) >= h->h_length))
    {
      while (*nv)
        if (!CheckNetmaskAddr((char *)&t, *nv++, h->h_length))
	  return 0;		/* found it in the adress list */
    }

  if (!CheckNetmaskAddr((char *)&t, (char *)a, sizeof(t)))
    return 0;			/* found it directly */

  return 1;
}

/*
 * The address at *peer is checked against the netmasks recorded in the
 * dstring masks. Masks is a list of netmasks in an ascii representation
 * and seperated by one or more of the delimiters " ,:\n".
 * A netmask shall be written in base 256 notation ``d.d.d.d'' in network byte
 * order. Both of the numbers 0 and 255 will do as wildcards.
 * Hostnames can also be placed in that list.
 *
 * Returns zero if o.k.
 */
int
CheckNetmask(masks, peer)
dstring *masks;
struct sockaddr_in *peer;
{
  struct hostent *hp;
  char *e, *p;
  int r, c;

  if (peer->sin_family != AF_INET)
    return 0;		/* unchecked, considered o.k. */
  
  if (peer->sin_addr.s_addr == INADDR_ANY)
    return 0;		/* must be a server socket. Considered o.k. */

  if (!masks || !masks->length)
    return 0;		/* no masks means checking turned off */

  p = masks->buf;
  p[masks->length] = '\0';

  hp = gethostbyaddr((char *)&peer->sin_addr.s_addr, 
  	sizeof(peer->sin_addr.s_addr), AF_INET);

  while (*p)
    {
      while (*p == ' ' || *p == ':' || *p == ',' || *p == '\n')
	p++;
      e = p;
      while (*e && *e != ' ' && *e != ':' && *e != ',' && *e != '\n')
	e++;
      c = *e;
      *e = '\0';
      debug1("CheckNetmask: %s -- ", p);
      r = CheckNetmaskString(p, hp, &peer->sin_addr.s_addr);
      *e = c;
      if (!r)
	{
	  debug("O.K.\n");
	  return 0;
	}
      debug("No.\n");
      p = e;
    }

  return 1;		/* security alert? */
}

/* 
 * Append the default netmask to the dstring at *d.
 * The default netmask is a list of broadcast addresses of all of the hosts
 * network interfaces. 
 * def_mask should be 0xff to construct Class C brodcast addresses or 0xffff to
 * construct Class B broadcast addresses.
 */
int
NetmaskDefault(d, def_mask, host, errp)
struct dstring **d;
unsigned long def_mask;
char *host;
dstring **errp;
{
  char *s, **nv;
  struct hostent *hp;

  if (!(hp = gethostbyname(host)))
    ERRRET(errno, "NetmaskDefault(): no gethostent(), no default.\n");

  dstring_append(d, -1, "localhost", 9);
  
  nv = hp->h_addr_list;
  while (nv && *nv)
    {
      (*nv)[hp->h_length-1] |= (def_mask      ) & 0xff;
      (*nv)[hp->h_length-2] |= (def_mask >>  8) & 0xff;
      (*nv)[hp->h_length-3] |= (def_mask >> 16) & 0xff;
      (*nv)[hp->h_length-4] |= (def_mask >> 24) & 0xff;
      s = inet_ntoa(*(struct in_addr *)(*nv++));
      dstring_append(d, -1, " ", 1);
      dstring_append(d, -1, s, 0);
    }
  return 0;
}


/* 
 * UdpSend(Str2Si("picasso", 1234, NULL), "message\n", 8, NULL)
 * 
 * The number of bytes sent is returned. On failure -1 is returned,
 * errno and (if possible) *errp are set.
 */
int
UdpSend(peer, buf, len, errp)
struct sockaddr_in *peer;
char *buf;
int len;
dstring **errp;
{
  int r;

  if (!peer)
    ERRRET(ENXIO, "UdpSend: no peer");

  if (server.udp_port_fd < 0)
    {
      if (!server.udp_port)
	ERRRET(EBADF, "UdpSend: no socket. Do ListenPort(SOCK_DGRAM...) first");
      debug1("Had no socket, but know port number. Calling ListenPort %d.\n",
             server.udp_port);
      if (ListenPort(SOCK_DGRAM, server.udp_port, NULL, errp))
        return -1;
    }

  if ((r = sendto(server.udp_port_fd, buf, len, 0, (struct sockaddr*)peer, sizeof(*peer))) < 0)
    ERR("UdpSend: sendto() failed");
  return r;
}

/*
 * UdpRecv5 returns the number of bytes actually recieved, or -1 on error 
 */
int
UdpRecv5(fd, buf, len, sip, errp)
int fd;
char *buf;
int len;
struct sockaddr_in *sip;
dstring **errp;
{
  int fl = sizeof(*sip);
  int r;

  if (sip)
    r = recvfrom(server.udp_port_fd, buf, len, 0, (struct sockaddr *)sip, &fl);
  else
    r = recv(server.udp_port_fd, buf, len, 0);
  if (r < 0)
    ERR("UdpRecv: recvfrom() failed");
  return r;
}

int
UdpRecv(buf, len, sip, errp)
char *buf;
int len;
struct sockaddr_in *sip;
dstring **errp;
{
  int r;

  if (server.udp_port_fd < 0)
    {
      if (!server.udp_port)
	ERRRET(EBADF, "UdpRecv: no socket. Do ListenPort(SOCK_DGRAM...) first");
      debug1("Had no socket, but know port number. Calling ListenPort %d.\n",
             server.udp_port);
      if (ListenPort(SOCK_DGRAM, server.udp_port, NULL, errp))
        return -1;
    }

  r = UdpRecv5(server.udp_port_fd, buf, len, sip, errp);

  if (sip && CheckNetmask(server.netmask, sip))
    {
      struct conn c;

      c.type = CTYPE_UDP;
      c.sa = *(struct sockaddr *)sip;
      ERR("UdpRecv: Packet from ");
      conn2dstr(errp, -1, &c);
      ERRRET(errno, " rejected: Netmask violation");
    }
  return r;
}

int
UdpSetTTL(fd, ttl, errp)
int fd;
int ttl;
dstring **errp;
{
#ifdef MULTICAST
  int false = 0;
  char ttlc = (char)ttl;
#endif

  if (ttl < 0)
    return 0;
#ifdef MULTICAST
  if (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL, &ttlc, 1) < 0)
    ERRRET(errno, "UdpSetTTL: setsockopt IP_MULTICAST_TTL failed");
  if (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP, (char *)&false, 1) < 0)
    ERRRET(errno, "UdpSetTTL: setsockopt IP_MULTICAST_LOOP failed");
#else
  ERRRET(0, "UdpSetTTL: compiled without multicast support");
#endif
  return 0;
}

/*
 * Returns: nonzero, *errp will point to a dstring describing what
 *          triggered the error. errno is valid. If no error, 
 *          0 is returned, but *errp may have warning message.
 * Effects: Entries in struct server are made, the port is listened and the
 *          caller should select()/accept() on server.???_port_fd.
 * type shall be one of SOCK_DGRAM or SOCK_STREAM optionally or'ed with 
 * SO_REUSEADDR.
 */
int
ListenPort(type, port, fdp, errp)
int type, port;
int *fdp;
dstring **errp;
{
  int rport, rfd, r;

  if ((r = ListenPort5(type, port, &rport, &rfd, errp)))
    return r;

  if (type & SOCK_STREAM)
    {
      server.tcp_port = rport;
      server.tcp_port_fd = rfd;
    }
  else
    {
      server.udp_port = rport;
      server.udp_port_fd = rfd;
    }
  if (fdp)
    *fdp = rfd;

  return 0;
}

int
ListenPort5(flags, port, portp, fdp, errp)
int flags, port;
int *portp, *fdp;
dstring **errp;
{
  struct sockaddr_in sa;
  int fd, size = SOCKBUFSIZE, ia = 1;

  if (!(Str2Si(NULL, port, &sa)))
    ERRRET(errno, "ListenPort: Str2Si: gethostbyname(INADDR_ANY)");

#if 0
  /*
   * The reuse addr feature is tricky. Although it is set with setsockopt(),
   * it only has an effect on a subsequent bind() call.
   * When a an address was already bound, bind() to the same address can only
   * succeed, if reuse addr is enabled. When a connection was established
   * and the previous socket dies (e.g. Process killed), another bind() will
   * only succeed with reuse addr or after a timeout of 60 seconds. I do not
   * know how to change this timeout. 5 seconds would be enough.
   *
   * On sunOS4, the timeout can be knocked down in the following way:
   * Bind with reuse addr first, then destroy the socket, then try again
   * with a fresh socket but without reuse addr. 
   * This hack may not work everywhere, but it should never do any harm.
   *
   * It does harm linux. 18.7.96 jw.
   */
  
  if ((fd = socket(sa.sin_family, flags & (SOCK_STREAM|SOCK_DGRAM), 0)) < 0)
    ERRRET(errno, "ListenPort: dummy socket()");
  if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&ia, sizeof(ia)) < 0)
    ERR("ListenPort: setsockopt: dummy SO_REUSEADDR (ignored)");
  bind(fd, (struct sockaddr *)&sa, sizeof(sa));
  close(fd);
#endif  

  if ((fd = socket(sa.sin_family, flags & (SOCK_STREAM|SOCK_DGRAM), 0)) < 0)
    ERRRET(errno, "ListenPort: socket()");

  if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&size, sizeof(size)) < 0)
    ERR("ListenPort: setsockopt: SO_RCVBUF (ignored)");

  ia = (flags & SO_REUSEADDR) ? 1 : (getenv("SO_REUSEADDR") ? 1 : 0);
  if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&ia, sizeof(ia)) < 0)
    ERR("ListenPort: setsockopt: SO_REUSEADDR (ignored)");
  if (bind(fd, (struct sockaddr *)&sa, sizeof(sa)) < 0)
    {
      ia = errno;
      close(fd);
      ERRRET(ia, "ListenPort: bind()");
    }

  /* get the port number back from the system */
  ia = sizeof(struct sockaddr_in);
  if (getsockname(fd, (struct sockaddr *)&sa, &ia) < 0)
    { 
      ia = errno; 
      close(fd); 
      ERRRET(ia, "ListenPort: getsockname()");
    }

  if (flags & SOCK_STREAM)
    {
      if (listen(fd, 5))
	{
	  ia = errno;
	  close(fd);
	  ERRRET(ia, "listen");
	}
    }
 
  if (portp)
    *portp = ntohs(sa.sin_port);
  if (fdp) 
    *fdp = fd;
  errno = 0;
  return 0;
}

static int timeo = 0;

static sigret_t 
timeo_func(SIGDEFARG) 
{ 
  timeo++; 
  SIGRETURN
}

/*
 * I received SIGPIPE  by writing on a closed TCP Socket under SunOS4.
 * In this case signal(SIGPIPE, SIG_IGN) is not sufficient, because the
 * handler must be reset after every signal.
 */
static sigret_t
sig_pipe_ign(SIGDEFARG)
{
  signal(SIGPIPE, sig_pipe_ign);
  SIGRETURN
}

/*
 * Returns: <0 on failure, errno is set and *errp contains a message.
 *          else >= 0: the filedescriptor resulting from accept(). 
 * Effects: accepts network connection on a port. SIGPIPE (may be caused 
 *          by send on broken stream socket connection) is ingnored.
 *          alarm() is used and reset.
 */
int
TcpPortAccept(errp)
dstring **errp;
{
  return TcpPortAccept3(server.tcp_port_fd, server.netmask, errp);
}

int
TcpPortAccept3(server_fd, mask, errp)
int server_fd;
dstring *mask;
dstring **errp;
{
  int il, clientfd;
  struct sockaddr_in sa;
  sigret_t (*sig) __P(SIGPROTOARG);

  il = sizeof(struct sockaddr_in);
  debug1("TcpPortAccept(%d)...\n", server_fd);
  signal(SIGPIPE, sig_pipe_ign);
  sig = (sigret_t (*)())signal(SIGALRM, timeo_func);
  timeo = 0;
  alarm(3);
  if ((clientfd = accept(server_fd, (struct sockaddr *)&sa, &il)) < 0)
    {
      il = errno;
      ERR("accept");
    }
  alarm(0);
  signal(SIGALRM, sig);
  if (CheckNetmask(mask, &sa))
    {
      char *s = "# general protection error: netmask violation.\r\n";
      struct conn c;

      write(clientfd, s, strlen(s));
      shutdown(clientfd, 2);
      close(clientfd);

      c.type = CTYPE_TCP;
      c.sa = *(struct sockaddr *)&sa;
      ERR("TcpPortAccept: connection from ");
      conn2dstr(errp, -1, &c);
      ERRRET(errno, " rejected: Netmask violation.");
      return -1;
    }
  debug2(" ... TcpPortAccept = %d (%d)\n", clientfd, timeo);
  if (clientfd < 0)
    errno = il;
  return clientfd;
}

/*
 * Returns: <0 on failure, errno is set and *errp contains a message.
 *          else 0.
 * Effects: closes a connection, If used on a server socket, stops listening
 *          and sets the corresponding filedescriptor in struct server to -1.
 *          If fd_client[] is not NULL, sanity checking is done and the fd
 *          entry of the corresponding struct conn is set to -1.
 * Caveat:  It is illegal to pass NULL as errp, unless fd is a serversocket.
 */
int                 
PortShutdown(clientfd, fd_client, errp)
int clientfd;
struct client **fd_client;
dstring **errp;
{
  if (clientfd == server.tcp_port_fd)
    {
      ASSERT(!fd_client || !fd_client[clientfd] || 
      	     fd_client[clientfd] == INVALID_CLIENT);
      close(server.tcp_port_fd);
      server.tcp_port_fd = -1;
      if (fd_client) fd_client[clientfd] = NULL;
      return 0;
    }
  if (clientfd == server.udp_port_fd)
    {
      ASSERT(!fd_client || !fd_client[clientfd] || 
      	     fd_client[clientfd] == INVALID_CLIENT);
      close(server.udp_port_fd);
      server.udp_port_fd = -1;
      if (fd_client) fd_client[clientfd] = NULL;
      return 0;
    }
  
  if (fd_client)
    {
      struct conn *c;

      if (!fd_client[clientfd])
	ERRRET(errno, "PortShutdown: clientfd not in fd_client");
      if (!(c = fd_client[clientfd]->i) || c->fd != clientfd)
	c = fd_client[clientfd]->o;
      ASSERT(c && c->fd == clientfd);	/* fd_client entry corrupt */
      if (!IS_CTYPE_TCP(c->type))
	ERRRET(errno, "PortShutdown: not a TCP connection");
      c->fd = -1;
    }

  if (shutdown(clientfd, 2) < 0) 
    { 
      dstring_appendn(errp, -1, "Portshutdown %d:", clientfd);
      dstring_appendn(errp, -1, "errno %d:", errno);
      ERRNL("shutdown() failed");
    }
  close(clientfd);
  if (fd_client) fd_client[clientfd] = NULL;
  return 0;
}

/*
 * Returns: <0 on failure, errno is set and *errp contains a message.
 *          else >= 0: the filedescriptor resulting from connect().
 * Effects: open a network connection to a socket that is listening and 
 *          accepting.  SIGPIPE (may be caused by send on broken stream 
 *          socket connection) is ingnored. alarm() is used and reset.
 *
 * Note:
 * Daemon-daemon connections must not send prompts in both directions.
 * As this could cause the two interpreters to enter a virtual life-lock.
 *
 * Convetion is, that the one who accepts may prompt, but the one who 
 * connects must not prompt.
 */
int
TcpPortConnect(hostname, port, timeout, errp)
char *hostname;
int port, timeout;
dstring **errp;
{
  struct sockaddr_in sv;
  int err, sock;
  /* void (*sig)(); */
  sigret_t (*sig) __P(SIGPROTOARG);

  if (!(Str2Si(hostname, port, &sv)))
    ERRRET(errno, "TcpPortConnect: Str2Si: gethostbyname()");
  if (timeout < 1)
    timeout = 3;

  if ((sock = socket(sv.sin_family, SOCK_STREAM, 0)) < 0)
    ERRRET(errno, "TcpPortConnect: socket()");
  signal(SIGPIPE, sig_pipe_ign);
  sig = (sigret_t (*)())signal(SIGALRM, timeo_func);
  timeo = 0;
  alarm(timeout);
  err = connect(sock, (struct sockaddr *)&sv, sizeof(sv));
  alarm(0);
  signal(SIGALRM, sig);
  if (err)
    {
      (void)ERR("TcpPortConnect: ");
      ERRRET(errno, (errno == EINTR && timeo) ? "Connection timed out" : 
                                                strerror(errno));
    }
  return sock;
}

/* ==================== end of message passing section =================== */

/* ====================== client lookup & management ===================== */
/*
 * return a client pointer for the given name. 
 * A '*' character may be appended to the name, to indicate a prefix match.
 * The client root pointer is usually NULL, unless you fear there are 
 * multiple clients that match. In this case you should call ClientByName
 * again and pass the address of the (nonzero!) next pointer obtained from
 * the previously returned client structure. Got it?
 *
 * The value at *root is never modified here. We could probably do with a
 * simple pointer instead of a pointer-pointer here, but as a pointer-pointer
 * is also returned, this looks nicer.
 *
 * ClientByName will return the adress of a NULL pointer when the client 
 * was not found. This NULL pointer should be modified to point to 
 * a valid client structure if a client is to be added to the list.
 */
struct client **
ClientByName(root, name)
struct client **root;
char *name;
{
  struct client **cl;
  int nlen;

  if (!root || !*root)
    root = &clients;
  
  if (!name)
    return root;

  nlen = strlen(name);

  if (nlen && (name[nlen-1] == '*'))
    nlen--;

  for (cl = root; *cl; cl = &(*cl)->next)
    if ((*cl)->name && (*cl)->name->buf && 
        !strncmp(name, (*cl)->name->buf, nlen) && 
        ((*cl)->name->buf[nlen] == '\0' || name[nlen] == '*'))
      break;
  return cl;
}


/* returns zero, if connection conn talks to peer client_peer, nonzero else */
static int
ConnMatchPeer(conn, client_peer)
struct conn *conn;
struct sockaddr *client_peer;
{
  if (!conn || !client_peer)
    return -1;
  /* 
   * Only two of the connection types can have peers.
   * And peers of different tribes are always different.
   * In any case, we do not really expect to handle any family
   * other than AF_INET correctly here.
   */
  if ((IS_CTYPE_TCP(conn->type) || conn->type == CTYPE_UDP) &&
      client_peer->sa_family == conn->sa.sa_family)
    {
      switch (client_peer->sa_family)
	{
	case AF_INET:
	  if (((struct sockaddr_in *)client_peer)->sin_port ==
	      ((struct sockaddr_in *)(&conn->sa))->sin_port)
	    {
	      /* 
	       * Need to canonify the address of our peer here. 
	       * We compare adresses that were both created by Str2Si().
	       */
	      struct sockaddr_in s_can;

	      if (!Str2Si(inet_ntoa(
	           ((struct sockaddr_in *)(&conn->sa))->sin_addr), 0, &s_can))
	        break;
	      if (s_can.sin_addr.s_addr == 
	          (((struct sockaddr_in *)client_peer)->sin_addr.s_addr))
	        return 0;
	    }
	  break;
	case AF_UNIX:
	  if (!strcmp(((struct sockaddr_un *)client_peer)->sun_path,
		      ((struct sockaddr_un *)(&conn->sa))->sun_path))
	    return 0;
	  break;
	default:
	  abort();
	}
    }
  return -1;
}

/*
 * Same as above, but probing for a known remote peer instead of
 * submarines. A client may have up to two peers. Probes both, if different.
 */
struct client **
ClientByPeer(root, client_peer)
struct client **root;
struct sockaddr *client_peer;
{
  struct client **cl;

  if (!root || !*root)
    root = &clients;
  for (cl = root; *cl; cl = &(*cl)->next)
    {
      struct conn *conn = (*cl)->i;

      if (!conn)
        conn = (*cl)->o;

      /* Done exactly once for each (different!) connection with client: */
      while (conn)
	{
	  if (!ConnMatchPeer(conn, client_peer))
	    return cl;

	  if (conn == (*cl)->i && conn != (*cl)->o)
	    conn = (*cl)->o;
	  else
	    break;
	}
    }
  return cl;
}

/*
 * As above, but starting with fd.
 * If a client is currently connected, we have a fildescriptor number
 * by which it can be identified. 
 * This function seems redundant, as we maintain a filedescriptor list that 
 * contains the client pointer. BUT, it is a nice way to obtain the adress 
 * of the last next pointer, when called with a negative fd.
 */
struct client **
ClientByFd(root, fd)
struct client **root;
int fd;
{
  struct client **cl;

  if (!root || !*root)
    root = &clients;
  for (cl = root; *cl; cl = &(*cl)->next)
    if ((fd >= 0) && 
        (((*cl)->i && ((*cl)->i->type!=CTYPE_INVALID) && ((*cl)->i->fd==fd)) ||
         ((*cl)->o && ((*cl)->o->type!=CTYPE_INVALID) && ((*cl)->o->fd==fd))))
      break;
  return cl;
}

/*
 * A lookup by Id is trivial, but not faster than the other... sigh...
 */
struct client **
ClientById(root, id)
struct client **root;
int id;
{
  struct client **cl;

  if (!root || !*root)
    root = &clients;
  for (cl = root; *cl; cl = &(*cl)->next)
    if ((*cl)->id == id)
      break;
  return cl;
}

/* print description of a connection into a dstring */
int
conn2dstr(d, off, conn)
struct dstring **d;
int off;
struct conn *conn;
{
  if (!conn)
    return dstring_append(d, off, NULL, 0);
  switch (conn->type)
    {
    case CTYPE_INVALID:
      return dstring_append(d, off, "invalid", 0);
    case CTYPE_TCP:
    case CTYPE_TCP_OR_TELNET:
    case CTYPE_TELNET:
    case CTYPE_UDP:
      return dstring_appendf(d, off, "%s:%s:%d", 
      		conn->type == CTYPE_UDP ? "UDP" : "TCP", 
      		inet_ntoa(((struct sockaddr_in *)(&conn->sa))->sin_addr),
		ntohs(((struct sockaddr_in *)(&conn->sa))->sin_port), 0);
    case CTYPE_WFD:
    case CTYPE_FD:
      {
        struct stat buf;
	char *t = "reg";

	if (fstat(conn->fd, &buf))
          return dstring_appendf(d, off, "nostat:%d", conn->fd, 0, 0, 0);

	     if (S_ISCHR (buf.st_mode)) t = "cdev";
	else if (S_ISBLK (buf.st_mode)) t = "bdev";
	else if (S_ISSOCK(buf.st_mode)) t = "sock";
	else if (S_ISFIFO(buf.st_mode)) t = "pipe";
        return dstring_appendf(d, off, "%s=%04x", 
		t, (*t == 'c' || *t == 'b') ? buf.st_rdev : buf.st_dev, 0, 0);
      }
    case CTYPE_BUF:
      return dstring_append(d, off, "buf???", 0);
    default:
      return dstring_append(d, off, "???", 0);
    }
}

/*
 * Parse_peer returns the portnumber at the end of the null-terminated 
 * string s. If there is none, ParsePeer returns the second argument,
 * which should be a default port number.
 * If the string s contains a port number, *offp (if offp is nonzero) will
 * recieve the offset of seperator.
 *
 * A peer in string form is a host followed by any
 * nonnumeric character (usually one of ,.:) followed
 * by a decimal port number.
 */
int
ParsePeer(s, p, ss)
char *s;
int p;
char **ss;
{
  char *e = s;

  while (*e)
    e++;
  e--;
  while (e > s && *e >= '0' && *e <= '9')
    e--;
  if (strchr(".,:;/\\", *e))
    {
      if (ss)
        *ss = e;
      e++;
      if (*e)
        p = atoi(e);
    }
  return p;
}

/* ==================== end of client management ======================= */

/* ================ filde descriptor artistics section ================== */

/* 
 * typical usage of the following two calls:
 *
 *        wfd = rfd = open(special_device, O_RDWR);
 *
 *        if (!jittr_is_selectable(wfd))
 *          if ((rfd = jittr_mk_selectable(wfd, &pid)) < 0)
 *            exit(0);
 *
 *        FD_SET(rfd, &rfdset);
 *        select(rfd+1, &rfdset, NULL, NULL, NULL);
 *
 *        read(rfd, ...);
 *        ioctl(wfd, BPPIOC_GETERR, &sta);
 *        write(wfd, ...);
 *
 *        close(rfd);
 *        waitpid(pid, &wstat, 0);
 *	  close(wfd);
 */

/*
 * jittr_is_selectable() checks how well the unix driver associated with fd
 * supports the select() system call. Especially if readability of fd is
 * detected corrrectly. It assumes that there is less than 9999 bytes available
 * for read on fd.  It may cause a read() to provoke a blocking select
 * thereafter.
 *
 * Call this only wuncet, it may take a few seconds.
 * Returns positive if select() appears to work on fd.
 * Returns 0, if the test result is "no".
 * Returns negative if the test failed.
 */
int
jittr_is_selectable(fd)
int fd;
{
  char buf[2];
  fd_set rfd;
  struct timeval tv;
  int i, pid;
#ifdef HAVE_UNION_WAIT
  union wait wstat;
#else
  int wstat;
#endif

  if ((pid = fork()) < 0)
    return -1;				/* test impossible, be pessimistic */
  if (pid)
    {
      wait(&wstat);
      if (WIFSIGNALED(wstat))		/* Bad news: Rescued by alarm. */
        return 0;	
      if (!WIFEXITED(wstat))		/* Completely screwed. */
        {
	  kill(pid, 9);			/* Shoot him in the foot. */
	  wait(&wstat);			/* We hope, he dies fast ... */
	  return -2;			/* test failed, be very pessimistic */
	}
      return WEXITSTATUS(wstat);	/* He made it, forward his answer. */
    }

  tv.tv_sec = 0;
  tv.tv_usec = 0;
  signal(SIGALRM, SIG_DFL);		/* I want the stake bloody! */
  alarm(1);
  for (i = 9999; i > 0; i--)
    {
      FD_ZERO(&rfd);
      FD_SET(fd, &rfd);
      if (!select(fd+1, &rfd, 0, 0, &tv))
        exit(1);			/* thumbs up, it would have blocked! */
      read(fd, buf, 1);
      /* 
       * Hmm, what can the return value of read tell us, 
       * when we don't know anything about the object
       * that fd is associated with?
       */
    }
  exit(0);				/* unbelievable: we read 9999 bytes? */
}


/* 
 * jittr_mk_selectable() forks a process that does blocking read on fd.
 * This read() may cause your system load average to increase by one. It 
 * reflects I/O-wait rather than CPU usage.
 *
 * Returns a pipe filedescriptor, which may be read under select() control.
 * Also returns the childs process id in *pidp, if pidp is nonzero.
 * Close the returned fildescriptor to make the child die. You should wait()
 * for any zombies then. Other operations e.g. write(), ioctl() are all
 * unsupported!
 * Expect that read() operations on the returned fildescriptor read in single 
 * byte granularity. Together with the second process, which copies the data
 * this limits efficiency of the fildescriptor.
 *
 * Returns negative when in trouble. Errno is set then.
 */
int
jittr_mk_selectable(fd, pidp)
int fd;
int *pidp;
{
  char buf[2];
  int pfd[2];
  int r, pid;

  if (pipe(pfd))
    return -1;

  if ((pid = fork()) < 0)
    return -1;
  
  if (pid)
    {
      if (pidp)
	*pidp = pid;
      close(pfd[1]);
      return pfd[0];
    }

  /* close everything */
  for (r = 255; r >= 0; r--)
    if (r != pfd[1] &&
#ifdef DEBUG
	(!debugfp || (r != fileno(debugfp))) &&
#endif
        r != fd)
      (void)close(r);

  pid = getppid();

  while (!kill(pid, 0))		/* in case he forgets us ... */
    {
      r = read(fd, buf, 1);

      if (r < 0)
        {
	  debug2("Device slave %d: read: %s.\n", (int)getpid(), 
	    strerror(errno));
          exit(1);
	}
      if (r > 0)
        {
	  int w = 0;

	  while (w >= 0 && w < r)
	    w += write(pfd[1], buf+w, r-w);
	  if (w < 0)
	    {
	      debug2("Device slave %d: write: %s.\n", (int)getpid(), 
		strerror(errno));
	      exit(2);
	    }
	}
    }
  debug2("Device slave pid %d suicide: Lost parent pid %d.\n", (int)getpid(),
  	pid);
  exit(0);
}
