/*
 * schedule.c -- a scheduler for timed events.
 * 
 * All code should be independant of realtime or timezones. 
 * Except for mktime() which should not be used without correction.
 *
 * 30.7.94 -- jw
 *
 * Calculating timezone-correction for mktime().
 * Can now truncate an exponential backoff. 
 * sched_parse_tv() & sched_format_tv() improved.
 * 14.7.96 -- jw
 *
 * Null termination in sched_format_timing() fixed. top looked so ugly.
 * 20.5.97, jw
 *
 * sched_parse_tv() now returns negative if mktime()/gmtime() fails
 */
#include <string.h>		/* strlen(), bzero()@4.1.3 */
#include <sys/time.h>		/* struct timeval */
#include <stdlib.h>		/* malloc(), free() */
#include "jittr/schedule.h"
#include "jittr/debug.h"

/*
#define STANDALONE
*/

#ifdef linux 
extern sprintf();
extern printf();
#endif

#ifdef sun
# include <stdio.h>
# ifdef FILE
#  include <strings.h>	/* bzero()@solaris */
# else
#  define bzero(s, n) memset(s, 0, n)
# endif
# include <stdlib.h>	/* malloc(), free() */
# ifdef __STDC__
#  include "jittr/sun_stdlib.h"
# endif
#endif

#ifdef __sgi
# include <bstring.h>  /* bzero()@irix */
#endif

/*
 * gcc@sunos4 and cc@irix warn about the timercmp macro in sys/time.h:
 * warning: suggest parentheses around && within ||
 *
 * We bring in our own correctly parenthesed xtimercmp:
 */
#define xtimercmp(tvp, uvp, cmp) \
	((tvp)->tv_sec cmp (uvp)->tv_sec || \
	 ((tvp)->tv_sec == (uvp)->tv_sec && (tvp)->tv_usec cmp (uvp)->tv_usec))

/*
 * If the first character in *pp is in the list, advance *pp by one
 * and return the index in the list. Otherwise return -1
 */
static int
get_token(pp, list)
char **pp;
char *list;
{
  char *l;

  for (l = list; *l; l++)
    if (*l == **pp)
      {
        (*pp)++;
	return l - list;
      }
  return -1;
}

static int
get_digit(pp, base)
char **pp;
int base;
{
  int r;
  char *l1 = "0123456789";
  char *l2 = "abcdef";
  char *l3 = "ABCDEF";

  if (base > 16 || base < 2)
    return -2;
  if (base < 16 && base > 10)
    l2[base -1] = l3[base-10] = '\0';
  else if (base < 10 && base >= 2)
    l1[base] = '\0';
  if (((r = get_token(pp, l1)) < 0) && ((r = get_token(pp, l2) + 10) < 10))
    r = get_token(pp, l3);  
  return r;
}

static int 
get_number(pp, base)
char **pp;
int base;
{
  int i, n = 0;
  char *p = *pp;

  while ((i = get_digit(pp, base)) >= 0)
    n = n * base + i;
  return (p == *pp) ? -1 : n;
}

/*
 * All of the following time examples are understood here.
 * Those starting with + are relative to now:
 *
 * +3m
 * +.25s
 * +3s10u
 * +240m.125s
 * +5m40.123
 * +5h40.123
 * +5:40
 * 3h45m12.5s_12:31:1994
 * 3:45:12.5_12-31-94
 * 3:45h,12/24/94
 *
 * Negative relative timestamps like +-3s or -3s are NOT understood here!
 * 
 * Absolute timestamps without a date are understood, when now is NON-zero.
 * They introduce the concept of "today":
 * 3:45h
 * 3h45m12.5s
 *
 * Returns 0 when sucessful, otherwise it returns a negative number, whose 
 * meaning you must lookup here in the source. Hmmm.
 */
int
sched_parse_tv(sp, tv, now)
char **sp;
struct timeval *tv, *now;
{
  static int localtime_off = 0;		/* used for correcting mktime() */
  char *s;
  int rel = 0, sec = 0, min = 0, hour = 0;
  int v, i;
  struct tm tm;

  if (!sp || !(s = *sp) || !tv)
    return -1;
  if (*s == '+')
    rel++;
  if (*s == '=' || *s == '+')
    s++;
  if (rel && !now)
    return -2;			/* relative spec needs the now origin */

  tv->tv_usec = tv->tv_sec = 0;
  bzero((char *)&tm, sizeof(tm));

  if ((v = get_number(&s, 10)) < 0)
    v = 0;
  switch (get_token(&s, ":hms-"))
    {
    case 0:
    case 1:
      tm.tm_hour = v;
      hour++;
      break;
    case 2:
      tm.tm_min = v;
      min++;
      break;
    case 4:
      return -3; 	/* cannot interpret negative relative timestamps */
    default:
      tm.tm_sec = v;
      sec++;
    }

  if ((v = get_number(&s, 10)) >= 0)
    switch (get_token(&s, ":hmsu"))
      {
      case 3:
	tm.tm_sec = v;
	sec++;
	break;
      case 4:
	tv->tv_usec = v;
	break;
      case 0:
      case 1:
      case 2:
	if (min || sec)
	  return -4;	/* colon, 'h', or 'm' after seconds!? Means what? */
	tm.tm_min = v;
	min++;
	break;
      default:
	if (hour && (*s != '.'))
	  {
	    tm.tm_min = v;
	    min++;
	  }
	else 
	  {
	    if (sec)
	      return -5;	/* unnamed trailing number after seconds */
	    tm.tm_sec = v;
	    sec++;
	  }
	break;
      }

  if ((v = get_number(&s, 10)) >= 0)
    if (sec)
      tv->tv_usec = v;
    else
      tm.tm_sec = v;

  if (!get_token(&s, "."))
    {
      i = 100000;
      while ((v = get_digit(&s, 10)) >= 0)
        {
          tv->tv_usec += i * v;
	  i /= 10;
	}
    }

  if (!get_token(&s, "s") && !tv->tv_usec)
    {
      if ((v = get_number(&s, 10)) >= 0)
	tv->tv_usec = v;
      get_token(&s, "u");
    }

  if (rel)
    {
      /* relative time takes no date */
      tv->tv_sec = tm.tm_hour;
      tv->tv_sec = tv->tv_sec * 60 + tm.tm_min;
      tv->tv_sec = tv->tv_sec * 60 + tm.tm_sec;

      tv->tv_sec += now->tv_sec;
      if ((tv->tv_usec += now->tv_usec) >= 1000000)
        {
	  tv->tv_sec++;
	  tv->tv_usec -= 1000000;
	}
      
      *sp = s;
      return 0;
    }

  if (!hour)
    return -6;			/* absolute spec. needs at least the hour */

  if (get_token(&s, ",_") < 0)
    {
      struct tm *now_tm;

      if (!now)
	{
	  debug("sched_parse_tv: Cannot guess the day, now ptr is NULL\n");
	  return -7;			/* absolute spec needs a date */
	}
      now_tm = gmtime((time_t *)&now->tv_sec);
      tm.tm_mon  = now_tm->tm_mon;
      tm.tm_mday = now_tm->tm_mday;
      tm.tm_year = now_tm->tm_year;
    }
  else
    {
      if ((tm.tm_mon = get_number(&s, 10) - 1) < 0)
	return -8;				/* month bad */
      if (get_token(&s, ":/-") < 0)
	return -9;				/* month-day seperator bad */
      if ((tm.tm_mday = get_number(&s, 10)) < 1)
	return -10;				/* day bad */
      if (get_token(&s, ":/-") < 0)
	return -11;				/* day-year seperator bad */
      if ((tm.tm_year = get_number(&s, 10)) < 70)
	return -12;				/* year bad */
    }

  if (tm.tm_year < 70)
    tm.tm_year = 70;
  if (tm.tm_year >= 1970)
    tm.tm_year -= 1900;

#if 0
  /* I found timegm() on sunos and (undocumented) on linux. */
  if ((tv->tv_sec = timegm(&tm)) == -1)
    return -13;				/* date cannot be represented in sec */
#else
  /* 
   * mktime() assumes the given fields to be in localtime; it ignores the
   * 0 in the tm.tm_gmtoff field. On sun, mktime() sets the fields tm.tm_gmtoff
   * and tm.tm_isdst correctly. Here the correction
   *   tv->tv_sec += tm.tm_gmtoff - 3600 * tm.tm_isdst;
   * would work. But this line is highly nonportable, as linux and irix5 do
   * not have the field tm.tm_gmtoff in their struct tm. Below I use
   * a linear regression method to calculate the effective GMT offset.
   *
   * 3.7.96 jw.
   */
  if ((tv->tv_sec = mktime(&tm)) == -1)
    return -13;				/* date cannot be represented in sec */
  tv->tv_sec += localtime_off;
  for (;;)
    {
      unsigned long s = mktime(gmtime((time_t *)&tv->tv_sec)) + localtime_off;
      int diff = tv->tv_sec - s;

      if (s == -1)
	return -14;		/* date can no longer be represented in sec */

      if (!diff)
	break;
      debug1("adjusting result of mktime() by %d seconds.\n", diff);
      tv->tv_sec += diff;
      localtime_off += diff;
    }
#endif

  *sp = s;
  return 0;
}

/* 
 * needs a nonzero now provided by gettimeofday 
 * to help resolving relative timings here.
 * Valid syntax examples are:
 *
 * +0.025s+12h*4*1.5=48h
 * [=====][==]OO [=] [=]
 *    \     \  \\  \  \_limit decr/incr. of repeat time interval.
 *     \     \  \\  \___exponential fallback factor, 2 if missing
 *      \     \  \\_____total repeat count, infinite if 0 or missing
 *       \     \  \_____repeat introducer, one repetition if not present
 *        \     \_______repeat time interval. single event, if missing
 *         \____________relative or absolute time spec of (first) event.
 */
int
sched_parse_timing(sp, ti, now)
char **sp;
struct sched_timing *ti;
struct timeval *now;
{
  int r;
  char *a = NULL;
  struct timeval nn;

  if (!sp && !*sp && !ti)
    return -1;			/* usage wrong */
  
  if (**sp == '+')
    a = *sp;
  if (a && !now)
    return -2;			/* relative spec, need now */

  if ((r = sched_parse_tv(sp, &ti->when, now)) < 0)
    return -100+r;		/* sched_parse_tv start time format wrong */
  TV2TIMING(&ti->when, ti);	/* init rest of timing structure */

  if (**sp == '*' && a)		/* interval may be skipped when relative */
    *sp = a;

  if (**sp != '+')
    return 0; 			/* single event, no repetitions */

  nn.tv_sec = nn.tv_usec = 0;	/* really relative spec */
  if ((r = sched_parse_tv(sp, &ti->iv, &nn)) < 0)
    return -200+r;		/* sched_parse_tv interval time format wrong */

  ti->max_rep = 2;

  if (get_token(sp, "*"))
    return 0;			/* single repetition, two events */

  if ((ti->max_rep = get_number(sp, 10)) < 0)
    ti->max_rep = 0;		/* infinite repetitions */

  if (get_token(sp, "*"))
    return 0;			/* no fallback between repeats */

  if ((((r = get_number(sp, 10)) >= 0) && !get_token(sp, ".")) ||
      !get_token(sp, "."))
    {
      double frac = 0.1;

      ti->exponent = (r > 0) ? (double)r : 0.0;
      while ((r = get_digit(sp, 10)) >= 0)
        {
	  ti->exponent += frac * (double)r;
	  frac = frac * .1;
	}
    }
  else
    ti->exponent = (r <= 0) ? 2.0 : (double)r;	/* default fallback is 2.0 */

  if (*(a = *sp) != '=')
    return 0;
  
  *a = '+';	/* tell sched_parse_tv, that it is a relative timestamp */
  r = sched_parse_tv(sp, &ti->li, &nn);
  *a = '=';
  if (r < 0)
    return -300+r;

  return 0;
}

/*
 * return the number of decimal digits to print the number 
 */
static int
num_len(n)
long n;
{
  long f, i;

  for (f = i = 1; i < 7; i++)
    {
      f = f * 10;
      if (n < f)
        break;
    }
  return (int)(i+1);
}

/* 
 * sched_timerdiff() returns the difference timeval in result, between future and
 * past.  If the diffeence is negative, result remains untouched and -1 is
 * returned.  Returns 0 on success.
 */
int
sched_timerdiff(result, future, past)
struct timeval *result, *future, *past;
{
  long s, u;

  s = future->tv_sec - past->tv_sec;
  if ((u = future->tv_usec - past->tv_usec) < 0)
    {
      u += 1000000;
      s--;
    }

  if (s < 0)
    {
      /* debug("sched_timerdiff: negative difference"); */
      return -1;
    }
  result->tv_sec = s;
  result->tv_usec = u;
  return 0;
}

/*
 * Tv is expected to be an absolute timestamp. An absolute timestamp spec is
 * printed when now is NULL. A relative spec is calcualted and printed, when
 * now is nonzero.
 * relative timstamps print like this:
 *	+1h15m2.502s
 * or if in the past:
 *	+-1h15m2.502s
 *
 * Fields that are 0 are ommitted. Except for minutes: When hours and seconds
 * are nonzero and minutes is 0, then 0m is inserted between them for
 * readability.
 */
int
sched_format_tv(p, len, tv, now)
char *p;
int len;
struct timeval *tv, *now;
{
  int l;
  
  if (!p || !len || !tv)
    return -1;			/* usage */

  if (now)
    {				/* make relative description */
      long s, u;
      struct tm tm;
      struct timeval diff;
      int neg = xtimercmp(tv, now, <);

      sched_timerdiff(&diff, neg ? now : tv, neg ? tv : now);
      s = diff.tv_sec;
      u = diff.tv_usec;

      tm.tm_sec = s % 60; s = s / 60;
      tm.tm_min = s % 60; s = s / 60;
      tm.tm_hour = s;

      l = 3+neg;  /* leading '+', optional '-', trailing 0, round down byte */
      if (tm.tm_hour)
        l += num_len(tm.tm_hour) + 1;
      if (tm.tm_min || (tm.tm_hour && tm.tm_sec))
        l += num_len(tm.tm_min) + 1;
      if (tm.tm_sec)
        l += num_len(tm.tm_sec) + 1;
      if (u)
        l += 7;			/* max. 6 digits and a period */
      l &= ~1;
      if (l > len)
        return -l;		/* need more string space */
      
      p[0] = '+';
      p[1] = '-';
      l = 1 + neg;
      if (tm.tm_hour) 
        { 
	  sprintf(p + l, "%dh", tm.tm_hour); 
	  l = strlen(p);
	}
      if (tm.tm_min || (tm.tm_hour && tm.tm_sec))
        {
	  sprintf(p + l, "%dm", tm.tm_min);
	  l = strlen(p);
	}
      if (tm.tm_sec || u)
        {
	  if (u)
	    sprintf(p + l, "%f", (double)tm.tm_sec + 0.000001 * (double)u);
	  else
	    sprintf(p + l, "%d", tm.tm_sec);
	  l = strlen(p);
	  if (u)	
	    while (p[l-1] == '0')
	      p[--l] = '\0';
	  p[l++] = 's';
	  p[l] = '\0';
	}

      if (l == neg + 1)		/* whoops, exact match */
        {
	  p[l++] = '0';
	  p[l] = '\0';
	}
    }
  else
    { 				/* absolute time format */
      struct tm *tm;

      if (!(tm = gmtime((time_t *)&tv->tv_sec)))
        return -1;		/* cannot convert time */

      l = 1 + 2 + 1 + 2 + 1 + 4 + 1;	/* ",MM/DD/YYYY\0" */
      l += num_len(tm->tm_hour) + 1 + 2 + 1;	/* HH:MMh */
      if (tm->tm_sec) l += num_len(tm->tm_sec) + 1;
      if (tv->tv_usec)
        l += 7;
      l &= ~1;
      if (l > len)
        return -l;		/* need more string space */

      l = 0; 
      sprintf(p, "%d:%02dh", tm->tm_hour, tm->tm_min); 
      l = strlen(p);
      if (tm->tm_sec || tv->tv_usec)
        {
	  if (tv->tv_usec)
	    sprintf(p + l, "%f", 
	    	    (double)tm->tm_sec + 0.000001 * (double)tv->tv_usec);
	  else
	    sprintf(p + l, "%d", tm->tm_sec);
	  l = strlen(p);
	  if (tv->tv_usec)
	    while (p[l-1] == '0')
	      p[--l] = '\0';
	  p[l++] = 's';
	  p[l] = '\0';
	}
      sprintf(p + l, ",%d/%d/%d", tm->tm_mon+1, tm->tm_mday, tm->tm_year+1900);
      l = strlen(p);
    }  
  return l;			/* printed format o.k. */
}

int
sched_format_timing(s, len, ti, now)
char *s;
int len;
struct sched_timing *ti;
struct timeval *now;
{
  int l, r;
  struct timeval nn;

  if (!s || !len || !ti)
    return -11;			/* usage */
  if ((l = sched_format_tv(s, len, &ti->when, now)) < 0)
    return l;
  
  if (ti->max_rep == 1)
    return l;			/* single event, no repetitions */

  nn.tv_sec = nn.tv_usec = 0;
  if ((r = sched_format_tv(s + l, len - l, &ti->iv, &nn)) < 0)
    return -((l - r + 1) & ~1);
  l += r;

  while ((r = l))
    {
      if (ti->max_rep == 2)
	break;
      r++;
      if (ti->max_rep)
	r += num_len(ti->max_rep);
      if (ti->exponent == 1.0)
        break;
      r++;
      if (ti->exponent != 2.0)
        r += num_len((int)ti->exponent) 
	  + (((double)((int)ti->exponent) == ti->exponent) ? 0 : 8);
      if (r >= len)
        return -((r + 1) & ~1);		/* found out, we get too long */

      if (1)				/* supress warning from cc@solaris */
        break;
    }

  if (ti->max_rep == 2)
    return l;			/* two occurances, no rep count */

  s[l++] = '*';
  s[l] = '\0';
  if (ti->max_rep)
    {
      sprintf(s + l, "%ld", ti->max_rep);
      l = strlen(s);
    }
  if (ti->exponent == 1.0)
    return l;			/* no fallback */

  s[l++] = '*';
  s[l] = '\0';
  if (ti->exponent != 2.0)
    {
      sprintf(s + l, "%f", ti->exponent);
      l = strlen(s);
      if ((double)((int)ti->exponent) != ti->exponent)
        while (s[l-1] == '0')
	  s[--l] = '\0';
    }
  if (ti->li.tv_sec || ti->li.tv_usec)
    {
      if ((r = sched_format_tv(s + l, len - l, &ti->li, &nn)) < 0)
	return -((l - r + 1) & ~1);
      s[l] = '=';
      l += r;
    }
  return l;
}

int
sched_make_action(ap, fn, flag, argv)
struct sched_action **ap;
int (*fn) __P((void **argv));
int flag;
void **argv;
{
  int i;
  struct sched_action *a;

  if (!ap || !fn || !argv)
    return -1;		/* usage */
  *ap = NULL;

  if (!(a = (struct sched_action *)malloc(sizeof(struct sched_action))))
    {
      return -2;	/* no mem */
    }

  for (i = 0; i < SCHED_ARGC; i++)
    {
      a->argv[i] = *argv;
      if (*argv)
        argv++;
    }
  a->fn = fn;
  a->flag = flag;
  a->next = NULL;
  *ap = a;
  return 0;
}

/*
 * May be called with a null pointer,
 * May be called to free up a sched_event structure, as the
 * action is the first element and no other elements are malloced.
 * May be called to free a chained list.
 */
void
sched_free_action(a)
struct sched_action *a;
{
  struct sched_action *n;
  int i;

  while (a)
    {
      for (i = 0; i < SCHED_ARGC; i++)
        {
	  if ((a->flag & (1 << i)) && a->argv[i])
	    free((char *)a->argv[i]);
          a->argv[i] = NULL;
	}
      a->fn = NULL;
      n = a->next;
      free((char *)a);
      a = n;
    }
}

static struct sched_event *root = NULL;
static struct sched_event *peek = NULL;

/*
 * Root is a chained list of events sorted by increasing "when"-timestamp.
 * sched_enter() inserts a new element into the chain maintaining the ordering.
 * FIXME: The search algorithm here here is linear, but could be heap-sort.
 */
int
sched_enter(e)
struct sched_event *e;
{
  struct sched_event **ep, *ne = NULL;
  struct timeval *tv = &e->time.when;
  
  peek = NULL;
  for (ep = &root; *ep; ep = &(*ep)->next)
    if (!(ne = *ep) || xtimercmp(tv, &ne->time.when, <))
      break;
  e->next = *ep;
  *ep = e;
  return e->id;
}

static unsigned long next_id = 1;

int
sched_enter_event(ti, a, flag)
struct sched_timing *ti;
struct sched_action *a;
int flag;
{
  struct sched_event *e;

  if (!ti || !a)
    return -1; 		/* usage */

  if (!(e = (struct sched_event *)malloc(sizeof(struct sched_event))))
    return -2;		/* no mem */
  e->action = *a;
  e->time = *ti;
  e->id = next_id++;
  if (flag & 1)
    free((char *)ti);
  if (flag & 2)
    free((char *)a);
  sched_enter(e);
  return e->id;
}

struct sched_event *
sched_alloc(stardate_fn, sec, usec, fn, arg1, arg2)
int (*stardate_fn) __P((struct timeval *));
int sec, usec;
int (*fn) __P((void **));
void *arg1, *arg2;
{
  struct sched_event *e;

  if (!(e = (struct sched_event *)malloc(sizeof(struct sched_event))))
    return NULL;
  e->id = next_id++;
  e->next = NULL;

  e->time.when.tv_sec = sec;
  e->time.when.tv_usec = usec;
  e->time.iv.tv_sec = e->time.iv.tv_usec = 0;
  e->time.li.tv_sec = e->time.li.tv_usec = 0;
  e->time.exponent = 1.0;
  e->time.count = 0;
  e->time.max_rep = 1;		/* a single event */

  if (stardate_fn)
    {
      stardate_fn(&e->time.when);
      if (sec || usec)
	{
	  e->time.iv.tv_sec = sec;
	  e->time.iv.tv_usec = usec;
	  e->time.max_rep = 0;	/* infinite repetitions */
	}
    }

  e->action.argv[0] = arg1;
  e->action.argv[1] = arg2;
  e->action.fn = fn;
  e->action.flag = 0;
  e->action.next = NULL;

  return e;
}

static struct sched_event **
lookup_event(id)
int id;
{
  struct sched_event **ep;

  if (!root)
    return NULL;
  for (ep = &root; *ep; ep = &(*ep)->next)
    if ((*ep)->id == id)
      return ep;
  return NULL;
}

struct sched_event *
sched_unlink_event(id)
int id;
{
  struct sched_event *e, **ep = lookup_event(id);

  if (!ep || !(e = *ep))
    return NULL;	/* no such event */
  peek = NULL;
  *ep = e->next; 
  e->next = NULL;
  return e;
}

int
sched_timeout(tv, now)
struct timeval *tv, *now;
{
  if (!root)
    return 0;		/* no events to wait for */

  if (now)		/* measure relative */
    {
      tv->tv_sec = root->time.when.tv_sec - now->tv_sec;
      if ((tv->tv_usec = root->time.when.tv_usec - now->tv_usec) < 0)
	{
	  tv->tv_usec += 1000000;
	  tv->tv_sec -= 1;
	}
    }
  else			/* measure absolute */
    *tv = root->time.when;
  return (tv->tv_usec < 0) ? -root->id : root->id;	/* when relative */
}

/*
 * Returns 0 if nothing done.
 * Otherwise it returns the positive ID of the processed event.
 *
 * How to improve timing accuracy here:
 *  If we are late, measure the average delay dx that we come to late.
 *  Share this value with sched_timeout(), which should subtract this value
 *  from its calculations. We expect to be called *not that late* next time.
 *  If we are early, do not return if we are within [-dx,0]. We do not expect
 *  that the I/O subsystem can call us again quick enough. Better busy loop
 *  until action is due, or run the action immediatly.
 */
int
sched_process(ret, now)
int *ret;
struct timeval *now;
{
  double d, n, l;
  int id;
  struct sched_event *e = root;

  if (!ret || !now)
    return -1;		/* caller should abort() */
    
  if (!root || xtimercmp(now, &root->time.when, <))
    return 0;		/* still nothing. call again later */

#ifdef DEBUGGG
  {
    char t[50];

    sched_format_tv(t, 49, now, &root->time.when);	/* args reversed! */
    debug1("sched_process: late %s\n", t);
  }
#endif

  e->time.count++;
  id = e->id;
  root = e->next;
  peek = NULL;

  *ret = e->action.fn(e->action.argv);
  if (*ret < 0 || (e->time.max_rep && e->time.count >= e->time.max_rep))
    sched_free_action((struct sched_action *)e);	/* last call */
  else
    {
      for (;;)
	{
	  e->time.when.tv_sec += e->time.iv.tv_sec;
	  if ((e->time.when.tv_usec += e->time.iv.tv_usec) >= 1000000)
	    {
	      e->time.when.tv_sec++;
	      e->time.when.tv_usec -= 1000000;
	    }
#ifdef DEBUGG
	  debug2("interval timer applied: %d.%06dsec\n",
		    (int)e->time.iv.tv_sec, (int)e->time.iv.tv_usec);
#endif
	  if (e->time.max_rep || xtimercmp(now, &e->time.when, <))
	    break;
	  
	  debug("timestamp of infinite repetition is in the past. Skipping.\n");
	  /* 
	   * This avoids fireworks when resuming from a breakpoint, or when
	   * recovering from a unix scheduling problem.
	   * "Multiple levels of scheduling should have cooperative strategies"
	   */
	}

      d = (double)e->time.iv.tv_usec + ((double)e->time.iv.tv_sec * 1000000.0);
      l = (double)e->time.li.tv_usec + ((double)e->time.li.tv_sec * 1000000.0);
      n = d * e->time.exponent;
      if ((d < l) ^ (n < l))	/* crossed the limit. Catch it. */
        {
	  e->time.exponent = 1.0;
          n = l;
	}
      e->time.iv.tv_sec = (int)(n / 1000000.0);
      e->time.iv.tv_usec = (int)(n - e->time.iv.tv_sec * 1000000.0);

      if (e->action.next)
	{				/* cycle actions */
	  struct sched_action a, *la;

	  la = (a = e->action).next;
	  while (la->next)
	    la = la->next;
	  e->action = *(a.next);
	  la->next = a.next;
	  a.next = NULL;
	  *(a.next) = a;
	}
      sched_enter(e);			/* reschedule */
    }
  return id;
}

int 
sched_peek_event(tp, ap)
struct sched_timing **tp;
struct sched_action **ap;
{
  int id;

  if (!ap && !tp)
    {
      peek = root;
      return root ? root->id : 0;	/* reset */
    }
  if (!peek)
    return 0;		/* no more */
  if (ap)
    *ap = &peek->action;
  if (tp)
    *tp = &peek->time;
  id = peek->id;
  peek = peek->next;
  return id; 		/* got one */
}

#ifdef STANDALONE
int
main(ac, av)
int ac;
char **av;
{
  int i;
  char *s;
  struct timeval tv, now;
  struct sched_timing ti;
  char buf[256];

  now.tv_sec = now.tv_usec = 0;


  /* use the portable interface from gettime.c */
#if 0
  gettimeofday(&now, 0);
#else
  gettimeval(&now);
#endif

  sched_format_tv(buf, 256, &now, NULL);
  printf("now=%ld.%06ld=%s\n", now.tv_sec, now.tv_usec, buf);

  s = av[1];
  i = sched_parse_tv(&s, &tv, &now);
  printf("r=%d chars=%d, tv=%ld.%06ld\n", i, s - av[1], tv.tv_sec, tv.tv_usec);
  s = av[1];
  i = sched_parse_timing(&s, &ti, &now);
  i = sched_format_timing(buf, 256, &ti, 0);
  printf("r=%d %s\n", i, buf);
  i = sched_format_timing(buf, 256, &ti, &now);
  printf("r=%d %s\n", i, buf);
  return 0;
}
#endif /* STANDALONE */
