/*
 * session.c -- jittr standard commands library (session management methods)
 *
 * Needed for the higher levels of the jittr protocol.
 *
 * 17.07.96 jw.
 */
#include <unistd.h>	/* write() */
#include <sys/types.h>  /* for sys/socket.h */
#include <sys/socket.h> /* struct sockaddr for net.h */
#include <sys/time.h>   /* struct timeval for net.h */
#include <netdb.h>	/* gethostent() */

#include "jittr/atom.h"		/* atoms */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"	/* struct cwp */
#include "jittr/net.h"		/* struct client, CLOCK_OFF_GRAN */
#include "jittr/doio.h"		/* struct server */
#include "jittr/schedule.h"	/* struct sched_action for top_cmd */

static int              id_cmd __P((int idx, int flag, void *dummy));
static int          prompt_cmd __P((int idx, int flag, void *dummy));
static int         netmask_cmd __P((int idx, int flag, void *dummy));
static int            date_cmd __P((int idx, int flag, void *dummy));

/*
 * date [+[-]]tstamp [prefix [suffix]]
 * date -sync +[-]time_offset
 * date -echo [prefix [suffix]]
 *
 * convert between absolute and relative timestamps. Tstamp defaults to "+0"
 * set the client inbound clock offset
 * print the client inbound clock offset
 *
 * XXX: Fixme: The code for the three algorithms is all mixed up.
 */
static int
date_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  struct client *cl = exec_context->who;
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen, r, echo = 0, synch = 0, rel = 0, neg = 0;
  char *s = jarg_first_word(&slen);
  struct timeval tv, now;
  char buf[50];

  while ((slen >= 2) && (*s == '-'))
    {
      if (!strncmp("-synchronize_client", s, slen))
	synch++;
      else if (!strncmp("-echo", s, slen))
        echo++;
      else
        return jittr_error("%s: unknown option %s.\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }

  if (synch || !echo)
    {
      if (!s)
	s = "+0";
      if (*s == '+')
	rel++;
      if (rel && s[1] == '-')
	{	
	  /* the library cannot parse negative timestamps so we cheat here */
	  neg++;
	  *(++s) = '+';
	}
    }

  /* we need a now-pointer, to resolve abvreviated timestamps. */
  if (!synch && !echo)
    jittr_gettimeval(&now);
  else
    {
      if (synch && !rel)
        return jittr_error("%s: sorry: need relative timestamp with -sync.\n",
	  jarg_name(), 0, 0, 0);
      now.tv_sec = now.tv_usec = 0;
    }

  if ((synch || !echo) && ((r = sched_parse_tv(&s, &tv, &now)) < 0))
    return jittr_error("%s: cannot parse timestamp: error code %d.\n",
      jarg_name(), r, 0, 0);
  
  if (synch)
    {
      /* 
       * Here is the rounding mechanism that saves our livelock prevention!
       * You should study the algorithm of ext_get_cmd() before you temper
       * with following two lines of code!
       */
      tv.tv_usec = (tv.tv_usec + (CLOCK_OFF_GRAN/2))/CLOCK_OFF_GRAN;
      tv.tv_usec *= CLOCK_OFF_GRAN;

#ifdef DEBUG
      sched_format_tv(buf, 49, &cl->off.tv, NULL);
      debug3("%s: time offset was: %d * %s\n", cl->name->buf,cl->off.sign,buf);
      sched_format_tv(buf, 49, &tv, NULL);
      debug3("%s: time offset now: %d * %s\n", cl->name->buf,neg ?-1:1, buf);
#endif
      cl->off.sign = neg ? -1 : 1;
      cl->off.tv = tv;		/* structure copy */
      if (!echo)
        return 0;
    }

  if (echo)
    {
      if (!cl->off.sign)
        return jittr_error("%s -echo: time adjustment for %s undefined.\n",
		jarg_name(), cl->name->buf, 0, 0);
      sched_format_tv(buf, 50, &cl->off.tv, &now);
      if (synch)
        s = jarg_next_word(&slen);
      dstring_append(rp, -1, s ? s : "# ", s ? slen : 2);
      dstring_append(rp, -1, "+-", (cl->off.sign < 0) ? 1 : 2); /* inverted! */
      dstring_append(rp, -1, buf + 1, 0);
      s = jarg_next_word(&slen);
      dstring_append(rp, -1, s ? s : "\n", s ? slen : 1);
      return 0;
    }

  if (neg)	/* correct our cheating */
    {
      /* convert it back into a relative timestamp */
      tv.tv_sec -= now.tv_sec;
      if ((tv.tv_usec -= now.tv_usec) < 0)
        {
	  tv.tv_sec--;
	  tv.tv_usec += 1000000;
	}
      /* now subtract it from the current time */
      tv.tv_sec = now.tv_sec - tv.tv_sec;
      if ((tv.tv_usec = now.tv_usec - tv.tv_usec) < 0)
        {
	  tv.tv_sec--;
	  tv.tv_usec += 1000000;
	}
    }

  /* tv is always an absolute timestamp here */
  sched_format_tv(buf, 50, &tv, rel ? NULL : &now);
  s = jarg_next_word(&slen);
  dstring_append(rp, -1, s ? s : "# ", s ? slen : 2);
  dstring_append(rp, -1, buf, 0);
  s = jarg_next_word(&slen);
  dstring_append(rp, -1, s ? s : " UTC\n" + (rel?0:4), s ? slen : (rel?5:1));
  return 0;
}

/*
 * netmask [-enforce|-all|-default] [-[read_]append] [netmasks/hosts ...]
 *
 * set or get the netmask of the internet services.
 * This shall be used to provide a minimum protection against intruders 
 * that would like to disturb your work.
 *
 * Existing connections are not harmed by narrowing the netmask unless -enforce
 * is specified.
 *
 * Connecting to and observing the system can be allowed to a broader range
 * of hosts than modifying the system by means of the -read_add option.
 */
static int
netmask_cmd(idx, flag, is_pwd)
int idx, flag;
void *is_pwd;
{
  struct client *cl = exec_context->who;
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen, append = 0, read_only = 0, enforce = 0, endopt = 0;
  char *s = jarg_first_word(&slen);

  if (!s)
    {
      dstring_append(rp, -1, "# ", 2);
      if (server.netmask && server.netmask_write)
        {
          int l = server.netmask->length;
          int wl = server.netmask_write->length;

	  if ((wl != l) ||
	      (strncmp(server.netmask->buf, server.netmask_write->buf, l)))
	    {
              dstring_append(rp, -1, " read {", 7);
              dstring_append(rp, -1, server.netmask->buf, l);
              dstring_append(rp, -1, "}\n# write {", 11);
              dstring_append(rp, -1, server.netmask_write->buf, wl);
              dstring_append(rp, -1, "}", 1);
	    }
	  else
            dstring_append(rp, -1, server.netmask->buf, l);
	}
      else if (server.netmask)
	dstring_append(rp, -1, server.netmask->buf, server.netmask->length);
      dstring_append(rp, -1, "\n", 1);
      return 1;
    }
  
  if (cl && cl->i && cl->i->read_only)
    return jittr_error("netmask not allowed: read_only!n", 0, 0, 0, 0);

  while (slen)
    {
      if (endopt || slen < 2 || *s != '-')  /* s == NULL implies slen == 0 */
        break;
      if (!strcmp("--", s)) 
        endopt++;
      else if (!strcmp("-all", s) || !strncmp("-unlimited", s, slen))
        { 
	  if (server.netmask)
	    free((char *)server.netmask);
	  server.netmask = NULL;
	  if (server.netmask_write)
	    free((char *)server.netmask_write);
	  server.netmask_write = NULL;
	}
      else if (!strncmp("-append", s, slen) || !strncmp("-add", s, slen))
	append++;
      else if (!strncmp("-read_append",s,slen) || !strncmp("-read_add",s,slen))
 	append++, read_only++;
      else if (!strncmp("-enforce", s, slen))
	enforce++;
      else if (!strncmp("-local", s, slen))
        {
	  dstring *errp = NULL;

 	  dstring_append(&server.netmask,       0, NULL, 0);
 	  dstring_append(&server.netmask_write, 0, NULL, 0);

	  NetmaskDefault(&server.netmask,     0xffff, Hostname(), &errp);
	  NetmaskDefault(&server.netmask_write, 0xff, Hostname(), &errp);
 	  if (dstring_length(errp))
	    return jittr_error("%s: NetmaskDefault(..., %s)\n", 
	    	jarg_name(), Hostname(), 0, 0);
	}
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }

  if (s && !append) 
    {
      dstring_append(&server.netmask,       0, NULL, 0);
      dstring_append(&server.netmask_write, 0, NULL, 0);
    }

  while (s)
    {
      if (dstring_length(server.netmask))
        dstring_append(&server.netmask, -1, " ", 1);
      dstring_append(&server.netmask, -1, s, 0);

      if (!read_only)
	{
          if (dstring_length(server.netmask_write))
	    dstring_append(&server.netmask_write, -1, " ", 1);
	  dstring_append(&server.netmask_write, -1, s, 0);
	}
      s = jarg_next_word(&slen);
    }
  
  if (enforce)
    {
      struct client *cl;
      char *r1 = "# new access restrictions apply, 'readonly' (netmask enforcement)\r\n";
      char *r0 = "# new access restrictions apply, 'read-write' (netmask enforcement)\r\n";

      s = "# new access restrictions apply, sorry (netmask enforcement)\r\n";
      for (cl = clients; cl; cl = cl->next)
        {
	  debug1("netmask: enforce check on client %s.\n", cl->name->buf);
	  if (cl->o && IS_CTYPE_TCP(cl->o->type) && (cl->o->fd >= 0))
	    {
	      if (CheckNetmask(server.netmask,
	      		       (struct sockaddr_in *)&cl->o->sa))
		{
		  write(cl->o->fd, s, strlen(s));
		  ConnShutdown(cl->o);
		}
	      else
	        {
		  read_only = cl->o->read_only;
		  cl->o->read_only = CheckNetmask(server.netmask_write, 
				    (struct sockaddr_in *)&cl->o->sa);
		  if (read_only < cl->o->read_only)
		    dstring_append(&cl->o->obuf, -1, r1, 0);
		  if (read_only > cl->o->read_only)
		    dstring_append(&cl->o->obuf, -1, r0, 0);
		}
	    }
	  if (cl->i && (cl->i != cl->o) && 
	      IS_CTYPE_TCP(cl->i->type) && (cl->i->fd >= 0))
	    {
	      if (CheckNetmask(server.netmask, 
	      		       (struct sockaddr_in *)&cl->i->sa))
		{
		  write(cl->i->fd, s, strlen(s));
		  ConnShutdown(cl->i);
		}
	      else
	        {
		  if (server.crypted_pw && *server.crypted_pw)
		    cl->i->read_only = 1;
		  else
		    cl->i->read_only = CheckNetmask(server.netmask_write, 
		      (struct sockaddr_in *)&cl->i->sa);

		  if (read_only < cl->o->read_only)
		    dstring_append(&cl->o->obuf, -1, r1, 0);
		  if (read_only > cl->o->read_only)
		    dstring_append(&cl->o->obuf, -1, r0, 0);
		}
	    }
	}
    }
  return 1;
}

/*
 * prompt [-c client] [-i client_id] on|plain|off [text [text_bad [text_more]]]
 */
static int
prompt_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int endopt = 0, slen;
  char *s = jarg_first_word(&slen);
  struct client *cl = exec_context->who;
  struct prompt *p;

  while (slen)
    {
      if (endopt || slen < 2 || *s != '-')  /* s == NULL implies slen == 0 */
        break;
      if (!strncmp("-client_name", s, slen))
        {
	  if (!(s = jarg_next_word(&slen)))
	    return jittr_error("%s: -c needs a parameter\n", jarg_name(),0,0,0);
	  if (!(cl = *ClientByName(NULL, s)))
	    return jittr_error("%s: no client named '%s'.\n", 
	    	jarg_name(), s, 0, 0);
	}
      else if (!strncmp("-id_of_client", s, slen))
        {
	  char *e;
	  int id;

	  if (!(s = jarg_next_word(&slen)))
	    return jittr_error("%s: -i needs a parameter\n", jarg_name(),0,0,0);
	  id = strtol(s, &e, 0);
	  if ((e < s+slen) || !(cl = *ClientById(NULL, id)))
	    return jittr_error("%s: no client with ID '%s'.\n", 
	    	jarg_name(), s, 0, 0);
	}
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }
  
  p = &cl->prompt;

  if (!s)
    {
      dstring_append(rp, -1, "# ", 2);
      switch (p->type)
        {
	  case PT_DISABLED: dstring_append(rp, -1, "PT_DISABLED", 11); break;
	  case PT_COMMENT:  dstring_append(rp, -1, "PT_COMMENT",  10); break;
	  case PT_PLAIN:    dstring_append(rp, -1, "PT_PLAIN",     8); break;
	  default:
	    ASSERT(0);		/* a new prompt type? */
	}
      dstring_append(rp, -1, " \"", 2); 
      if (p->prompt_ok)
        dstring_appendq(rp, -1, p->prompt_ok->buf, p->prompt_ok->length);
      dstring_append(rp, -1, "\" \"", 3);
      if (p->prompt_bad)
        dstring_appendq(rp, -1, p->prompt_bad->buf, p->prompt_bad->length);
      dstring_append(rp, -1, "\" \"", 3);
      if (p->prompt_more)
        dstring_appendq(rp, -1, p->prompt_more->buf, p->prompt_more->length);
      dstring_append(rp, -1, "\"\n", 2); 
      return 1;
    }

       if (!strncmp("on",    s, slen)) p->type = PT_COMMENT;
  else if (!strncmp("plain", s, slen)) p->type = PT_PLAIN;
  else if (!strncmp("off",   s, slen)) p->type = PT_DISABLED;
  else return jittr_error("%s: on|plain|off expected instead of %s.\n", 
  	jarg_name(), s, 0, 0);
  
  if ((s = jarg_next_word(&slen))) dstring_append(&p->prompt_ok, 0, s, slen);
  if ((s = jarg_next_word(&slen))) dstring_append(&p->prompt_bad, 0, s, slen);
  if ((s = jarg_next_word(&slen))) dstring_append(&p->prompt_more, 0, s, slen);
  return 1;
}

/*
 * id [-numeric] [-set clientname]|[prefix [suffix]]
 * id -query [prefix [suffix]]
 * set or get name of asking client. The second form requests the name of the 
 * asked daemon.
 *
 * TODO: When setting a new name of the client, we must find out if this client
 * is already known. If so, and if this client is connected, we have a problem.
 * otherwise we must unify ourselves with this client without leaving dangling
 * pointers, both is not easy.
 */
static int
id_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  struct client *cl = exec_context->who;
  int slen, n = 0;
  char *s = jarg_first_word(&slen);
  
  if (s && !strncmp("-query", s, slen))
    {
      s = jarg_next_word(&slen);
      dstring_append(rp, -1, s ? s : "id -s ", s ? slen : 6);
      dstring_append(rp, -1, server.name->buf, server.name->length);
      s = jarg_next_word(&slen);
      dstring_append(rp, -1, s ? s : "\n", s ? slen : 1);
      return 1;
    }
  if (s && !strncmp("-set", s, slen))
    {
      struct client **c;

      if (!(s = jarg_next_word(&slen)))
        return jittr_error("%s: -set: parameter missing\n", jarg_name(),0,0,0);

      if (*(c = ClientByName(NULL, s)) && (*c != cl))
        {
	  /* 
	   * client named s already exists. This is complicated.
	   * If this client is connected, we must reject the change.
	   *
	   * If he is disconnected, we flag both his connections invalid, 
	   * truncate his name and take over.
	   * He is food for the garbage collector then. Whoops, what garbage
	   * collector?
	   */
	  if (TurnClient(c, 0))
	    return jittr_error("%s: Connections to a different client '%s' exist: Id=%d. %d\n", jarg_name(), s, (*c)->id, "What shall we do with him?");

	  /* The client is dead, long live the client! */
	}
      if (slen == server.name->length && !strncmp(s, server.name->buf, slen))
        return jittr_error("%s: Cannot name client identical to myself: %s.\n", 
		jarg_name(), s, 0, 0);
      dstring_append(&cl->name, 0, s, slen);
      return 1;
    }
  if (s && !strncmp("-numeric", s, slen) && ++n)
      s = jarg_next_word(&slen);
  dstring_append(rp, -1, s, slen);
  if (n)
    dstring_appendf(rp, -1, "%d", cl->id, 0, 0, 0);
  else
    dstring_append(rp, -1, cl->name->buf, cl->name->length);
  s = jarg_next_word(&slen);
  atom_append(REPLY_ATOM, -1, s ? s : "\n", slen);
  return 1;
}

static char *help[] =
{
  "",   "",
  "id [-numeric]|[-set clientname]|[prefix [suffix]]",
	"return id, return or set name of asking client",
  "id -query [prefix [suffix]]",
	"return name of the daemon itself",
  "prompt [-c client|-i id]",
  	"print the type and the three prompt strings",
  "prompt [-c client|-i id] on|plain|off [text [textbad [textmore]]]",
  	"set the type and/or text how we prompt to this (or the specified) client",
  "netmask [-all|-local] [-append] [netmasks/hostnames ...]",
  	"set or get the netmask used with all internet services. Default is no netmask, free access",
  "netmask -read_add [netmasks/hostnames ...]",
  	"extend the netmask used with internet services to allow clients in which may only observe",
  "netmask -enforce",
  	"close all connections that would violate the current netmask",
  "date [+[-]]tstamp [prefix [suffix]]",
  	"convert between relative and absolute timestamps in UTC. Tstamp defaults to \"+0\"",
  "date -sync +[-]time_offset",
  	"record an adjustment for absolute timestamps on commands from this client. The adjustment for MET DST timezone is '+-2h'",
  "date -echo [prefix [suffix]]",
  	"echo the recorded client time offset with inverted sign",
  NULL
};

int
jittr_session_init()
{
  atom_command(   HELP_CMD_ATOM, NULL,    atom_help_cmd, (void *)help);
  atom_command(     ID_CMD_ATOM, "id",           id_cmd, (void *)0);
  atom_command( PROMPT_CMD_ATOM, "prompt",   prompt_cmd, (void *)0);
  atom_command(NETMASK_CMD_ATOM, "netmask", netmask_cmd, (void *)0);
  atom_command(   DATE_CMD_ATOM, "date",       date_cmd, (void *)0);
  return 0;
}
