/*
 * user.c -- jittr standard commands library (User frinedly methods)
 *
 * Everything that is not really needed by the system but helpful
 * to the curious hacker. Split from jcmd.c due to filesize.
 *
 * 07.07.96 jw.
 */
#include <sys/types.h> 		/* for sys/socket.h */
#include <sys/socket.h>		/* struct sockaddr for net.h */
#include <sys/time.h>  		/* struct timeval for net.h */
#include <fcntl.h>		/* O_RDONLY */
#include <unistd.h>		/* setuid() */
#include <errno.h>		/* strerror(), errno */

#include "jittr/atom.h"		/* atoms */
#include "jittr/parse.h"	/* jarg_*() methods */
#include "jittr/jittr.h"	/* struct cwp */
#include "jittr/net.h"		/* struct client */
#include "jittr/doio.h"		/* jittr_make_client() */
#include "jittr/schedule.h"	/* struct sched_action for top_cmd */
#include "jittr/action.h"	/* struct jittr_action for top_cmd */
#include "jittr/version.h"	/* symbol VERSION for nicer help */

#ifndef LINUX
extern char *crypt __P((char *key, char *salt));
#endif

static int              cd_cmd __P((int idx, int flag, void *is_pwd));
static int             top_cmd __P((int idx, int flag, void *dummy));
static int             who_cmd __P((int idx, int flag, void *dummy));
static int             dir_cmd __P((int idx, int flag, void *is_ls));
static int            quit_cmd __P((int idx, int flag, void *dummy));
static int           debug_cmd __P((int idx, int flag, void *dummy));
static int          setuid_cmd __P((int idx, int flag, void *dummy));
static int          source_cmd __P((int idx, int flag, void *dummy));
static int           comma_cmd __P((int idx, int flag, void *dummy));
static int            pass_cmd __P((int idx, int flag, void *dummy));

struct pass_data
{
  int (*fn) __P((void *cl, char *buf, int len));
  void *data;
  dstring *prompt;
};

static int
pass_input(cli, bufp, len)
void *cli;
char *bufp;
int len;
{
  struct client *cl = (struct client *)cli;
  struct pass_data *prev = (struct pass_data *)cl->input_notify_data;
  struct conn *conn;
  struct dstring *buf;
  int bytes = 0, intr = 0;

  buf = *jittr_ptr2ds(bufp, DSHINT_CLIENT, cli);
  if (buf != (conn = cl->i)->ibuf) conn = cl->o;

  /* ignore leading linefeeds */
  while (len && ((*bufp == '\n') || (*bufp == '\r')))
    {
      bufp++;
      len--;
    }
  
  /* ignore trailing linefeeds */
  while (len && ((bufp[len-1] == '\n') || (bufp[len-1] == '\r')))
    len--;
  
  if (!len)
    intr = 1;

  if (!intr && ((bytes = jittr_purge_telnet(bufp, len)) < 0))
    {
      /* IAC WONT TIMING_MARK */
      dstring_append(&conn->obuf, -1, "\377\374\006", 3);
      intr = 1;
    }

  if (!intr && !bytes)
    return buf->length = 0;

  bufp[bytes] = '\0';

  if (!intr && server.crypted_pw && *server.crypted_pw &&
      strcmp(crypt(bufp, server.crypted_pw), server.crypted_pw))
    {
      debug1("pass_input: '%s': bad password, try again\n", bufp);
      cl->i->read_only = 1;
      return buf->length = 0;
    }
   
  debug1("pass_input: password %s, bye!\n", intr ? "interrupt" : "matches");
  if (intr != 1)
    cl->i->read_only = CheckNetmask(server.netmask_write,
      (struct sockaddr_in *)&cl->i->sa);
  cl->input_notify_fn = prev->fn;
  cl->input_notify_data = prev->data;
  free((char *)(cl->prompt.prompt_ok));
  cl->prompt.prompt_ok = prev->prompt;
  free((char *)prev);
  if (cl->i->type == CTYPE_TELNET || cl->i->type == CTYPE_TCP_OR_TELNET)
    {
      /* telnet: IAC WONT ECHO */
      dstring_append(&cl->o->obuf, -1, "\xff\xfc\x01", 3);	
    }
  else if (cl->i->type == CTYPE_FD && isatty(cl->i->fd))
    {
      debug("pass_cmd: tty_LocalEcho on not impl.\n");
    }
  
  dstring_appendn(&cl->o->obuf, -1, "# readonly=%d\n", cl->i->read_only);
  return buf->length = 0;
}

/* 
 * pass [plain_text_pwd]
 *
 * Attempt to promote from readonly to read-write mode.
 */
static int
pass_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  struct client *cl = exec_context->who;
  int slen;
  char *s = jarg_first_word(&slen);
  char *pw = atom_value(idx);
  struct pass_data *pass_data;

  if (flag & TRACE_WRITE)
    {
      server.crypted_pw = pw;
      return 0;
    }

  if (!pw)
    return 0;

  if (s)
    {
      if (!strcmp(crypt(s, pw), pw))
	{
	  cl->i->read_only = CheckNetmask(server.netmask_write,
	    (struct sockaddr_in *)&cl->i->sa);
	  debug2("pass: password matched. Client %s read_only=%d\n", 
	    cl->name->buf, cl->i->read_only);
	}
      else
        {
          cl->o->read_only = 1;
	  debug1("pass: password bad. Client %s read_only=1\n", cl->name->buf);
	}
      return 0;
    }

  if (!(pass_data = (struct pass_data *)calloc(sizeof(struct pass_data), 1)))
    {
      debug("pass: out of memory\n");
      return 0;
    }
  
  pass_data->fn     = cl->input_notify_fn;
  pass_data->data   = cl->input_notify_data;
  pass_data->prompt = cl->prompt.prompt_ok;

  cl->prompt.prompt_ok = NULL;
  dstring_append(&cl->prompt.prompt_ok, 0, server.name->buf, 0);
  dstring_append(&cl->prompt.prompt_ok, -1, ": enter password: \r\n", 0);
  cl->input_notify_fn = pass_input;
  cl->input_notify_data = pass_data;
  if (cl->i->type == CTYPE_TELNET || cl->i->type == CTYPE_TCP_OR_TELNET)
    {
      /* telnet: IAC WILL ECHO */
      dstring_append(rp, -1, "\xff\xfb\x01", 3);	
    }
  else
    {
      /*
       * A prompt is provoked by the telnet's IAC DO ECHO response .
       * For all other connection types, we issue a prompt now.
       */
      dstring_append(rp, -1, "#", 1);
      dstring_append(rp, -1, cl->prompt.prompt_ok->buf, 0);
      dstring_append(rp, -1, "\n", 1);
    }
  
  if (cl->i->type == CTYPE_FD && isatty(cl->i->fd))
    {
      debug("pass_cmd: tty_LocalEcho off not impl.\n");
    }

  return 0;
}


/* 
 * , text ...
 *
 * The comma command appends text to stderr.
 */
static int
comma_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[STDERR_ATOM]->value;
  struct client *cl = exec_context->who;
  int slen;
  char *s = jarg_first_word(&slen);

  dstring_append(rp, -1, cl->name->buf, cl->name->length);
  dstring_append(rp, -1, ":", 1);

  while (s)
    {
      dstring_append(rp, -1, " ", 1);
      dstring_append(rp, -1, s, slen);
      s = jarg_next_word(&slen);
    }
  atom_append(STDERR_ATOM, -1, "", 0);

  return 0;
}

/* 
 * source [-v] filename 
 */
static int
source_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  int slen = 0;
  char *s = jarg_first_word(&slen);
  struct client *cl;
  int fd = -1, fileflag = 1, verbose = 0;

  if (slen > 1 && !strncmp("-verbose", s, slen))
    {
      verbose++;
      s = jarg_next_word(&slen);
    }

  if (!s)
    return jittr_error("%s: filename missing.\n", jarg_name(), 0, 0, 0);
  if (!strcmp("-", s))
    fd = fileflag = 0;
  else if ((fd = open(s, O_RDONLY, 0)) < 0)
    return jittr_error("%s: cannot open file %s: (errno %d) %s.\n", 
    	jarg_name(), s, errno, strerror(errno));
  if (!(cl = jittr_make_client(fileflag ? "file" : NULL, fd)))
    return jittr_error("%s %s failed.\n", jarg_name(), s, 0, 0);
  cl->i->type = CTYPE_FD;
  cl->prompt.type = PT_DISABLED;
  if (verbose)
    cl->o = NULL;	/* causes messages on stderr */
  else
    {
      cl->o = &cl->_o;
      cl->o->fd = -1;	/* causes the output to be discarded */
      cl->o->type = CTYPE_WFD;
      cl->o->ibuf = NULL;
      cl->o->obuf = NULL;
    }
  if (fileflag)
    {
      dstring_append(&cl->name, -1, "://", 3);
      dstring_append(&cl->name, -1, Hostname(), 0);
      dstring_append(&cl->name, -1, "/", 1);
      dstring_append(&cl->name, -1, s, 0);
    }

  return 0;
}


/*
 * debug on|off|filename
 */
static int
debug_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
#ifdef DEBUG
  char *s = jarg_first_word(NULL);

  dstring **rp = &atoms[REPLY_ATOM]->value;
  if (!s)
    {
      if (!debugfp)
        {
	  dstring_append(rp, -1, "# debugging (to ", 0);
	  dstring_append(rp, -1, debugfilename, 0);
	  dstring_append(rp, -1, ") is currently disabled.\n", 0);
	}
      else
        {
	  dstring_append(rp, -1, "# writing debug output to ", 0);
	  dstring_append(rp, -1, debugfilename, 0);
	  dstring_append(rp, -1, "\n", 1);
	}
      return 0;
    }

  if (!strcmp("off", s))
    {
      if (!debugfp) 
        return jittr_error("%s: debug is already off.\n", jarg_name(), 0, 0, 0);
      debug1("closing debug file %s. =============================\n",
        debugfilename);
      if (debugfp != stderr)
        fclose(debugfp);
      debugfp = NULL;
    }
  else
    {
      if (!strcmp("on", s))
        s = NULL;
      if (!debugfp)
        fclose(debugfp);
      OPENDEBUG(s ? s : debugfilename);
      debug1("opening debug file %s. =============================\n",
        debugfilename);
    }
  return 0;
#else
  return jittr_error("%s: libjittr has been compiled without debug support.\n",
  	jarg_name(), 0, 0, 0);
#endif
}

/*
 * setuid [uid|username [gid|groupname]]
 */
static int
setuid_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  char *u = jarg_first_word(NULL);

  return jittr_setuid(u, jarg_next_word(NULL));
}

/* 
 * quit
 * Another convenience command. It is equivalent to
 *
 * 	set NewAtom -nop
 *
 * This works because jittr_loop checks the permanance of atom 0.
 * Two quit commands after another are equivalent to
 *
 *	set 0 -nop; del 0
 *
 * From network connections the parameter word 'server' must be provided.
 */
static int
quit_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  struct client *cl = exec_context->who;
  char *s = jarg_first_word(NULL);

  /* shutdown this client only ? */
  if (cl && cl->i && 
      (cl->i->type == CTYPE_UDP || IS_CTYPE_TCP(cl->i->type)) &&
      (!s || strcasecmp("server", s)))
    return TurnClient(ClientByFd(NULL, cl->i->fd), 1);

  if (atoms[NEW_ATOM]->a.flag & A_PERMANENT)
    {
      dstring **rp = &atoms[REPLY_ATOM]->value;

      dstring_append(rp, -1, "# Process may continue until scheduled commands are done.\n", 0);
      dstring_append(rp, -1, "# Reenter your quit command to force immediate shutdown.\n", 0);
      atoms[NEW_ATOM]->a.flag &= ~A_PERMANENT;
    }
  else
    atom_delete(NEW_ATOM);	/* another quit command? Hmmm. O.k. Exit now! */
  return 1;
}

/*
 * cd [path]
 * pwd [prefix [suffix]]
 *
 * set or get clients working point.
 */
static int
cd_cmd(idx, flag, is_pwd)
int idx, flag;
void *is_pwd;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  struct client *cl = exec_context->who;
  int slen;
  char *s = jarg_first_word(&slen);

  if (is_pwd)
    {
      /* should pwd detect a cwp change since scheduled? */
      dstring_append(rp, -1, s, slen);
      dstring_appendf(rp, -1, "%d", cl->cwp.atom, 0,0,0);
      s = jarg_next_word(&slen);
      atom_append(REPLY_ATOM, -1, s ? s : "\n", slen);
      return 1;
    }

  /* we are cd */
  if ((idx = atom_lookup(s ? s : "0")) < 0)
    {
      char buf[1024];

      /*
       * This hack makes the link component ".." work with the cd command.
       * It won't work with other commands, unless explicitly implemented.
       * This hack is copied to dir_cmd(). Keep in sync.
       */
      if ((jittr_mkpath(jittr_getcwd(), s, buf) < 0) ||
          ((idx = atom_lookup(buf)) < 0))
	return jittr_error("cd: atom %s (or %s) not found.\n", s, buf, 0, 0);
    }
  if (cl->cwp.atom != exec_context->where.atom)
    jittr_error("cd %d: client %s already changed cwp: %d (expected %d). ",
	    idx, cl->name->buf, cl->cwp.atom, exec_context->where.atom);
  cl->cwp.atom = idx;
  return 1;
}

/*
 * top [-all] [-abs]		list scheduler queue
 * top -kill id [id ...]	shrink scheduler queue
 */
static int
top_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen, endopt = 0;
  int id, toplimit = 12, fixed = 0, suflen;
  char *suffix;
  char *s = jarg_first_word(&slen);
  struct sched_timing *t;
  struct sched_action *a;
  struct timeval now;
  char buf[80];
  long n;
  
  while (slen)
    {
      if (endopt || slen < 2 || *s != '-')  /* s == NULL implies slen == 0 */
        break;
      if (!strncmp("-all", s, slen))
	toplimit = 999;			/* almost unlimited */
      else if (!strncmp("-fixed", s, slen) || !strncmp("-absolute", s, slen))
	fixed++;
      else if (!strncmp("-delete", s, slen) || !strncmp("-kill", s, slen))
	{
	  struct sched_event *e;

	  while ((s = jarg_next_word(&slen)))
	    {
	      if (!(e = sched_unlink_event(atoi(s))))
		return jittr_error("%s: kill failed: event id %s not found.\n",
			    jarg_name(), s, 0, 0);
	      sched_free_action((struct sched_action *)e);
	    }
	  return 1;
	}
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }

  suffix = jarg_next_word(&suflen);

  jittr_gettimeval(&now);
  id = sched_peek_event(NULL, NULL);
  id = sched_peek_event(&t, &a);
  while (id > 0 && toplimit-- > 0)
    {
      n = t->max_rep;
      if (t->max_rep)
        t->max_rep -= t->count;	/* because sched_format_timing looks there */
      sched_format_timing(buf, 80, t, fixed ? NULL : &now);
      t->max_rep = n;

      dstring_append(rp, -1, s ? s : "#", s ? slen : 1);

      /* XXX: sprintf has a purify problem here: It reads one byte too far */
      dstring_appendf(rp, -1, "%3d @%-*s ", id, fixed ? 36 : 24, buf, 0, 0);

      if (a->fn == jittr_process)
        {
	  struct jittr_action *ja = (struct jittr_action *)a->argv;
	  dstring_append(rp, -1, "{", 1);
	  dstring_append(rp, -1, atom_name(ja->atom), 0);
	  if (ja->args)
	    {
	      n = ja->args->length - ((char *)ja->args->data-ja->args->buf) - 1;
	      dstring_append(rp, -1, " ", 1);
	      if (parse_needsq((char *)ja->args->data, n) & ~1)
	        dstring_appendq(rp, -1, (char *)ja->args->data, n);
	      else
	        dstring_append(rp, -1, (char *)ja->args->data, n);
	    }
	  dstring_appendf(rp, -1, "} %c=%s",
		(ja->type == RT_CLIENT) ? 'c' : 'a',
		(ja->type == RT_CLIENT) ? ja->reply.client->name->buf : 
					  atom_name(ja->reply.atom), 0, 0);
	}
      else
	dstring_appendf(rp, -1, "[%08x %08x %08x ...]", 
		(long)a->fn, (long)a->argv[0], (long)a->argv[1], 0);
      atom_append(REPLY_ATOM, -1, suffix ? suffix : "\n", suffix ? suflen : 1);
      id = sched_peek_event(&t, &a);
    }

  return 1;
}  

/*
 * Every atom is a directory of links. This command lists them.
 * dir [prefix [suffix]]
 * ls [-1] [-long] [path [prefix [suffix]]]
 *
 * 'dir' is equivalent to 'ls -long' and '-long' implies '-1'.
 * '-1' forces one entry per line, otherwise ls lists all names in one line.
 * '-l' first column shows name-of-the-link, second column shows 
 *      contents-of-link (where it points to), and (if it has already been
 *      resolved), a third column shows the target atom index number.
 * '-L' resolves links or unresolves invalid links.
 * '-u' print directory unsorted (default is sort by name).
 * '-i' print directory sorted by atom indices.
 * Path defaults to ".", an must be specified, when prefix / suffix are used.
 * When path is specified, prefix defaults to <path>/.
 *
 * The entry "." is always present and refers to the current atom.
 */
static int
dir_cmd(idx, flag, is_ls)
int idx, flag;
void *is_ls;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int slen, endopt = 0, suflen, prelen;
  int needslash = 0, n, l, single = 0, lopt = is_ls ? 0 : 1;
  int flags = JDIR_SORT_LINK;
  char *suffix = NULL, *prefix = NULL;
  struct dstring *d, **dp;
  void *ptr;
  char *name, *s = jarg_first_word(&slen);
  static char pad[] = "                          ";
#define PAD(n) (((n) >= sizeof(pad)) ? "" : (pad + (n)))

  while (slen)
    {
      if (endopt || slen < 2 || *s != '-')  /* s == NULL implies slen == 0 */
        break;
      if (!strncmp("-long", s, slen))
	lopt = 1;
      else if (!strcmp("-1", s))
	single = 1;
      else if (!strncmp("-Link", s, slen))
	flags |= JDIR_AUTO;
      else if (!strcmp("-u", s))
	flags &= ~JDIR_SORT_MASK;
      else if (!strcmp("-i", s))
	{
	  flags &= ~JDIR_SORT_MASK;
	  flags |= JDIR_SORT_IDX;
	}
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }
  if (lopt)
    single = 1;

  if ((idx = jittr_getcwd()) < 0)
    return jittr_error("%s: no exec_context?\n", jarg_name(), 0, 0, 0);

  if (slen && ((idx = atom_lookup(s)) < 0))
    {
      char buf[1024];
      /* This hack is copied from cd_cmd(). Keep in sync. */
      if ((jittr_mkpath(jittr_getcwd(), s, buf) < 0) ||
          ((idx = atom_lookup(buf)) < 0))
	return jittr_error("%s: atom %s (or %s) not found.\n", 
			   jarg_name(), s, buf, 0);
    }

  if (!(prefix = jarg_next_word(&prelen)))
    {
      prefix = s;
      prelen = slen;
      if (slen && s[slen-1] != '/') needslash++;
    }

  if (!(suffix = jarg_next_word(&suflen)))
    {
      suffix = "\n";
      suflen = 1;
    }

  l = atoms[idx]->name ? atoms[idx]->name->length : 0;
  dstring_append(rp, -1, prefix, prelen);
  if (needslash) dstring_append(rp, -1, "/", 1);
  dstring_appendf(rp, -1, lopt ? ".%s\"%s\"%s idx=%d" : ".",
  	PAD(1 + (sizeof(pad) >> 1)), atom_name(idx), PAD(l), idx, 0, 0);

  ptr = jittr_opendir(idx, JDIR_SAVE|flags);
  while ((dp = jittr_readdir(ptr, &name)))
    {
      dstring_append(rp, -1, single ? suffix : " ", single ? suflen : 1);
      if (single)
        {
	  dstring_append(rp, -1, prefix, prelen);
	  if (needslash) dstring_append(rp, -1, "/", 1);
	}
      dstring_append(rp, -1, name, l = strlen(name));
      if (lopt)
	{
	  l += sizeof(pad) >> 1;
	  if (!(d = *dp))	/* I never saw that one. Let us wait, .... */
	    {
	      dstring_appendf(rp, -1, "%s\"\"%s cyclic", PAD(l), PAD(0), 0, 0);
	      continue;		/* cyclic link frustration trap */
	    }

	  n = -dstring_appendf(rp, -1, "%s\"", PAD(l), 0, 0, 0);
	  n += dstring_appendq(rp, -1, d->buf, d->length);
	  dstring_appendf(rp, -1, "\"%s ", PAD(n), 0, 0, 0);
	  switch ((int)d->data)
	    {
	      case JLINK_UNRESOLVED:
		dstring_append(rp, -1, "atom", 4);
		break;
	      case JLINK_UNRESOLVABLE:	/* non-atom link */
		dstring_append(rp, -1, "value", 5);
		break;
	      default:	/* JLINK_IS_RESOLVED(d->data):  atom index */
		dstring_appendf(rp, -1, "idx=%d", (int)d->data, 0, 0, 0);
	    }
	}
    }
  jittr_closedir(ptr);
#undef PAD

  atom_append(REPLY_ATOM, -1, suffix, suflen);
  return 1;
}

static int 
who_access_fn(s, cl, buf, len)
struct server *s;
struct client *cl;
char *buf;
int len;
{
  static dstring *d = NULL;

  int idx = (int)(s->access_data);
  struct timeval tv;

#if 0
  debug3("who_access_fn: %d bytes from (%d) %s\n", len, cl->id, cl->name->buf);
#endif

  jittr_gettimeval(&tv);
  dstring_appendn(&d, 0, "%d ", cl->id);
  dstring_append(&d, -1, cl->name->buf, cl->name->length);

  /* must write as NULL client, so that readonly clients can write */
  jittr_atom_set(idx, 0, d->buf, d->length, &tv, NULL, 0);

  return 0;
}


/*
 * who [-all] [-long]		list all clients
 */
static int
who_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  dstring **rp = &atoms[REPLY_ATOM]->value;
  int all = 0, verbose = 0, suflen;
  char *suffix;
  struct client *cl;
  int slen, endopt = 0;
  char *s = jarg_first_word(&slen);
  
  while (slen)
    {
      if (endopt || slen < 2 || *s != '-')  /* s == NULL implies slen == 0 */
        break;
      if (!strncmp("-long", s, slen))
	verbose++;
      else if (!strncmp("-all", s, slen))
	all++;
      else
	return jittr_error("%s: unknown option %s\n", jarg_name(), s, 0, 0);
      s = jarg_next_word(&slen);
    }

  
  suffix = jarg_next_word(&suflen);
  for (cl = clients; cl; cl = cl->next)
    {
      if (!all && 
          (!cl->o || (cl->o->fd < 0)) && 
	  (!cl->i || (cl->i->fd < 0)))
        continue;	/* XXX: how about udp clients? */

      dstring_append(rp, -1, s, slen);
      if (verbose)
        {
	  dstring_appendf(rp, -1, "%3d %-20s ", cl->id, cl->name->buf, 0, 0);
	  if (cl->cwp.atom)
	    dstring_appendf(rp, -1, "cwp=%-2d ", cl->cwp.atom, 0, 0, 0);
          dstring_appendf(rp, -1, "in={%s %-2d ",
	    con_type_name[cl->i ? cl->i->type : 0],
	    cl->i ? cl->i->fd : -1, 0, 0);
	  conn2dstr(rp, -1, cl->i);
	  if (cl->o == cl->i)
	    dstring_appendf(rp, -1, "} out=in off=", 13);
	  else
	    {
	      dstring_appendf(rp, -1, "} out={%s %d ",
		con_type_name[cl->o ? cl->o->type : 0],
		cl->o ? cl->o->fd : -1, 0, 0);
	      conn2dstr(rp, -1, cl->o);
	      dstring_append(rp, -1, "} off=", 6);
	    }
	  if (!cl->off.sign)
	    dstring_append(rp, -5, NULL, 0);
	  else
	    {
	      char t[50];
	      struct timeval now;

	      now.tv_sec =  now.tv_usec = 0;

	      sched_format_tv(t, 49, &cl->off.tv, &now);
	      dstring_append(rp, -1, "+-", (cl->off.sign > 0) ? 1 : 2);
	      dstring_append(rp, -1, t+1, 0);
	    }
	}
      else
        dstring_append(rp, -1, cl->name->buf, cl->name->length);
      atom_append(REPLY_ATOM, -1, suffix ? suffix : "\n", suffix ? suflen : 1);
    }
  return 1;
}

/* 
 * untrace idx
 *
 * removes all trace callbacks from atom idx, referring to the current client.
 * this command is an alias to
 * 	trace idx -delete -client . *
 */
static int
untrace_cmd(idx, flag, dummy)
int idx, flag;
void *dummy;
{
  struct exec_context *context;
  struct trace *tr;
  struct client *cl_opt = NULL;
  int slen;
  struct trace *ntp, **tp;
  char *s;
  
  if (!(s = jarg_first_word(&slen)))
    return jittr_error("%s: atom name missing.", jarg_name(), s, 0, 0);

  if ((idx = atom_lookup(s)) < 0)
    return jittr_error("%s: atom '%s' not found.", jarg_name(), s, 0, 0);
  
  if (!exec_context)
    return jittr_error("%s: need an exec_context.\n", jarg_name(), 0, 0, 0);

  cl_opt = exec_context->who;
  tp = &atoms[idx]->trace;

  while ((tr = *tp))
    {
      context = (struct exec_context *)tr->data;
      ntp = tr->next;

      if (cl_opt != context->who)
	{
	  tp = &tr->next;
	  continue;
	}
      atom_trace_delete(tp);
      *tp = ntp;
    }
  return 1;
}


static char *help[] =
{
  "Helpful commands for the human user are:",   "",
  "cd [path]",
  	"change clients working point. Default '0'",
  "pwd [prefix [suffix]]",
	"return clients working point",
  "who [-long] [prefix [suffix]]",
	"list all clients in short or long form",
  "get who",
        "returns index and name of the client who was at last active",
  "top [-all] [-absolute] [prefix [suffix]]",
  	"list scheduler queue. Default: stop after 12 entries and print relative timestamps",
  "top -kill id",
  	"delete scheduled event No. id",
  "debug on|off|filename",
  	"disable or change debugging behaviour",
  "source [-v] filename",
  	"open a unix file and interpret it as jittr commands",
  "setuid [unix-uid|username [unig-gid|groupname]]",
  	"change the user id and group id of this process. Default parameters are 'nobody nobody'. Be sure to run this command if the daemon process is started as root from some system rc script",
  "dir [-u|-i] [prefix [suffix]]",
	"list links from the current working point atom. -u lists them unsorted, -i sorts by atom index",
  "ls [-u|-i|-l|-1|-L] [prefix [suffix]]",
	"similar to dir, options like unix ls(1)",
  "untrace idx", "alias to 'trace idx -client . -delete *'",
  "quit [server]",
  	"close all(!) connections and exit(!) the daemon process as soon as the scheduler is drained. If this command is sent through a TCP or UDP network connection, the keyword 'server' must be provided. Quit (without a parameter) can be used to disconnect a telnet connection",
  ", message_text ...",	"write client name and message text to stderr. All clients that have said 'trace stderr' can talk or chat like in an irc channel. See also 'id -set'",
  "set pass SQVikaFU3Ove6", "",
  "pass [clear_password]", "check password and enable write access, if password and netmasks match. Prompts for password, if called without parameter",
  "get jittr_version", "print the version number and date",
  "",	VERSION,
  NULL
};

int
jittr_user_init()
{
  atom_command(HELP_CMD_ATOM, NULL,    atom_help_cmd, (void *)help);
  atom_command(           -1, "cd",           cd_cmd, (void *)0);
  atom_command(           -1, "pwd",          cd_cmd, (void *)1);
  atom_command(           -1, "who",         who_cmd, (void *)0);
  atom_command(           -1, "top",         top_cmd, (void *)0);
  atom_command(           -1, "dir",         dir_cmd, (void *)0);
  atom_command(           -1, "ls",          dir_cmd, (void *)1);
  atom_command(           -1, "debug",     debug_cmd, (void *)0);
  atom_command(           -1, "source",   source_cmd, (void *)0);
  atom_command(           -1, "setuid",   setuid_cmd, (void *)0);
  atom_command(           -1, "pass",       pass_cmd, (void *)0);
  atom_command(           -1, "quit",       quit_cmd, (void *)0);
  atom_command(           -1, "untrace", untrace_cmd, (void *)0);
  atom_command(           -1, ",",         comma_cmd, (void *)0);

  server.access_data = (void *)atom_lookup("who");
  server.access_fn = who_access_fn;
  server.crypted_pw = NULL;
  (*atom_trace_lookup(&atoms[atom_lookup("pass")]->trace, NULL, NULL))->flag |= TRACE_WRITE;

  return 0;
}
