/*
 * openurl_ftp.c -- ftp protocol for openurl
 *
 * 24.1.98, jw
 */

#include <errno.h>
#include <pwd.h>		/* passwd */
#include <sys/utsname.h>	/* utsname */
#include <sys/time.h>		/* timeval */
#include <sys/types.h>		/* fd_set */
#include <sys/socket.h>         /* SOCK_STREAM */
#include <netinet/in.h>		/* struct sockaddr_in */
#include <netdb.h>              /* gethostbyname(name) */

#include "openurl.h"

#define ESTIMATED_PACKET_SIZE 1024

/*
 * portable lookup of own Hostname
 */
static char *
Hostname()
{
  static char h[256];
#if defined(SOLARIS) || (defined(SYSV) && !defined(ISC))
  struct utsname utsnam;
 
  uname(&utsnam);
  strncpy(h, utsnam.nodename, 255);
#else
  gethostname(h, 255);
#endif
  return h;
}

static int fetch_line(dstring **dp, int fd, int tout)
{
  int r;
  fd_set rf;
  struct timeval tv;

  dstring_append(dp, 0, NULL, 255);
  (*dp)->length = 0;

  for (;;)
    {
      FD_ZERO(&rf);
      FD_SET(fd, &rf);
      tv.tv_sec = (*dp)->length ? (tout >> 3) : tout;
      tv.tv_usec = 0;

      if (select(FD_SETSIZE, &rf, (fd_set *)0, (fd_set *)0, &tv) < 0)
	return -1;		/* damned error ?? */
      
      if (!FD_ISSET(fd, &rf))
	return -2;		/* timeout */

      if ((r = read(fd, (*dp)->buf+(*dp)->length, 1)) < 0)
	return -3;		/* read error */
      if (!r)
        continue;
      (*dp)->length += r;
      if ((*dp)->buf[(*dp)->length-1] == '\n' || 
          (*dp)->buf[(*dp)->length-1] == '\r')
	{
	  if ((*dp)->length < 2)
	    {
	      (*dp)->length = 0;
	      continue;
	    }
	  else
	    {
              (*dp)->buf[--(*dp)->length] = '\0';
	      break;
	    }
	}
    }
  return atoi((*dp)->buf);
}

int OpenUrlFtp(struct url_info *u)
{
  int fd, id;
  dstring *d = NULL;
  dstring *s = NULL;
  struct passwd *ppp;
  int listen_port, lfd, dfd;
  struct sockaddr_in sa;
  int ia = sizeof(struct sockaddr_in);
  struct hostent *hp;
  unsigned char listen_haddr[4];
  unsigned char hostname[256];

  if (!(hp = gethostbyname(Hostname())))
    return -3;		/* cannot identify own host */
  listen_haddr[0] = (unsigned char)hp->h_addr_list[0][0];
  listen_haddr[1] = (unsigned char)hp->h_addr_list[0][1];
  listen_haddr[2] = (unsigned char)hp->h_addr_list[0][2];
  listen_haddr[3] = (unsigned char)hp->h_addr_list[0][3];
  strcpy(hostname, hp->h_name);

  if (!(ppp = getpwuid(getuid())))
    return -2;		/* cannot identify myself */

  dstring_append(&u->proto, 0, "ftp", 0);
  if ((fd = TcpPortConnect(u->host->buf, u->port, u->timeout, &d)) <= 0)
    {
      if (d) dstring_append(&u->path, 0, d->buf, d->length); 
      if (d) free((char *)d);
      return -1;		/* connect to ftp server failed */
    }
  
  for (;;)
    {
#define IFFETCH(v, cmp) if (((v) = fetch_line(&d, fd, u->timeout)) cmp)
      IFFETCH(id, == 0) continue;
      if (id != 220) break;
      if (d->buf[3] == '-') continue;

      dstring_append(&u->server, 0, d->buf+4, d->length-4);

      write(fd, "USER anonymous\r\n", 16);
      IFFETCH(id, != 331) break;	/* guest o.k. send e-mail addr */

      dstring_append(&s, 0, "PASS ", 5);
      dstring_append(&s, -1, ppp->pw_name, 0);
      dstring_append(&s, -1, "@", 1);
      dstring_append(&s, -1, hostname, 0);
      dstring_append(&s, -1, "\r\n", 2);
      write(fd, s->buf, s->length);
      for (;;)
        {
          IFFETCH(id, != 230) break;	/* guest o.k. restrictions apply */
	  if (d->buf[3] != '-') break;
	}
      if (id != 230) break;

      write(fd, "TYPE I\r\n", 8);
      IFFETCH(id, != 200) break;	/* Type set to I. */

      write(fd, "SIZE ", 5);
      write(fd, u->path->buf, u->path->length);
      write(fd, "\r\n", 2);
      IFFETCH(id, != 213 && id != 500) break;	/* length of file */
      if (id == 213)
        u->len = atoi(d->buf+4);

      write(fd, "MDTM ", 5);
      write(fd, u->path->buf, u->path->length);
      write(fd, "\r\n", 2);
      IFFETCH(id, != 213 && id != 500) break;	/* last modified JJJJMMDDhhmmss UTC */
      if (id == 213)
        dstring_append(&u->modified, 0, d->buf+4, d->length+4);

      if (ListenPort5(SOCK_STREAM, 0, &listen_port, &lfd, &d))
        {
	  id = -1;
	  break;
	}
      dstring_appendn(&s, 0, "PORT %d,", listen_haddr[0]);
      dstring_appendn(&s, -1, "%d,", listen_haddr[1]);
      dstring_appendn(&s, -1, "%d,", listen_haddr[2]);
      dstring_appendn(&s, -1, "%d,", listen_haddr[3]);
      dstring_appendn(&s, -1, "%d,", (listen_port >> 8) & 0xff);
      dstring_appendn(&s, -1, "%d\r\n", (listen_port  ) & 0xff);
      write(fd, s->buf, s->length);
      IFFETCH(id, != 200) break;	/* PORT successful */

      write(fd, "RETR ", 5);
      write(fd, u->path->buf, u->path->length);
      write(fd, "\r\n", 2);
      IFFETCH(id, != 150) break;	/* data connection opened */
#undef IFFETCH

      if ((dfd = TcpPortAccept3(lfd, NULL, &d)) < 0) break;

      /*
       * The ftp protocol is designed to keep the control connection open 
       * during data transfer. The client should await a
       * "226 Transfer complete." message and send a "QUIT" message then.  The
       * Transfer is complete and connections shall be destructed after the
       * server agrees with "221 Goodbye."
       *
       * In this application we return a fildedescriptor of the data connection
       * and have no longer control of the control connection. Our caller
       * cannot manage the control connection, as he does not know about it.
       *
       * All the ftp-servers I tested continue sending data even after the
       * control connection is closed here. If your server behaves differently,
       * you may try to implement an ugly CloseUrl()....
       */
      close(lfd);
      id = 0;
      
      break;	/* don't ever loop */
    }

  if (id) 
    {
      if (d) dstring_append(&u->path, 0, d->buf, d->length); 
      if (d) free((char *)d);
      dfd = (id < 0) ? id : -id;
    }

  close(fd);
  return dfd;	/* negative on error */
}
