/*
 * npPlugin.c
 *
 * Netscape Plug-in based on the experience gathered with plugin.c
 * and based on simple.c by dp Suresh <dp@netscape.com>
 *
 * 26.09.97 jw
 * 26.02.98,jw: found source of sporadic xerror: NPP_SetWindow gets called with
 *              a new windowid.
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#if defined(sun) && !defined(FILE)
# include <sys/select.h>
#endif
#include <stdlib.h>
#include <sys/time.h>	/* struct timeval@irix5 */
#include <strings.h>	/* bcopy() */
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#if defined(__sun) && defined(__GNUC__)
extern char *strdup();
extern int strncasecmp();
extern int isalnum();
extern int kill();
#endif

#include "debug.h"
#include "npapi.h"			/* includes <X11/Xlib.h> */
#ifndef OPENDEBUGTRUNC
# define OPENDEBUGTRUNC(name) do { truncate(name, 0); OPENDEBUG(name); } while (0)
#endif

static const char version[] = "Plugin Loader V1.0, jw 26Feb98";

extern unsigned char binary[];
extern int binary_len;

extern char PLUGIN_NAME[];
extern char PLUGIN_MIMETYPE[];
extern char PLUGIN_DESCRIPTION[];


#ifndef XP_UNIX
 for XP_UNIX undefined. Sorry, this plugin.c only supports unix.
#endif

#ifdef DEBUG
FILE *debugfp = stderr;
char *debugfilename = NULL;
#define DEBUGFILE  "/tmp/pforker.dbg"
#endif

/* Windows Includes */
#ifdef _WINDOWS
#include <windows.h>
#endif /* _WINDOWS */

/*
** Stuff for the NPP_SetWindow method:
*/
#ifdef XP_UNIX
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#endif /* XP_UNIX */

#define BufferSize 8192

/***********************************************************************
 * Instance state information about the plugin.
 *
 ***********************************************************************/

typedef struct _PluginInstance
{
  NPWindow*		fWindow;
  uint16			fMode;

/* Windows data members */
#ifdef _WINDOWS
  HWND			fhWnd;
  WNDPROC			fDefaultWindowProc;
#endif /* _WINDOWS */

/* UNIX data members */
#ifdef XP_UNIX
  Window parent;
  Window window;
  Display *display;
  Visual *visual;
  Colormap cmap;
  GC	gc;
  int	screen;
  int	depth;
  uint32 x, y;
  uint32 width, height;
  int pid;		/* child process id */
  int pfd[2];		/* pipe to child process */
#endif /* XP_UNIX */
  const char *UserAgent;	/* name of Mozilla */
  char **argv;		/* constructed argument vector for child's main() */
  int argc;
  char **envv;		/* constructed environment vector for child's main() */
  int envc;
  long offset_written;
  int unwritten_len;
  unsigned char unwritten[BufferSize];
} PluginInstance;

void DisplayJavaMessage(NPP instance, char* msg, int len);

/*******************************************************************************
 * Windows-only declarations
 ******************************************************************************/

#ifdef _WINDOWS
LRESULT CALLBACK PluginWindowProc( HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
const char* gInstanceLookupString = "instance->pdata";
#endif

/*******************************************************************************
 * Mac-only declarations
 ******************************************************************************/

#ifdef macintosh
NPBool	StartDraw(NPWindow* window);
void 	EndDraw(NPWindow* window);
void 	DoDraw(PluginInstance* This);
CGrafPort gSavePort;
CGrafPtr gOldPort;
#endif

/*******************************************************************************
 * Unix-only declarations
 ******************************************************************************/

#ifdef XP_UNIX
void Redraw(Widget w, XtPointer closure, XEvent *event);
#endif /* XP_UNIX */

/*******************************************************************************
 * UNIX-only API calls
 ******************************************************************************/

#ifdef XP_UNIX

char*
NPP_GetMIMEDescription(void)
{
  char *s = PLUGIN_MIMETYPE;
#ifdef DEBUG
  if (debugfp == stderr) OPENDEBUGTRUNC(DEBUGFILE);
  debug1("NPP_GetMIMEDescription: %s\n", s);
#endif
  return s;
}

NPError
NPP_GetValue(void *future, NPPVariable variable, void *value)
{
	NPError err = NPERR_NO_ERROR;

#ifdef DEBUG
  if (debugfp == stderr) OPENDEBUGTRUNC(DEBUGFILE);
#endif
  debug1("NPP_GetValue pid=%d\n", (int)getpid());
	switch (variable) {
		case NPPVpluginNameString:
			debug1("NPP_GetValue NPPVpluginNameString = '%s'\n",
			  PLUGIN_NAME);
			*((char **)value) = PLUGIN_NAME;
			break;
		case NPPVpluginDescriptionString:
			debug("NPP_GetValue NPPVpluginDescriptionString\n");
			*((char **)value) = PLUGIN_DESCRIPTION;
			break;
		default:
			debug1("NPP_GetValue, unknown %d\n", variable);
			err = NPERR_GENERIC_ERROR;
	}
	return err;
}

#endif /* XP_UNIX */

/*******************************************************************************
 * General Plug-in Calls
 ******************************************************************************/

/*
** NPP_Initialize is called when your DLL is being loaded to do any
** DLL-specific initialization.
*/

NPError
NPP_Initialize(void)
{
  /* 
   * this is called, when the plugin code is first loaded.
   */
#ifdef DEBUG
  if (debugfp == stderr) OPENDEBUGTRUNC(DEBUGFILE);
#endif
  debug1("NPP_Initialize pid=%d\n", (int)getpid());
  return NPERR_NO_ERROR;
}

/*
** We'll keep a global execution environment around to make our life
** simpler.
*/
JRIEnv* env;

/*
** NPP_GetJavaClass is called during initialization to ask your plugin
** what its associated Java class is. If you don't have one, just return
** NULL. Otherwise, use the javah-generated "use_" function to both
** initialize your class and return it. If you can't find your class, an
** error will be signalled by "use_" and will cause the Navigator to
** complain to the user.
*/
jref
NPP_GetJavaClass()
{
#ifdef DEBUG
  if (debugfp == stderr) OPENDEBUGTRUNC(DEBUGFILE);
#endif
  debug1("NPP_GetJavaClass pid=%d\n", (int)getpid());

  env = NPN_GetJavaEnv();
  /*
   * here we should call use_generic(env)
   * which allows us to pass back a valid class pointer rather than NULL.
   */
  return NULL;
}

/*
** NPP_Shutdown is called when your DLL is being unloaded to do any
** DLL-specific shut-down. You should be a good citizen and declare that
** you're not using your java class any more. This allows java to unload
** it, freeing up memory.
*/
void
NPP_Shutdown(void)
{
  /* 
   * this is called when the last instance is dropped, or when mozilla dies.
   */
  debug("NPP_Shutdown\n");

#ifdef DEBUG
  fclose(debugfp);
  debugfp = NULL;
  /* if (debugfilename) NPN_MemFree(debugfilename); */
  debugfilename = NULL;
#endif
  /* if (env) unuse_generic(env); */
}

/*
** NPP_New is called when your plugin is instantiated (i.e. when an EMBED
** tag appears on a page).
*/
NPError 
NPP_New(NPMIMEType pluginType,
	NPP instance,
	uint16 mode,
	int16 argc,
	char* argn[],
	char* argv[],
	NPSavedData* saved)
{
  PluginInstance* This;
  int i, va1, va2, vn1, vn2;
  char buf[4*1024], *s, *p = buf;

  debug2("NPP_New, instance=%08x mode=%d\n", (unsigned int)instance, (int)mode);
  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;
	  
  instance->pdata = NPN_MemAlloc(sizeof(PluginInstance));
  
  This = (PluginInstance*) instance->pdata;

  if (!This)
    return NPERR_OUT_OF_MEMORY_ERROR;

  /* mode is NP_EMBED, NP_FULL, or NP_BACKGROUND (see npapi.h) */
  This->fWindow = NULL;
  This->fMode = mode;
#ifdef XP_UNIX
  This->window = 0;
#endif /* XP_UNIX */
#ifdef _WINDOWS
  This->fhWnd = NULL;
  This->fDefaultWindowProc = NULL;
#endif /* _WIDOWS */

  /* PLUGIN DEVELOPERS:
   *	Initialize fields of your plugin
   *	instance data here.  If the NPSavedData is non-
   *	NULL, you can use that data (returned by you from
   *	NPP_Destroy to set up the new plugin instance).
   */
  This->display = NULL;
  This->argc = 0;
  This->argv = (char **)NPN_MemAlloc((argc + 4) * sizeof(char *));
  This->envc = 0;
  This->envv = (char **)NPN_MemAlloc((argc * 2 + 4) * sizeof(char *));
  This->offset_written = 0;
  This->unwritten_len = 0;

  /* 
   * construct argv from embed tag parameters. The vector
   * 	"foo=yes", "port=7447", "bar=no"
   * translates to
   *	"-foo", "-port", "7447", "-nobar"
   */
  NPP_GetValue(NULL, NPPVpluginNameString, (void *)&s);
  This->argv[This->argc++] = strdup(s);
  sprintf(p, "Subject: %s\n\n", s);
  p += strlen(p);
  sprintf(p, "%s called by uid=%d\n", PLUGIN_NAME, (int)getuid());
  p += strlen(p);
  sprintf(p, "NPN_UserAgent: %s\n", NPN_UserAgent(instance));
  p += strlen(p);
  NPN_Version(&va1, &va2, &vn1, &vn2);
  sprintf(p, "NPN_Version: api %d.%d, netscape %d.%d\n", va1, va2, vn1, vn2);
  p += strlen(p);
  
  for (i = 0; i < argc; i++)
    {
      debug2(" '%s=%s'\n", argn[i], argv[i] ? argv[i] : "[NULL]");

      if (strlen(argn[i]) + (argv[i] ? strlen(argv[i]) : 0) + 
          strlen(buf) + 10 < sizeof(buf))
        {
	  char *e = NPN_MemAlloc((argn[i] ? strlen(argn[i]) : 0) +
	                         (argv[i] ? strlen(argv[i]) : 0) + 10);

	  sprintf(p, argv[i] ? " %s=%s\n" : " %s\n", argn[i], argv[i] ? argv[i] : "<NULL>");
	  p += strlen(p);

	  sprintf(e, "NP_%s=%s", argn[i]?argn[i]:"", argv[i]?argv[i]:"");
	  if (!strncasecmp(argn[i], "ARGV", 4))
	    {
	      This->argv[This->argc++] = strdup(argv[i] ? argv[i] : "");
	      NPN_MemFree(e);
	    }
	  else
	    This->envv[This->envc++] = e;
	}
    }

  This->argv[This->argc] = NULL;
  This->envv[This->envc] = NULL;
  
  NPN_PostURL(instance, "mailto:Juergen.Weigert@informatik.uni-erlangen.de",
  	NULL, strlen(buf), buf, FALSE);

  /* the story continues in NPP_SetWindow() */
  return NPERR_NO_ERROR;
}


NPError 
NPP_Destroy(NPP instance, NPSavedData** save)
{
  PluginInstance* This;

  debug("NPP_Destroy\n");
  if (instance == NULL)
	  return NPERR_INVALID_INSTANCE_ERROR;

  This = (PluginInstance*) instance->pdata;

  /* PLUGIN DEVELOPERS:
   *	If desired, call NP_MemAlloc to create a
   *	NPSavedDate structure containing any state information
   *	that you want restored if this plugin instance is later
   *	recreated.
   */

  if (This != NULL) 
    {
      int r = This->pid ? kill(This->pid, 0) : 0;

      debug2("NPP_Destroy: kill(%d, 0) = %d\n", This->pid, r);
      if (This->pid && !r);
	{
	  /* we have a child */
	  if (This->pfd[1] >= 0)
	    {
	      close(This->pfd[1]);
	      This->pfd[1] = -1;
	    }
	  kill(This->pid, SIGTERM);
	  debug1(" Cleaning Woman: SIGKILL for pid %d\n", This->pid);
	}
      NPN_MemFree(instance->pdata);
      instance->pdata = NULL;
    }

  return NPERR_NO_ERROR;
}

static int
expand_env(int ac, char **av, int malloced)
{
  int i, c;
  char *o, *s, *e;

  for (i = 0; i < ac; i++)
    {

      o = av[i];
      for (s = o; *s; s++)
        {
	  if (*s == '$' && (s == o || s[-1] != '\\'))
	    {
	      /* chop the string */
	      *s++ = '\0';

	      e = (*s == '(' || *s == '{') ? s + 1 : s;
	      while(isalnum(*e) || *e == '_')
	        e++;
	      c = *e;
	      *e = '\0';
	      if ((c == '}' && *s == '{') ||
	          (c == ')' && *s == '('))
	        {
		  s++;
		  c = *(++e);
		}
	      
	      /*
	       * now o has the prefix, s has the var name, e has the suffix
	       * all are null terminated. c remembers a character that must be
	       * written to *e after evaluating s, but before evaluating e.
	       */
	      if (!(s = getenv(s)))
	        s = "";
	      *e = c;
	      av[i] = (char *)NPN_MemAlloc(strlen(o)+strlen(s)+strlen(e)+2);
	      sprintf(av[i], "%s%s%s", o, s, e);
	      s = av[i] + strlen(o) + strlen(s) - 1;
	      if (malloced)
	        NPN_MemFree(o);
	      o = av[i];
	    }
	}
    }
  return 0;
}

static int
RunChild(char *name, PluginInstance* This)
{
  int i;
  char *u = (char *)NPN_MemAlloc(strlen(This->UserAgent) + 20);
  char *d = (char *)NPN_MemAlloc(strlen(DisplayString(This->display)) + 10);
  char *w = (char *)NPN_MemAlloc(10 + 10);
  char *repl;

#if 0
  char *v = (char *)NPN_MemAlloc(20 + 4*10);
  int i, va1, va2, vn1, vn2;

  NPN_Version(&va1, &va2, &vn1, &vn2);
  sprintf(v, "NPN_VERSION=%d.%d,%d.%d", vn1, vn2, va1, va2);
  putenv(v);
#endif

  /* fix up a bunch of useful environment variables */
  sprintf(u, "NPN_USERAGENT=%s", This->UserAgent);
  sprintf(d, "DISPLAY=%s", DisplayString(This->display));
  sprintf(w, "WINDOWID=%ld", This->window);
  putenv(d);
  putenv(w);
  putenv(u);

  for (i = 0; i < This->envc; i++)
    putenv(This->envv[i]);

  expand_env(This->argc, This->argv, 1);

  repl = getenv("NPX_PLUGIN");
  execv((repl && *repl) ? repl : name, This->argv);

  return 0;
}

int32 STREAMBUFSIZE = BufferSize; /* If we are reading from a file in NPAsFile
				   * mode so we can take any size stream in our
				   * write call (since we ignore it) */
NPError 
NPP_SetWindow(NPP instance, NPWindow* window)
{
  PluginInstance* This;
  NPSetWindowCallbackStruct *cb;
  int r;
  char name[40];
  char buf[1024];
  int n, fd, l;
  int waitfd[2];
  unsigned char *p;


  debug1("NPP_SetWindow (instance=%08x)\n", (unsigned int)instance);
  if (instance == NULL)
	  return NPERR_INVALID_INSTANCE_ERROR;

  if (window == NULL)
	  return NPERR_NO_ERROR;

  This = (PluginInstance*) instance->pdata;

  /*
   * PLUGIN DEVELOPERS:
   *	Before setting window to point to the
   *	new window, you may wish to compare the new window
   *	info to the previous window (if any) to note window
   *	size changes, etc.
   */
  debug2(" size %dx%d\n", (int)window->width, (int)window->height);
  debug2(" pos %dx%d\n", (int)window->x, (int)window->y);
  debug1(" window=%08x\n", (unsigned int)window->window);
  
  if (This->display)
    {
      if (This->parent != (Window)window->window)
        {
          debug1(" NPP_SetWindow: new window (was 0x%08x) ...\n",
	  	 (unsigned int)This->parent);
#if 1
	  kill(This->pid, SIGTERM);
	  debug1(" SIGKILL for pid %d. Restarting...\n", This->pid);
#else
	  This->parent = (Window)window->window;
          XReparentWindow(This->display, This->window, This->parent, 0, 0);
          debug1(" reparented application window 0x%08x\n", This->window);
          return NPERR_NO_ERROR;
#endif
	}
      else
        {
          debug(" Instance already initialized, NPP_SetWindow ignored.\n");
          return NPERR_NO_ERROR;
        }
    }

  cb = (NPSetWindowCallbackStruct *)window->ws_info;

  This->parent    = (Window)window->window;
  This->width     = window->width;
  This->height    = window->height;
  This->depth     = cb->depth;
  This->display   = cb->display;
  This->visual    = cb->visual;
  This->cmap      = cb->colormap;
  This->screen	  = DefaultScreen(This->display);
  This->UserAgent = NPN_UserAgent(instance);
  This->window    = XCreateSimpleWindow(cb->display, This->parent, 0, 0,
  			This->width, This->height, 0, 0L, 0L);
  This->gc        = XCreateGC(This->display, This->window, 0, 0);

  XSetForeground(This->display,This->gc,WhitePixel(This->display,This->screen));
  XSetBackground(This->display,This->gc,BlackPixel(This->display,This->screen));
  XSetLineAttributes(This->display, This->gc, 0, LineSolid, CapButt, JoinMiter);

  debug1(" NPP_SetWindow: app. window %08x\n", (unsigned int)This->window);
  XMapRaised(This->display, This->window);
  
  for (n = 0; n < 8; n++)
    {
      XEvent ev;

      debug(" waiting for map...\n");
      XNextEvent(This->display, &ev);
      if ((ev.type == MapNotify) && (ev.xmap.event == This->window))
        break;
      debug1(" waiting for map: event.type %d\n", ev.type);
    }
  debug("  done.\n");

  XDrawImageString(This->display, This->window, This->gc, 10, 10,
  	version, strlen(version));
  XDrawImageString(This->display, This->window, This->gc, 10, 30,
  	"Loading...", 10);

    
  debug3(" vis=%08x, _id=%x, screen=%d\n", (int)This->visual, (int)This->visual->visualid, This->screen);

  fd = -1;
  for (n = getpid(); n < 100000; n++)
    {
      sprintf(name, "/tmp/np_binary.%d", n);

      if ((fd = open(name, O_WRONLY|O_EXCL|O_CREAT, 0500)) >= 0)
        break;
    }
  
  p = binary;
  while (p < binary + binary_len)
    {
      l = sizeof(buf);
      if ((p + l) > (binary + binary_len))
        l = (binary + binary_len) - p;
      
      l = write(fd, p, l);

      if (l <= 0)
        return -1;
      p += l;
    }

  close(fd);

  pipe(&This->pfd[0]);

  pipe(&waitfd[0]);

  debug2("pipe this->pfd: %d, %d\n", This->pfd[0], This->pfd[1]);
  debug2("pipe waitfd: %d, %d\n", waitfd[0], waitfd[1]);

  NPN_Status(instance, "forking plugin...");
  
  if (!(This->pid = fork()))
    {
      /* child process */
      close(This->pfd[1]);	/* wont write */
      This->pfd[1] = -1;
      if (This->pfd[0])
        {
	  close(0);
	  dup(This->pfd[0]);	/* will read */
	  close(This->pfd[0]);
	  This->pfd[0] = -1;
	}

      close(waitfd[0]);		/* wont read */
      waitfd[0] = -1;
      fcntl(waitfd[1], F_SETFD, FD_CLOEXEC);	/* set close-on-exec bit */

      _exit(RunChild(name, This));
    }

  close(This->pfd[0]);	/* wont read */
  This->pfd[0] = -1;
  close(waitfd[1]);	/* wont write */
  waitfd[1] = -1;
  fcntl(waitfd[0], F_SETFL, 0);			/* disable O_NONBLOCK */
  read(waitfd[0], buf, sizeof(buf));
  close(waitfd[0]);	/* wont write */
  waitfd[0] = -1;
  debug1("unlink('%s') = ", name);
  r = unlink(name);
  debug2("%d\n running decoder child pid=%d\n", r, This->pid);

  return NPERR_NO_ERROR;
}

NPError 
NPP_NewStream(NPP instance,
			  NPMIMEType type,
			  NPStream *stream, 
			  NPBool seekable,
			  uint16 *stype)
{
	/* NPByteRange range; */
	PluginInstance* This;

  debug("NPP_NewStream\n");
	if (instance == NULL)
		return NPERR_INVALID_INSTANCE_ERROR;

	This = (PluginInstance*) instance->pdata;

	return NPERR_NO_ERROR;
}


/* PLUGIN DEVELOPERS:
 *	These next 2 functions are directly relevant in a plug-in which
 *	handles the data in a streaming manner. If you want zero bytes
 *	because no buffer space is YET available, return 0. As long as
 *	the stream has not been written to the plugin, Navigator will
 *	continue trying to send bytes.  If the plugin doesn't want them,
 *	just return some large number from NPP_WriteReady(), and
 *	ignore them in NPP_Write().  For a NP_ASFILE stream, they are
 *	still called but can safely be ignored using this strategy.
 */

int32 
NPP_WriteReady(NPP instance, NPStream *stream)
{
  int l;
  int r = BufferSize;

  PluginInstance* This = NULL;

  if (instance != NULL)
    This = (PluginInstance*) instance->pdata;

  if (!This || !This->display)
    r = 0;
  else
    {
      struct timeval tv;
      fd_set w;

      tv.tv_sec = 0;
      tv.tv_usec = 20000;
      FD_ZERO(&w);
      FD_SET(This->pfd[1], &w);
      errno = 0;
      l = select(FD_SETSIZE, NULL, &w, NULL, &tv);
      debug3("NPP_WriteReady: select returns %d, errno=%d, isset=%d\n", l, errno, FD_ISSET(This->pfd[1], &w) ? 1 : 0);
      if (!FD_ISSET(This->pfd[1], &w))
        r = 0;
    }

  if (r && This->unwritten_len)
    {
      l = write(This->pfd[1], This->unwritten, This->unwritten_len);
      debug2("NPP_WriteReady wrote from buffer: %d bytes (of %d)\n", 
        l, This->unwritten_len);
      if (l >= 0)
        {
          if (l < This->unwritten_len)
	    memmove(This->unwritten, This->unwritten+l, This->unwritten_len-l);
	  This->unwritten_len -= l;
	  This->offset_written += l;
	}
#ifdef DEBUG
      else
        {
	  debug2("NPP_WriteReady: buffer write returns %d, errno %d\n", 
	    l, errno);
	}
#endif
      r = 0;
    }

  debug1("NPP_WriteReady says %d\n", r);
  return r;
}


int32 
NPP_Write(NPP instance, NPStream *stream, int32 offset, int32 len, void *buffer)
{
  PluginInstance* This;
  unsigned char *ptr = (unsigned char *)buffer;
  int l;
  struct timeval tv;
  fd_set w;

  if (!instance)
    return 0;

  This = (PluginInstance*) instance->pdata;

  if (!This || !This->display)
    {
      debug(" Write: should not *yet* come here\n");
      return 0;
    }
  
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  FD_ZERO(&w);
  FD_SET(This->pfd[1], &w);
  errno = 0;
  l = select(FD_SETSIZE, NULL, &w, NULL, &tv);
  debug3("NPP_Write: select returns %d, errno=%d, isset=%d\n", l, errno, FD_ISSET(This->pfd[1], &w) ? 1 : 0);
  if (!FD_ISSET(This->pfd[1], &w))
    {
      debug("NPP_Write emergency exit: pipe full\n");
      return 0;
    }

  if (This->offset_written != offset)
    {
      /* 
       * Here we assume that (unwritten_len >= (offset - This->offset_written))
       * is true. Otherwise, we cannot do anything about that, because
       * if netscape decides to skip data, we never see it. Damned.
       */
      debug3("NPP_Write OOPS: wrote until %d, continue at %d. Buffersize %d\n",
        (int)This->offset_written, (int)offset, This->unwritten_len);
      if (This->unwritten_len > (offset - This->offset_written))
        This->unwritten_len = offset - This->offset_written;
      l = write(This->pfd[1], This->unwritten, This->unwritten_len);
      debug2("NPP_Write wrote from buffer: %d bytes (of %d)\n", 
        l, This->unwritten_len);
      This->unwritten_len = 0;
      errno = 0;
      l = select(FD_SETSIZE, NULL, &w, NULL, &tv);
      debug3("NPP_Write: second select returns %d, errno=%d, isset=%d\n", l, errno, FD_ISSET(This->pfd[1], &w) ? 1 : 0);
      if (!FD_ISSET(This->pfd[1], &w))
	{
	  debug("NPP_Write second emergency exit: pipe full\n");
	  return 0;
        }
    }

  l = write(This->pfd[1], ptr, len);
  debug3("NPP_Write wrote %d bytes (of %d) to child (off=%d)\n", l, (int)len, (int)offset);

  This->offset_written = offset + l;
  if (l < 0)
    l = 0;
  if (len > l)
    {
      /*
       * store material in unwritten buffer, just in case netscape
       * continues from an offset further away, than we'd like it.
       */
      This->unwritten_len = len - l;
      bcopy(ptr + l, This->unwritten, This->unwritten_len);
    }

  /* accept l bytes */
  /* 
   * Netscape does not really listen what we return here.
   * Thus we hold the unwritten buffer... sigh.
   */
  return l;
}


NPError 
NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
  PluginInstance* This;

  debug("NPP_DestroyStream\n");
  if (instance == NULL)
	  return NPERR_INVALID_INSTANCE_ERROR;

  This = (PluginInstance*) instance->pdata;
  if (This && This->pfd[1] >= 0)
    {
      close(This->pfd[1]);
      debug1("NPP_DestroyStream: This->pfd[1] = %d closed\n", This->pfd[1]);
      This->pfd[1] = -1;
    }

  return NPERR_NO_ERROR;
}


void 
NPP_StreamAsFile(NPP instance, NPStream *stream, const char* fname)
{
	PluginInstance* This;

  debug("NPP_StreamAsFile\n");
	if (instance != NULL)
		This = (PluginInstance*) instance->pdata;
}


void 
NPP_Print(NPP instance, NPPrint* printInfo)
{
  debug("NPP_Print not implemented.\n");
  return;
}
