#include <string.h>

#include "pyradecoder.h"
#include "common.h"
#include "filter.p"


/* ========================================================================= */
/* Returns the 'temp_ref' of the found intra picture or -1 if no intra
   picture header has been found */
int ParsePyraJunkForIntra(PyraGlobal *pyraData, Byte **next_p, Byte *end,
			  int lastjunk)
{
  int codingType;


  do {

    *next_p = PassToStreamBufferPyra(pyraData->bs, *next_p, end);

    if (pyraData->bs->BufState == GOB_PENDING && lastjunk)
      InitStreamParsingPyra(pyraData->bs);

    if (pyraData->bs->BufState == GOB_LOADED) {
      /* We have already loaded a GOB; is it GOB 0? */
      if (ShowPyraGOBnumber(pyraData->bs->ByteBuffer) == 0) {
	/* Yeah, it is GOB 0; Which coding type has the new picture? */
	/* Check the temporal reference */
	pyraData->next_temp_ref = ShowPyraTempRef(pyraData->bs->ByteBuffer);
	codingType = ShowPyraCodingType(pyraData->bs->ByteBuffer);
	if (codingType == PICTURE_CODING_TYPE_INTRA)
	  return pyraData->next_temp_ref;
	else {
	  if (*next_p == end && lastjunk)
	    pyraData->bs->BufState = GOB_EMPTY;
	  else {
	    pyraData->bs->BufState = GOB_COMPLETE;
	    pyraData->bs->BufferPtr = pyraData->bs->GOBend - 3;
	    pyraData->bs->ValidBits = 0;
	  }
	}
      } else {
	if (*next_p == end && lastjunk)
	  pyraData->bs->BufState = GOB_EMPTY;
	else {
	  pyraData->bs->BufState = GOB_COMPLETE;
	  pyraData->bs->BufferPtr = pyraData->bs->GOBend - 3;
	  pyraData->bs->ValidBits = 0;
	}
      }
    } else if (pyraData->bs->BufState == GOB_PENDING ||
	       pyraData->bs->BufState == GOB_EMPTY) {
      return -1;
    }
  } while (1);
}


/* ========================================================================= */
int SkipUntilIntra(PyraGlobal *pyraData, Byte **next_p, Byte *end,
		   int lastjunk, int t)
{
  int codingType;


  do {

    *next_p = PassToStreamBufferPyra(pyraData->bs, *next_p, end);

    if (pyraData->bs->BufState == GOB_LOADED) {
      /* We have already loaded a GOB; is it GOB 0? */
      if (ShowPyraGOBnumber(pyraData->bs->ByteBuffer) == 0) {
	/* Yeah, it is GOB 0; Which coding type has the new picture? */
	/* Check the temporal reference */
	pyraData->next_temp_ref = ShowPyraTempRef(pyraData->bs->ByteBuffer);
	codingType = ShowPyraCodingType(pyraData->bs->ByteBuffer);
	/* Do init if it not done yet */
	if (pyraData->first) {
	  pyraData->PicSize = ShowPyraPicSize(pyraData->bs->ByteBuffer);
	  InitPyraDecoder(pyraData);
	  InitBlockDecoder(pyraData, pyraData->width);
	}
      } else {
	/* This is not GOB 0; */
	codingType = pyraData->PicType;
	if (pyraData->GOBFrameID != ShowPyraGFID(pyraData->bs->ByteBuffer)) {
	  /* The ptype has changed! So lets wait for a picture sync */
	  codingType = PICTURE_CODING_TYPE_UNKNOWN;
	}
      }

      /* What time do we have? */
      if (t < pyraData->next_temp_ref) {
	/* Not too old; so lets wait. */
	PyraFinishPicture(pyraData);
	pyraData->temp_ref = t;
	return 1;
      } else if (t == pyraData->next_temp_ref &&
		 codingType == PICTURE_CODING_TYPE_INTRA &&
		 !pyraData->skip_until_next_tr) {
	/* That's the right time and the picture is intra coded */
	pyraData->awaitIntra = 0;
	return 1;
      } else {
	/* Sorry, this picture is already too old or not intra or we are
	   just skipping everything */
	pyraData->bs->BufState = GOB_EMPTY;
	if ((*next_p = FindPyraPictSync(*next_p, end)) == NULL) {
	  *next_p = end;
	  if (lastjunk)
	    PyraFinishPicture(pyraData);
	  return lastjunk;
	}
      }
    } else if (pyraData->bs->BufState == GOB_PENDING ||
	       pyraData->bs->BufState == GOB_EMPTY) {
      if (lastjunk && *next_p == end) {
	PyraFinishPicture(pyraData);
	return 1;
      } else
	return 0;
    }
  } while (1);
}


/* ========================================================================= */

/* ========================================================================= */
/* This Funktion returns '1' if pyraData->newframe should be displayed
   and '0' if it needs more data */
/* If 'lastjunk' is set, the data after the last found Sync is also passed
   to 'PyraDecodeGOB'. So be careful that there is a complete GOB after the
   last Sync in the buffer! */
int HandlePyraDataJunk(PyraGlobal *pyraData, Byte **next_p, Byte *end,
		       int lastjunk)
{
    /* Copies the data buffer to the stream buffer.
     * Copy stops at a sync mark.
     * next points to the first data byte within the
     * data buffer which has not been copied.
     */
  int       returncode = 1;
  int       ngob;

  if (pyraData->decGOBsCounter == pyraData->GOBs_in_pict) { /*!!MS*/
      /* We have decoded successfully all GOBs in this picture */
      /* Reset the GOB counters */
      pyraData->decGOBsCounter = 0;
      memset(pyraData->decGOBs, 0, MAX_GOBS * sizeof(unsigned char));
  }

#if 0
  /* Finish this picture if we have lost some data */
  if (pyraData->decGOBsCounter != 0 &&
      pyraData->next_temp_ref != pyraData->temp_ref) {
    /* We have lost the remaining part of the picture */
    pyraData->prev_temp_ref = pyraData->temp_ref;
    pyraData->temp_ref = pyraData->next_temp_ref;
    PyraFinishPicture(pyraData);
    return 1;
  }

  if (pyraData->decGOBsCounter == pyraData->GOBs_in_pict) {
    /* We have decoded successfully all GOBs in the last picture */
    /* Reset the GOB counters */
    pyraData->decGOBsCounter = 0;
    memset(pyraData->decGOBs, 0, MAX_GOBS * sizeof(unsigned char));
  }
#endif

  *next_p = PassToStreamBufferPyra(pyraData->bs, *next_p, end);
 
  if (lastjunk && pyraData->bs->BufState == GOB_PENDING) {
    InitStreamParsingPyra(pyraData->bs);
  }

  while (pyraData->bs->BufState == GOB_LOADED) {

    ngob = ShowPyraGOBnumber(pyraData->bs->ByteBuffer);
    if (pyraData->decGOBs[ngob] == 1) {
      /* We have already decoded a GOB with this number */
      /* So let's finish the picture */
      PyraFinishPicture(pyraData);
      return 2;
    }
    if (ngob == 0) {
      if (pyraData->decGOBsCounter > 0 && !pyraData->first) {
	/* A new picture start; and we didn't finish the last picture! */
	/* So lets do concealment */
	PyraFinishPicture(pyraData);
	return 1;
      }
    }

    /* We have a complete GOB which can be decoded now. */
    if (!PyraDecodeGOB(pyraData))
      returncode = 2;

    if (*next_p == end && lastjunk)
      pyraData->bs->BufState = GOB_EMPTY;

    if (pyraData->decGOBsCounter == pyraData->GOBs_in_pict) {
      return returncode;
    }

    /* Copy the remaining data bytes to the stream buffer. */
    *next_p = PassToStreamBufferPyra(pyraData->bs, *next_p, end);

    if (lastjunk && pyraData->bs->BufState == GOB_PENDING) {
      InitStreamParsingPyra(pyraData->bs);
    }
  }

  return 0;
}

/* ========================================================================= */
/* ========================================================================= */
int HandlePyraCompleteGOBs(PyraGlobal *pyraData, Byte *next, Byte *end)
{
  if (pyraData->decGOBsCounter == pyraData->GOBs_in_pict) { /*!!MS*/
      /* We have decoded successfully all GOBs in this picture */
      /* Reset the GOB counters */
      pyraData->decGOBsCounter = 0;
      memset(pyraData->decGOBs, 0, MAX_GOBS * sizeof(unsigned char));
  }

  pyraData->bs->ByteBuffer = next;
  pyraData->bs->BufferPtr = end;

  do {
      InitStreamParsingPyra(pyraData->bs);
      /* We have a complete GOB which can be decoded now. */
      if (!PyraDecodeGOB(pyraData))
          return 0;
      pyraData->bs->ByteBuffer = FindPyraSync(pyraData->bs->BufferPtr - sizeof(unsigned long), end);
  } while (pyraData->bs->BufferPtr < end);

  return 1;
}
