/*
 * main.c - 
 *
 * 15.10.97, jw.
 */
#ifdef LINUX
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

/* #define VERIFY_DEC_PICTURE */
#define COMPUTE_PSNR

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef LINUX
#include <sys/time.h>
#endif
#include <unistd.h>
#include <string.h>
#include <sys/fcntl.h>	/* for O_RDONLY */
#include <time.h>
#include <sys/time.h>	/* for struct timeval */

#define DEFINE_GLOBALS


#include "defs.h"
#include "pyra/decoder/main.h"

#include "options.h"
#include "config.h"	/* Options initializers */
#include "dispconf.h"	/* Options initializers */
#include <openurl.h>	/* OpenUrl(), struct url_info */
#include <jittr/schedule.h>	/* sched_timerdiff */
extern FILE *fdopen();

/* Prototype files */
#include "h263decoder.h"
#include "pyradecoder.h"
#include "common.h"
#include "display.h"
#include "filter.p"

#ifndef WITHOUT_CHACODEC
#include "chacodec.h"
/*#include "chacodec.p"*/
#endif

#ifdef COMPUTE_PSNR
static int  psnr;
static char *origName;
static int  firstFrame, lastFrame;
static char *protName;
static FILE *fp_prot;
#endif

typedef union Global_union {
  PyraGlobal *pyra;
  H263Global *h263;
} Global;


#ifdef DEBUG
FILE *debugfp = stderr;
char *debugfilename = NULL;
#endif

/* Define mbuiltins for jittr library */
struct mbuiltin *mbuiltins = NULL;

static OptionsStruct Init_Options =
{
  NULL,
  USE_VIS,
  USE_DGA,
  USE_SHM,
  DISPLAY,
  0,		/* Default for cover_window: use own toplevel */
  THREAD_YUV,
  GRAY,
  PSEUDOCOLOR,
  EXPAND,
  SIZE_QCIF,
  0,		/* Default for fps */
  LOOP,
  AUDIO
};

OptionsStruct Options;

#if defined(__sun) && defined(FILE)
# define memmove(d, s, n) bcopy(s, d, n)
#endif

static void ExitWithUsage(char *msg, int ret)
{
#define O(txt, def, opt) fprintf(stderr, "  -%s. Default: -%s%s.\n", txt, \
def ? "" : "no", opt);
  fprintf(stderr, "Usage: %s [-[no]option ...] files.[hpy|h263] ...\n", Options.argv0);
#ifndef WITHOUT_DGA
  O("dga        Use Direct Graphics Access", USE_DGA, "dga");
#endif
  O("windowid   Use given window as parent", 1, " own toplevel");
  O("nodisplay  Don't show the decoded frames", DISPLAY, "display");
  O("expand     Double the size of the image", EXPAND, "expand");
  O("gray       Show video in grayscale", GRAY, "gray");
  O("noaudio    Disable audio", AUDIO, "audio");
/*  O("loop       Repeat the input stream infinitly", LOOP, "loop"); */
  O("pseudo     Show video in PseudoColor (8Bit)", PSEUDOCOLOR,
    "pseudo (TrueColor)");
  O("qcif       Show image in QCIF size", 1, SIZE_QCIF ? "qcif" : "cif");
#ifndef WITHOUT_SHM
  O("noshm      Don't use shared memory", USE_SHM, "shm");
#endif
#ifndef WITHOUT_THREADS
  O("threads    Spawn a thread for color conversion", THREAD_YUV, "threads");
#endif
#ifndef WITHOUT_VIS
  O("vis        Use UltraSparc Visual Instruction Set", USE_VIS, "vis");
#endif
  O("fps        If set, use given framerate", 1, "nofps (unlimited)");
  O("quiet      run in silence", 0, "not quiet");
  if (msg)
    fprintf(stderr, "\nHmmm: %s\n", msg);
  exit(ret);
}


/*
 * This WaitFrame() is not good for network transmissions.
 * Consider Buffers here!!!
 *
 * Returns negative, if too late.
 */
int
WaitFrame(int temp_ref, int fps)
{
  struct timeval tv, fut, now;

  static int prev_temp_ref = -100;
  static struct timeval prev_tv;

  if (!fps)
    return 0;

  if (!prev_tv.tv_sec)
    {
      prev_temp_ref = temp_ref;
      gettimeofday(&prev_tv, NULL);
      return 0;
    }
  
  if (fps < 0)
    {
      int i;

#if 1
      tv.tv_usec = 1001000/30;
#else
      tv.tv_usec = this->timebasenom / this->timebasedenom; /* FIXME */
#endif
      i = temp_ref - prev_temp_ref;
      prev_temp_ref = temp_ref;
      /* ASSERT(i == this->trd); */

      if (i < 0) i += 256;

      tv.tv_usec *= i;
      tv.tv_sec = tv.tv_usec / 1000000;
      tv.tv_usec -= 1000000 * tv.tv_sec;
    }
  else
    {
      tv.tv_usec = 1000000/fps;
      tv.tv_sec = 0;
    }
/*  debug1("frametime %d usec, ", (int)tv.tv_usec); */

  
  fut.tv_sec = tv.tv_sec + prev_tv.tv_sec;
  fut.tv_usec = tv.tv_usec + prev_tv.tv_usec;
  if (fut.tv_usec > 1000000)
    {
      fut.tv_sec++;
      fut.tv_usec -= 1000000;
    }

  prev_tv.tv_sec = fut.tv_sec;
  prev_tv.tv_usec = fut.tv_usec;

  gettimeofday(&now, NULL);
  if (sched_timerdiff(&tv, &fut, &now))
    {
      /* already too late */
      return -1;
    }
  select(0, NULL, NULL, NULL, &tv); 
/*  debug1("waiting %d usec\n", (int)tv.tv_usec); */
  return 1;
}


/* *************************************************************************/
/* *************************************************************************/
int VerifyDecPic(int *method, Global *these, int nl, int t)
{
  char *decPictBaseName = "decPict";
  char    decPictName[100];
  int     trel = 0;
  int     errorPos;
  Picture *decPic, *inPic;


  decPic = (Picture *)malloc(sizeof(Picture));

  switch (method[nl]) {
  case H263:
    if (t)
      trel = (t / these[nl].h263->trd);
    decPic->w = these[nl].h263->coded_picture_width;
    decPic->h = these[nl].h263->coded_picture_height;
    decPic->ws = these[nl].h263->coded_picture_width;
    decPic->hs = these[nl].h263->coded_picture_height;
    decPic->y = these[nl].h263->newframe[0];
    decPic->u = these[nl].h263->newframe[1];
    decPic->v = these[nl].h263->newframe[2];
    break;
  case PYRA:
    if (t)
      trel = (t / these[nl].pyra->trd);
    decPic->w = these[nl].pyra->DecPic->w;
    decPic->h = these[nl].pyra->DecPic->h;
    decPic->ws = these[nl].pyra->DecPic->ws;
    decPic->hs = these[nl].pyra->DecPic->hs;
    decPic->y = these[nl].pyra->DecPic->y;
    decPic->u = these[nl].pyra->DecPic->u;
    decPic->v = these[nl].pyra->DecPic->v;
    break;
  }

  inPic = AllocPicture(decPic->w, decPic->h);
  inPic->ws = decPic->ws;
  inPic->hs = decPic->hs;

  sprintf(decPictName, "%s%d", decPictBaseName, nl);
  if (!ReadPicture(decPictName, CAT_SEQ_TYPE, trel, inPic)) {
    fprintf(stderr, "%s for verifying needed!\n", decPictName);
    exit(4);
  }
  if ((errorPos = FindFirstDifferenceBetweenSubPictures(inPic, decPic)) >= 0) {
    printf("\nFehler bei Layer >%d<, t = %d;  Position %d\n", nl, t, errorPos);
  }

  /* Free memory */
  free(decPic);
  FreePicture(inPic);

  return(errorPos);
}


/* *************************************************************************/
/* *************************************************************************/
void InitLayerReferences(int *method, Global *these, int nl, int lastDecLayer)
{
  if (lastDecLayer >= 0 && !these[nl].pyra->first) {
    switch (method[lastDecLayer]) {
    case H263:
      MV2MVField(these[lastDecLayer].h263->MV,
		 these[lastDecLayer].h263->modemap,
		 these[lastDecLayer].h263->coded_picture_width / 16,
		 these[lastDecLayer].h263->trd, these[nl].pyra->mvf,
		 these[nl].pyra->trd);
      break;
    case PYRA:
      SampleUpMVField(these[lastDecLayer].pyra->mvf,
		      these[lastDecLayer].pyra->trd, these[nl].pyra->mvf,
		      these[nl].pyra->trd);
      break;
    }
  } else

  if (lastDecLayer < 0)
    lastDecLayer = nl + 1;
  switch (method[lastDecLayer]) {
  case H263:
    these[nl].pyra->SpatRefPic->w =
      these[lastDecLayer].h263->coded_picture_width;
    these[nl].pyra->SpatRefPic->h =
      these[lastDecLayer].h263->coded_picture_height;
    these[nl].pyra->SpatRefPic->ws = these[nl].pyra->SpatRefPic->w;
    these[nl].pyra->SpatRefPic->hs = these[nl].pyra->SpatRefPic->h;
    these[nl].pyra->SpatRefPic->y = these[lastDecLayer].h263->newframe[0];
    these[nl].pyra->SpatRefPic->u = these[lastDecLayer].h263->newframe[1];
    these[nl].pyra->SpatRefPic->v = these[lastDecLayer].h263->newframe[2];
    break;
  case PYRA:
    LinkPicture(these[lastDecLayer].pyra->DecPic, these[nl].pyra->SpatRefPic);
    break;
  }
}



/* *************************************************************************/
/* *************************************************************************/
void PrintDecInfo(DecInfo *dInfo) {

#ifndef WITHOUT_CHACODEC
  /* Reed-Solomon packetization */
  printf("\t-prs  switch on decoding from Reed-Solomon packed stream [%s]\n",
	 dInfo->prs ? "ON" : "OFF");
  printf("\t-sPL  ring at packet loss and show some information [%s]\n",
	 dInfo->showPckLoss ? "ON" : "OFF");
#endif

  /* Display */
  printf("\t-dF turn off the display\n");
  printf("\t-dS <num> display size [%d]\n", dInfo->displaySize);
  printf("\t    SQCIF: 1\n");
  printf("\t    QCIF : 2\n");
  printf("\t    CIF  : 3\n");
  printf("\t    CIF4 : 4\n");
  printf("\t    CIF16: 5\n");
  printf("\t-windowid <num> window ID (0: use top level window) [%ld].\n",
	 dInfo->windowid);
  /* General */
  printf("\t-fps <num> frames per second [%d] (0: as fast as possible)\n",
	 dInfo->fps);
  printf("\t-tl <num> top layer; best decoded quality\n [%d]",
	 dInfo->topLayer);
  printf("\t-quiet run quietly (default: verbose)\n");
  printf("\t-loop  [num] decode the sequence 'num' times. Without number: endless loop."
	 "(default: no loop)\n");
#ifdef COMPUTE_PSNR
  printf("\t-psnr  compute PSNR default [OFF]\n");
  printf("\t-orig  basename of original sequence [%s]\n",
	 (origName ? origName : ""));
  printf("\t-fF    first frame [%d]\n", firstFrame);
  printf("\t-lF    last frame [%d]\n", lastFrame);
  printf("\t-prot  protocol file name (default: stdout)\n");
#endif
}


void Usage(char *argv[]) {
  printf("\nScalVico decoder (version 1.0)\n");
  printf("Copyright (c) 1998 LNT, Uni Erlangen, Germany\n");
  printf("Type <%s -H> for advanced help\n", argv[0]);
  printf("Usage: %s\n", argv[0]);
}


void VerboseUsage(int argc, char *argv[], DecInfo *dInfo) {
  int i;

  for(i=0; *(argv+i) != NULL; ++i) {
    /* Help utilities */
    if((!strcmp(argv[i],"-help")) || (!strcmp(argv[i],"-H")) 
       || (!strcmp(argv[i],"-U"))) {
      Usage(argv);
      printf("Options:\n");
      PrintDecInfo(dInfo);
      printf("\n\tDefault options in square brackets are specified in "
	     "main.h\n");
      printf("\n\tFor the Options that need a number for each layer you "
	     "have to specify\n\t*exactly* >%d< numbers!\n", NUM_LAYERS);
      exit(0);
    }
    if((!strcmp(argv[i],"-h")) || (!strcmp(argv[i],"-u"))) {
      Usage(argv);
      exit(0);
    }
  }
}


DecInfo *NewDecInfo()
{
  DecInfo     *dInfo = (DecInfo *)malloc(sizeof(DecInfo));


  if(!dInfo)
    exit(-1);

#ifndef WITHOUT_CHACODEC
  /* Reed-Solomon packetization */
  dInfo->prs = DEFAULT_PACKET_RS;
  dInfo->showPckLoss = DEFAULT_SHOW_PACKET_LOSS;
#endif

  /* General configurations */
  dInfo->displayFormat   = DEFAULT_DISPLAY_FORMAT;
  dInfo->displaySize     = DEFAULT_DISPLAY_SIZE;
  dInfo->windowid        = DEFAULT_WINDOWID;
  dInfo->fps = DEFAULT_FRAMES_PER_SECOND;
  dInfo->topLayer = DEFAULT_TOP_LAYER;
  dInfo->quiet = DEFAULT_QUIET_FLAG;
  dInfo->loop = DEFAULT_LOOP;

#ifdef COMPUTE_PSNR
  psnr = 0;
  origName = NULL;
  firstFrame = 0;
  lastFrame = 0;
  protName = NULL;
  fp_prot = stdout;
#endif

  return(dInfo);
}


char **EvaluateArgumentsDecInfo(int argc, char *argv[], DecInfo *dInfo)
{
  argv++;
  argc--;
  while(*argv && **argv == '-' && argv[0][1]) {

#ifndef WITHOUT_CHACODEC
    /* Reed-Solomon packetization */
    if(!strcmp(*argv,"-prs")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      dInfo->prs = !(dInfo->prs);
      continue;
    }
    /* Percentage packet loss */
    if(!strcmp(*argv,"-sPL")) {
      argv++;
      dInfo->showPckLoss = !(dInfo->showPckLoss);
      continue;
    }
#endif

    /* Switch display */
    if(!strcmp(*argv,"-dF")) {
      argv++;
      dInfo->displayFormat = !(dInfo->displayFormat);
      continue;
    }
    /* Display size */
    if(!strcmp(*argv,"-dS")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      dInfo->displaySize = atoi(*argv++);
      continue;
    }
    if(!strcmp(*argv,"-qcif")) {
      argv++;
      dInfo->displaySize = QCIF;
      continue;
    }
    if(!strcmp(*argv,"-cif")) {
      argv++;
      dInfo->displaySize = CIF;
      continue;
    }
    if(!strcmp(*argv,"-cif4")) {
      argv++;
      dInfo->displaySize = CIF4;
      continue;
    }
    if(!strcmp(*argv,"-cif16")) {
      argv++;
      dInfo->displaySize = CIF4;
      continue;
    }
    /* Window ID */
    if(!strcmp(*argv,"-windowid") || !strcmp(*argv,"-id") ||
       !strcmp(*argv,"-window")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      dInfo->windowid = strtol(*argv++, NULL, 0);
      continue;
    }
    /* Frames per second */
    if(!strcmp(*argv,"-fps")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      dInfo->fps = atoi(*argv++);
      continue;
    }
    /* Top layer to decode */
    if(!strcmp(*argv,"-tl")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      dInfo->topLayer = atoi(*argv++);
      continue;
    }
    /* Switch quiet flag */
    if(!strcmp(*argv,"-quiet")) {
      argv++;
      dInfo->quiet = 1;
      continue;
    }
    /* Frames per second */
    if(!strcmp(*argv,"-loop")) {
      argv++;
      dInfo->loop = -1;
      if(argc >= 0 && isdigit(**argv)) {
	dInfo->loop = atoi(*argv++);
      }
      continue;
    }

#ifdef COMPUTE_PSNR
    /* Switch on psnr measurement */
    if(!strcmp(*argv,"-psnr")) {
      argv++;
      psnr = 1;
      continue;
    }
    /* Name of original video sequence */
    if(!strcmp(*argv,"-orig")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      if (origName) free(origName);
      origName = (char *)malloc((strlen(*(++argv)) + 1) * sizeof(char));
      strcpy(origName,*argv++);
      continue;
    }
    /* First frame in original sequence */
    if(!strcmp(*argv,"-fF")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      firstFrame = atoi(*argv++);
      continue;
    }
    /* First frame in original sequence */
    if(!strcmp(*argv,"-lF")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      argv++;
      lastFrame = atoi(*argv++);
      continue;
    }
    /* Name of original video sequence */
    if(!strcmp(*argv,"-prot")) {
      if(--argc < 0) {Usage(argv); printf("%d\n",argc); exit(0);}
      if (protName) free(protName);
      protName = (char *)malloc((strlen(*(++argv)) + 1) * sizeof(char));
      strcpy(protName,*argv++);
      if (strcmp(protName, "-") == 0) {
	fp_prot = stdout;
      } else {
	if ((fp_prot = fopen(protName, "w")) == NULL) {
	  fprintf(stderr, "Can not open protocol file >%s<\n", protName);
	  exit (0);
	}
      }
      continue;
    }
#endif

    printf("Unknown option '%s'\n", *argv);
    PrintDecInfo(dInfo);
    exit(1);
    break;
  }
  return(argv);
}




/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- RunFileDecoding -- Decode bitstream files
 *
 * Author:      K.S.
 *
 * Created:     2-Jul-98
 *
 * Purpose:     -
 *
 * Options:     -
 *
 * Description: 
 *
 * See also:    -
 *
 * Modified:    -
 *
 *****************************************************************************/
int RunFileDecoding(int nLayers, FILE *stream[NUM_LAYERS], Picture *dispPict,
		    int fps, int quiet, int loop)
/***********************************************************CommentEnd********/
{
  int first = 1;
  int t;
  int lastDecLayer;
  int nl;
  int method[NUM_LAYERS];
  Global these[NUM_LAYERS];
  Byte *databuf[NUM_LAYERS];
  Byte *next[NUM_LAYERS], *end[NUM_LAYERS];
  int cont = 0;
  int action;
  int dSize;
  Picture picture, *pic = NULL;
#ifdef COMPUTE_PSNR
  int              t_psnr = firstFrame, t_psnr_end;
  Picture          *origPict = NULL;
  if (psnr) {
    origPict = ClonePicture(dispPict);
  }
#endif


  /****************/
  /* Looping loop */
  /****************/
  do {
    /*******************/
    /* Initializations */
    /*******************/
    for (nl = 0; nl < nLayers; nl++) {
      cont |= (1 << nl);
      if (first) {
	databuf[nl] = (Byte *) malloc(PACKET_SIZE);
      }
      end[nl] = databuf[nl] + PACKET_SIZE;
      next[nl] = end[nl];
      CheckInBuffer(stream[nl], databuf[nl], PACKET_SIZE, &next[nl], &end[nl]);
      /* Decide on method by looking at the first sync */
      if (first) {
	method[nl] = NONE;
	if (databuf[nl][0] == 0 && databuf[nl][1] == 0)
	  if (databuf[nl][2] == 1) {
	    method[nl] = PYRA;
	    these[nl].pyra = NewPyraDecoder();
	    these[nl].pyra->trd = 1;
	  } else if (databuf[nl][2] & 0x80) {
	    method[nl] = H263;
	    these[nl].h263 = NewH263Decoder();
	    these[nl].h263->quiet = quiet;
	    these[nl].h263->trd = 1;
	  }
	if (method[nl] == NONE) {
	  fprintf(stderr, "Streamfile no. %d contains undecodable data\n", nl);
	  fprintf(stderr, "\nPerhaps you forgot the '-prs' flag?\n\n");
	  return(0);
	}
      }
    }
    first = 0;

    /*************/
    /* Film loop */
    /*************/
    for (t = 0; cont; t++) {
      t &= 0xff;   /* Count module 256 (8 bit) */
      if (!quiet)
	printf("\rt: %4d", t);

      /**************/
      /* Layer loop */
      /**************/
      lastDecLayer = -1;
      for (nl = nLayers - 1; nl >= 0; nl--) {
	if (cont & (1 << nl)) {
	  switch (method[nl]) {
	  case H263:
	    /* Is these anything to decode in this layer at this time */
	    /* In a error free environment we should always have a pict sync
	       at this point.
	       If the buffer is empty, lets check in some bytes */
	    if (next[nl] + 3 >= end[nl])
	      CheckInBuffer(stream[nl], databuf[nl], PACKET_SIZE,
			    &next[nl], &end[nl]);
	    these[nl].h263->next_temp_ref = ShowH263TempRef(next[nl]);
	    if (these[nl].h263->next_temp_ref != t)
	      break;

	    do {
	      action = HandleH263DataJunk(these[nl].h263, &next[nl], end[nl],
					  feof(stream[nl]));

	      if (action > 0) {
		picture.w = these[nl].h263->coded_picture_width;
		picture.ws = picture.w;
		picture.h = these[nl].h263->coded_picture_height;
		picture.hs = picture.h;
		if (these[nl].h263->pb_frame) {
		  picture.y = these[nl].h263->bframe[0];
		  picture.u = these[nl].h263->bframe[1];
		  picture.v = these[nl].h263->bframe[2];
		}
		picture.y = these[nl].h263->newframe[0];
		picture.u = these[nl].h263->newframe[1];
		picture.v = these[nl].h263->newframe[2];
#ifdef VERIFY_DEC_PICTURE
		if (VerifyDecPic(method, these, nl, t) >= 0)
		  exit(13);
#endif
		break;
	      } else {
		CheckInBuffer(stream[nl], databuf[nl], PACKET_SIZE,
			      &next[nl], &end[nl]);
	      }
	    } while (1);
	    /* End condtition for layer nl */
	    if (feof(stream[nl]) && next[nl] >= end[nl])
	      cont &= ~(1 << nl);
	    if (these[nl].h263->temp_ref == (t & 0xff)) {
	      lastDecLayer = nl;
	      pic = &picture;
	    }
	    break;

	  case PYRA:
	    /* Is these anything to decode in this layer at this time */
	    /* In a error free environment we should always have a pict sync
	       at this point.
	       If the buffer is empty, lets check in some bytes */
	    if (next[nl] + 4 >= end[nl])
	      CheckInBuffer(stream[nl], databuf[nl], PACKET_SIZE,
			    &next[nl], &end[nl]);
	    these[nl].pyra->next_temp_ref = ShowPyraTempRef(next[nl]);
	    if (these[nl].pyra->next_temp_ref != t)
	      break;
	    /* We want to decode now; so lets set 'trd' for the motion vect. */
	    these[nl].pyra->trd = t - these[nl].pyra->temp_ref;
	    if (these[nl].pyra->trd < 0)
	      these[nl].pyra->trd += 256;

	    /* Init baselayer flag */
	    if (!(these[nl].pyra->baseLayerFlag = ((lastDecLayer >= 0) ? 0 : 1)))
	      /* Init spatial reference */
	      InitLayerReferences(method, these, nl, lastDecLayer);

	    do {
	      action = HandlePyraDataJunk(these[nl].pyra, &next[nl], end[nl],
					  feof(stream[nl]));

	      if (action == 1) {
#ifdef VERIFY_DEC_PICTURE
		if (VerifyDecPic(method, these, nl, t) >= 0)
		  exit(13);
#endif
		break;
	      } else { /* action == 0 means we have to fill up the data buffer */
		CheckInBuffer(stream[nl], databuf[nl], PACKET_SIZE,
			      &next[nl], &end[nl]);
	      }
	    } while (1);
	    /* End condtition for layer nl */
	    if (feof(stream[nl]) && next[nl] >= end[nl])
	      cont &= ~(1 << nl);
	    if (these[nl].pyra->temp_ref == (t & 0xff)) {
	      pic = these[nl].pyra->DecPic;
	      lastDecLayer = nl;
	    }
	    break;
	  }
	}
      }
      /*********************/
      /* End of Layer loop */
      /*********************/
      /* Display */
      WaitFrame(t, fps);
      if (lastDecLayer >= 0) {
	if (pic->ws == dispPict->ws)
	  yuv2rgb(pic);
	else {
	  if (pic->ws < dispPict->ws)
	    dSize = dispPict->ws / pic->ws;
	  else
	    dSize = pic->ws / dispPict->ws;
	  switch(dSize) {
	  case 1:  dSize = 0; break;
	  case 2:  dSize = 1; break;
	  case 4:  dSize = 2; break;
	  case 8:  dSize = 3; break;
	  case 16: dSize = 4; break;
	  case 32: dSize = 5; break;
	  case 64: dSize = 6; break;
	  default:
	    fprintf(stderr, "Can't handle size difference for display!\n");
	    exit(1);
	  }
	  if (pic->ws < dispPict->ws)
	    MultInterpolateFastPictureSub(dSize, pic, dispPict);
	  else
	    MultDownsamplePictureSub(dSize, pic, dispPict);
	  yuv2rgb(dispPict);
	}
      }

#ifdef COMPUTE_PSNR
      if (psnr) {
	if (pic->ws != dispPict->ws)
	  pic = dispPict;
	if (cont)
	  t_psnr_end = t;
	else
	  t_psnr_end = lastFrame;
	while (t_psnr <= t_psnr_end) {
	  if (origName == NULL || !ReadPicture(origName, SEP_RAW_TYPE,
					       t_psnr, origPict)) {
	    fprintf(stderr, "Can't find input sequence \"%s\"\n",
		    (origName ? origName : ""));
	    exit(-1);
	  }
	  fprintf(fp_prot, "%2.2f\n",
		  PSNRSubPictures(origPict->y, pic->y,
				  origPict->ws, origPict->hs, origPict->w));
	}
      }
#endif

    }
    /********************/
    /* End of Film loop */
    /********************/
    if (loop) {
      for (nl = nLayers - 1; nl >= 0; nl--)
	rewind(stream[nl]);
    }
  } while (loop--);
  /***********************/
  /* End of Looping loop */
  /***********************/

  /************/
  /* Clean up */
  /************/
  for (nl = 0; nl < nLayers; nl++) {
    switch (method[nl]) {
    case H263:
      DisinitH263Decoder(these[nl].h263);
      FreeH263Decoder(these[nl].h263);
      break;
    case PYRA:
      DisinitPyraDecoder(these[nl].pyra);
      FreePyraDecoder(these[nl].pyra);
      break;
    }
    free(databuf[nl]);
  }

#ifdef COMPUTE_PSNR
  if (psnr) {
    FreePicture(origPict);
  }
#endif

  return(1);
}



#ifndef WITHOUT_CHACODEC

int ShouldWeDecodeH263Now(int t, int nl, InterleaverStack *ilStack_p,
			  Byte **next_p, Byte **end_p, int *end_f_p,
			  Bitstream *bs, int *sync_p, int showPckLoss)
{
  int next_t;


  while (bs->BufState != GOB_LOADED) {
    if ((*next_p = PassToStreamBufferH263(bs, *next_p, *end_p)) == *end_p) {
      if (*end_f_p && bs->BufState == GOB_PENDING) {
	InitStreamParsingH263(bs);
	break;
      }

      if (bs->BufState == GOB_PENDING && (bs->BufferPtr - bs->ByteBuffer) > 4
	  && t != ShowH263TempRef(bs->ByteBuffer))
	return 0;

      if (bs->BufState != GOB_PENDING && LayerTooFarInFront(nl, ilStack_p))
	return 0;

      if (!GetNextInterleaverData(nl, ilStack_p, next_p, end_p, end_f_p)) {

	*sync_p = 0;
	if (showPckLoss) {
	  printf("\a");
	  printf("\n\n Switch off Layer %d\n\n",nl);
	}
	bs->BufState = GOB_EMPTY;
	return 0;
      }

      if (ilStack_p->status[ilStack_p->videoDecIL[nl]] != 0 &&
	  !(ilStack_p->status[ilStack_p->videoDecIL[nl]] & (1 << nl)) &&
	  bs->BufState == GOB_PENDING) {
	fprintf(stderr, "Need more packetblocks for depacking this stream!\n");
	exit(12);
      }
    }
  }
  if (!(*sync_p))
    return 0;

  next_t = ShowH263TempRef(bs->ByteBuffer);
  if (next_t == t)
    return 1;

  return 0;
}


int ShouldWeDecodePyraNow(int t, int nl, InterleaverStack *ilStack_p,
			  Byte **next_p, Byte **end_p, int *end_f_p,
			  Bitstream *bs, int *sync_p, int showPckLoss)
{
  int next_t;


  while (bs->BufState != GOB_LOADED) {
    if ((*next_p = PassToStreamBufferPyra(bs, *next_p, *end_p)) == *end_p) {
      if (*end_f_p && bs->BufState == GOB_PENDING) {
	InitStreamParsingPyra(bs);
	break;
      }

      if (bs->BufState == GOB_PENDING && (bs->BufferPtr - bs->ByteBuffer) > 4
	  && t != ShowPyraTempRef(bs->ByteBuffer))
	return 0;

      if (bs->BufState != GOB_PENDING && LayerTooFarInFront(nl, ilStack_p))
	return 0;

      if (!GetNextInterleaverData(nl, ilStack_p, next_p, end_p, end_f_p)) {

	*sync_p = 0;
	if (showPckLoss) {
	  printf("\a");
	  printf("\n\n Switch off Layer %d\n\n",nl);
	}
	bs->BufState = GOB_EMPTY;
	return 0;
      }

      if (ilStack_p->status[ilStack_p->videoDecIL[nl]] != 0 &&
	  !(ilStack_p->status[ilStack_p->videoDecIL[nl]] & (1 << nl)) &&
	  bs->BufState == GOB_PENDING) {
	fprintf(stderr, "Need more packetblocks for depacking this stream!\n");
	exit(12);
      }
    }
  }
  if (!(*sync_p))
    return 0;

  next_t = ShowPyraTempRef(bs->ByteBuffer);
  if (next_t == t)
    return 1;

  return 0;
}



int Resync(int t, int nl, int nLayers, int method, Global *these,
	   FILE *stream, InterleaverStack *ilS,
	   Byte **next_p, Byte **end_p, int *end_f_p,
	   int *sync, int *syncTime)
{
  int dt = 0;
  int ilParse_f;
  int switchOffAgain;


  if (syncTime[nl] >= 0) {
    if (t == syncTime[nl])
      if (nl < nLayers - 1 && method == PYRA)
	if (!sync[nl+1]) {
	  /* If the reference layer is out of sync, we can't sync */
	  syncTime[nl] = -1;
	  these[nl].pyra->bs->BufState = GOB_EMPTY;
	}
    if (t == syncTime[nl]) {
      /* We've got it! Let's go on decoding */
      sync[nl] = 1;
      syncTime[nl] = -1;
      return 1;
    } else if (syncTime[nl] >= 0)
      return 0;
  }

  if (method == PYRA && !sync[nl+1] &&
      ilS->ilNumber[ilS->videoDecIL[nl]] >ilS->ilNumber[ilS->videoDecIL[nl+1]])
    return 0;

  do {
    switchOffAgain = 0;
    ChaRecDec(stream, ilS);
    if (!GetNextInterleaverData(nl, ilS, next_p, end_p, end_f_p))
      switchOffAgain = 1;
    else {
      /* We've got one packet */
      switch (method) {
      case H263:
	syncTime[nl] = ParseH263JunkForIntra(these[nl].h263, next_p, *end_p,
					    *end_f_p);
	break;
      case PYRA:
	syncTime[nl] = ParsePyraJunkForIntra(these[nl].pyra, next_p, *end_p,
					    *end_f_p);
	break;
      }

      if (syncTime[nl] >= 0) {
	dt = t - syncTime[nl];
	if (ABS(dt) >= 128) {
	  if (dt < 0) {
	    syncTime[nl] = -1;
	    switchOffAgain = 1;
	  }
	} else
	  if (dt > 0) {
	    syncTime[nl] = -1;
	    switchOffAgain = 1;
	  }
      }
      if (syncTime[nl] >= 0) {
	if (nl == nLayers - 1) {  /* Baselayer */
	  if (dt == 0) {    /* Found a sync for the present time */
	    sync[nl] = 1;
	    syncTime[nl] = -1;
	    return 1;
	  } else
	    return 0;
	} else {
	  if (sync[nl+1]) {   /* The reference layer is in sync */
	    if (dt == 0) {    /* Found a sync for the present time */
	      sync[nl] = 1;
	      syncTime[nl] = -1;
	      return 1;
	    } else
	      return 0;
	  } else {      /* The reference layer is out of sync */
	    dt = syncTime[nl + 1] - syncTime[nl];
	    if (ABS(dt) >= 128) {
	      if (dt < 0) {
		syncTime[nl] = -1;
		switchOffAgain = 1;
	      }
	    } else 
	      if (dt > 0) {
		syncTime[nl] = -1;
		switchOffAgain = 1;
	      }
	  }

	  if (syncTime[nl] >= 0)
	    if (dt == 0) {    /* Found a sync for the present time */
	      sync[nl] = 1;
	      syncTime[nl] = -1;
	      return 1;
	    } else
	      return 0;
	}
      }
    }

    if (switchOffAgain) {
      switch (method) {
      case H263:
	these[nl].h263->bs->BufState = GOB_EMPTY;
	break;
      case PYRA:
	these[nl].pyra->bs->BufState = GOB_EMPTY;
	break;
      }
      syncTime[nl] = -1;
    }

    if (nl < nLayers - 1) {    /* Not baselayer */
      ilParse_f = (ilS->ilNumber[ilS->videoDecIL[nl]] <=
		   ilS->ilNumber[ilS->videoDecIL[nl+1]]);
    } else
      ilParse_f = 1;
    if ((ilS->status[ilS->videoDecIL[nl]] & (1 << nl)) == 0)
      ilParse_f = 0;

  } while (ilParse_f);

  return 0;
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- RunPRSDecoding -- Decode Reed-Solomon packetized stream
 *
 * Author:      K.S.
 *
 * Created:     2-Jul-98
 *
 * Purpose:     -
 *
 * Options:     -
 *
 * Description: 
 *
 * See also:    -
 *
 * Modified:    -
 *
 *****************************************************************************/
int RunPRSDecoding(FILE *stream, Picture *dispPict, int fps, int topLayer,
		   int quiet, int loop, int showPckLoss)
/***********************************************************CommentEnd********/
{
  int              first = 1;
  int              t;
  int              i;
  int              lastDecLayer;
  int              nLayers, nl;
  int              sync[NUM_LAYERS], syncTime[NUM_LAYERS];
  int              method[NUM_LAYERS];
  Global           these[NUM_LAYERS];
  Byte             *next[NUM_LAYERS], *end[NUM_LAYERS];
  int              cont0 = 0, cont = 0;
  int              action;
  int              dSize;
  Picture          picture, *pic = NULL;
  InterleaverStack ilStack; 
  int              end_f[NUM_LAYERS];
  int              packetSize;
#ifdef COMPUTE_PSNR
  int              t_psnr = firstFrame, t_psnr_end;
  Picture          *origPict = NULL;
  if (psnr) {
    origPict = ClonePicture(dispPict);
  }
#endif


  /************************/
  /* Init channel decoder */
  /************************/
  /* Create GF(2^l): initialize N, LOG, EXP, GFMULTAB and stack. */
  GFInit(8);   /* 8 == bits per byte */
  /****************/
  /* Looping loop */
  /****************/
  do {
    /* Read file header */
    ReadPrsFileHeader(stream, &packetSize, &(ilStack.nIlmax),
		      &(ilStack.k_il_info), &(ilStack.pckMode), &nLayers,
		      method);

    for (i = 0; i < NUM_IL; i++) {
      if (first) {
	ilStack.il[i] = (Byte *)malloc(packetSize*ilStack.nIlmax*sizeof(Byte));
	ilStack.pckRec[i] = (int *)malloc(ilStack.nIlmax * sizeof(int));
      }
      memset(ilStack.pckRec[i], 0, ilStack.nIlmax * sizeof(int));
      for (nl = 0; nl < NUM_LAYERS; nl++) {
	if (first) {
	  ilStack.pckValid[i][nl] = (int *)malloc(ilStack.nIlmax *sizeof(int));
	}
	memset(ilStack.pckValid[i][nl], 0, ilStack.nIlmax * sizeof(int));
      }
      ilStack.ilNumber[i] = i;
      ilStack.status[i] = 0;
      ilStack.nRecPck[i] = 0;
      ilStack.bytesPerPck[i] = packetSize;
      ilStack.numLayer[i] = nLayers;
    }
    ChaRecDec(stream, &ilStack);

    /*******************/
    /* Initializations */
    /*******************/
    for (nl = 0; nl < nLayers; nl++) {
      cont0 |= (1 << nl);
      cont = cont0;
      sync[nl] = 1;
      syncTime[nl] = -1;

      ilStack.videoDecIL[nl] = 0;
      ilStack.videoDecLayerPck[nl] = -1;

      next[nl] = NULL;
      end[nl] = NULL;
      end_f[nl] = 0;

      if (first) {
	/* Decide on method by looking at the first sync */
	switch (method[nl]) {
	case PYRA:
	  these[nl].pyra = NewPyraDecoder();
	  these[nl].pyra->trd = 1;
	  break;
	case H263:
	  these[nl].h263 = NewH263Decoder();
	  these[nl].h263->quiet = quiet;
	  these[nl].h263->trd = 1;
	  break;
	default:
	  fprintf(stderr, "Layer no. %d in interleaver contains undecodable "
		  "data\n", nl);
	  return(0);
	}
      }
    }
    first = 0;

    /*************/
    /* Film loop */
    /*************/
    for (t = 0; cont; t++) {
      t &= 0xff;   /* Count module 256 (8 bit) */
      if (!quiet)
	printf("\rt: %4d", t);

      /**************/
      /* Layer loop */
      /**************/
      lastDecLayer = -1;
      for (nl = nLayers - 1; nl >= topLayer; nl--) {
	if (cont & (1 << nl)) {

	  /* Check the sockets */
	  ChaRecDec(stream, &ilStack);

	  /* Switch of the layer if the reference layer is out of sync */
	  if (nl != nLayers - 1 && method[nl] != H263)
	    /* Switch of the layer if the layer below is out of sync */
	    if (!sync[nl + 1]) {
	      if (sync[nl])
		switch (method[nl]) {
		case H263:
		  these[nl].h263->bs->BufState = GOB_EMPTY;
		  break;
		case PYRA:
		  these[nl].pyra->bs->BufState = GOB_EMPTY;
		  break;
		}
	      sync[nl] = 0;
	    }


	  /* Resync if necessary */
	  if (!sync[nl])
	    if (!Resync(t, nl, nLayers, method[nl], these, stream, &ilStack,
			next + nl, end + nl, end_f + nl, sync, syncTime))
	      continue;

	  switch (method[nl]) {
	  case H263:

	    /* Is these anything to decode in this layer at this time */
	    /* In a error free environment we should always have a pict sync
	       at this point.
	       If the buffer is empty, lets check in some bytes */
	    if (!ShouldWeDecodeH263Now(t,nl, &ilStack, next+nl, end+nl, end_f+nl,
				       these[nl].h263->bs, sync+nl, showPckLoss))
	      break;

	    /*************************/
	    /* Picture decoding loop */
	    /*************************/
	    do {
	      ChaRecDec(stream, &ilStack);
	      action = HandleH263DataJunk(these[nl].h263, &next[nl], end[nl],
					  end_f[nl]);

	      if (action > 0) {
		picture.w = these[nl].h263->coded_picture_width;
		picture.ws = picture.w;
		picture.h = these[nl].h263->coded_picture_height;
		picture.hs = picture.h;
		if (these[nl].h263->pb_frame) {
		  picture.y = these[nl].h263->bframe[0];
		  picture.u = these[nl].h263->bframe[1];
		  picture.v = these[nl].h263->bframe[2];
		}
		picture.y = these[nl].h263->newframe[0];
		picture.u = these[nl].h263->newframe[1];
		picture.v = these[nl].h263->newframe[2];
#ifdef VERIFY_DEC_PICTURE
		if (VerifyDecPic(method, these, nl, t) >= 0)
		  exit(13);
#endif
		break;
	      } else {
		if (next[nl] == end[nl])
		  if (!GetNextInterleaverData(nl, &ilStack, &next[nl], &end[nl],
					      &end_f[nl])) {
		    /* Packet loss */
		    sync[nl] = 0;
		    if (showPckLoss) {
		      printf("\a");
		      printf("\n\n Switch off Layer %d\n\n",nl);
		    }
		    these[nl].h263->bs->BufState = GOB_EMPTY;
		    H263FinishPicture(these[nl].h263);
		    break;
		  }
	      }
	    } while (1);
	    if (these[nl].h263->temp_ref == (t & 0xff)) {
	      lastDecLayer = nl;
	      pic = &picture;
	    }
	    break;

	  case PYRA:

	    /* Is these anything to decode in this layer at this time */
	    /* In a error free environment we should always have a pict sync
	       at this point.
	       If the buffer is empty, lets check in some bytes */
	    if (!ShouldWeDecodePyraNow(t,nl, &ilStack, next+nl, end+nl, end_f+nl,
				       these[nl].pyra->bs, sync+nl, showPckLoss))
	      break;

	    /* We want to decode now; so lets set 'trd' for the motion vect. */
	    these[nl].pyra->trd = t - these[nl].pyra->temp_ref;
	    if (these[nl].pyra->trd < 0)
	      these[nl].pyra->trd += 256;

	    /* Init baselayer flag */
	    if (!(these[nl].pyra->baseLayerFlag = ((lastDecLayer >= 0) ? 0 : 1)))
	      /* Init spatial reference */
	      InitLayerReferences(method, these, nl, lastDecLayer);

	    /*************************/
	    /* Picture decoding loop */
	    /*************************/
	    do {
	      ChaRecDec(stream, &ilStack);
	      action = HandlePyraDataJunk(these[nl].pyra, &next[nl], end[nl],
					  end_f[nl]);

	      if (action == 1) {

#ifdef VERIFY_DEC_PICTURE
		if (VerifyDecPic(method, these, nl, t) >= 0)
		  exit(13);
#endif
		break;
	      } else { /* action == 0 means we have to fill up the data buffer */
		if (next[nl] == end[nl])
		  if (!GetNextInterleaverData(nl, &ilStack, &next[nl],
					      &end[nl], &end_f[nl])) {
		    /* Packet loss */
		    sync[nl] = 0;
		    if (showPckLoss) {
		      printf("\a");
		      printf("\n\n Switch off Layer %d\n\n",nl);
		    }
		    these[nl].pyra->bs->BufState = GOB_EMPTY;
		    PyraFinishPicture(these[nl].pyra);
		    break;
		  }
	      }
	    } while (1);
	    if (these[nl].pyra->temp_ref == (t & 0xff)) {
	      pic = these[nl].pyra->DecPic;
	      lastDecLayer = nl;
	    }
	    break;
	  }
	}
      }
      /*********************/
      /* End of Layer loop */
      /*********************/
      /* Display */
      WaitFrame(t, fps);
      if (lastDecLayer >= 0) {
	if (pic->ws == dispPict->ws)
	  yuv2rgb(pic);
	else {
	  if (pic->ws < dispPict->ws)
	    dSize = dispPict->ws / pic->ws;
	  else
	    dSize = pic->ws / dispPict->ws;
	  switch(dSize) {
	  case 1:  dSize = 0; break;
	  case 2:  dSize = 1; break;
	  case 4:  dSize = 2; break;
	  case 8:  dSize = 3; break;
	  case 16: dSize = 4; break;
	  case 32: dSize = 5; break;
	  case 64: dSize = 6; break;
	  default:
	    fprintf(stderr, "Can't handle size difference for display!\n");
	    exit(1);
	  }
	  if (pic->ws < dispPict->ws)
	    MultInterpolateFastPictureSub(dSize, pic, dispPict);
	  else
	    MultDownsamplePictureSub(dSize, pic, dispPict);
	  yuv2rgb(dispPict);
	}
      }

      /* Reset status in ilStack for not decoded layers */
      for (nl = 0; nl < topLayer; nl++)
	for (i = 0; i < NUM_IL; i++)
	  ilStack.status[i] &= ~(1 << nl);

      /* Test End of File */
      if (feof(stream)) {
	cont = 0;
	for (i = 0; i < NUM_IL; i++)
	  if (ilStack.status[i] != 0) {
	    cont = cont0;
	    break;
	  }
      }

#ifdef COMPUTE_PSNR
      if (psnr) {
	if (pic->ws != dispPict->ws)
	  pic = dispPict;
	if (cont)
	  t_psnr_end = t;
	else
	  t_psnr_end = lastFrame;
	while (t_psnr <= t_psnr_end) {
	  if (origName == NULL || !ReadPicture(origName, SEP_RAW_TYPE,
					       t_psnr, origPict)) {
	    fprintf(stderr, "Can't find input sequence \"%s\"\n",
		    (origName ? origName : ""));
	    exit(-1);
	  }
	  fprintf(fp_prot, "%2.2f\n",
		  PSNRSubPictures(origPict->y, pic->y,
				  origPict->ws, origPict->hs, origPict->w));
	  t_psnr++;
	}
      }
#endif

    }
    /********************/
    /* End of Film loop */
    /********************/

    if (loop)
      rewind(stream);
  } while (loop--);
  /***********************/
  /* End of Looping loop */
  /***********************/

  /************/
  /* Clean up */
  /************/
  for (nl = 0; nl < nLayers; nl++) {
    switch (method[nl]) {
    case H263:
      if (!these[nl].h263->first)
	DisinitH263Decoder(these[nl].h263);
      FreeH263Decoder(these[nl].h263);
      break;
    case PYRA:
      if (!these[nl].pyra->first)
	DisinitPyraDecoder(these[nl].pyra);
      FreePyraDecoder(these[nl].pyra);
      break;
    }
  }
  GFFree();
  for (i = 0; i < NUM_IL; i++) {
    free(ilStack.il[i]);
    free(ilStack.pckRec[i]);
    for (nl = 0; nl < NUM_LAYERS; nl++)
      free(ilStack.pckValid[i][nl]);
  }

#ifdef COMPUTE_PSNR
  if (psnr) {
    FreePicture(origPict);
  }
#endif

  return(1);
}
#endif



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PDecoder -- Scalable pyramid decoder (ScalVico)
 *
 * Author:      K.S.
 *	        
 * Created:     3-Feb-98
 *	        
 * Purpose:     -
 * 	        
 * Options:     -pS <num> packet size
 *                  [1024]
 *              -pL <num> <num> <num> <num> percentage packet loss
 *                  [0] [0] [0] [0]
 *              -dF turn off the display
 *              -dS <num> display size
 *                  SQCIF: 1
 *                  QCIF : 2
 *                  CIF  : 3
 *                  CIF4 : 4
 *                  CIF16: 5
 *              -fps <num> frames per second (0: as fast as possible).
 *	        -quiet run quietly
 *
 *	        Default options in square brackets are specified in main.h
 *
 *
 * Description: Call:
 *                      PDecoder <options> bitstream0 bitstream1 ...
 *
 * See also:    -
 *
 * Modified:    -
 *
 *****************************************************************************/
void main(int argc, char *argv[])
/***********************************************************CommentEnd********/
{
  DecInfo        *dInfo = NewDecInfo();
  int nLayers;
  FILE *stream[NUM_LAYERS];
  Picture *dispPict;




  Options = Init_Options; 	/* struct copy */
  Options.argv0 = argv[0];

#if 0
  while(*argv && **argv == '-' && argv[0][1]) 
    {
      char *o = *argv++;
      int v = strncmp("no", o+1, 2) ? 1 : 0;

      o += v ? 1 : 3;
           if (!strcmp(o, "gray"))	Options.visgray = v;
      else if (!strcmp(o, "grey"))	Options.visgray = v;
      else if (!strcmp(o, "pseudo"))	Options.vis8 = v;
      else if (!strcmp(o, "vis"))	Options.use_vis = v;
      else if (!strcmp(o, "dga"))	Options.use_dga = v;
      else if (!strcmp(o, "shm"))	Options.use_shm = v;
      else if (!strcmp(o, "shmem"))	Options.use_shm = v;
      else if (!strcmp(o, "display"))	Options.display = v;
      else if (!strcmp(o, "thread"))	Options.thread_yuv = v;
      else if (!strcmp(o, "threads"))	Options.thread_yuv = v;
      else if (!strcmp(o, "expand"))	Options.visexpand = v;
      else if (!strcmp(o, "audio"))	Options.audio = v;
      else if (!strcmp(o, "qcif"))	Options.qcif = v;
      else if (!strcmp(o, "cif"))	Options.qcif = !v;
      else if (!strcmp(o, "fps"))	Options.fps = atoi(*argv++);
      else if (!strcmp(o, "quiet"))	quiet = 1;
      else if (!strcmp(o, "window"))	
        Options.cover_window = strtol(*argv++, NULL, 0);
      else if (!strcmp(o, "windowid"))	
        Options.cover_window = strtol(*argv++, NULL, 0);
      else if (!strcmp(o, "id"))	
        Options.cover_window = strtol(*argv++, NULL, 0);
/*    else if (!strcmp(o, "loop"))	Options.loop = v;	*/
      else 
        ExitWithUsage("Unknown commandline option.", 1);
    }
#endif

  /* Evalute command line arguments and/or print usage */
  argv = EvaluateArgumentsDecInfo(argc, argv, dInfo);
  VerboseUsage(argc, argv, dInfo);

  /* Set options struct */
  Options.cover_window = dInfo->windowid;
  Options.fps = dInfo->fps;

  /* Print a short hello message */
  if (!dInfo->quiet) {
    printf("\n\n                  ScalVico\n\n");
    printf("    Uwe Horn, Jrgen Weigert, Klaus Stuhlmller\n\n\n");
  }

  if (!Options.fps && !dInfo->quiet)
    {
      if (Options.audio)
	fprintf(stderr, "Audio disabled, -fps missing.\n");
      Options.audio = 0;
    }

  /* Open input files */
  if (!*argv)
    ExitWithUsage("Name of input file is missing.", 0);
  for (nLayers = 0; nLayers < NUM_LAYERS && *argv; nLayers++, argv++) {
    if (!strcmp(*argv, "-"))
      stream[nLayers] = stdin;    /* Read stream from standard input */
    else {
#ifdef LINUX
      if (!(stream[nLayers] = fopen(argv[0], "r")))
	fprintf(stderr, "%s: ", *argv),perror("fopen"),exit(0);
#else
      struct url_info u;
      int fd = OpenUrl(*argv, &u, O_RDONLY);

      if (fd < 0 || !(stream[nLayers] = fdopen(fd, "r"))) {
	perror(*argv);
	exit(1);
      }

      if (!dInfo->quiet)
	fprintf(stderr, "mime/type: %s, length=%d, mtime='%s'\n",
		u.mime->buf, u.len, u.modified ? u.modified->buf : "<>");
      FreeUrlInfo(&u);
#endif
    }
  }

  /*******************/
  /* Initializations */
  /*******************/
  SetupPyraTables();
  /* init_display */
  switch (dInfo->displaySize) {
  case QCIF:
    Options.qcif = 1;
    init_display(QCIF_WIDTH, QCIF_HEIGHT);
    dispPict = AllocPicture(QCIF_WIDTH, QCIF_HEIGHT);
    break;
  case CIF:
    init_display(CIF_WIDTH, CIF_HEIGHT);
    dispPict = AllocPicture(CIF_WIDTH, CIF_HEIGHT);
    break;
  case CIF4:
    init_display(CIF4_WIDTH, CIF4_HEIGHT);
    dispPict = AllocPicture(CIF4_WIDTH, CIF4_HEIGHT);
    break;
  case CIF16:
    init_display(CIF16_WIDTH, CIF16_HEIGHT);
    dispPict = AllocPicture(CIF16_WIDTH, CIF16_HEIGHT);
    break;
  default:
    fprintf(stderr, "PDecoder: display size >%d< not supported\n",
	    dInfo->displaySize);
    exit(1);
  }
  InitFilters(MAX(CIF_WIDTH,dispPict->w), MAX(CIF_HEIGHT,dispPict->h));

  /*********/
  /* Do it */
  /*********/
#ifndef WITHOUT_CHACODEC
  if (dInfo->prs) {
    if (nLayers != 1) {
      fprintf(stderr, "Need exactly one stream file for Reed-Solomon "
	      "packed stream decoding.\n");
      exit(1);
    }
    RunPRSDecoding(stream[0], dispPict, dInfo->fps, dInfo->topLayer,
		   dInfo->quiet, dInfo->loop, dInfo->showPckLoss);
  } else
#endif
    RunFileDecoding(nLayers - dInfo->topLayer, stream + dInfo->topLayer,
		    dispPict, dInfo->fps, dInfo->quiet, dInfo->loop);

  
  /************/
  /* Clean up */
  /************/
#ifdef COMPUTE_PSNR
  if (psnr && fp_prot != stdout) {
    fclose(fp_prot);
  }
#endif

  DisinitFilters();
  FreePicture(dispPict);
  for (nLayers--; nLayers >= 0; nLayers--) {
    fclose(stream[nLayers]);
  }
  sleep(1);
  exit_display();
  if (!dInfo->quiet) {
    printf("\n\n                *************************\n");
    printf("                *        The End        *\n");
    printf("                *************************\n\n\n");
  }
  return;
}

