#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include "pyradecoder.h"
#include "common.h"
#include "filter.p"


void SetupPyraTables()
{
  SetupExtVectTables();
  InitInterpolator();
}


PyraGlobal *NewPyraDecoder() {
  PyraGlobal *pyraData;

  pyraData = NewPyraDecoderNoBuffer();
  InitStreamBuffer(pyraData->bs, 1);

  return(pyraData);
}


PyraGlobal *NewPyraDecoderNoBuffer() {
  PyraGlobal *pyraData = (PyraGlobal *) malloc(sizeof(PyraGlobal));


  pyraData->ptype = 0;

  pyraData->mvf.mode = NULL;
  pyraData->mvf.mx = NULL;
  pyraData->mvf.my = NULL;
  pyraData->DecPic = NULL;
  pyraData->TempRefPic = NULL;
  pyraData->SpatRefPic = (Picture *)malloc(sizeof(Picture));
  pyraData->bs = (Bitstream *) malloc(sizeof(Bitstream));
  InitStreamBuffer(pyraData->bs, 0);

  pyraData->first = 1;

  pyraData->awaitIntra = 1;
  pyraData->skip_until_next_tr = 0;

  pyraData->decGOBsCounter = 0;

  pyraData->GOBs_in_pict = 100000;  /* Just an utopic high number */

  pyraData->BlockPyra[3] = AllocPicture(2,2);
  pyraData->BlockPyra[2] = AllocPicture(4,4);
  pyraData->BlockPyra[1] = AllocPicture(8,8);
  pyraData->BlockPyra[0] = AllocPicture(16,16);

  return(pyraData);
}


void FreePyraDecoder(PyraGlobal *pyraData)
{
  free(pyraData->SpatRefPic);

  FreeStreamBuffer(pyraData->bs);
  free(pyraData->bs);

  free(pyraData);
}


void InitPyraDecoder(PyraGlobal *pyraData)
{
  int size;


  pyraData->first = 0;

  switch (pyraData->PicSize) {
  case SQCIF:
    pyraData->width = SQCIF_WIDTH;
    pyraData->height = SQCIF_HEIGHT;
    pyraData->widthSub = SQCIF_WIDTH;
    pyraData->heightSub = SQCIF_HEIGHT;
    pyraData->MB_rows_in_GOB = 1;
    pyraData->MBs_in_row = SQCIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->MBs_in_GOB = SQCIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->GOBs_in_pict = SQCIF_HEIGHT / MACROBLOCK_SIZE;
    break;
  case QCIF:
    pyraData->width = QCIF_WIDTH;
    pyraData->height = QCIF_HEIGHT;
    pyraData->widthSub = QCIF_WIDTH;
    pyraData->heightSub = QCIF_HEIGHT;
    pyraData->MB_rows_in_GOB = 1;
    pyraData->MBs_in_row = QCIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->MBs_in_GOB = QCIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->GOBs_in_pict = QCIF_HEIGHT / MACROBLOCK_SIZE;
    break;
  case CIF:
    pyraData->width = CIF_WIDTH;
    pyraData->height = CIF_HEIGHT;
    pyraData->widthSub = CIF_WIDTH;
    pyraData->heightSub = CIF_HEIGHT;
    pyraData->MB_rows_in_GOB = 1;
    pyraData->MBs_in_row = CIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->MBs_in_GOB = CIF_WIDTH / MACROBLOCK_SIZE;
    pyraData->GOBs_in_pict = CIF_HEIGHT / MACROBLOCK_SIZE;
    break;
  case CIF4:
    pyraData->width = CIF4_WIDTH;
    pyraData->height = CIF4_HEIGHT;
    pyraData->widthSub = CIF4_WIDTH;
    pyraData->heightSub = CIF4_HEIGHT;
    pyraData->MB_rows_in_GOB = 2;
    pyraData->MBs_in_row = CIF4_WIDTH / MACROBLOCK_SIZE;
    pyraData->MBs_in_GOB = CIF4_WIDTH / MACROBLOCK_SIZE * 2;
    pyraData->GOBs_in_pict = CIF4_HEIGHT / MACROBLOCK_SIZE / 2;
    break;
  case CIF16:
    pyraData->width = CIF16_WIDTH;
    pyraData->height = CIF16_HEIGHT;
    pyraData->widthSub = CIF16_WIDTH;
    pyraData->heightSub = CIF16_HEIGHT;
    pyraData->MB_rows_in_GOB = 4;
    pyraData->MBs_in_row = CIF16_WIDTH / MACROBLOCK_SIZE;
    pyraData->MBs_in_GOB = CIF16_WIDTH / MACROBLOCK_SIZE * 4;
    pyraData->GOBs_in_pict = CIF16_HEIGHT / MACROBLOCK_SIZE / 4;
    break;
  case CUSTOM_SOURCE_FORMAT:
    pyraData->MB_rows_in_GOB = 1;
    pyraData->MBs_in_row = pyraData->width / 16;
    pyraData->MBs_in_GOB = pyraData->width / 16;
    pyraData->GOBs_in_pict = pyraData->height / 16;
    if (pyraData->height > 800) {
      pyraData->MB_rows_in_GOB = 4;
      pyraData->MBs_in_GOB *= 4;
      pyraData->GOBs_in_pict /= 4;
    } else if (pyraData->height > 400) {
      pyraData->MB_rows_in_GOB = 2;
      pyraData->MBs_in_GOB *= 2;
      pyraData->GOBs_in_pict /= 2;
    }
    break;
  default:
    printf("ERROR: Illegal input format\n");
    exit(-1);
    break;
  }

  /* Allocate the motion vector field */
  pyraData->mvf.w = pyraData->width / MACROBLOCK_SIZE;
  pyraData->mvf.h = pyraData->height / MACROBLOCK_SIZE;
  size = pyraData->mvf.w * pyraData->mvf.h * sizeof(short);
  pyraData->mvf.mode = (short *)malloc(size);
  pyraData->mvf.mx = (short *)malloc(size);
  pyraData->mvf.my = (short *)malloc(size);

  pyraData->DecPic = AllocPicture(pyraData->width, pyraData->height);
  pyraData->DecPic->ws = pyraData->widthSub;
  pyraData->DecPic->hs = pyraData->heightSub;
  pyraData->TempRefPic = CloneResetPicture(pyraData->DecPic);

  /* Init size difference to spatial reference layer */
  if (!pyraData->baseLayerFlag)
    switch(pyraData->widthSub / pyraData->SpatRefPic->ws) {
    case 1:  pyraData->dSize = 0; break;
    case 2:  pyraData->dSize = 1; break;
    case 4:  pyraData->dSize = 2; break;
    case 8:  pyraData->dSize = 3; break;
    case 16: pyraData->dSize = 4; break;
    case 32: pyraData->dSize = 5; break;
    case 64: pyraData->dSize = 6; break;
    default:
      fprintf(stderr, "Only 1,2,4,8,16,32 and 64 are allowed size division "
	      "factors! (%d)\n",pyraData->widthSub/pyraData->SpatRefPic->ws);
      exit(1);
    }
  else
    pyraData->dSize = -1;
	assert(pyraData->GOBs_in_pict<19);
}

void DisinitPyraDecoder(PyraGlobal *pyraData)
{
    if (!pyraData->first) {
        pyraData->first = 1;

        /* Free the motion vector field */
        free(pyraData->mvf.mode);
        free(pyraData->mvf.mx);
        free(pyraData->mvf.my);

        FreePicture(pyraData->DecPic);
        FreePicture(pyraData->TempRefPic);
    }
}


void PyraFinishPicture(PyraGlobal *pyraData)
{
  int gn;

  if (!pyraData->first) {
    if (pyraData->decGOBsCounter == 0)
      MultInterpolateFastPictureSub(pyraData->dSize,
				    pyraData->SpatRefPic, pyraData->DecPic);
    else {
      for (gn = 0; gn < pyraData->GOBs_in_pict; gn++)
	if (!pyraData->decGOBs[gn]) {
	  pyraData->GOBNumber = gn;
	  PyraConcealGOB(pyraData);
	}
      memset(pyraData->decGOBs, 0, MAX_GOBS * sizeof(unsigned char));
      pyraData->decGOBsCounter = 0;
    }
  }
}
