#include <stdlib.h>
#include <string.h>

#include "Util.h"
#include "pyradecoder.h"

#include "common.h"
#include "filter.p"
#include "Codebooks.p"
#include "HuffDecode.p"

#include "h263decoder.h"
#include "h263decoder.p"


#define CSCALING (ROUND(CSCALE*scale))




void InitBlockDecoder(PyraGlobal *pyraData, int w) {
  int i, j, n;

  pyraData->Yrowbytes = w;
  pyraData->Crowbytes = w/2;

  for (j=0; j<2; j++)
    for (i=0; i<4; i++)
      pyraData->YRowOffset[(j*4)+i] = j+(i*w);

  for (j=0; j<2; j++)
    for (i=0; i<4; i++)
      pyraData->CRowOffset[(j*4)+i] = j+(i*(w/2));

  n = 0;
  for (j=0; j<16; j+=8) {
    for (i=0; i<16; i+= 8) {
      pyraData->Y4x4BlockOffset[n++] = j    *w+i;
      pyraData->Y4x4BlockOffset[n++] = j    *w+i+4;
      pyraData->Y4x4BlockOffset[n++] = (j+4)*w+i;
      pyraData->Y4x4BlockOffset[n++] = (j+4)*w+i+4;
    }
  }

  n = 0;
  for (j=0; j<8; j+=4) {
    for (i=0; i<8; i+= 4) {
      pyraData->C4x4BlockOffset[n++] = j*pyraData->Crowbytes+i;
    }
  }
}

static void ReconstructVector(PyraGlobal *pyraData, Byte *dec, float scale,
			      float *v) {

  int *offset = pyraData->CurrRowOffset;

  dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[0])));
  dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[1])));
  dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[2])));
  dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[3])));
  dec++;
  dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[4])));
  dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[5])));
  dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[6])));
  dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[7])));
}

static void ReconstructTransformedVector(PyraGlobal *pyraData, Byte *dec,
					 float scale, float *v, int sy) {
  int *offset = pyraData->CurrRowOffset;

  switch (sy) {
  case 0:
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[4])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[5])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[6])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[7])));
    dec++;
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[0])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[1])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[2])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[3])));
    break;
  case 1:
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[3])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[2])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[1])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[0])));
    dec++;
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[7])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[6])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[5])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[4])));
    break;
  case 2:
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[7])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[6])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[5])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[4])));
    dec++;
    dec[0] = ROUNDCROPBYTE(dec[0] + (scale*(v[3])));
    dec[offset[1]] = ROUNDCROPBYTE(dec[offset[1]] + (scale*(v[2])));
    dec[offset[2]] = ROUNDCROPBYTE(dec[offset[2]] + (scale*(v[1])));
    dec[offset[3]] = ROUNDCROPBYTE(dec[offset[3]] + (scale*(v[0])));
    break;
  }
}

/* ========================================================================= */
/* Shell 1: */
/* ========================================================================= */
static void DecodeShell1Vector(PyraGlobal *pyraData, Byte *dec, int scale) {
  int Index = pyraData->GetShell1HuffCode(pyraData->bs);
  if (GetOneBit(pyraData->bs) == 1) scale = -scale;
  ReconstructVector(pyraData, dec, scale, pyraData->Shell1Codebook + 8*Index);
}

static void ParseShell1Vector(PyraGlobal *pyraData) {
  pyraData->GetShell1HuffCode(pyraData->bs);
  GetOneBit(pyraData->bs);
}

/* ========================================================================= 
 * Shell 2: 
 * Interpretation of bits: b3 b2 b1 b0
 * b3: sign, negative if zero, positive otherwise
 * b2 b1 b0: 3 bit offset -or-
 * b0: flip, b2 b1: 2 bit offset
 * ========================================================================= */

static int Shell2GroupSym[800] = {
2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,1,1,1,1,1,1,1,1,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
3,3,3,3,3,3,3,3,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,2,2,2,2,1,1,1,1,3,3,3,3,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,
1,1,1,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,
2,2,2,2,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,
1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,3,3,3,3,1,1,1,1,1,1,1,1,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,3,3,3,3,2,2,2,2,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,3,3,3,3,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,3,3,3,3,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,
2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,3,3,3,3,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,1,1,1,1,
1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};

static void DecodeShell2Vector(PyraGlobal *pyraData, Byte *dec, int scale) {
  int sign, trans, sy, Index;

  Index = pyraData->GetShell2HuffCode(pyraData->bs);
  sign = GetOneBit(pyraData->bs);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(pyraData->bs);
  else trans = 0;

  if (sign==1) scale = -scale;

  if (trans==1) {
    ReconstructTransformedVector(pyraData, dec, scale,
				 pyraData->Shell2Codebook + 8*Index,sy-1);
  } else {
    ReconstructVector(pyraData, dec, scale, pyraData->Shell2Codebook + 8*Index);
  }
}

static void ParseShell2Vector(PyraGlobal *pyraData) {
  int sign, trans, sy, Index;

  Index = pyraData->GetShell2HuffCode(pyraData->bs);
  sign = GetOneBit(pyraData->bs);

  sy = Shell2GroupSym[Index];
  if (sy != 0)
    trans = GetOneBit(pyraData->bs);
  else trans = 0;
}

/* ========================================================================= */
/* Decoding of 2 vectors */
/* ========================================================================= */

static void Decode2Vectors (PyraGlobal *pyraData, Byte *z, int scale) {

  float v[8];
  int mode;

  mode = pyraData->GetShellCombHuffCode(pyraData->bs) + 1; 
  
  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1Vector(pyraData, z, scale);
    break;
  case 2: 
    DecodeShell2Vector(pyraData, z, scale);
    break;
  case 3:
    DecodeExteriorVector(pyraData, v);
    ReconstructVector(pyraData, z, scale, v);
    break;
  }
  switch (mode & 3) {
  case 0:
    break;
  case 1:
    DecodeShell1Vector(pyraData, z+2, scale);
    break;
  case 2: 
    DecodeShell2Vector(pyraData, z+2, scale);
    break;
  case 3:
    DecodeExteriorVector(pyraData, v);
    ReconstructVector(pyraData, z+2, scale, v);
    break;
  }
}

static void Parse2Vectors (PyraGlobal *pyraData) {

  int mode;

  mode = pyraData->GetShellCombHuffCode(pyraData->bs) + 1; 
  
  switch ((mode >> 2) & 3) {
  case 0:
    break;
  case 1:
    ParseShell1Vector(pyraData);
    break;
  case 2: 
    ParseShell2Vector(pyraData);
    break;
  case 3:
    ParseExteriorVector(pyraData);
    break;
  }
  switch (mode & 3) {
  case 0:
    break;
  case 1:
    ParseShell1Vector(pyraData);
    break;
  case 2: 
    ParseShell2Vector(pyraData);
    break;
  case 3:
    ParseExteriorVector(pyraData);
    break;
  }
}

/* ========================================================================= */
/* Decoding of different Macroblock types */
/* ========================================================================= */

static void SetupBlockDecoder(PyraGlobal *pyraData, int coding_mode) {

  switch (coding_mode) {
  case 0:
    pyraData->Shell1Codebook = Shell1SPYCodebook;
    pyraData->Shell2Codebook = Shell2SPYCodebook;
    pyraData->GetClassHuffCode = GetSPYClassHuffCode;
    pyraData->GetCBPHuffCode = GetIntraCBPYHuffCode;
    pyraData->GetShellCombHuffCode = GetIntraShellCombYHuffCode;
    pyraData->GetShell1HuffCode = GetShell1SPYHuffCode;
    pyraData->GetShell2HuffCode = GetShell2SPYHuffCode;
    pyraData->GetClassHuffCode = GetSPYClassHuffCode;
    pyraData->CurrRowOffset = pyraData->YRowOffset;
    break;
  case 1:
    pyraData->Shell1Codebook = Shell1SPCCodebook;
    pyraData->Shell2Codebook = Shell2SPCCodebook;
    pyraData->GetClassHuffCode = GetSPCClassHuffCode;
    pyraData->GetCBPHuffCode = GetIntraCBPCHuffCode;
    pyraData->GetShellCombHuffCode = GetIntraShellCombCHuffCode;
    pyraData->GetShell1HuffCode = GetShell1SPCHuffCode;
    pyraData->GetShell2HuffCode = GetShell2SPCHuffCode;
    pyraData->GetClassHuffCode = GetSPCClassHuffCode;
    pyraData->CurrRowOffset = pyraData->CRowOffset;
    break;
  case 2:
    pyraData->Shell1Codebook = Shell1TPYCodebook;
    pyraData->Shell2Codebook = Shell2TPYCodebook;
    pyraData->GetClassHuffCode = GetTPYClassHuffCode;
    pyraData->GetCBPHuffCode = GetInterCBPYHuffCode;
    pyraData->GetShellCombHuffCode = GetInterShellCombYHuffCode;
    pyraData->GetShell1HuffCode = GetShell1TPYHuffCode;
    pyraData->GetShell2HuffCode = GetShell2TPYHuffCode;
    pyraData->GetClassHuffCode = GetTPYClassHuffCode;
    pyraData->CurrRowOffset = pyraData->YRowOffset;
    break;
  case 3:
    pyraData->Shell1Codebook = Shell1TPCCodebook;
    pyraData->Shell2Codebook = Shell2TPCCodebook;
    pyraData->GetClassHuffCode = GetTPCClassHuffCode;
    pyraData->GetCBPHuffCode = GetInterCBPCHuffCode;
    pyraData->GetShellCombHuffCode = GetInterShellCombCHuffCode;
    pyraData->GetShell1HuffCode = GetShell1TPCHuffCode;
    pyraData->GetShell2HuffCode = GetShell2TPCHuffCode;
    pyraData->GetClassHuffCode = GetTPCClassHuffCode;
    pyraData->CurrRowOffset = pyraData->CRowOffset;
    break;
  }
}


static void PyraDecode16x16MB(PyraGlobal *pyraData,
			 Byte *ydec, Byte *udec, Byte *vdec,
			 int scale, int mode)
{

  int ccp, cbpy1, cbpy2, cbpc;

  if (mode == 0)
    ccp = GetIntraCCPHuffCode(pyraData->bs) + 1;
  else
    /* this is somewhat misleading */
    ccp = GetInterCCPHuffCode(pyraData->bs) + 1; 

  if ((ccp&4)||(ccp&2))
    SetupBlockDecoder(pyraData, mode<<1);

  if (ccp&4) {
  
    cbpy1 = pyraData->GetCBPHuffCode(pyraData->bs) + 1;
  
    if ( cbpy1 & 128 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[0], scale);

    if ( cbpy1 & 64 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[1], scale);

    if ( cbpy1 & 32 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[2], scale);

    if ( cbpy1 & 16 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[3], scale);

    if ( cbpy1 & 8 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[4], scale);

    if ( cbpy1 & 4 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[5], scale);

    if ( cbpy1 & 2 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[6], scale);

    if ( cbpy1 & 1 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[7], scale);

  }

  if (ccp & 2) {
    cbpy2 = pyraData->GetCBPHuffCode(pyraData->bs) + 1;

    if ( cbpy2 & 128 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[8], scale);

    if ( cbpy2 & 64 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[9], scale);

    if ( cbpy2 & 32 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[10], scale);

    if ( cbpy2 & 16 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[11], scale);

    if ( cbpy2 & 8 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[12], scale);

    if ( cbpy2 & 4 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[13], scale);

    if ( cbpy2 & 2 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[14], scale);

    if ( cbpy2 & 1 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[15], scale);

  }

  if (ccp & 1) {

    SetupBlockDecoder(pyraData, (mode<<1)+1);

    cbpc = pyraData->GetCBPHuffCode(pyraData->bs) + 1;

    if (cbpc & 128)
      Decode2Vectors(pyraData, udec+pyraData->C4x4BlockOffset[0], CSCALING);

    if (cbpc & 64)
      Decode2Vectors(pyraData, udec+pyraData->C4x4BlockOffset[1], CSCALING);

    if (cbpc & 32)
      Decode2Vectors(pyraData, udec+pyraData->C4x4BlockOffset[2], CSCALING);

    if (cbpc & 16)
      Decode2Vectors(pyraData, udec+pyraData->C4x4BlockOffset[3], CSCALING);

    if (cbpc & 8) 
      Decode2Vectors(pyraData, vdec+pyraData->C4x4BlockOffset[0], CSCALING);

    if (cbpc & 4)
      Decode2Vectors(pyraData, vdec+pyraData->C4x4BlockOffset[1], CSCALING);

    if (cbpc & 2)
      Decode2Vectors(pyraData, vdec+pyraData->C4x4BlockOffset[2], CSCALING);

    if (cbpc & 1)
      Decode2Vectors(pyraData, vdec+pyraData->C4x4BlockOffset[3], CSCALING);
  }
}

static void PyraParse16x16MB(PyraGlobal *pyraData, int mode)
{

  int ccp, cbpy1, cbpy2, cbpc;

  if (mode == 0)
    ccp = GetIntraCCPHuffCode(pyraData->bs) + 1;
  else
    /* this is somewhat misleading */
    ccp = GetInterCCPHuffCode(pyraData->bs) + 1; 

  if ((ccp&4)||(ccp&2))
    SetupBlockDecoder(pyraData, mode<<1);

  if (ccp&4) {
  
    cbpy1 = pyraData->GetCBPHuffCode(pyraData->bs) + 1;
  
    if ( cbpy1 & 128 )
      Parse2Vectors(pyraData);
    if ( cbpy1 & 64 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 32 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 16 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 8 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 4 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 2 )	    
      Parse2Vectors(pyraData);
    if ( cbpy1 & 1 )	    
      Parse2Vectors(pyraData);

  }

  if (ccp & 2) {
    cbpy2 = pyraData->GetCBPHuffCode(pyraData->bs) + 1;

    if ( cbpy2 & 128 )
      Parse2Vectors(pyraData);
    if ( cbpy2 & 64 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 32 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 16 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 8 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 4 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 2 )	    
      Parse2Vectors(pyraData);
    if ( cbpy2 & 1 )	    
      Parse2Vectors(pyraData);

  }

  if (ccp & 1) {

    SetupBlockDecoder(pyraData, (mode<<1)+1);

    cbpc = pyraData->GetCBPHuffCode(pyraData->bs) + 1;

    if (cbpc & 128) 
      Parse2Vectors(pyraData);
    if (cbpc & 64) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 32) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 16) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 8) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 4) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 2) 	    
      Parse2Vectors(pyraData);
    if (cbpc & 1) 	    
      Parse2Vectors(pyraData);
  }
}

#ifndef PYRA_DCT_BASELAYER
static void PyraDecode8x8MB(PyraGlobal *pyraData, Byte *ydec, Byte *udec,
			    Byte *vdec, int scale, int mode)
{

  int ccp;

  if (mode == 0)
    ccp = GetIntraCCP6HuffCode(pyraData->bs) + 1;
  else {
    fprintf(stderr,"GetInterCCPHuffCode not implemented\n");
  }
    

  /* initializes codebooks and parsing functions */
  SetupBlockDecoder(pyraData, mode<<1);

  if ( ccp & 32 )
    Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[0], scale);

  if ( ccp & 16 )
    Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[1], scale);

  if ( ccp & 8 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[2], scale);

  if ( ccp & 4 )
      Decode2Vectors(pyraData, ydec+pyraData->Y4x4BlockOffset[3], scale);

  SetupBlockDecoder(pyraData, (mode<<1)+1);

  if (ccp & 2) 
    Decode2Vectors(pyraData, udec+pyraData->C4x4BlockOffset[0], CSCALING);

  if (ccp & 1)
    Decode2Vectors(pyraData, vdec+pyraData->C4x4BlockOffset[0], CSCALING);
  
}
#endif

/* ========================================================================= */

int PyraDecodeHeader(PyraGlobal *pyraData) {
  int     header;
  int     gfid;
  int     ptype;
  int     code = 1;


  header = GetBits(pyraData->bs,24);   /* Get Sync */

  header = GetBits(pyraData->bs,5);    /* Get GOB */
  pyraData->GOBNumber    = header;
  switch(header) {
  case 0: 
    /* Get temporal reference */
    pyraData->prev_temp_ref = pyraData->temp_ref;
    pyraData->temp_ref     = GetBits(pyraData->bs,8);
    pyraData->trd = pyraData->temp_ref - pyraData->prev_temp_ref;
    if (pyraData->trd < 0)
      pyraData->trd += 256;
    /* Get PTYPE */
    ptype = ShowBits(pyraData->bs, 6);
    /* Get picture size */
    pyraData->PicSize      = GetBits(pyraData->bs,3);
    if (pyraData->PicSize == 6) {
      pyraData->widthSub = GetBits(pyraData->bs,11); /*! new init! !*/
      pyraData->heightSub = GetBits(pyraData->bs,11);
      pyraData->width = ((pyraData->widthSub + (MACROBLOCK_SIZE - 1)) /
	MACROBLOCK_SIZE) * MACROBLOCK_SIZE;
      pyraData->height = ((pyraData->heightSub + (MACROBLOCK_SIZE - 1)) /
	MACROBLOCK_SIZE) * MACROBLOCK_SIZE;
      ptype = (ptype & 0x3c) | ShowBits(pyraData->bs,2);
    }
    /* Reset GFID */
    if (1/*ptype != pyraData->ptype || pyraData->GOBFrameID > 3*/) {
      pyraData->ptype = ptype;
      pyraData->GOBFrameID = -1;
    }
    /* Get picture type and scal type */
    pyraData->PicType      = GetOneBit(pyraData->bs);
    pyraData->ScalType     = GetBits(pyraData->bs,2);
#ifdef DEBUG
    fprintf(stderr,"Start of new picture, Timestamp=%d, Size=%d, Type=%d, ", 
	    pyraData->temp_ref, pyraData->PicSize, pyraData->PicType);
#endif

    pyraData->decGOBsCounter = 0;
    memset(pyraData->decGOBs, 0, MAX_GOBS * sizeof(unsigned char));

    break;
  default:
    gfid = GetBits(pyraData->bs,2);
    if (pyraData->GOBFrameID >= 0) {
      if (pyraData->GOBFrameID != gfid) {
	pyraData->GOBFrameID = 4;
	/* We need error concealment since the picture type has changed */
	code = 0;
      }
    } else
      pyraData->GOBFrameID = gfid;

#ifdef DEBUG
    fprintf(stderr,"Start of new GOB, Number=%d, FrameID=%d, ", 
	    pyraData->GOBNumber,pyraData->GOBFrameID); 
#endif
    break;
  }
  pyraData->currQuant = GetBits(pyraData->bs, 5);
  pyraData->currScale = SCALE_MIN + SCALE_STEP * pyraData->currQuant;
#ifdef DEBUG
  fprintf(stderr,"Scale = %d\n", pyraData->currScale);
#endif

  if (pyraData->decGOBsCounter == 0)
    SwapPictures(&(pyraData->TempRefPic), &(pyraData->DecPic));

  if (pyraData->first) {
    InitPyraDecoder(pyraData);
    InitBlockDecoder(pyraData, pyraData->width);
  }

  return code;
}

static int DecodeDiff(Bitstream *bs) {
  int x = GetDiffHuffCode(bs);

  if (x) {
    x = 2*x;
    return ( GetOneBit(bs) ? -x : x);
  } else 
    return(0);
}

#ifndef PYRA_DCT_BASELAYER
static void DecodeIntraMeans(Byte *m, int color, Bitstream *bs) {

  m[0] = 128 + DecodeDiff(bs);

  if (color==1) return;

  m[1] = m[0] + DecodeDiff(bs);
  m[2] = m[0] + DecodeDiff(bs);
  m[3] = ROUND( ((-2*m[0]) + (3*m[1]) + (3*m[2])) / 4.0) + DecodeDiff(bs);
}

static void PredictFromBaseLayerIntraBlock(PyraGlobal *pyraData, Picture *dec,
					   int sx, int sy, Bitstream *bs) {

  int coded;
  int oldWidth;

  /* Get mean values from bitstream (layer 3 of block pyramid) */
  DecodeIntraMeans(pyraData->BlockPyra[3]->y, 0, bs);
  DecodeIntraMeans(pyraData->BlockPyra[3]->u, 1, bs);
  DecodeIntraMeans(pyraData->BlockPyra[3]->v, 1, bs);

  /* Interpolate to layer 2 */
  InterpolatePictureFast(pyraData->BlockPyra[3], pyraData->BlockPyra[2]);

  /* nothing coded for layer 2 */

  /* Interpolate to layer 1 */
  InterpolatePictureFast(pyraData->BlockPyra[2], pyraData->BlockPyra[1]);

  coded = (GetOneBit(bs) == 0);

  if (coded) {
    oldWidth = pyraData->width; /* save old picture width */
    InitBlockDecoder(pyraData, 8); /* change picture width for block pyramid */
    PyraDecode8x8MB(pyraData, pyraData->BlockPyra[1]->y,
		    pyraData->BlockPyra[1]->u, pyraData->BlockPyra[1]->v,
		    pyraData->currScale / 2, 0);
    InitBlockDecoder(pyraData, oldWidth); /* restore picture width */
  }

  InterpolatePictureFast(pyraData->BlockPyra[1], pyraData->BlockPyra[0]);

  CopyBlockToPicture(pyraData->BlockPyra[0]->y, dec->y, sx, sy, 16,16,dec->w);
  sx >>= 1; sy >>= 1;
  CopyBlockToPicture(pyraData->BlockPyra[0]->u, dec->u, sx, sy, 8, 8,dec->w/2);
  CopyBlockToPicture(pyraData->BlockPyra[0]->v, dec->v, sx, sy, 8, 8,dec->w/2);
}
#endif


int PyraDecodeGOB(PyraGlobal *pyraData) {
  int coded = 0;
  int pmode, mvx, mvy;
  int k;
  int mx, my, mn;
  int offset;


  pyraData->bs->BufState = GOB_COMPLETE;

  if (!PyraDecodeHeader(pyraData)) {
    PyraConcealGOB(pyraData);
    pyraData->decGOBsCounter++;
    pyraData->decGOBs[pyraData->GOBNumber] = 1;
    pyraData->bs->BufState = GOB_EMPTY;
    return 0;
  }

  /* For an Intra coded picture in an enhancement layer we can interpolate
     the whole picture */
  if (!pyraData->decGOBsCounter && !pyraData->PicType &&
      !pyraData->baseLayerFlag)
    MultInterpolateFastPictureSub(pyraData->dSize, pyraData->SpatRefPic,
				  pyraData->DecPic);

  /* Counter, Flags, etc.... */
  pyraData->decGOBsCounter++;
  pyraData->decGOBs[pyraData->GOBNumber] = 1;

#ifndef PYRA_DCT_BASELAYER
  if ((pyraData->PicType==0) && (pyraData->baseLayerFlag)) {
    DecodeDPCMLayer(pyraData);
    return(2);
  }
#endif

  mn = pyraData->GOBNumber * pyraData->MBs_in_GOB;
  my = MACROBLOCK_SIZE * mn / pyraData->MBs_in_row;

  for (k = 0; k < pyraData->MB_rows_in_GOB; k++, my += MACROBLOCK_SIZE) {
    for (mx = 0; mx < pyraData->width; mx += MACROBLOCK_SIZE, mn++) {

      if (pyraData->PicType==0) {   /* Intra coded picture */
	if (!pyraData->baseLayerFlag) {
	  coded = (GetOneBit(pyraData->bs) == 0);
	}
	pmode = 2;
	pyraData->mvf.mode[mn] = MODE_INTRA;
      } else {   /* Inter coded */

	pmode = GetPPredTypeHuffCode(pyraData->bs);
	coded = (pmode&1);

	if (!(pmode & 2)) {   /* Inter MB */
	  /* Set MV predictor */
	  if (pyraData->baseLayerFlag)
	    H263MVPrediction(&(pyraData->mvf), mx >> 4, my >> 4, k == 0, NULL);
	  mvx = pyraData->mvf.mx[mn];
	  mvy = pyraData->mvf.my[mn];

	  /* Clip MV predictor */
	  mvx = CLIP(mvx, -(2*mx), 2*(pyraData->width - mx - 16));
	  mvy = CLIP(mvy, -(2*my), 2*(pyraData->height - my - 16));
	  mvx += GetMVHuffCode(pyraData->bs) - 32;
	  mvy += GetMVHuffCode(pyraData->bs) - 32;
	  if (mvx > 31)
	    mvx -= 64;
	  else if (mvx < -32)
	    mvx += 64;
	  if (mvy > 31)
	    mvy -= 64;
	  else if (mvy < -32)
	    mvy += 64;
	  pyraData->mvf.mx[mn] = mvx;
	  pyraData->mvf.my[mn] = mvy;
	  pyraData->mvf.mode[mn] = MODE_INTER;
	  reconstructPyra(pyraData, mx, my, 1, 0, 0);
	} else {
	  pyraData->mvf.mode[mn] = MODE_INTRA;
	  if (pyraData->baseLayerFlag) {
	    pyraData->mvf.mx[mn] = 0;
	    pyraData->mvf.my[mn] = 0;
	  }
	}
      }

      /* motion compensation or interpolation depending on the
       * coding mode. In any case pyraData->DecPic shall contain the predicted
       * signal.
       */

      if (pmode & 2) {   /* Intra MB */
	if (!pyraData->baseLayerFlag) {  /* not the base layer so we
					    can form a prediction */
	  if (pyraData->PicType != PICTURE_CODING_TYPE_INTRA)
	    MultInterpolateMacroBlockSub(pyraData->dSize, pyraData->SpatRefPic,
					 pyraData->DecPic, mx, my);
	} else { /* Intra blocks in the base layer
		    need special treatment  */

#ifdef PYRA_DCT_BASELAYER
	  const int quiet = 1;
	  int       fault;

	  H263getIFrameMB(pyraData->bs, mx, my, pyraData->DecPic->w,
			  pyraData->DecPic->h, pyraData->DecPic->y,
			  pyraData->DecPic->u, pyraData->DecPic->v,
			  &pyraData->currQuant, quiet, &fault);
	  coded = 0;
#else
	  /* Forms a prediction for layer 0 of block pyramid from
	     intra coded data */
	  PredictFromBaseLayerIntraBlock(pyraData, pyraData->DecPic, mx, my,
					 pyraData->bs);
	  /* coded/uncoded bit for 16x16 block */
	  coded = (GetOneBit(pyraData->bs) == 0);
#endif
	}
      }

      if (coded) {
#ifdef DEBUG
	fprintf(stderr,"*");
#endif

	offset = (mx + my * pyraData->width / 2) >> 1;
	PyraDecode16x16MB(pyraData, pyraData->DecPic->y+mx+my*pyraData->width,
			  pyraData->DecPic->u + offset,
			  pyraData->DecPic->v + offset,
			  pyraData->currScale, !(pmode & 2));
      } else {
#ifdef DEBUG
	fprintf(stderr,".");
#endif
      }
    }
  }
#ifdef DEBUG
  fprintf(stderr,"\n");
#endif

  return 1;
}


int PyraParseGOB(PyraGlobal *pyraData, int *mbInd, int *mbQuant, int offset) {
  int coded = 0;
  int pmode, mvx, mvy;
  int k;
  int mx, my, mn;
  int bsInd = 8 * (pyraData->bs->BufferPtr - pyraData->bs->ByteBuffer) +
    pyraData->bs->ValidBits;
  int prev_temp_ref = pyraData->prev_temp_ref;
  

  pyraData->bs->BufState = GOB_COMPLETE;

  if (!PyraDecodeHeader(pyraData)) {
    PyraConcealGOB(pyraData);
    pyraData->decGOBsCounter++;
    pyraData->decGOBs[pyraData->GOBNumber] = 1;
    pyraData->bs->BufState = GOB_EMPTY;
    return 0;
  }
  pyraData->decGOBsCounter++;
  pyraData->decGOBs[pyraData->GOBNumber] = 1;

  pyraData->prev_temp_ref = prev_temp_ref;
  pyraData->trd = pyraData->temp_ref - prev_temp_ref;

  mn = pyraData->GOBNumber * pyraData->MBs_in_GOB;
  my = MACROBLOCK_SIZE * mn / pyraData->MBs_in_row;

  for (k = 0; k < pyraData->MB_rows_in_GOB; k++, my += MACROBLOCK_SIZE) {
    for (mx = 0; mx < pyraData->width; mx += MACROBLOCK_SIZE, mn++) {

      if (mbInd != NULL)
	if (mx == 0 && k == 0)
	  mbInd[mn] = bsInd + offset;
	else
	  mbInd[mn] = 8 * (pyraData->bs->BufferPtr - pyraData->bs->ByteBuffer) +
	  pyraData->bs->ValidBits + offset;
      if (mbQuant != NULL)
	mbQuant[mn] = pyraData->currQuant;

      if (pyraData->PicType==0) {   /* Intra coded picture */
	if (!pyraData->baseLayerFlag)
	  coded = (GetOneBit(pyraData->bs) == 0);
	pmode = 2;
	pyraData->mvf.mode[mn] = MODE_INTRA;
      } else {   /* Inter coded */

	pmode = GetPPredTypeHuffCode(pyraData->bs);
	coded = (pmode&1);

	if (!(pmode & 2)) {   /* Inter MB */
	  /* Set MV predictor */
	  if (pyraData->baseLayerFlag)
	    H263MVPrediction(&(pyraData->mvf), mx >> 4, my >> 4, k == 0, NULL);
	  mvx = pyraData->mvf.mx[mn];
	  mvy = pyraData->mvf.my[mn];

	  /* Clip MV predictor */
	  mvx = CLIP(mvx, -(2*mx), 2*(pyraData->width - mx - 16));
	  mvy = CLIP(mvy, -(2*my), 2*(pyraData->height - my - 16));
	  mvx += GetMVHuffCode(pyraData->bs) - 32;
	  mvy += GetMVHuffCode(pyraData->bs) - 32;
	  if (mvx > 31)
	    mvx -= 64;
	  else if (mvx < -32)
	    mvx += 64;
	  if (mvy > 31)
	    mvy -= 64;
	  else if (mvy < -32)
	    mvy += 64;
	  pyraData->mvf.mx[mn] = mvx;
	  pyraData->mvf.my[mn] = mvy;
	  pyraData->mvf.mode[mn] = MODE_INTER;
	} else {
	  pyraData->mvf.mode[mn] = MODE_INTRA;
	  if (pyraData->baseLayerFlag) {
	    pyraData->mvf.mx[mn] = 0;
	    pyraData->mvf.my[mn] = 0;
	  }
	}
      }

      /* to do: motion compensation or interpolation depending on the
       * coding mode. In any case pyraData->DecPic shall contain the predicted
       * signal.
       */

      if (pmode & 2) {   /* Intra MB */
	if (pyraData->baseLayerFlag != 0) {/* the base layer */

#ifdef PYRA_DCT_BASELAYER
	  const int quiet = 1;
	  int       fault;

	  H263getIFrameMB(pyraData->bs, mx, my, pyraData->DecPic->w,
			  pyraData->DecPic->h, pyraData->DecPic->y,
			  pyraData->DecPic->u, pyraData->DecPic->v,
			  &pyraData->currQuant, quiet, &fault);
	  coded = 0;
#else
	  /* Forms a prediction for layer 0 of block pyramid from
	     intra coded data */
	  PredictFromBaseLayerIntraBlock(pyraData, pyraData->DecPic, mx, my,
					 pyraData->bs);
	  /* coded/uncoded bit for 16x16 block */
	  coded = (GetOneBit(pyraData->bs) == 0);
#endif
	}
      }

      if (coded) {
#ifdef DEBUG
	fprintf(stderr,"*");
#endif

	PyraParse16x16MB(pyraData, !(pmode & 2));
      } else {
#ifdef DEBUG
	fprintf(stderr,".");
#endif
      }
    }
  }
#ifdef DEBUG
  fprintf(stderr,"\n");
#endif

  return 1;
}


void PyraConcealGOB(PyraGlobal *pyraData)
{
  int gobsize = pyraData->MBs_in_GOB * MACROBLOCK_SIZE * MACROBLOCK_SIZE *
    sizeof(unsigned char);
  int mx, my, sx, sy;


#if 0
  if (pyraData->baseLayerFlag) {
    memcpy(pyraData->DecPic->y + pyraData->GOBNumber * gobsize,
	   pyraData->TempRefPic->y + pyraData->GOBNumber * gobsize, gobsize);
    gobsize /= 4;
    memcpy(pyraData->DecPic->u + pyraData->GOBNumber * gobsize,
	   pyraData->TempRefPic->u + pyraData->GOBNumber * gobsize, gobsize);
    memcpy(pyraData->DecPic->v + pyraData->GOBNumber * gobsize,
	   pyraData->TempRefPic->v + pyraData->GOBNumber * gobsize, gobsize);
  } else {
#endif
    if (pyraData->width == pyraData->SpatRefPic->w) {
      memcpy(pyraData->DecPic->y + pyraData->GOBNumber * gobsize,
	     pyraData->SpatRefPic->y + pyraData->GOBNumber * gobsize, gobsize);
      gobsize /= 4;
      memcpy(pyraData->DecPic->u + pyraData->GOBNumber * gobsize,
	     pyraData->SpatRefPic->u + pyraData->GOBNumber * gobsize, gobsize);
      memcpy(pyraData->DecPic->v + pyraData->GOBNumber * gobsize,
	     pyraData->SpatRefPic->v + pyraData->GOBNumber * gobsize, gobsize);
    } else if (pyraData->width == pyraData->SpatRefPic->w << 1) {
      my = pyraData->GOBNumber * (MACROBLOCK_SIZE / 2);
      sy = my >> 1;
      for (mx = 0; mx < pyraData->MBs_in_GOB * (MACROBLOCK_SIZE / 2);
	   mx += (MACROBLOCK_SIZE / 2)) {
	InterpolateBlock(pyraData->SpatRefPic->y, pyraData->DecPic->y,
			 mx, my, 8, 8,
			 pyraData->SpatRefPic->w, pyraData->SpatRefPic->h);
	sx = mx >> 1;
	InterpolateBlock(pyraData->SpatRefPic->u, pyraData->DecPic->u,
			 sx, sy, 4, 4, pyraData->SpatRefPic->w / 2,
			 pyraData->SpatRefPic->h / 2);
	InterpolateBlock(pyraData->SpatRefPic->v, pyraData->DecPic->v,
			 sx, sy, 4, 4, pyraData->SpatRefPic->w / 2,
			 pyraData->SpatRefPic->h / 2);
      }
    }
#if 0
  }
#endif
}

static void DecodeDPCMImage(Bitstream *bs, Byte *pic, int w, int h, int iw) {
  int i, j;
  int p1, p2, p3, p, d;


  p = 128;
  for (i=0; i<w; i++) {
    d = DecodeDiff(bs);
    pic[i] = p+d;
    p = pic[i];
  } 

  for (j=1; j<h; j++) {
    p = pic[(j-1)*iw];
    d = DecodeDiff(bs);
    pic[j*iw] = p+d;

    for (i=1; i<w; i++) {

      p1=pic[(j-1)*iw+i-1];
      p2=pic[(j-1)*iw+i];
      p3=pic[j*iw+i-1];
      p = ROUND( ((-2*p1) + (3*p2) + (3*p3)) / 4.0);
      d = DecodeDiff(bs);
      pic[j*iw+i] = p+d;
    }
  }
}

void DecodeDPCMLayer(PyraGlobal *pyraData) {
  int w = pyraData->DecPic->w;
  int wc = w / 2;
  int wSub, hSub, wSubc, hSubc;

  wSub = pyraData->widthSub;
  hSub = pyraData->heightSub;
  wSubc = wSub / 2;
  hSubc = hSub / 2;

  DecodeDPCMImage(pyraData->bs, pyraData->DecPic->y, wSub, hSub, w);
  DecodeDPCMImage(pyraData->bs, pyraData->DecPic->u, wSubc, hSubc, wc);
  DecodeDPCMImage(pyraData->bs, pyraData->DecPic->v, wSubc, hSubc, wc);

  /* Counter, Flags, etc.... */
  {
    int i;

    pyraData->decGOBsCounter = pyraData->GOBs_in_pict;
    for (i = pyraData->GOBs_in_pict - 1; i >= 0; i--)
      pyraData->decGOBs[i] = 1;
  }
}
