#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include "defs.h"
#include "structs.h"
#include "Util.h"
#include "bitOut.h"
#include "pyraencoder.h"

#include "filter.p"
#include "Lattice.h"
#include "motion.h"
#include "pyracoder.p"
#include "common.h"

static Picture * ActBlockPyra[4], * DecBlockPyra[4];
static BlockMemStruct *act_blk, *pre_blk, *dec_blk;
static CodedPatternStruct *block_patt;

void InitBlockPyra() {

  int l;
  int w = 16;
  int h = 16;

  for (l=0; l<4; l++) {

    ActBlockPyra[l] = (Picture *) malloc (sizeof(Picture));
    ActBlockPyra[l]->w = w;
    ActBlockPyra[l]->h = h;
    ActBlockPyra[l]->y = malloc(w*h);
    ActBlockPyra[l]->u = malloc(w*h/4);
    ActBlockPyra[l]->v = malloc(w*h/4);

    DecBlockPyra[l] = (Picture *) malloc (sizeof(Picture));
    DecBlockPyra[l]->w = w;
    DecBlockPyra[l]->h = h;
    DecBlockPyra[l]->y = malloc(w*h);
    DecBlockPyra[l]->u = malloc(w*h/4);
    DecBlockPyra[l]->v = malloc(w*h/4);

    w = w/2;
    h = h/2;
  }
  act_blk = (BlockMemStruct *) malloc(sizeof(BlockMemStruct));
  pre_blk = (BlockMemStruct *) malloc(sizeof(BlockMemStruct));
  dec_blk = (BlockMemStruct *) malloc(sizeof(BlockMemStruct));
  block_patt = (CodedPatternStruct *) malloc(sizeof(CodedPatternStruct));
}

/* ========================================================================= */
static void EncodeDPCM(Byte *actual, Byte *decoded, int w, int h, int iw,
		       Bitstr *bs)
{
  int i, j;
  int p1, p2, p3, p, d, r;

  p1 = 128;

  for (i=0; i<w; i++) {
    d = actual[i] - p1;
    r = EncodeDiff(d, bs);
    decoded[i] = p1 + r;
    p1 = decoded[i];
  } 

  for (j=1; j<h; j++) {
    p1 = decoded[(j-1)*iw];
    d = actual[j*iw] - p1;
    r = EncodeDiff(d, bs);
    decoded[j*iw] = p1 + r;
    for (i=1; i<w; i++) {
      p1=decoded[(j-1)*iw+i-1];
      p2=decoded[(j-1)*iw+i];
      p3=decoded[j*iw+i-1];
      p = ROUND( ((-2*p1) + (3*p2) + (3*p3)) / 4.0);
      d = actual[j*iw+i] - p;
      r = EncodeDiff(d, bs);
      decoded[j*iw+i] =  p + r;
    }
  }
}


int EncodePictDPCM(Picture *act, Picture *dec, Bitstr *bs)
{
  const int  bsIndStart = bs->ind;
  int w = act->w;
  int wc = w/2;
  int ws = act->ws, hs = act->hs;
  int wsc = ws/2, hsc = hs/2;


  EncodeDPCM(act->y, dec->y, ws, hs, w, bs);
  EncodeDPCM(act->u, dec->u, wsc, hsc, wc, bs);
  EncodeDPCM(act->v, dec->v, wsc, hsc, wc, bs);

  return(bs->ind - bsIndStart);
}

/* ============================================================================== */


static void CodeIntraMeans(Byte *m, int n, Bitstr *bs)
{
  int d,r;
  int p1, p2, p3, p;

  d = m[0] - 128;
  r = EncodeDiff(d, bs);
  m[0] = 128 + r;

  if (n==1) return;

  p1 = m[0];
  d = m[1] - p1;
  r = EncodeDiff(d, bs);
  m[1] = p1 + r;

  d = m[2] - p1;
  r = EncodeDiff(d, bs);
  m[2] = p1 + r;

  p1 = m[0];
  p2 = m[1];
  p3 = m[2];
  p = ROUND( ((-2*p1) + (3*p2) + (3*p3)) / 4.0);

  d = m[3] - p;
  r = EncodeDiff(d, bs);
  m[3] = p + r;
}

void Code16x16IntraBlock(Picture *Actual, Picture *Decoded, int bx, int by,
			 int scale1, int scale0, Bitstr *bs) {
  int i, j, n;


  /* ------------------------------------- */
  /* Copy actual image data to block */
  n = 0;
  for (j=0; j<16; j++) 
    for (i=0; i<16; i++) 
      ActBlockPyra[0]->y[n++] = Actual->y[(by+j)*Actual->w+bx+i];

  n = 0;
  for (j=0; j<8; j++) 
    for (i=0; i<8; i++) {
      ActBlockPyra[0]->u[n] = Actual->u[((by>>1)+j)*(Actual->w>>1)+(bx>>1)+i];
      ActBlockPyra[0]->v[n] = Actual->v[((by>>1)+j)*(Actual->w>>1)+(bx>>1)+i];
      n++;
    }
  /* ------------------------------------- */
  /* Make block pyramid */
  DownsamplePicture(ActBlockPyra[0], ActBlockPyra[1]);
  DownsamplePicture(ActBlockPyra[1], ActBlockPyra[2]);
  DownsamplePicture(ActBlockPyra[2], ActBlockPyra[3]);
  /* ------------------------------------- */
  /* Code lowpass */
  CodeIntraMeans(ActBlockPyra[3]->y,4, bs);
  CodeIntraMeans(ActBlockPyra[3]->u,1, bs);
  CodeIntraMeans(ActBlockPyra[3]->v,1, bs);
  /* ------------------------------------- */
  /* Upsample up to layer 1, don't code layer 2 */
  InterpolatePictureFast(ActBlockPyra[3], DecBlockPyra[2]);
  InterpolatePictureFast(DecBlockPyra[2], DecBlockPyra[1]);
  /* ------------------------------------- */
  /* Code 8x8 block */

  Get8x8MB(ActBlockPyra[1], 0, 0, act_blk);
  Get8x8MB(DecBlockPyra[1], 0, 0, pre_blk);

  Quantize8x8MB(act_blk, pre_blk, scale1, MODE_INTRA, block_patt);
  Send8x8LowestIntraMB(dec_blk, pre_blk, scale1, block_patt, bs);

  Put8x8MB(DecBlockPyra[1], 0, 0, dec_blk);
  /* ------------------------------------- */
  /* Upsample to layer 0 */
  InterpolatePictureFast(DecBlockPyra[1], DecBlockPyra[0]);
  /* ------------------------------------- */
  /* Code 16x16 block */

  Get16x16MB(ActBlockPyra[0], 0, 0, act_blk);
  Get16x16MB(DecBlockPyra[0], 0, 0, pre_blk);

  Quantize16x16MB(act_blk, pre_blk, scale0, MODE_INTRA, block_patt);
  Send16x16LowestIntraMB(dec_blk, pre_blk, scale0, block_patt, bs);

  Put16x16MB(DecBlockPyra[0], 0, 0, dec_blk);
  /* ------------------------------------- */
  /* Copy decoded block back to image */
  n = 0;
  for (j=0; j<16; j++) 
    for (i=0; i<16; i++) 
      Decoded->y[(by+j)*Actual->w+bx+i] = DecBlockPyra[0]->y[n++];

  n = 0;
  for (j=0; j<8; j++) 
    for (i=0; i<8; i++) {
      Decoded->u[((by>>1)+j)*(Actual->w>>1)+(bx>>1)+i] = DecBlockPyra[0]->u[n];
      Decoded->v[((by>>1)+j)*(Actual->w>>1)+(bx>>1)+i] = DecBlockPyra[0]->v[n];
      n++;
    }
}
/* ========================================================================= */
/* Encoding of predicted blocks */

void Code16x16PredBlock(Picture * act_pict, Picture * pre_pict,
			Picture * dec_pict, int bx, int by, int scale,
			int layerMode, int mode, int mvx, int mvy,
			int lambda, Bitstr *bs) {

  BlockMemStruct act_blk, pre_blk, dec_blk;
  CodedPatternStruct block_patt;
  int bsIndStart = bs->ind;
  int costU = 0, costP;


  if (lambda > 0) {
    costU = MacroBlockPictureSSE(bx, by, act_pict, pre_pict, INT_MAX);
    Send16x16MB(&dec_blk, &pre_blk, scale, layerMode, mode, 1,
		&block_patt, mvx, mvy, bs);
    costU += lambda * (bs->ind - bsIndStart);
    bs->ind = bsIndStart;
  }

  Get16x16MB(act_pict, bx, by, &act_blk);
  Get16x16MB(pre_pict, bx, by, &pre_blk);
  Quantize16x16MB(&act_blk, &pre_blk, scale, mode, &block_patt);
  Send16x16MB(&dec_blk, &pre_blk, scale, layerMode, mode, 0,
	      &block_patt, mvx, mvy, bs);
  Put16x16MB(dec_pict, bx, by, &dec_blk);

  if (lambda > 0) {
    costP = lambda * (bs->ind - bsIndStart);
    costP += MacroBlockPictureSSE(bx, by, act_pict, dec_pict, costU - costP);
    if (costU < costP) {
      bs->ind = bsIndStart;
      Send16x16MB(&dec_blk, &pre_blk, scale, layerMode, mode, 1,
		  &block_patt, mvx, mvy, bs);
      Put16x16MB(dec_pict, bx, by, &dec_blk);
    }
#ifdef PROTOCOL
    if (fp_prot_g != NULL && protFormat_g >= 3) {
      if (costU < costP)
	fprintf(fp_prot_g, " ");
      else
	fprintf(fp_prot_g, "x");
    }
#endif
  }
}

int BlockMSE(int *b1, int *b2, int size) {
  int i, diff, mse;

  mse = 0;

  for (i=0; i<size; i++) {
    diff = b1[i] - b2[i];
    mse += (diff*diff);
  }

  return((int) ((float) mse / (float)size));
}

void Code8x8PredBlock(Picture * act_pict, Picture * pre_pict,
		      Picture * dec_pict, int bx, int by, int scale,
		      int layerMode, int mode, int mvx, int mvy, Bitstr *bs) {

  BlockMemStruct act_blk, pre_blk, dec_blk;
  CodedPatternStruct block_patt;

  Get8x8MB(act_pict, bx, by, &act_blk);
  Get8x8MB(pre_pict, bx, by, &pre_blk);

  Quantize8x8MB(&act_blk, &pre_blk, scale, mode, &block_patt);
  Send8x8MB(&dec_blk, &pre_blk, scale, layerMode, mode, &block_patt,
	    mvx, mvy, bs);
      
  Put8x8MB(dec_pict, bx, by, &dec_blk);
}
/* ========================================================================= */
