#include <stdlib.h>

#include "defs.h"
#include "structs.h"
#include "Util.h"
#include "bitOut.h"
#include "pyraencoder.h"
#include "Lattice.h"

#include "motion.h"
#include "pyracoder.p"


/* ========================================================================= */
/* Accessing image blocks */

static void Get4x4Block(Byte *image, int w, int x, int y, int *vector) {
 /* Gets a 4x4 block out of image which has width w and height h into vector */
  Byte *col;

  col = image+y*w+x;

  vector[0] = *col; col+=w;
  vector[1] = *col; col+=w;
  vector[2] = *col; col+=w;
  vector[3] = *col; 

  col = image+y*w+x+1;

  vector[4] = *col; col+=w;
  vector[5] = *col; col+=w;
  vector[6] = *col; col+=w;
  vector[7] = *col; 

  col = image+y*w+x+2;

  vector[8] = *col; col+=w;
  vector[9] = *col; col+=w;
  vector[10] = *col; col+=w;
  vector[11] = *col; 

  col = image+y*w+x+3;

  vector[12] = *col; col+=w;
  vector[13] = *col; col+=w;
  vector[14] = *col; col+=w;
  vector[15] = *col; 

}

/* ------------------------------------------------------------------------- */

static void Get8x8Block(Byte *image, int w, int x, int y, int *vector) {
  Get4x4Block(image, w, x,   y,   vector); vector+=16;
  Get4x4Block(image, w, x+4, y,   vector); vector+=16;
  Get4x4Block(image, w, x,   y+4, vector); vector+=16;
  Get4x4Block(image, w, x+4, y+4, vector); vector+=16;
}

/* ========================================================================= */
/* Getting Macroblocks */

void Get8x8MB(Picture *pic, int x, int y, BlockMemStruct *blk) {
  Get8x8Block(pic->y, pic->w, x, y, blk->y);

  Get4x4Block(pic->u, pic->w/2, x/2, y/2, blk->u);
  Get4x4Block(pic->v, pic->w/2, x/2, y/2, blk->v);
}

/* ------------------------------------------------------------------------- */

void Get16x16MB(Picture *pic, int x, int y, BlockMemStruct *blk) {
  Get8x8Block(pic->y, pic->w, x,   y,   blk->y);
  Get8x8Block(pic->y, pic->w, x+8, y,   blk->y+64);
  Get8x8Block(pic->y, pic->w, x,   y+8, blk->y+128);
  Get8x8Block(pic->y, pic->w, x+8, y+8, blk->y+192);

  Get8x8Block(pic->u, pic->w/2, x/2, y/2, blk->u);
  Get8x8Block(pic->v, pic->w/2, x/2, y/2, blk->v);
}

/* ========================================================================== 
 * ==========================================================================
 */

static void Put4x4Block(Byte *image, int w, int x, int y, int *vector) {

  Byte *col;

  col = image+y*w+x;

  *col = vector[0] ; col+=w;
  *col = vector[1] ; col+=w;
  *col = vector[2] ; col+=w;
  *col = vector[3] ; 

  col = image+y*w+x+1;

  *col = vector[4] ; col+=w;
  *col = vector[5] ; col+=w;
  *col = vector[6] ; col+=w;
  *col = vector[7] ; 

  col = image+y*w+x+2;

  *col = vector[8] ; col+=w;
  *col = vector[9] ; col+=w;
  *col = vector[10] ; col+=w;
  *col = vector[11] ;

  col = image+y*w+x+3;

  *col = vector[12] ; col+=w;
  *col = vector[13] ; col+=w;
  *col = vector[14] ; col+=w;
  *col = vector[15] ;

}

static void Put8x8Block(Byte *image, int w, int x, int y, int *vector) {
  Put4x4Block(image, w, x,   y,   vector); vector+=16;
  Put4x4Block(image, w, x+4, y,   vector); vector+=16;
  Put4x4Block(image, w, x,   y+4, vector); vector+=16;
  Put4x4Block(image, w, x+4, y+4, vector); vector+=16;
}

/* ========================================================================= */
/* Putting Macroblocks */

void Put8x8MB(Picture * pic, int x, int y, BlockMemStruct *blk) {
  Put8x8Block(pic->y, pic->w, x, y, blk->y);

  Put4x4Block(pic->u, pic->w/2, x/2, y/2, blk->u);
  Put4x4Block(pic->v, pic->w/2, x/2, y/2, blk->v);
}
/* ------------------------------------------------------------------------- */

void Put16x16MB(Picture * pic, int x, int y, BlockMemStruct *blk) {
  Put8x8Block(pic->y, pic->w, x, y, blk->y);
  Put8x8Block(pic->y, pic->w, x+8, y, blk->y+64);
  Put8x8Block(pic->y, pic->w, x, y+8, blk->y+128);
  Put8x8Block(pic->y, pic->w, x+8, y+8, blk->y+192);

  Put8x8Block(pic->u, pic->w/2, x/2, y/2, blk->u);
  Put8x8Block(pic->v, pic->w/2, x/2, y/2, blk->v);
}

/* ========================================================================= */
/* lattice point memory for storing quantized lattice points */
/* 0..31 = 32 Y vectors, 32..39 = 8 U vectors, 41..48 = 8 V vectors */

static LatticePointInfo *LPImem[48];

void InitLatticeQuantizer() {

  int i;

  for (i=0; i<48; i++) {

    LPImem[i] = (LatticePointInfo *) malloc(sizeof(LatticePointInfo));
  }

}
/* ========================================================================= */
void Quantize8x8MB(BlockMemStruct *act_blk, BlockMemStruct *pre_blk,
		   int BScale, int CodingMode, CodedPatternStruct *block_patt)
{
  int n, m;
  float scale;
  int shell1, shell2;
  int *yact, *uact, *vact;
  int *ypre, *upre, *vpre;

  yact = act_blk->y; uact = act_blk->u; vact = act_blk->v;
  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  
  scale = (float) BScale;

  m = 0; /* index into combined shell */

  block_patt->ccp = 0;

  for (n=0; n<8; n+=2, m++) {
    shell1 = QuantizeVect(yact,   ypre  , scale, LPImem[n]);
    shell2 = QuantizeVect(yact+8, ypre+8, scale, LPImem[n+1]);

    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->ccp <<= 1; block_patt->ccp |= ((block_patt->comb_shell[m]) ? 1 : 0); 

    yact += 16; ypre += 16;
  }

  /* Smaller quantizer step sizes for chroma signals */

  scale = ROUND(BScale*CSCALE);
#if 0
  if (CodingMode == MODE_INTER)
    scale = ROUND(BScale*0.7);
  else
    scale = ROUND(BScale*0.75);
#endif

  /* 4x4 U-block */
  for (n=8; n<10; n+=2, m++) {
    shell1 = QuantizeVect(uact, upre, scale, LPImem[n]);
    shell2 = QuantizeVect(uact+8, upre+8, scale, LPImem[n+1]);

    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->ccp <<= 1; block_patt->ccp |= ((block_patt->comb_shell[m]) ? 1 : 0); 

    uact += 16; upre += 16;
  }

  /* 4x4 V-block */
  for (n=10; n<12; n+=2, m++) {
    shell1 = QuantizeVect(vact, vpre, scale, LPImem[n]);
    shell2 = QuantizeVect(vact+8, vpre+8, scale, LPImem[n+1]);

    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->ccp <<= 1; block_patt->ccp |= (block_patt->comb_shell[m]) ? 1 : 0; 

    vact += 16; vpre += 16;
  }
}
/* ========================================================================= */

void Send8x8MB(BlockMemStruct *dec_blk, BlockMemStruct *pre_blk, int BScale,
	       int LayerMode, int CodingMode, CodedPatternStruct *block_patt,
	       int mvx, int mvy, Bitstr *bs) {
  int n, m;
  float scale;
  /*int coded4;*/
  int mb_mode;
  int tabmode;
  int *ypre, *upre, *vpre;
  int *ydec, *udec, *vdec;

  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  ydec = dec_blk->y; udec = dec_blk->u; vdec = dec_blk->v;

  /*
  for (m=0; m<12; m++) {
    coded4 += (block_patt->comb_shell[m] ? 1 : 0);
  }

  if (coded4==1) {
    for (m=0; m<12; m++) {
      block_patt->comb_shell[m] = 0;
    }
    block_patt->ccp = 0;
  }
  */

  m = 0;

  /* Mode for switching between different tables */
  if (CodingMode == MODE_INTER) tabmode = 2; else tabmode = 0;

  if (LayerMode ==  PICTURE_CODING_TYPE_INTRA) {
    if (block_patt->ccp==0) {
      EmitBits(1,1,"MB not coded", bs);
    } else {
      EmitBits(0,1,"MB coded", bs);
      Put8x8CCP(block_patt->ccp-1,tabmode, bs);
    }
#ifdef COUNT_BITS
    bitCountP->totalBits += 1;
    bitCountP->headerBits += 1;
#endif
  } else if (LayerMode ==  PICTURE_CODING_TYPE_INTER) {
    if (CodingMode == MODE_INTER) {
      if ((block_patt->ccp == 0) && (mvx == 0) && (mvy == 0) )
	mb_mode = 0; /* Inter-Layer, uncoded Inter-Block */
      else
	mb_mode = 1; /* Inter-Layer, coded Inter-Block */
    } else if (CodingMode == MODE_INTRA) {
      if (block_patt->ccp==0) 
	mb_mode = 2; /* Inter-Layer, uncoded Intra-Block */
      else
	mb_mode = 3; /* Inter-Layer, coded Intra-Block */
    }

    PutMode(mb_mode, bs);

    switch(mb_mode) {
    case 0:
    case 2:
      break;
    case 1:
      PutMV(mvx, mvy, bs);
    case 3:
      Put8x8CCP(block_patt->ccp-1,tabmode, bs);
      break;
    }
  }

  scale = (float) BScale;

  for (n=0; n<8; n+=2, m++) {

    if (block_patt->comb_shell[m])
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

    ydec += 16; ypre += 16;
  }

  /* Smaller step sizes for chroma signals */

  scale = ROUND(BScale*CSCALE);
#if 0
  if (CodingMode == MODE_INTER)
    scale = ROUND(BScale*0.7);
  else
    scale = ROUND(BScale*0.75);
#endif

  /* For chroma signals add one to tabmode */
  tabmode = tabmode + 1;

  for (n=8; n<10; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(udec,   upre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(udec+8, upre+8, scale, LPImem[n+1], tabmode);

    udec += 16; upre += 16;
  }

  for (n=10; n<12; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n],  tabmode, bs);
    RecVect(vdec,   vpre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1],tabmode, bs);
    RecVect(vdec+8, vpre+8, scale, LPImem[n+1], tabmode);

    vdec += 16; vpre += 16;
  }
}

/* ========================================================================= */
void Send8x8LowestIntraMB(BlockMemStruct *dec_blk, BlockMemStruct *pre_blk, 
			int BScale, CodedPatternStruct *block_patt, Bitstr *bs) {
  int n, m;
  float scale;
  int tabmode;
  int *ypre, *upre, *vpre;
  int *ydec, *udec, *vdec;

  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  ydec = dec_blk->y; udec = dec_blk->u; vdec = dec_blk->v;

  tabmode = 0;
  m = 0;

  if (block_patt->ccp==0) {
    EmitBits(1,1,"MB not coded", bs);
  } else {
    EmitBits(0,1,"MB coded", bs);
    Put8x8CCP(block_patt->ccp-1,tabmode, bs);
  }
#ifdef COUNT_BITS
  bitCountP->totalBits += 1;
  bitCountP->headerBits += 1;
#endif

  scale = (float) BScale;

  for (n=0; n<8; n+=2, m++) {

    if (block_patt->comb_shell[m])
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

    ydec += 16; ypre += 16;
  }

  scale = ROUND(BScale*CSCALE);
#if 0
  scale = ROUND(BScale*0.75);
#endif

  /* For chroma signals add one to tabmode */
  tabmode = tabmode + 1;

  for (n=8; n<10; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(udec,   upre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(udec+8, upre+8, scale, LPImem[n+1], tabmode);

    udec += 16; upre += 16;
  }

  for (n=10; n<12; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n],  tabmode, bs);
    RecVect(vdec,   vpre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1],tabmode, bs);
    RecVect(vdec+8, vpre+8, scale, LPImem[n+1], tabmode);

    vdec += 16; vpre += 16;
  }
}

/* ========================================================================= */
void Quantize16x16MB(BlockMemStruct *act_blk, BlockMemStruct *pre_blk,
		     int BScale, int CodingMode,
		     CodedPatternStruct *block_patt)
{
  int n, m;
  float scale;
  int shell1, shell2;
  int *yact, *uact, *vact;
  int *ypre, *upre, *vpre;

  yact = act_blk->y; uact = act_blk->u; vact = act_blk->v;
  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  
  scale = (float) BScale;

  m = 0; /* index into combined shell */

  block_patt->cbpy[0] = 0;

  /* upper two 8x8 blocks */
  for (n=0; n<16; n+=2, m++) {
    shell1 = QuantizeVect(yact,   ypre  , scale, LPImem[n]);
    shell2 = QuantizeVect(yact+8, ypre+8, scale, LPImem[n+1]);

    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->cbpy[0] <<= 1; block_patt->cbpy[0] |= ((block_patt->comb_shell[m]) ? 1 : 0); 

    yact += 16; ypre += 16;
  }
  block_patt->cbpy[1] = 0;

  /* lower two 8x8 blocks */

  for (n=16; n<32; n+=2, m++) {
    shell1 = QuantizeVect(yact,   ypre  , scale, LPImem[n]);
    shell2 = QuantizeVect(yact+8, ypre+8, scale, LPImem[n+1]);

    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->cbpy[1] <<= 1; block_patt->cbpy[1] |= ((block_patt->comb_shell[m]) ? 1 : 0); 

    yact += 16; ypre += 16;
  }

  block_patt->cbpc = 0; /* reset cbpc */

  /* Smaller quantizer step sizes for chroma signals */

  scale = ROUND(BScale*CSCALE);

  /* 8x8 U-block */
  for (n=32; n<40; n+=2, m++) {
    shell1 = QuantizeVect(uact, upre, scale, LPImem[n]);
    shell2 = QuantizeVect(uact+8, upre+8, scale, LPImem[n+1]);
    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->cbpc <<= 1; block_patt->cbpc |= ((block_patt->comb_shell[m]) ? 1 : 0); 

    uact += 16; upre += 16;
  }

  /* 8x8 V-block */
  for (n=40; n<48; n+=2, m++) {
    shell1 = QuantizeVect(vact, vpre, scale, LPImem[n]);
    shell2 = QuantizeVect(vact+8, vpre+8, scale, LPImem[n+1]);
    block_patt->comb_shell[m] = (shell1<<2)|shell2;
    block_patt->cbpc <<= 1; block_patt->cbpc |= (block_patt->comb_shell[m]) ? 1 : 0; 

    vact += 16; vpre += 16;
  }

  block_patt->ccp = ((block_patt->cbpy[0] ? 1 : 0)<<2) | ((block_patt->cbpy[1] ? 1 : 0)<<1) | ((block_patt->cbpc ? 1 : 0));
}

/* ========================================================================= */

void Send16x16MB(BlockMemStruct *dec_blk, BlockMemStruct *pre_blk, int BScale,
		 int LayerMode, int CodingMode, int uncoded_f,
		 CodedPatternStruct *block_patt,
		 int mvx, int mvy, Bitstr *bs) {
  int n, m;
  float scale;
  /*int coded4;*/
  int mb_mode;
  int tabmode;
  int *ypre, *upre, *vpre;
  int *ydec, *udec, *vdec;

  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  ydec = dec_blk->y; udec = dec_blk->u; vdec = dec_blk->v;

  /* Check how many coded 4x4 blocks are contained */

  /*
  for (m=0; m<24; m++) {
    coded4 += (block_patt->comb_shell[m] ? 1 : 0);
  }
  */
  /* Don't code the block if it contains only less than three
   * coded 4x4 blocks.
   */
  /*
  if ((coded4>0)&&(coded4<=2)) {
    for (m=0; m<24; m++) {
      block_patt->comb_shell[m] = 0;
    }
    block_patt->ccp = block_patt->cbpy[0] = block_patt->cbpy[1] = block_patt->cbpc = 0;
  }
  */
  /* Mode for switching between different tables */
  if (CodingMode == MODE_INTER) tabmode = 2; else tabmode = 0;


  /*************************************************/

  if (LayerMode == PICTURE_CODING_TYPE_INTRA) {
    if (block_patt->ccp==0 || uncoded_f) {
      EmitBits(1,1,"MB not coded", bs);
    } else {
      EmitBits(0,1,"MB coded", bs);
      PutCCP(block_patt->ccp-1,tabmode, bs);
    }
#ifdef COUNT_BITS
    bitCountP->totalBits += 1;
    bitCountP->headerBits += 1;
#endif
  } else if (LayerMode == PICTURE_CODING_TYPE_INTER) {
    if (CodingMode == MODE_INTER) {
      if (block_patt->ccp == 0 || uncoded_f)
	mb_mode = 0; /* Inter-Layer, uncoded Inter-Block */
      else
	mb_mode = 1; /* Inter-Layer, coded Inter-Block */
    } else if (CodingMode == MODE_INTRA) {
      if (block_patt->ccp==0 || uncoded_f) 
	mb_mode = 2; /* Inter-Layer, uncoded Intra-Block */
      else
	mb_mode = 3; /* Inter-Layer, coded Intra-Block */
    }

    PutMode(mb_mode, bs);

    switch(mb_mode) {
    case 0:
      PutMV(mvx, mvy, bs);
    case 2:
      break;
    case 1:
      PutMV(mvx, mvy, bs);
    case 3:
      PutCCP(block_patt->ccp-1,tabmode, bs);
      break;
    }
  }

#ifdef PROTOCOL
  if (fp_prot_g != NULL && protFormat_g >= 3) {
    if (!uncoded_f) {
      if ((block_patt->cbpy[0] || block_patt->cbpy[1]))
	fprintf(fp_prot_g, "*");
      else
	fprintf(fp_prot_g, " ");
      if (block_patt->cbpc && !uncoded_f)
	fprintf(fp_prot_g, "c");
      else
	fprintf(fp_prot_g, " ");
    }
  }
#endif

  if (uncoded_f) {
    /* Copy prediction */
    memcpy(ydec, ypre, 256 * sizeof(int));
    memcpy(udec, upre, 64 * sizeof(int));
    memcpy(vdec, vpre, 64 * sizeof(int));
  } else {
    scale = (float) BScale;

    if (block_patt->cbpy[0]) 
      PutCBP(block_patt->cbpy[0]-1,tabmode, bs);

    m = 0;
    for (n=0; n<16; n+=2, m++) {

      if (block_patt->comb_shell[m]) 
	PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

      SendVect(LPImem[n]  , tabmode, bs);
      RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
      SendVect(LPImem[n+1], tabmode, bs);
      RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

      ydec += 16; ypre += 16;
    }

    if (block_patt->cbpy[1]) 
      PutCBP(block_patt->cbpy[1]-1,tabmode, bs);

    for (n=16; n<32; n+=2, m++) {

      if (block_patt->comb_shell[m]) 
	PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

      SendVect(LPImem[n]  , tabmode, bs);
      RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
      SendVect(LPImem[n+1], tabmode, bs);
      RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

      ydec += 16; ypre += 16;
    }

    /* Smaller step sizes for chroma signals */
    scale = ROUND(BScale*CSCALE);

    /* For chroma signals add one to tabmode */
    tabmode = tabmode + 1;

    if (block_patt->cbpc)
      PutCBP(block_patt->cbpc-1,tabmode, bs);

    for (n=32; n<40; n+=2, m++) {

      if (block_patt->comb_shell[m]) 
	PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

      SendVect(LPImem[n]  , tabmode, bs);
      RecVect(udec,   upre,   scale, LPImem[n],   tabmode);
      SendVect(LPImem[n+1], tabmode, bs);
      RecVect(udec+8, upre+8, scale, LPImem[n+1], tabmode);

      udec += 16; upre += 16;
    }

    for (n=40; n<48; n+=2, m++) {

      if (block_patt->comb_shell[m]) 
	PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

      SendVect(LPImem[n],  tabmode, bs);
      RecVect(vdec,   vpre,   scale, LPImem[n],   tabmode);
      SendVect(LPImem[n+1],tabmode, bs);
      RecVect(vdec+8, vpre+8, scale, LPImem[n+1], tabmode);

      vdec += 16; vpre += 16;
    }
  }
}

void Send16x16LowestIntraMB(BlockMemStruct *dec_blk, BlockMemStruct *pre_blk,
			    int BScale, CodedPatternStruct *block_patt,
			    Bitstr *bs) {
  int n, m;
  float scale;
  int tabmode;
  int *ypre, *upre, *vpre;
  int *ydec, *udec, *vdec;

  ypre = pre_blk->y; upre = pre_blk->u; vpre = pre_blk->v;
  ydec = dec_blk->y; udec = dec_blk->u; vdec = dec_blk->v;

  tabmode = 0;
  m = 0;

  if (block_patt->ccp==0) {
    EmitBits(1,1,"MB not coded", bs);
  } else {
    EmitBits(0,1,"MB coded", bs);
    PutCCP(block_patt->ccp-1,tabmode, bs);
  }
#ifdef COUNT_BITS
  bitCountP->totalBits += 1;
  bitCountP->headerBits += 1;
#endif

#ifdef PROTOCOL
  if (fp_prot_g != NULL && protFormat_g >= 3) {
    if (block_patt->cbpy[0] || block_patt->cbpy[1])
      fprintf(fp_prot_g, "*");
    else
      fprintf(fp_prot_g, " ");
    if (block_patt->cbpc)
      fprintf(fp_prot_g, "c");
    else
      fprintf(fp_prot_g, " ");
  }
#endif

  scale = (float) BScale;

  if (block_patt->cbpy[0]) 
    PutCBP(block_patt->cbpy[0]-1,tabmode, bs);

  for (n=0; n<16; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

    ydec += 16; ypre += 16;
  }

  if (block_patt->cbpy[1]) 
    PutCBP(block_patt->cbpy[1]-1,tabmode, bs);

  for (n=16; n<32; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(ydec,   ypre,   scale, LPImem[n]  , tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(ydec+8, ypre+8, scale, LPImem[n+1], tabmode);

    ydec += 16; ypre += 16;
  }

  /* Smaller step sizes for chroma signals */

  scale = ROUND(BScale*CSCALE);

  /* For chroma signals add one to tabmode */
  tabmode = tabmode + 1;

  if (block_patt->cbpc) 
    PutCBP(block_patt->cbpc-1,tabmode, bs);

  for (n=32; n<40; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n]  , tabmode, bs);
    RecVect(udec,   upre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1], tabmode, bs);
    RecVect(udec+8, upre+8, scale, LPImem[n+1], tabmode);

    udec += 16; upre += 16;
  }

  for (n=40; n<48; n+=2, m++) {

    if (block_patt->comb_shell[m]) 
      PutCombShell(block_patt->comb_shell[m]-1,tabmode, bs);

    SendVect(LPImem[n],  tabmode, bs);
    RecVect(vdec,   vpre,   scale, LPImem[n],   tabmode);
    SendVect(LPImem[n+1],tabmode, bs);
    RecVect(vdec+8, vpre+8, scale, LPImem[n+1], tabmode);

    vdec += 16; vpre += 16;
  }
}
